// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef GAME_H
#define GAME_H

#include <sigc++/signal.h>
#include <glibmm/ustring.h>
#include <sigc++/connection.h>
#include <memory>
#include <list>

#include "rectangle.h"
#include "army.h"
#include "fight.h"
#include "game-parameters.h"
#include "player.h"
#include "defs.h"

class NextTurn;
class GameScenario;
class Stack;
class Player;
class Fight;
class StackTile;

/** Controls a game.
  * 
  * Manages the big and small map, the game scenario and who's turn it is, etc.
  * It's mostly a puppeteer class that connects the various other classes with
  * signals and callbacks. 
  *
  */
class Game
{
 public:
    Game(GameScenario* gameScenario, NextTurn *nextTurn);
    ~Game();

    void end_turn();

    void startGame(); // initiate game flow
    void loadGame();
    void stopGame(); // stop game flow, clean up
    // save current game, returns true if successful
    bool saveGame(Glib::ustring file);

    static GameScenario *getScenario();
    
    // signals
    sigc::signal<void, Player *, unsigned int> next_player_turn;
    sigc::signal<void, Player *> game_loaded;
    sigc::signal<void, Player *> game_over;
    sigc::signal<void, Player *> player_died;
    sigc::signal<void> game_stopped;
    sigc::signal<void, Stack*, std::list<Vector<int> >, Vector<int>, Vector<int> > stack_moves;
    sigc::signal<void, Stack*, std::list<Vector<int> > > raft_moves;
    sigc::signal<void, Fight::Result, int, double, double> fight_started;
    sigc::signal<void, std::list<Army*> >reinforcements_arrived;

    
    void addPlayer(Player *p);

 private:
    static Game *current_game;

    // locks/unlocks the input widgets during computer turns
    void lock_inputs();
    void unlock_inputs();
    void maybe_reinforce();

    //! Called whenever a stack has changed, updates the map etc.
    void stackUpdate(Stack* s);
    //! Called whenever players fight
    void on_fight_started(Fight &fight);
    void stack_arrives_on_tile(Stack *stack, Vector<int> tile);
    void stack_leaves_tile(Stack *stack, Vector<int> tile);
    void on_stack_halted(Stack *stack);
    void on_stack_stopped();
    void on_stack_starts_moving(Stack *stack, std::list<Vector<int> > points, Vector<int> blocked, Vector<int> stopped_short);
    void on_raft_starts_moving(Stack *stack, std::list<Vector<int> > points);

    void on_player_died(Player *player);
    void init_turn_for_player(Player* p);
    void deselect_selected_stack();

    GameScenario* d_gameScenario;
    NextTurn* d_nextTurn;


    bool input_locked;

    std::list<sigc::connection> connections[MAX_PLAYERS + 1];
};

#endif
