// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "config.h"
#include <sstream>
#include <algorithm>
#include <sigc++/functors/mem_fun.h>

#include "playerlist.h"
#include "armysetlist.h"

#include "xmlhelper.h"
#include "history.h"
#include "stacklist.h"
#include "real_player.h"
#include "GameMap.h"

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
#define debug(x)

Glib::ustring Playerlist::d_tag = "playerlist";
Playerlist* Playerlist::s_instance = 0;
Player* Playerlist::d_activeplayer = 0;

Playerlist* Playerlist::getInstance()
{
    if (s_instance == 0)
        s_instance = new Playerlist();

    return s_instance;
}

Playerlist* Playerlist::getInstance(XML_Helper* helper)
{
    if (s_instance)
        deleteInstance();

    s_instance = new Playerlist(helper);

    return s_instance;
}

void Playerlist::deleteInstance()
{
    if (s_instance)
    {
        delete s_instance;
        s_instance = 0;
    }
}

Playerlist::Playerlist()
{
    d_activeplayer = 0;
}

Playerlist::Playerlist(XML_Helper* helper)
{
    d_activeplayer=0;
    //load data. This consists currently of two values: size (for checking
    //the size; we don't use it yet) and active(which player is active)
    //we do it by calling load with playerlist as string
    load(Playerlist::d_tag, helper);

    helper->registerTag(Player::d_tag, sigc::mem_fun(this, &Playerlist::load));
}

Playerlist::~Playerlist()
{
    iterator it = begin();
    while (!empty())
        it = flErase(it);
}

bool Playerlist::checkPlayers()
{
    bool last = false;
    bool dead = false;
    debug("checkPlayers()");
    iterator it = begin ();
	  
    while (it != end ())
    {
        debug("checkPlayers() iter");
        if ((*it)->isDead() || (*it)->isImmortal())
        {
            debug("checkPlayers() dead?");
            it++;
            continue;
        }

        if ((*it)->getStacklist()->hasGeneral() == false)
        {
            debug("checkPlayers() city?");
            iterator nextit = it;
            nextit++;

            (*it)->kill();
	    if (getNoOfPlayers() == 1)
	      last = true;
            splayerDead.emit(*it);
            dead = true;
	    if (last)
	      break;

            it = nextit;    // do this at the end to catch abuse of invalid it
        } else {
            debug("checkPlayers() inc");
            ++it;
        }
    }
  return dead;
}

void Playerlist::nextPlayer()
{
    debug("nextPlayer()");

    iterator it;

    if (!d_activeplayer)
    {
        it = begin();
    }
    else
    {
        for (it = begin(); it != end(); ++it)
        {
            if ((*it) == d_activeplayer)
            {
                it++;
                break;
            }
        }
    }

    // in case we have got a dead player, continue iterating. This breaks
    // if we have ONLY dead players which we assume never happens.
    while ((it == end()) || ((*it)->isDead()))
    {
        if (it == end())
        {
            it = begin();
            continue;
        }
        it++;
    }

    d_activeplayer = (*it);
    debug("got player: " <<d_activeplayer->getName())
}

Player* Playerlist::getPlayer(guint32 id) const
{
  IdMap::const_iterator it = d_id.find(id);
  if (it == d_id.end())
    return NULL;
  return (*it).second;
}

guint32 Playerlist::getNoOfPlayers() const
{
    unsigned int number = 0;

    for (const_iterator it = begin(); it != end(); it++)
    {
        if (!(*it)->isDead())
            number++;
    }

    return number;
}

Player* Playerlist::getFirstLiving() const
{
    for (const_iterator it = begin(); ; it++)
        if (!(*it)->isDead())
            return (*it);
}

bool Playerlist::save(XML_Helper* helper) const
{
    //to prevent segfaults
    if (!d_activeplayer)
        d_activeplayer = (*begin());

    bool retval = true;

    retval &= helper->openTag(Playerlist::d_tag);
    retval &= helper->saveData("active", d_activeplayer->getId());

    for (const_iterator it = begin(); it != end(); it++)
        retval &= (*it)->save(helper);

    retval &= helper->closeTag();

    return retval;
}

void Playerlist::add(Player *player)
{
  push_back(player);
  d_id[player->getId()] = player;
}

bool Playerlist::load(Glib::ustring tag, XML_Helper* helper)
{
    static guint32 active = 0;

    if (tag == Playerlist::d_tag) //only called in the constructor
    {
        helper->getData(active, "active");
        return true;
    }

    if (tag != Player::d_tag)
        return false;

    Player* p = Player::loadPlayer(helper);
    if(p == 0)
        return false;

    //insert player...
    add(p);

    if (p->getId() == active)
      d_activeplayer = p;

    return true;
}

Playerlist::iterator Playerlist::flErase(Playerlist::iterator it)
{
    delete (*it);
    return erase (it);
}

guint32 Playerlist::countPlayersAlive () const
{
  guint32 numAlive = 0; 

  for (const_iterator it = begin (); it != end (); it++)
    {
      if ((*it)->isDead () == true)
	continue;
      numAlive++;
    }
  return numAlive;
}

bool Playerlist::isEndOfRound() const
{
  //check to see if all players have moved this round.
  //do all players have the same number of history:end_turn events?
  if (d_activeplayer == NULL)
    return false;
  guint32 count = d_activeplayer->countEndTurnHistoryEntries();
  for (const_iterator it = begin(); it != end(); it++)
    {
      if (*it == d_activeplayer)
	continue;
      if (count != (*it)->countEndTurnHistoryEntries())
	return false;
    }
  return true;
}

void Playerlist::clearAllActions()
{
  for (iterator it = begin(); it != end(); it++)
    (*it)->clearActionlist();
}
