/* Move all of the deletes and renames to their final resting place.
   There are versions for exact and inexact patching.

  Copyright (C) 2003, 2004 Walter Landry
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA */

#include "boost/filesystem/operations.hpp"
#include "boost/filesystem/convenience.hpp"
#include "Moved_Path.hpp"
#include <list>
#include "move_original_away.hpp"
#include "check_symlink_hierarchy.hpp"
#include "Command_Info.hpp"
#include "Checksums.hpp"

using namespace std;
using namespace boost;
namespace fs=boost::filesystem;
using fs::path;

extern void unsafe_move_list
(const path &root,
 const list<pair<pair<path,path>,string> > &move_list);

/* Generic version for exact and inexact patching. */

void move_to_destination(list<Moved_Path> &moved, const path &deleted_path,
                         const path &source,
                         list<pair<string,string> > &move_conflicts,
                         const Checksums &checksums, bool exact_patching)
{
  /* Sort the moved list so that topmost destinations come first. */

  moved.sort(Moved_Path_destination_cmp);

  /* Now move everything (adds, renames and deletes). */

  list<pair<pair<path,path>,string> > move_list;
  for(list<Moved_Path>::iterator i=moved.begin(); i!=moved.end(); ++i)
    {
      if(!exact_patching)
        {
          /* If the file is going into a directory, that directory may
             already exist as a symlink or file.  So we need to move that
             symlink or file out of the way. */
          
          list<path> checked_list;
          path checked_path(i->destination.branch_path());
          while(!checked_path.empty())
            {
              checked_list.push_front(checked_path);
              checked_path=checked_path.branch_path();
            }
          
          for(list<path>::iterator j=checked_list.begin();
              j!=checked_list.end(); ++j)
            {
              if(lexists(source/(*j)) && (symbolic_link_exists(source/(*j))
                                          || !is_directory(source/(*j))))
                {
                  move_original_away(source,source/(*j),move_conflicts,
                                     checksums);
                }
            }
        }

      /* This check makes sure that we are not patching through a
         symlink.  It isn't really required for inexact patching,
         since any symlink directories should have been already
         removed.  But it is needed for exact patching, and it makes
         me sleep better in any case. */
      check_symlink_hierarchy(source,source/i->destination);

      if(Command_Info::verbosity>=verbose)
        cout << "Creating directories: "
             << (i->destination.branch_path()).native_file_string() << endl;

      fs::create_directories((source/i->destination).branch_path());
      if(!exact_patching)
        {
          if(lexists(source/i->destination))
            {
              move_original_away(source,source/i->destination,move_conflicts,
                                 checksums);
            }
        }

      if(Command_Info::verbosity>=verbose)
        cout << "Renaming: " << (deleted_path/i->initial).native_file_string()
             << "\t" << i->destination.native_file_string() << endl;

      /* Need to prune out the deleted files and files that had add
         conflicts so we don't try to move their inventory id */

      if(i->initial.leaf().substr(0,8)!=",,delete"
         && !i->inventory_id.empty())
        {
          move_list.push_back(make_pair(make_pair(deleted_path/i->initial,
                                                  i->destination),
                                        i->inventory_id));
        }
      rename(source/deleted_path/i->initial,source/i->destination);
    }
  unsafe_move_list(source,move_list);
}

/* Wrapper versions for the exact and inexact patching cases. */

void move_to_destination(list<Moved_Path> &moved, const path &deleted_path,
                         const path &source,
                         list<pair<string,string> > &move_conflicts,
                         const Checksums &checksums)
{
  move_to_destination(moved,deleted_path,source,move_conflicts,checksums,
                      false);
}

void move_to_destination(list<Moved_Path> &moved, const path &deleted_path,
                         const path &source)
{
  list<pair<string,string> > move_conflicts;
  Checksums checksums;
  move_to_destination(moved,deleted_path,source,move_conflicts,checksums,
                      true);
}
