<?php
/*
	This file is part of Astairs.

    Astairs is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    Astairs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Astairs.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2006, 2007, 2008, 2009 AFIDE
    
	Auteurs:

	Said Ladjani <sladjani@yahoo.fr>
	Remi Cocquet <remi.cocquet@gmail.com>

*/


/**
 * Obtenir la liste des bases du server en cours de connection
 * @return string retourne la requete a executer pour rcuprer la liste des bases de donnes
 */
function getDataBases (){
	$sql = "
		SELECT pdb.datname AS datname,
			(SELECT spcname FROM pg_catalog.pg_tablespace pt WHERE pt.oid=pdb.dattablespace) AS tablespace
		FROM pg_database pdb, pg_user pu
		WHERE pdb.datdba = pu.usesysid
		AND pdb.datallowconn
		ORDER BY pdb.datname";
 	return $sql;
}

/**
 * Obtenir la liste des schemas de la base a la quelle on est connecte
 * @return string retourne la requete a executer pour rcuprer la liste des schemas
 */
function getSchemas() {
	$where = "WHERE nspname NOT LIKE 'pg\\\\_%' AND nspname NOT LIKE 'information_schema'";
	$sql = "
		SELECT pn.nspname,
			pu.usename AS nspowner,
			pg_catalog.obj_description(pn.oid, 'pg_namespace') AS nspcomment
		FROM pg_catalog.pg_namespace pn LEFT JOIN pg_catalog.pg_user pu ON (pn.nspowner = pu.usesysid)
		{$where} ORDER BY nspname";
	 
	return $sql;
}

/**
 * Obtenir toutes les fonctions d'une base
 * @param string $schema schma  utiliser
 * @return string requte permetant le retour de la liste des fonctions avec leurs proprits
 */
function getFunctions($schema = 'public') {
	//choix du schma
	$where = "n.nspname = '$schema'";

	//construction de la requte
	$sql = "
				SELECT
					p.proname,
					p.oid AS prooid,
					pl.lanname AS prolanguage,
					pg_catalog.format_type(p.prorettype, NULL) AS proresult,
					prosrc AS prodefinition,
					p.proretset,
					proisstrict,
					provolatile,
					prosecdef,
					pg_catalog.oidvectortypes(p.proargtypes) AS proarguments,
					pg_catalog.obj_description(p.oid, 'pg_proc') AS procomment,
					p.proname || ' (' || pg_catalog.oidvectortypes(p.proargtypes) || ')' AS proproto,
					CASE WHEN p.proretset THEN 'setof ' ELSE '' END || pg_catalog.format_type(p.prorettype, NULL) AS proreturns,
					u.usename AS proowner
				FROM
					pg_catalog.pg_proc p
				INNER JOIN pg_catalog.pg_namespace n ON n.oid = p.pronamespace
				INNER JOIN pg_catalog.pg_language pl ON pl.oid = p.prolang
				LEFT JOIN pg_catalog.pg_user u ON u.usesysid = p.proowner
				WHERE NOT p.proisagg
				AND {$where}
				ORDER BY p.proname, proresult";
	return $sql;
}

/**
 * Obtenir la liste des types n'appartenant pas aux types systme
 * @param string $schema shema utilis pour le retour des types
 * @return string requte sql retournant les types du schma pass en paramtre
 */
function getTypes($schema = 'public') {
	//dfinition du shma  utiliser
	$where = "n.nspname = '{$schema}'";
	//pas d'affichage des fonctions systme
	$where2 = "AND c.relnamespace NOT IN (SELECT oid FROM pg_catalog.pg_namespace WHERE nspname LIKE 'pg\\\\_%')";

	// Create type filter
	$tqry = "'c'";

	//cration de la requte
	$sql = "
				SELECT
					t.typname AS basename,
					pg_catalog.format_type(t.oid, NULL) AS typname,
					pu.usename AS typowner,
					t.typtype,
					pg_catalog.obj_description(t.oid, 'pg_type') AS typcomment
				FROM (pg_catalog.pg_type t
					LEFT JOIN pg_catalog.pg_namespace n ON n.oid = t.typnamespace)
					LEFT JOIN pg_catalog.pg_user pu ON t.typowner = pu.usesysid
				WHERE (t.typrelid = 0 OR (SELECT c.relkind IN ({$tqry}) FROM pg_catalog.pg_class c WHERE c.oid = t.typrelid {$where2}))     
				AND t.typname !~ '^_'
				AND {$where}            
				ORDER BY typname";
	return $sql;
}

/**
 * Creates a new function.
 * @param $funcname The name of the function to create
 * @param $args A comma separated string of types
 * @param $returns The return type
 * @param $definition The definition for the new function
 * @param $language The language the function is written for
 * @param $flags An array of optional flags
 * @param $setof True if it returns a set, false otherwise
 * @param $replace (optional) True if OR REPLACE, false for normal
 * @return 0 success
 */
function createFunction($proto, $returns='returns', $language='pgsql', $volatile = 'v', $isstrict = 'f', $secdef = 'f', $definition='definition', $owner = 'pgsql',$comment, $schema='public', $replace = true) {
//	$sql = "
//
//--
//-- Name : FUNCTION {$schema}.{$proto};
//--";
	$sql .= "
CREATE";
	if ($replace)
		$sql .= " OR REPLACE";
	$sql .= " FUNCTION {$schema}.{$proto}";

	// For some reason, the returns field cannot have quotes...
	$sql .= "
	RETURNS {$returns} AS";
	$sql .= "
\$BODY\$".$definition . "\$BODY\$";

	$sql .= "
	LANGUAGE '{$language}'";
	
	switch ($volatile){
		case "s" :
			$sql .= " STABLE";
			break;
		case "i" :
			$sql .= " IMMUTABLE";
			break;
		default :
			$sql .= " VOLATILE";
	}
	if($isstrict = 't')
		$sql .= "
	RETURNS NULL ON NULL INPUT";
	
	if($secdef = 'v')
		$sql .= "
	SECURITY DEFINER;";
	$sql .= "
ALTER FUNCTION {$schema}.{$proto} OWNER TO $owner;";
	if($comment != '')
	$sql .= "
COMMENT ON FUNCTION {$schema}.{$proto} IS '".clean($comment)."';";
	
	return $sql;
}

/**
 * Creates a new type
 * @param ...
 * @return 0 success
 */
function createType($typname, $typin, $typout, $typlen, $typdef, $typelem, $typdelim, $typbyval, $typalign, $typstorage) {
	$this->fieldClean($typname);
	$this->fieldClean($typin);
	$this->fieldClean($typout);

	$sql = "
		CREATE TYPE \"{$typname}\" (
		INPUT = \"{$typin}\",
		OUTPUT = \"{$typout}\",
		INTERNALLENGTH = {$typlen}";
	if ($typdef != '')
		$sql .= ", DEFAULT = {$typdef}";
	if ($typelem != '')
		$sql .= ", ELEMENT = {$typelem}";
	if ($typdelim != '')
		$sql .= ", DELIMITER = {$typdelim}";
	if ($typbyval)
		$sql .= ", PASSEDBYVALUE, ";
	if ($typalign != '')
		$sql .= ", ALIGNMENT = {$typalign}";
	if ($typstorage != '')
		$sql .= ", STORAGE = {$typstorage}";

	$sql .= ")";

	return $sql;
}

/**
 * Cleans (escapes) a string
 * @param $str The string to clean, by reference
 * @return The cleaned string
 */
function clean(&$str) {
    if ($str === null) return null;
    $str = str_replace("\r\n","\n",$str);
    if (function_exists('pg_escape_string'))
        $str = pg_escape_string($str);
    else
        $str = addslashes($str);
    return $str;
}
?>
