<?php
/*
	This file is part of Astairs.

    Astairs is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    Astairs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Astairs.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2006, 2007, 2008, 2009 AFIDE
    
	Auteurs:

	Said Ladjani <sladjani@yahoo.fr>
	Remi Cocquet <remi.cocquet@gmail.com>

*/

/**
 * Dfinition d'un DoEtape.
 * Un DoEtape est l'objet de base servant  gerer l'avancement des
 * Etapes d'un parcours.
 * Ainsi, chaque objet d'avancement des diffrents types d'Etapes 
 * d'un parcours doivent hriter de cette classe.
 * 
 * @package etape
 */
class DoEtape extends DoNode{
	
	/**
	 * @var integer Idendifiant de la liste chaine dans laquelle s'intgre l'Etape.
	 */
	var $entity;
	
	/**
	 * Constructeur
	 * 
	 * Creation de l'objet et initialisation
	 * de ses attributs hrits.
	 * 
	 * @param integer $formation la formation de l'individu
	 * @param integer $individu l'individu qui effectue la formation
	 */
	function DoEtape($formation,$individu){
		messageDebug("DoEtape::DoEtape($formation,$individu)");
		parent::DoNode($formation, $individu);
	}
	
	/**
	 * Initialisation.
	 * Initialise les attributs propres a l'objet.
	 * 
	 * @param integer $parcours Identifiant du parcours courant.
	 * @param integer $etape Identifiant de l'tape courante.
	 * @param integer $type Identifiant du type de l'tape courante.
	 */
	function initDoEtape($parcours,$etape, $type){
		messageDebug("DoEtape::initDoEtape($parcours,$etape, $type)");
		parent::initDoNode($parcours,$etape, $type);
	}
	
//-----------------------------------------------------
//			METHODES PUBLIQUES
//-----------------------------------------------------
	
	/**
	 * Dbut de l'tape.
	 * Initialise l'avancement de l'tape courante.
	 * Cration de l'identifiant rsultat et enregistrement de la date de dbut de l'tape.
	 * Si l'identifiant rsultat existe dj, une erreur est gnre.
	 * 
	 * @return integer idenfiant rsultat en base de donnes
	 */
	function beginCurrentEtape(){
		messageDebug("DoEtape::beginCurrentEtape()");
		$res_resultat=get_resultat_etape($this->getFormation(), parent::getIndividu(), $this->getParcours(), $this->getEtape());
		if(pg_num_rows($res_resultat)==1){
			$row_resultat=pg_fetch_row($res_resultat);
			$id_resultat=$row_resultat[0];
			$debut_resultat=$row_resultat[5];
			$duree_resultat=$row_resultat[6];
		}
		if(!empty($id_resultat)){
			if(!empty($duree_resultat)){
				die("Erreur DoEtape::beginCurrentEtape(".$this->getEtape()."): l'tape a dj t commence et est dj termine.<br/>");
			}else{
				die("Erreur DoEtape::beginCurrentEtape(".$this->getEtape()."): l'tape a dj t commence et n'est pas termine.<br/>");				
			}
		}else{
			$this->resultat=set_resultat_debut($this->getFormation(), parent::getIndividu(), $this->getParcours(), $this->getEtape() );
			$row_resultat=get_resultat($this->resultat);
			$this->debut=$row_resultat[5];
			$this->duree=$row_resultat[6];//ajout Rmi
		}
		return $this->resultat;
	}

	/**
	 * Construction des objets associs  l'avancement.
	 * L'Objet associ  l'avancement, de type Etape, est construit  partir
	 * de l'identifiant Resultat cr par la methode beginCurrentEtape(). Elle doit
	 * donc avoir t appelle auparavent. Si non, une erreur est gnre.
	 * L'objet Resultat, associ au type de l'Etape, est lui aussi construit. Vide, il devra etre correctement
	 * initialis afin d'etre ensuite pass en paramettres de la methode endCurrentEtape().
	 * 
	 * @param object NodeFactory &$obj_NodeFactory La fabrique d'objet Etape.
	 * @return object Node  L'objet correspondant  l'tape (au DoNode) 
	 */
	function &performCurrentEtape(&$obj_EtapeFactory){
		messageDebug("DoEtape::performCurrentEtape($obj_NodeFactory)");
		parent::perform($obj_EtapeFactory);
		if(!empty($this->resultat)&&is_object($obj_EtapeFactory)){
			$this->obj_node=&$obj_EtapeFactory->createEtape($this);
			$this->obj_resultat= ResultatEtapeFactory::createResultatEtape($this->getType());
		}else{
			die("Erreur DoEtape::performCurrentEtape(".$obj_EtapeFactory."): L'tape n'a pas t commence.<br>");
		}
		return $this->obj_node;
	}

	/**
	 * Termine l'tape.
	 * Enregistre la date de fin de l'tape et detruit les attributs de l'objet inutiles (resultat et obj_node). 
	 * Cette mthode ncssite un identifiant Resultat, et donc un appel prcdant  la methode beginCurrentEtape().
	 * Elle ncessite aussi un objet Resultat en parametre, qui est obtenu par la methode getCurrentResultat()
	 * 
	 * @param object ResultatEtape $obj_resultat : Resultat associ  l'objet associ  l'avancement.
	 * @return boolean true ou false suivant le succs ou non de la mthode.
	 */
	function endCurrentEtape($obj_resultat=null){
		messageDebug("DoEtape::endCurrentEtape($obj_resultat)");
		$res_parent=parent::end();
		if($res_parent&&!empty($this->resultat)){
			$this->duree=set_resultat_fin($this->resultat);
			if(!$this->duree){
				die("Erreur DoEtape::endCurrentEtape(): L'tape n'a pas t termine.<br>");
				return false;
			}else{
				$row_resultat=get_resultat($this->resultat);
				$this->duree=$row_resultat[6];	
				
				return true;
			}
			
		}
		die("Erreur DoEtape::endCurrentEtape(): L'tape ne peut tre termine car elle n'a pas t commence.<br>");
		return false;
		
	}	

	/**
	 * Indique si l'tape est finie ou non.
	 * @return boolean TRUE ou FALSE.
	 */
	function isEnd(){
		if(!$this->isBegin())return false;
		$row_resultat=get_resultat($this->getResultat());
//		print_rr($row_resultat);
		if(empty($row_resultat[6]))return false;
		else return true;	
	}

	/**
	 * Indique si l'tape est commence ou non.
	 * @return boolean TRUE ou FALSE.
	 */
	function isBegin(){
		if(empty($this->resultat))return false;
		
		$row_resultat=get_resultat($this->getResultat());
		if(empty($row_resultat[5]))return false;
		else return true;	
	}	
	
	/**
	 * Obtenir le status courrant de l'Etape.
	 * Le status est obtenu  partir des attributs de l'objet.
	 * L'tape peut avoir les status suivants:
	 * DO_ETAPE_NONE, DO_ETAPE_BEGIN et DO_ETAPE_END.
	 */
	function getCurrentStatus(){
		$status=DO_ETAPE_NONE;
		if(!empty($this->resultat)){
			$row_resultat=get_resultat($this->resultat);
			if(!empty($row_resultat[5])){
				$status=DO_ETAPE_BEGIN;
				if(!empty($row_resultat[6]))$status=DO_ETAPE_END;
			}	
		}
		return $status;	
	}
	
	/**
	 * Calculer et obtenir le status de l'Etape.
	 * Le status est obtenu  partir des donnes en base.
	 * L'tape peut avoir les status suivants:
	 * DO_ETAPE_NONE, DO_ETAPE_BEGIN et DO_ETAPE_END.
	 */
	 function getStatus(){
		$status=DO_ETAPE_NONE;
		$res_resultat=get_resultat_etape($this->getFormation(), parent::getIndividu(), $this->getParcours(), $this->getEtape());
		if(pg_num_rows($res_resultat)==1){
			$row_resultat=pg_fetch_row($res_resultat);
			$id_resultat=$row_resultat[0];
			$duree_resultat=$row_resultat[6];
		}
		if(!empty($id_resultat)){
			$status=DO_ETAPE_BEGIN;
			if(!empty($duree_resultat)){
				$status=DO_ETAPE_END;
			}
		}
		return $status;	
	}
	
	/**
	 * Obtenir l'objet source correspondant  l'tape.
	 * Contrairement  la mthode &getObjEtape(), l'objet retourn n'est pas celui li  l'avancement, cr par la methode perfom(),
	 * mais un objet nouvellement contruit.
	 * @param object EtapeFactory &$obj_EtapeFactory La fabrique pour constuire l'objet retourn
	 * @return object Etape l'objet correspondant  l'tape
	 */		
	function getSrcEtape(&$obj_EtapeFactory){
		return $obj_EtapeFactory->createEtape($this);
	}
	
	/**
	 * Comparer 2 objets DoEtape.
	 * Si les attributs element, individu, node, node_type et entity de l'objet sont
	 * identiques  ceux de l'objet pass en parametre, la mthode retourne true, false sinon.
	 * 
	 * @param object DoEtape L'objet auquel on compare le DoEtape.
	 * @return boolean true si les principaux attributs sont identiques, false sinon. 
	 */
	function compareTo($obj_DoEtape){
		//echo "coucou<br>";
		if(!is_object($obj_DoEtape))return false;
		
		
		
		if(
			$this->element == $obj_DoEtape->element &&
			$this->individu == $obj_DoEtape->individu &&
			$this->node == $obj_DoEtape->node &&
			$this->node_type == $obj_DoEtape->node_type &&
			$this->entity == $obj_DoEtape->entity 
		){
			return true;
		}
		else{
			return false;
		}
	}

//-----------------------------------------------------
//			ACCESSEURS
//-----------------------------------------------------

	/**
	 * Obtenir l'identifiant du parcours pourlequel cet objet d'avancement est utilis.
	 * @return integer identifiant du parcours
	 */
	function getParcours(){
		return parent::getEntity();
	}
	
	/**
	 * Obtenir l'identifiant de la formation pour laquelle cet objet d'avancement est utilis.
	 * @return integer identifiant de la formation
	 */
	function getFormation(){
		return parent::getElement();
	}

	/**
	 * Obtenir le type de l'tape 
	 * @return integer le type de l'tape 
	 */	
	function getType(){
		return parent::getNodeType();
	}
	
	/**
	 * 
	 * Obtenir l'identifiant de l'tape 
	 * @return integer l'identifiant de l'tape 
	 */		
	function getEtape(){
		return parent::getNode();
	}

	/**
	 * Obtenir l'objet correspondant  l'tape 
	 * Contrairement  la mthode &getSrcEtape(), l'objet retourn est pas celui li  l'avancement, cr par la methode perfom().
	 * @return object Etape l'objet correspondant  l'tape
	 */		
	function &getObjEtape(){
		return parent::getObjNode();
	}
	
	/**
	 * Obtenir l'identifiant resultat en base de donnes.
	 * @return integer l'identifiant resultat en base de donnes.
	 */	
	function getResultat(){
		return $this->resultat;
	}

	/**
	 * Obtenir l'objet Resultat correspondant  l'Etape en cours.
	 * @return object Resultat l'identifiant resultat en base de donnes.
	 */		
	function &getCurrentResultat(){
		return parent::	getCurrentResultat();
	}
	
	/**
	 * Indique si l'Etape  donn lieu  une correction.
	 * Retourne toujours false, car une tape n'est pas corrige par dfaut.
	 * @return boolean true si l'Etape  t corrige, false sinon.
	 */
	function isCorrected(){
		return false;
	}
	
	/**
	 * Indique si l'Etape est une tentative.
	 * @return boolean true si l'Etape  est une tentative, false sinon.
	 */
	function isTentative(){
		return false;
	}	
}
?>
