<?php
/*
	This file is part of Astairs.

    Astairs is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    Astairs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Astairs.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2006, 2007, 2008, 2009 AFIDE
    
	Auteurs:

	Said Ladjani <sladjani@yahoo.fr>
	Remi Cocquet <remi.cocquet@gmail.com>

*/

/**
 * Fonctions d'appel au fonctions PL/PGSQL pour les formations.
 * 
 * @package formation
 * @modulegroup plpgsql
 * @module plpgsql_formation
 */
include ("$root/lib/formation/plpgsql_deprecated_formation.inc");

/**
 * Obtenir le formateur d'une formation. (OLD)
 * Recupere l'identifiant et la chaine 'nom prenom' du formateur.
 * 
 * @param integer  l'identifiant de la formation
 * @return integer resultat de la requete :(id int4,nom_prenom text)
 */
function get_formation_formateur_old($formation) {
	$req = "select distinct num_user, textcat( textcat(nom, text ' '), prenom)
						from UTILISATEURS
						where num_user in (
							select distinct g_num_user
							from gestionnaire
							where
							g_num_cal=$formation
							);";
	return (pg_query($req));
}
 
/**
 * Obtenir le formateur d'une formation. 
 * Recupere l'identifiant et la chaine 'nom prenom' du formateur.
 * 
 * @param integer  l'identifiant de la formation
 * @return integer resultat de la requete :(id int4,nom_prenom text)
 */
function get_formation_formateur($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_get_formation_formateur($formation) as (id int4,nom_prenom text, formation int4)";
		$res = pg_query($sql);
	} else {
		$res = get_formation_formateur_old($formation);
	}
	return $res;
}

/**
 * Obtenir la classe d'une formation (OLD)
 * 
 * Recupere l'identifiant et le libelle.
 * 
 * @param integer :  l'identifiant de formation
 * @return integer : resultat requete :(id int4,libelle varchar)
 */
function get_formation_classes_old($formation) {
	$req = "	select id,nom from classe 
		    			where id in (	select distinct g_num_gest_g from GESTIONNAIRE where g_num_cal=$formation)";
	//echo "$req<br>";		    			
	return pg_query($req);
}

/**
 * Obtenir la classe d'une formation 
 * 
 * Recupere l'identifiant et le libelle.
 * 
 * @param integer :  l'identifiant de formation
 * @return integer : resultat requete :(id int4,libelle varchar)
 */
function get_formation_classes($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_get_formation_classes($formation) as (id int4,libelle varchar,pere int4,min int4,max int4)";
		$res = pg_query($sql);
	} else
		$res = get_formation_classes_old($formation);
	return $res;
}

/**
 *  Obtenir les informations d'une formation.(OLD)
 *
 * @param  integer : le numero de la formation
 * @return  integer : l'entier resultat de la requete (id int4, libelle varchar, parcours int4, date_debut date, date_fin date, langue_defaut int4)
 */
function get_formation_old($formation) {
	$req = "SELECT DISTINCT num_cal AS id, nom_cal AS libelle, g_num_parcours as parcours, d1.jour AS date_debut, d2.jour AS date_fin, split_part(langue,';',1) AS langue_defaut
		        FROM calendrier, gestionnaire, dates d1, dates d2
		        WHERE num_cal=g_num_cal
		        AND num_cal=$formation
		        AND datedeb=d1.num_date AND datefin=d2.num_date;";
	//echo $req."<BR>";
	return pg_query($req);
}

/**
 *  Obtenir les informations d'une formation.
 *
 * @param  integer : le numero de la formation
 * @return  integer : l'entier resultat de la requete :(id int4, libelle varchar, parcours int4, date_debut date, date_fin date, langue_defaut int4)
 */
function get_formation($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_get_formation($formation) AS (id int4, libelle varchar, parcours int4, date_debut date, date_fin date, langue_defaut int4,ressources_fusionnees bool )";
//		echo "sql=$sql<br>";
		$res = pg_query($sql);
	} else
		$res = get_formation_old($formation);
	return $res;
}

/**
	Obtenir l'alerte pour une date de formation (OLD)
	@param integer : numero formation
	@param date : date de formation
	@return integer : resultat requete (id,libelle,formation,date,jours,texte)
*/
function get_formation_alerte_old($formation,$date){
	$sql="select A.num_alert,A.intitule,A.num_cal,D.jour ,A.periode,A.texte 
				from alert A,dates D where num_cal=$formation and 
				A.num_date=D.num_date and D.jour='$date'";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}
 /**
	Obtenir l'alerte pour une date de formation 
	@param integer : numero formation
	@param date : date de formation
	@return integer : resultat requete (id,libelle,formation,date,jours,texte)
*/
function get_formation_alerte($formation,$date){
	if (VERSION_BASE > VERSION) {
		//$datetimestamp=date_to_timestamp($date);
		$sql="select * from contraintes.af_get_formation_alerte($formation,'$date') as 
					(id int4,libelle varchar,formation int4,date date,jours int4,texte text)";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		$res= get_formation_alerte_old($formation,$date);
	return $res;
}

/**
 *    Obtenir les alertes d'une formation (OLD)
 *
 *   @param integer : le numero de formation
 *   @return interger : le resultat de la requete
 */
function get_formation_alertes_old($formation) {
	$sql = "select num_alert,intitule,num_cal,jour,periode,texte from alert A,dates D where num_cal=$formation and A.num_date=D.num_date order by jour";
	return (pg_query($sql));
}

/**
 *    Obtenir les alertes d'une formation
 *
 *   @param integer : le numero de formation
 *   @return interger : le resultat de la requete
 */
function get_formation_alertes($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_get_formation_alertes($formation) as  
					(id int4,libelle varchar,formation int4,date date,jours int4,texte text) order by date";
		$res = pg_query($sql);
	} else
		$res = get_formation_alertes_old($formation);
	return $res;
}

/**
    Obtenir les alertes d'une formation pour une p�riode donn�e (OLD)
*
*   @param integer : le numero de formation
*   @param date : le debut de p�riode
*   @param date : la fin de p�riode
*   @return integer : le resultat de la requete
*/
function get_formation_alertes_periode_old($formation,$datedebut,$datefin){
    $sql="SELECT D.jour FROM dates D,alert A
                WHERE D.num_date=A.num_date
                AND A.num_cal=$formation
                AND  D.jour BETWEEN '$datedebut' AND   '$datefin'";
	$sql="select A.num_alert,A.intitule,A.num_cal,D.jour,A.periode,A.texte from dates D,alert A
				WHERE D.num_date=A.num_date
                AND A.num_cal=$formation
                AND  D.jour BETWEEN '$datedebut' AND   '$datefin'";
    $res=pg_query($sql);
    return $res;
}
/**
    Obtenir les alertes d'une formation pour une p�riode donn�e (OLD)
*
*   @param integer : le numero de formation
*   @param date : le debut de p�riode
*   @param date : la fin de p�riode
*   @return integer : le resultat de la requete
*/
function get_formation_alertes_periode($formation,$datedebut,$datefin){
    if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_alertes_periode($formation,'$datedebut','$datefin') as 
					(id int4,libelle varchar,formation int4,date date,jours int4,texte text)";
		return pg_query($sql);
    }
    else
        $res = get_formation_alertes_periode_old($formation,$datedebut,$datefin);
    return $res;
}

/**
 *   Obtenir les  espaces d'une formation. (OLD)
 *
 *   @param integer : le numero de la formation
 *   @param integer : le resultat de la requete  (id int4, libelle varchar, int4 formation)
 */
function get_formation_espaces_old($formation) {
	$sql = "select num_espace, titre, $formation from ressource_esp where num_espace in (select num_espace from ressource_gest where num_cal=$formation)";
	$res = pg_query($sql);
	return $res;
}

/**
 *   Obtenir les  espaces d'une formation.
 *
 *   @param integer : le numero de la formation
 *   @param integer : le resultat de la requete  (id int4, libelle varchar, int4 formation)
 */
function get_formation_espaces($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_espaces($formation) AS (id int4, libelle varchar, int4 formation)";
		$res = pg_query($sql);
	} else
		$res = get_formation_espaces_old($formation);

	return $res;
}

/**
 *   Obtenir les sections d'un espace pour une formation (OLD)
 *
 *   @param integer : le numero de formation
 *   @param integer : le numero de l'espace
 *   @return integer : le resultat de la requete (id int4, libelle varchar, icone int4, formation int4, espace int4)
 */
function get_formation_sections_espace_old($formation, $espace) {
	$sql = "select num_section, titre, icone, $formation,$espace from ressource_sec where num_section in
	            (select num_section from ressource_gest where num_cal=$formation and num_espace=$espace)";
	$res = pg_query($sql);
	return $res;
}

/**
 *   Obtenir les sections d'un espace pour une formation
 * 
 *   @param integer : le numero de formation
 *   @param integer : le numero de l'espace
 *   @return integer : le resultat de la requete (id int4, libelle varchar, icone int4, formation int4, espace int4)
*/
function get_formation_sections_espace($formation, $espace) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_sections_espace($formation,$espace) AS (id int4, libelle varchar, icone int4, formation int4, espace int4)";
		$res = pg_query($sql);
	} else
		$res = get_formation_sections_espace_old($formation, $espace);
	return ($res);
}

/**
 *   Obtenir les fichiers ressources d'une section d'un espace pour une formation (OLD)
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 *   @param integer : le numero de section
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_ressources_espace_section_old($formation, $espace, $section) {
	$sql = "select num_gest_res, libelle, num_fic, num_cal, num_espace, num_section, datedeb, datefin, nouveau, motcle, type from ressource_gest  where num_cal=$formation and num_espace=$espace and num_section=$section ";
	$res = pg_query($sql);
	return $res;
}

/**
 *   Obtenir les fichiers ressources d'une section d'un espace pour une formation
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 *   @param integer : le numero de section
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_ressources_espace_section($formation, $espace, $section) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_fichiers_ressources_espace_section($formation,$espace, $section) AS (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute interval, mots_cles text, type int4 )";
		$res = pg_query($sql);
		//echo " </br>$sql</br>";
		return $res;
	} else
		return get_formation_ressources_espace_section_old($formation, $espace, $section);
}

/**
 * Cr�e une formation (OLD)
 *
 * @param varchar : le nom de la formation
 * @param integer : la date de debut
 * @param integer : la date de fin
 * @param integer : l'identifiant de la classe
 * @param integer : l'identifiant du formateur
 * @param integer : l'identifiant du parcours
 * @param integer : l'identifiant de la langue par d�faut
 * @param array : un tableau contenant les identifiants des langues de la formation/
 * 
 * @return integer : le numero de la formation ainsi cr�ee
 */
function set_formation_old($libelle, $debut, $fin, $classe, $formateur, $parcours, $langue_defaut, $langues) {
	$sql = "insert into calendrier(nom_cal,datedeb,datefin,langue) values('$nomcal',$debut,$fin,'$langue');";
	$ins = pg_query($sql);
	$oid = pg_getlastoid($ins);
	$res = pg_query("SELECT num_cal FROM calendrier WHERE OID=$oid;");
	return pg_result($res, 0, 0);
}

/**
 * Cr�e une formation 
 *
 * @param varchar : le nom de la formation
 * @param integer : la date de debut
 * @param integer : la date de fin
 * @param integer : l'identifiant de la classe
 * @param integer : l'identifiant du formateur
 * @param integer : l'identifiant du parcours
 * @param integer : l'identifiant de la langue par d�faut
 * @param array : un tableau contenant les identifiants des langues de la formation/
 * 
 * @return integer : le numero de la formation ainsi cr�ee
 */
function set_formation($libelle, $debut, $fin, $classe, $formateur, $parcours, $langue_defaut, $langues) {
	if (VERSION_BASE > VERSION) {
		$chaine_langues = int_array_to_plpgsql_array($langues);
		$sql = "SELECT * FROM contraintes.af_set_formation(varchar '$libelle', '$debut',  '$fin',$classe, $formateur, $parcours, $langue_defaut, $chaine_langues);";
		echo "$sql<br>";
		$ins = pg_query($sql);
		return pg_fetch_result($ins, 0, 0);
	} else
		return set_formation_old($libelle, $debut, $fin, $classe, $formateur, $parcours, $langue_defaut, $langues);
}

/**
	Mettre � jour une formation (OLD)
	@param integer : formation
	@param varchar : le nom de la formation
	@param date : la date de debut
	@param date : la date de fin
	@param integer : l'identifiant de la langue par d�faut + langues dispo (ex : french;english)
*/
function maj_formation_old($formation,$libelle,$debut,$fin,$langue_defaut){
	$num_debut=get_num_date($debut);
	$num_fin=get_num_date($fin);
	$sql="update calendrier set nom_cal='$libelle',datedeb=$num_debut,datefin=$num_fin,langue='$langue_defaut' 
				where num_cal=$formation";
	//echo "sql=$sql<br>";
	$res=pg_query($sql);
}
/**
	Mettre � jour une formation 
	@param integer : formation
	@param varchar : le nom de la formation
	@param date : la date de debut
	@param date : la date de fin
	@param integer : l'identifiant de la langue par d�faut
*/
function maj_formation($formation,$libelle,$debut,$fin,$langue_defaut){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_maj_formation($formation,'$libelle','$debut','$fin',$langue_defaut)";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		$res=maj_formation_old($formation,$libelle,$debut,$fin,$langue_defaut);
	return $res;
}

/**
 * Obtenir les participants d'une formation. (OLD)
 * Les participants sont les identifiants du formateur et de la classe d'une formation.
 * 
 * @param integer : l'identifiant de la formation
 * @param integer : l'entier r�sultat de la requete (formation int4, formateur int4, classe int4)
 */
function get_formation_participants_old($formation) {
	$sql = "SELECT DISTINCT g_num_cal, g_num_user, g_num_gest_g FROM gestionnaire WHERE g_num_cal=$formation";
	$res = pg_query($sql);
	return $res;
}

/**
 * Obtenir les participants d'une formation.
 * Les participants sont les identifiants du formateur et de la classe d'une formation.
 * 
 * @param integer : l'identifiant de la formation
 * @param integer : l'entier r�sultat de la requete (formation int4, formateur int4, classe int4)
 */
function get_formation_participants($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_get_formation_participants($formation) as (formation int4, formateur int4, classe int4)";
		//echo "sql=$sql<br>";
		$res = pg_query($sql);
	} else
		$res = get_formation_participants_old($formation);
	return $res;
}

/**
* Obtenir les ressources d'une formation (OLD)
* 
* En plus des champs du resultat de la fonction de la nouvelle base,
* cette fonction de substitution retourne le type de la ressource en dernier champ tuple.
* 
* @param integer : l'identifiant de la formation
* @param integer : l'entier r�sultat de la requete (ressource int4, espace int4, section int4, debut_disponibilite date, fin_disponibilite date, duree_nouveaute int4, formation int4, type char)
*/
function get_formation_ressources_old($formation) {
	$sql = "(SELECT G.num_gest_res, F.num_fic as num, T.num_themedir, T.nom_theme, E.num_espace, E.titre as espace, S.num_section, S.titre as section, G.libelle, G.num_cal, G.type,G.motcle
	        FROM ressource_fic F, THEMEDIR T, ressource_sec S, ressource_esp  E, ressource_gest G
	        WHERE F.num_themedir=T.num_themedir
	        AND G.num_fic=F.num_fic
	        AND G.num_section=S.num_section
	        AND G.num_espace=E.num_espace
	        AND G.num_cal=$formation
	        AND G.type='f'
	        )
	        UNION
	        (SELECT G.num_gest_res, F.num_text as num, T.num_themedir, T.nom_theme, E.num_espace, E.titre as espace, S.num_section, S.titre as section, G.libelle, G.num_cal, G.type,G.motcle
	        FROM ressource_txt F, THEMEDIR T, ressource_sec S, ressource_esp  E, ressource_gest G
	        WHERE F.num_themedir=T.num_themedir
	        AND G.num_fic=F.num_text
	        AND G.num_section=S.num_section
	        AND G.num_espace=E.num_espace
	        AND G.num_cal=$formation
	        AND G.type='t'
	        )
	        ORDER BY  nom_theme, espace,section, libelle ;
	        ";
	$res = pg_query($sql);
	return $res;
}
/** Obtenir les infos des ressources d'une formation
	@param integer : numero de formation
	@param integer : les infos r�sultats des ressources
*/
function get_formation_ressources($formation,$all="TRUE") {
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_ressources($formation, $all) 
		as(ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut timestamp,fin timestamp,nouveau interval,icone int4) order by libelle_espace,libelle_section";
//		echo "$sql<br>";
		$res=pg_query($sql);
	} 
	else
		$res = get_formation_ressources_old($formation);
	return $res;
}

/**
	Obtenir les info d'une ressource d'une formation (old)
	@param integer : numero de formation
	@param integer : numero espace
	@param integer : numero section
	@param integer : numero ressource
	@param integer : les infos r�sultats de la ressource
*/
function get_formation_ressource_old($formation,$espace,$section,$ressource){

}
/**
	Obtenir les info d'une ressource d'une formation 
	@param integer : numero de formation
	@param integer : numero espace
	@param integer : numero section
	@param integer : numero ressource
	@param integer : les infos r�sultats de la ressource
*/
function get_formation_ressource($formation,$espace,$section,$ressource){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_ressource($formation,$espace,$section,$ressource) 
		as(fichier_ressource int4,fichier_ressource_bis int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut timestamp,fin timestamp,nouveau interval,icone int4)";
		//echo $sql;
		$res=pg_query($sql); 
	}
	else
		$res=get_formation_ressource_old($formation,$espace,$section,$ressource);
	return $res;
}

	



/**
*   Associer une ressource � une formation (OLD)
* 
* 
*   @param integer : l'identifiant de la formation
*   @param integer : l'identifiant de la ressource
*   @param integer : l'identifiant de l'espace
*   @param integer : l'identifiant de la section
*   @param date : la date de d�but de disponibilite
*   @param date : la date de fin de disponibilite
*   @param integer : la dur�e de nouveaut�
*   @param char :  le type de la ressource (t=texte, f=fichier)
* 
*   @return bool : true ou false (ou t ou f) selon que l'association se soit bien pass�e ou non
*/
function set_formation_ressource_old($formation, $ressource, $espace, $section, $debut, $fin, $nouveaute, $type) {
	$sql = "UPDATE ressource_gest SET num_cal=$formation,num_espace=$espace,num_section=$section,datedeb=$debut,datefin=$fin,nouveau=$nouveaute
	            WHERE num_gest_res=$ressource AND type='$type';";
	$upd = pg_query($sql);
	return $upd;
}

/**
 *   Associer une ressource � une formation
 * 
*   @param integer : l'identifiant de la formation
*   @param integer : l'identifiant de la ressource
*   @param integer : l'identifiant de l'espace
*   @param integer : l'identifiant de la section
*   @param date : la date de d�but de disponibilite
*   @param date : la date de fin de disponibilite
*   @param integer : la dur�e de nouveaut�
* 
*   @return bool : true ou false (ou t ou f) selon que l'association se soit bien pass�e ou non
*/
function set_formation_ressource($formation, $ressource, $espace, $section, $debut, $fin, $nouveaute, $type = "") {
	if (VERSION_BASE > VERSION) {
		$date_deb=str_replace('/','-',$debut);
		$date_fin=str_replace('/','-',$fin);
		
		$sql = "SELECT * FROM contraintes.af_set_formation_ressource($formation, $ressource,'$date_deb','$date_fin','$nouveaute days',$espace,$section);";
		//echo "sql=$sql<br>";
		$upd = pg_query($sql);
		return $upd;
	} else
		$res = set_formation_ressource_old($formation, $ressource, $espace, $section, $debut, $fin, $nouveaute, $type);
	return ($res);
}

/**
 * Obtenir les langues d'une formation.(OLD)
 * 
 * @param integer : l'identifiant de la formation
 * @return integer : le resultat de la requete (id int4,libelle varchar, code varchar, locale varchar, formation int4)
 */
function get_formation_langues_old($formation) {
	$sql = "SELECT * FROM _af_get_formation_langues($formation) AS (id int4, libelle varchar, langue varchar, locale varchar, formation int4) ;";
	$res = pg_query($sql);
	return $res;
}

/**
 * Obtenir les langues d'une formation.
 * 
 * @param integer : l'identifiant de la formation
 * @return integer : le resultat de la requete (id int4,libelle varchar, code varchar, locale varchar, formation int4)
 */
function get_formation_langues($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_langues($formation) AS (id int4,libelle varchar, code varchar, locale varchar, formation int4) ;";
		//echo "sql=$sql<br>";
		$res = pg_query($sql);
		return $res;
	} else
		$res = get_formation_langues_old($formation);
	return $res;
}

/**
 *	Obtenir les droits (du formateur) d'une formation.(OLD)
 *  
 *	@param integer : l'identifiant de la formation
 *	@return integer : le resultat de la requete (id int4,libelle varchar)
 */
function get_formation_droits_old($formation) {
	$sql="select num_droit,libelle_droit from droit where num_droit in 
				(select g_num_droit from gestionnaire where g_num_cal=$formation)";
	return pg_query($sql);
}

/**
 *	Obtenir les droits (du formateur) d'une formation.
 *  
 *	@param integer : l'identifiant de la formation
 *	@return integer : le resultat de la requete (id int4,libelle varchar)
 */
function get_formation_droits($formation) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_get_formation_droits($formation) as (id int4,libelle varchar)";
		$res = pg_query($sql);
	} else
		$res = get_formation_droits_old($formation);
	return $res;
}

/**
 *	Ajoute un droit � une formation. (OLD)
 * 
 *	Le droit est pour le formateur de la formation
 *
 *	@param integer : l'identifiant de la formation
 *	@param integer : numero droit
 *
 *	@return bool : resultat requete
 */
function set_formation_droit_old($formation, $droit,$formateur,$filiere,$classe,$parcours) {
	$sql="insert into gestionnaire (g_num_user,g_num_gest_c,g_num_gest_g,g_num_cal,g_num_droit,g_num_parcours)
         values('$formateur','$filiere','$classe','$formation','$droit','$parcours')";
		 //echo "sql=$sql<br>";
	return pg_query($sql);
}

/**
 *	Ajoute un droit � une formation. 
 * 
 *	Le droit est pour le formateur de la formation
 *
 *	@param integer : l'identifiant de la formation
 *	@param integer : numero droit
 *
 *	@return bool : resultat requete
 */
function set_formation_droit($formation, $droit,$formateur,$filiere,$classe,$parcours) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_set_formation_droit($formation,$droit)";
		$res = pg_query($sql);
	} else
		$res = set_formation_droit_old($formation, $droit,$formateur,$filiere,$classe,$parcours);
	return $res;
}

/**
 *	Supprime un droit � une formation. (OLD)
 * //TODO retourner 't' ou 'f'
 *	Le droit est pour le formateur de la formation
 *
 *	@param integer : l'identifiant de la formation
 *	@param integer : numero droit
 *
 *	@return bool : resultat requete
 */
function del_formation_droit_old($formation, $droit) {
	$sql="delete from gestionnaire where g_num_cal=$formation and g_num_droit=$droit";
	return pg_query($sql);
}

/**
 *	Supprime un droit � une formation
 * 
 *	Le droit est pour le formateur de la formation
 * //TODO retourner 't' ou 'f'
 *	@param integer : l'identifiant de la formation
 *	@param integer : numero droit
 
 *
 *	@return bool : resultat requete
 */
function del_formation_droit($formation, $droit) {
	if (VERSION_BASE > VERSION) {
		$sql = "select * from contraintes.af_del_formation_droit($formation,$droit)";
		$res = pg_query($sql);
	} else
		$res = del_formation_droit_old($formation, $droit);
	return $res;
}

/**
	Obtenir tous les evenements d'une formation  (OLD)
	@param integer : formation
	@return integer : resultat requete (formation,libelle,date,duree,modalite,intervention,intervenant,salle)
*/
function get_formation_evenements_old($formation){
	$sql="select gnum_cal,libelle,
				cast(D.jour as timestamp)+cast(hdeb as time),
				extract(epoch from (cast(hfin as time)-cast(hdeb as time))),
				type,numtype,num_intervenant,num_salle
				FROM dates D,gest_cal g 
				WHERE D.num_date=g.num_date  and demi_jour!='C'
				AND gnum_cal=$formation";
	//echo "sql=$sql<br>";
	return pg_query($sql);									
}
/**
	Obtenir tous les evenements d'une formation  
	@param integer : formation
	@return integer : resultat requete (formation,libelle,date,duree,modalite,intervention,intervenant,salle)
*/
function get_formation_evenements($formation){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_evenements($formation)
				as (formation int4,libelle varchar,date timestamp,duree interval,modalite int4,intervention int4,intervenant int4,salle int4)";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_evenements_old($formation);
	return $res;
}

/**
	Obtenir les evenements d'une formation pour une periode donn�e (OLD)
	@param integer : le numero de formation
	@param date : debut periode
	@param date : fin periode
	@return integer : resultat requete(n�formation,libelle,date,duree,modalite,intervention,intervenant,salle)
*/
function get_formation_evenements_periode_old($formation,$datedeb,$datefin){
	$sql="SELECT gnum_cal,libelle,
				cast(D.jour as timestamp)+cast(hdeb as time),
				extract(epoch from (cast(hfin as time)-cast(hdeb as time))),
				numtype,type,num_intervenant,num_salle
				FROM dates D,gest_cal g 
				WHERE D.num_date=g.num_date
				AND gnum_cal=$formation";
				if(strtotime($datedeb)!=strtotime($datefin))
					$sql .=" AND  D.jour BETWEEN '$datedeb' AND   '$datefin'";
				else
					$sql .="and D.jour='$datedeb'";
	$sql="SELECT gnum_cal,libelle,
				cast(D.jour as timestamp)+cast(hdeb as time),
				cast(hfin as time)-cast(hdeb as time),
				type,numtype,num_intervenant,num_salle
				FROM dates D,gest_cal g 
				WHERE D.num_date=g.num_date
				AND gnum_cal=$formation";
				if(strtotime($datedeb)!=strtotime($datefin))
					$sql .=" AND  D.jour BETWEEN '$datedeb' AND   '$datefin'";
				else
					$sql .="and D.jour='$datedeb'";
	//echo "sql=$sql<br>";
	return pg_query($sql);
	
}
 /**
	Obtenir les evenements d'une formation pour une periode donn�e 
	@param integer : le numero de formation
	@param date : debut periode
	@param date : fin periode
	@return integer : resultat requete(n�formation,libelle,date,duree,modalite,intervention,intervenant,salle)
*/
function get_formation_evenements_periode($formation,$datedeb,$datefin){
	 if (VERSION_BASE > VERSION) {
		// transforme les date en timestamp
		$datedeb=date_to_timestamp($datedeb);
		$datefin=date_to_timestamp($datefin);
		// si cours pour la journ�e en cours
		if($datedeb==$datefin)
			$datefin=$datedeb + 86400;
		$sql="select * from contraintes.af_get_formation_evenements_periode($formation,cast(to_timestamp($datedeb) as timestamp without time zone),cast(to_timestamp($datefin) as timestamp without time zone) )
					as (formation int4,libelle varchar,date timestamp,duree interval,modalite int4,intervention int4,intervenant int4,salle int4)";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	 }
	 else
		$res=get_formation_evenements_periode_old($formation,$datedeb,$datefin);
	 return $res;
}

/**
	Obtenir un �venement d'une formation (OLD)
	@param integer : numero formation
	@param date : la date de l'evenement
	@param time : heure de debut
	@return integer : resultat requete(n�formation,libelle,date,duree,modalite,intervention,intervenant,salle)
*/
function get_formation_evenement_old($formation,$date,$debut){
	$debut=substr($debut,0,-3);
	$sql="select gnum_cal,libelle,
				cast(D.jour as timestamp)+cast(hdeb as time),
				extract(epoch from (cast(hfin as time)-cast(hdeb as time))),
				numtype,type,num_intervenant,num_salle
				FROM dates D,gest_cal g 
				where G.gnum_cal=$formation and D.num_date=G.num_date and D.jour='$date' and hdeb='$debut'";
	$sql="select gnum_cal,libelle,
				cast(D.jour as timestamp)+cast(hdeb as time),
				cast(hfin as time)-cast(hdeb as time),
				numtype,type,num_intervenant,num_salle
				FROM dates D,gest_cal g 
				where G.gnum_cal=$formation and D.num_date=G.num_date and D.jour='$date' and hdeb='$debut'";
	return pg_query($sql);
}
/**
	Obtenir un �venement d'une formation 
	@param integer : numero formation
	@param date : la date de l'evenement
	@param time : heure de debut
	@return integer : resultat requete(n�formation,libelle,date,duree,modalite,intervention,intervenant,salle)
*/
function get_formation_evenement($formation,$date,$debut){
	if (VERSION_BASE > VERSION) {
		//echo "date=$date et debut=$debut<br>";
		$dt=$date." ".$debut;
		//echo "dt=$dt<br>";
		$date_deb=dbstamp2tstamp($dt);
		//echo "date_deb=$date_deb<br>";
		$sql="select * from contraintes.af_get_formation_evenement($formation,cast(to_timestamp($date_deb) as timestamp without time zone))
				as (formation int4,libelle varchar,date timestamp,duree interval,modalite int4,intervention int4,intervenant int4,salle int4)";
		
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_evenement_old($formation,$date,$debut);
	return $res;
}

/**
	Enregistre un evenement d'une formation (OLD)
	@param integer : numero formation
	@param date : la date de l'evenement
	@param string : periode
	@param string : heure debut
	@param string : heure fin
	@param string : intitule
	@param integer : modalite
	@param integer : intervention
	@param integer : intervenant
	@param integer : salle
*/
function set_formation_evenement_old($formation,$date,$periode,$hdeb,$hfin,$libelle,$modalite,$intervention,$intervenant,$salle){
	$num_date=get_num_date($date);
	if($intervenant=="")
		$intervenant='NULL';
	if($salle=="")
		$salle='NULL';
	
	$sql="insert into gest_cal(gnum_cal,num_date,demi_jour,hdeb,hfin,libelle,type,numtype,num_intervenant,num_salle) 
				values($formation,$num_date,'$periode','$hdeb','$hfin','$libelle',$modalite,$intervention,$intervenant,$salle)";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}
/**
	Enregistre un evenement d'une formation 
	@param integer : numero formation
	@param date : la date de l'evenement
	@param string : periode
	@param string : heure debut
	@param string : heure fin
	@param string : intitule
	@param integer : modalite
	@param integer : intervention
	@param integer : intervenant
	@param integer : salle
*/
function set_formation_evenement($formation,$intitule,$date,$periode,$debut,$fin,$intervention,$modalite,$intervenant,$salle){
	if (VERSION_BASE > VERSION) {
		// nouveau timestamp
		$dt=$date." ".$debut;
		$new_date=dbstamp2tstamp($dt);	
		// duree
		$tab_deb=explode(":",$debut);
		$tab_fin=explode(":",$fin);
		$d=get_hms(hms_to_secondes($tab_fin) - hms_to_secondes($tab_deb));
		$duree=$d["heure"].":".$d["minute"].":".$d["seconde"];	
		$sql="select * from contraintes.af_set_formation_evenement($formation,'$intitule',cast(to_timestamp($new_date) as timestamp without time zone),'$duree',$modalite,$intervention,$intervenant,$salle)";
		//echo "sql=$sql<br>";				
		$res=pg_query($sql);
	}
	else
		$res=set_formation_evenement_old($formation,$date,$periode,$debut,$fin,$intitule,$modalite,$intervention,$intervenant,$salle);
	return $res;
}

/**
	Mettre � jour un evenement de formation (OLD)
	*@param integer : le numero de formation
*   @param date : la date de l'evenement
*   @param char : la periode (M -> matin , A -> apr�s-midi , T -> journ�e enti�re)
*   @param varchar : heure de debut
*   @param varchar : heure de fin
*   @param varchar : intitul�
*   @param integer : le type d'intervention
*   @param integer : la modalite (1 -> Synchrone , 2 -> Asynchrone ou 3 -> Presentiel)
*   @param integer : le numero d'intervenant
*   @param integer : le numero de salle
*   @param integer : l'heure du cours
*   @param integer : la date du cours
*/
function maj_formation_evenement_old($formation,$intitule,$date,$periode,$debut,$fin,$type,$modalite,$intervenant,$salle,$odate,$odebut){
	// numero de date avant modif
	$onum_date=get_num_date($odate);
	// numero de date apr�s modif
	$num_date=get_num_date($date);
	$odebut=substr($odebut,0,-3);
	if($intervenant=="")
		$intervenant='NULL';
	if($salle=="")
		$salle='NULL';
	$sql="update gest_cal set 
				num_date=$num_date,
				demi_jour='$periode',
				hdeb='$debut',
				hfin='$fin',
				libelle='$intitule',
				type=$type,
				numtype=$modalite,
				num_intervenant=$intervenant,
				num_salle=$salle 
				where hdeb='$odebut' and num_date=$onum_date";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}
/**
	Mettre � jour un evenement de formation 
	@param integer : le numero de formation
*   @param varchar : intitul�
*   @param date : la date de l'evenement
*   @param char : la periode (M -> matin , A -> apr�s-midi , T -> journ�e enti�re)
*   @param varchar : heure de debut
*   @param varchar : heure de fin

*   @param integer : le type d'intervention
*   @param integer : la modalite (1 -> Synchrone , 2 -> Asynchrone ou 3 -> Presentiel)
*   @param integer : le numero d'intervenant
*   @param integer : le numero de salle
	@param date : l'ancienne date du cours
*   @param varchar : l'ancienne heure du cours
*   
*/
function  maj_formation_evenement($formation,$intitule,$date,$periode,$debut,$fin,$type,$modalite,$intervenant,$salle,$odate,$odebut){
	 if (VERSION_BASE > VERSION){
		// ancien timestamp en base
		$dt=$odate." ".$odebut;
		$old_date=dbstamp2tstamp($dt);
		// nouveau timestamp
		$dt=$date." ".$debut;
		$new_date=dbstamp2tstamp($dt);
		// duree
		$tab_deb=explode(":",$debut);
		$tab_fin=explode(":",$fin);
		$d=get_hms(hms_to_secondes($tab_fin) - hms_to_secondes($tab_deb));
		
		$duree=$d["heure"].":".$d["minute"].":".$d["seconde"];
		$sql="select * from contraintes.af_maj_formation_evenement($formation,'$intitule',cast(to_timestamp($new_date) as timestamp without time zone),'$duree',$modalite,$type,$intervenant,$salle,cast(to_timestamp($old_date) as timestamp without time zone))";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	 }
	 else
		$res=maj_formation_evenement_old($formation,$intitule,$date,$periode,$debut,$fin,$type,$modalite,$intervenant,$salle,$odate,$odebut) ;
	 return $res;
}

/**
	Retourne un tableau de conflits pour un evenement de formation (OLD)
	@param integer : numero formation
	@param date : la date de l'evenement
	@param string : heure debut
	@param string : heure fin
	@param string : ancien debut
	@param mode : ('add' ou 'edit')
	@param integer : 1=>copie
	@return array : 1 heures, 2 classe, 3 salle, 4 intervenant
*/
function check_formation_evenement_conflits_old($formation,$date,$debut,$fin,$odebut,$mode,$copie,$salle=0,$intervenant=0){
	 $tab=array();
	 $tab[]=0;
	 // teste heure debut et fin
	 $sql="select count(*) from dates D,gest_cal G
				where D.num_date=G.num_date and D.jour='$date' 
				and ((G.hdeb>='$debut' and G.hdeb<='$fin') or (G.hdeb<='$debut' and G.hfin>='$debut'))
				and G.gnum_cal=$formation";
	 if($mode=='edit' && $copie!=1)
		$sql .=" and G.hdeb!='$odebut'";
	 $res=pg_query($sql);
	 if(pg_fetch_result($res,0,0)>0)
		$tab[]=1;
		
	 // teste si le groupe n'est pas pris � ces heures dans d'autres formations
    //  recup du numero de classe
	$sql="select distinct g_num_gest_g from gestionnaire where g_num_cal=$formation";
	$res=pg_query($sql);
	$classe=pg_fetch_result($res,0,0);
	$sql="select count(*) from dates D,gest_cal G
				where D.num_date=G.num_date and D.jour='$date' 
				and ((G.hdeb>='$debut' and G.hdeb<='$fin') or (G.hdeb<='$debut' and G.hfin>='$debut')) 
				and G.gnum_cal in 
				(select g_num_cal from gestionnaire where g_num_gest_g=$classe) 
				and G.gnum_cal!=$formation 
				and G.hdeb!='$odebut'";
	 $res=pg_query($sql);
	 if(pg_fetch_result($res,0,0)>0)
		$tab[]=2;
		
	 // teste si la salle est libre
	 $sql="select count(*) from dates D,gest_cal G
				where D.num_date=G.num_date and D.jour='$date' 
				and ((G.hdeb>='$debut' and G.hdeb<='$fin') or (G.hdeb<='$debut' and G.hfin>='$debut')) 
				and G.num_salle=$salle 
				and not(G.hdeb='$odebut' and G.gnum_cal=$formation)";
	 //echo "sql=$sql<br>";
	$res=pg_query($sql);
	 if(pg_fetch_result($res,0,0)>0)
		$tab[]=3;
		
	 // teste si l'intervenant est libre
	 $sql="select count(*) from dates D,gest_cal G
				where D.num_date=G.num_date and D.jour='$date' 
				and ((G.hdeb>='$debut' and G.hdeb<='$fin') or (G.hdeb<='$debut' and G.hfin>='$debut')) 
				and G.num_intervenant=$intervenant 
				and not(G.hdeb='$odebut' and G.gnum_cal=$formation)";
	$res=pg_query($sql);
	 if(pg_fetch_result($res,0,0)>0)
		$tab[]=4;
	 //print_r($tab);
	 return $tab;
}
/**
	Retourne un tableau de conflits pour un evenement de formation 
	@param integer : numero formation
	@param date : la date de l'evenement
	@param string : heure debut
	@param string : heure fin
	@param string : ancien debut
	@param mode : ('add' ou 'edit')
	@param integer : 1=>copie
	@return array : 1 heures, 2 classe, 3 salle, 4 intervenant
*/
function check_formation_evenement_conflits($formation,$date,$debut,$fin,$odebut,$mode,$copie,$salle=0,$intervenant=0){
	 if (VERSION_BASE > VERSION){
		$tab=array();
		//$date=dbstamp2tstamp($date);
		if($odebut=="")
			$odebut=$debut;
		else
			$odebut=substr($odebut,0,-3);
		$sql="select * from contraintes.af_check_formation_evenement($formation,'$date',
					cast('$debut' as time),cast('$fin' as time),cast('$odebut' as time),'$mode',$copie) ";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
		$row=pg_fetch_result($res,0,0);
		if(sizeof($row)>0){
			$t=explode("=",$row);
			$ta=substr($t[1],1,-1);
			$tab=explode(",",$ta);
		
		}
		
	 }
	 else
		$tab=check_formation_evenement_conflits_old($formation,$date,$debut,$fin,$odebut,$mode,$copie,$salle,$intervenant);
	 return $tab;
}

/**
    Modifier les dates d'une formation (OLD)
*
*   @param integer : le numero de formation
*   @param integer : le numero de  date de debut
*   @param integer : le numero de date de fin
*   @return integer : le resultat de la requete
*/
function maj_formation_dates_old($formation,$debut,$fin){
	//echo "deb=$debut et fin=$fin<br>";
	$num_fin=get_num_date($fin);
	$num_debut=get_num_date($debut);
    $sql="UPDATE calendrier SET datedeb=$num_debut, datefin=$num_fin WHERE num_cal=$formation";
    //echo "sql=$sql<br>";
    $maj=pg_query($sql);
    return $maj;
}
/**
    Modifier les dates d'une formation
*
*   @param integer : le numero de formation
*   @param integer : le numero de  date de debut
*   @param integer : le numero de date de fin
*   @return integer : le resultat de la requete
*/
function maj_formation_dates($formation,$debut,$fin){
    if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_maj_formation_dates($formation,'$debut','$fin')";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
    }
    else
        $res = maj_formation_dates_old($formation,$debut,$fin);
    return $res;
}

/**
    Modifier les langues d'une formation (OLD)
*
*   @param integer : le numero de formation
*   @param string : la liste des langues
*   @return integer : le resultat de la requete
*/
function maj_formation_langues_old($formation,$langue){
    $sql="UPDATE calendrier SET langue='$langue'  WHERE num_cal=$formation";
    $maj=pg_query($sql);
    return $maj;
}
/**
    Modifier les langues d'une formation
*
*   @param integer : le numero de formation
*   @param array : les anciennes langues
*   @param array : les nouvelles langues
*   @return integer : le resultat de la requete
*/
function maj_formation_langues($formation,$langue_insert,$langue_delete){
    if (VERSION_BASE > VERSION) {
		$langue_ajout = int_array_to_plpgsql_array($langue_insert);
		$langue_sup = int_array_to_plpgsql_array($langue_delete);
        $sql="select * from contraintes.af_maj_formation_langues($formation,$langue_ajout,$langue_sup)";
		$res=pg_query($sql);
    }
    else
        $res = maj_formation_langues_old($formation,$langue);
    return $res;
}

/**
	Teste s'il y a des evenements de formation en dehors des dates (OLD)
	@param integer : formation
	@param date : la date de debut
	@param date : la date de fin
	@return bool : true si evenement en dehors des dates, false sinon
*/
function check_formation_evenements_hors_periode_old($formation,$datedeb,$datefin){
	$sql="SELECT COUNT(*)
                           FROM Dates D, gest_cal G
                           WHERE D.num_date=G.num_date
                           AND (D.jour <'$datedeb' OR D.jour>'$datefin')
                           AND G.gnum_cal=$formation";
	$sql="SELECT *
                           FROM Dates D, gest_cal G
                           WHERE D.num_date=G.num_date
                           AND (D.jour <'$datedeb' OR D.jour>'$datefin')
                           AND G.gnum_cal=$formation and demi_jour<>'C'";
     //echo "sql=$sql<br>";
	$result=pg_query($sql);
	$nb=pg_num_rows($result);
	//echo "nb=$nb<br>";
	if( pg_fetch_result($result,0)==0)
		return 'f';
	return 't';
		 //return pg_fetch_result($res,0,0);
}
/**
	Teste s'il y a des evenements de formation en dehors des dates 
	@param integer : formation
	@param date : la date de debut
	@param date : la date de fin
	@return bool : true si evenement en dehors des dates, false sinon
*/
function check_formation_evenements_hors_periode($formation,$datedeb,$datefin){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_is_formation_evenements_hors_periode($formation,'$datedeb','$datefin')";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		return check_formation_evenements_hors_periode_old($formation,$datedeb,$datefin);
	return pg_fetch_result($res,0,0);
}

/**
	Supprime les evenements d'une journee de formation (OLD)
	@param integer : formation
	@param date : date de la journ�e
	
	@return bool : true ok false sinon
*/
function del_formation_evenements_old($formation,$date){
	$dt=get_num_date($date);
	$sql="delete from gest_cal where num_date=$dt and gnum_cal=$formation";
	//echo "sql=$sql<br>";
	return pg_query($sql);

}
/**
	Supprime les evenements d'une journee de formation
	@param integer : formation
	@param date : date de la journ�e
	
	@return bool : true ok false sinon
*/
function del_formation_evenements($formation,$date){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_del_formation_evenements($formation,'$date')";
		$res=pg_query($sql);
		//echo "sql=$sql<br>";
	}
	else
		$res=del_formation_evenements_old($formation,$date);
	return $res;
}

/**
	Supprime un evenement d'une journee de formation (OLD)
	@param integer : formation
	@param date : date de la journ�e
	@param : string : debut
	return bool : true ok false sinon
*/
function del_formation_evenement_old($formation,$date,$debut){
	$dt=get_num_date($date);
	$deb=substr($debut,0,-3);
	$sql="delete from gest_cal where num_date=$dt and gnum_cal=$formation and hdeb='$deb'";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}
/**
	Supprime un evenement d'une journee de formation
	@param integer : formation
	@param date : date de la journ�e
	@param : string : debut
	return bool : true ok false sinon
*/
function del_formation_evenement($formation,$date,$debut){
	if (VERSION_BASE > VERSION) {
		$dt=$date." ".$debut;
		//echo "dt=$dt<br>";
		$date_deb=dbstamp2tstamp($dt);
		$sql="select * from contraintes.af_del_formation_evenement($formation,cast(to_timestamp($date_deb) as timestamp without time zone))";
		$res=pg_query($sql);
		//echo "sql=$sql<br>";
	}
	else
		$res=del_formation_evenement_old($formation,$date,$debut);
	return $res;
}

/**
	Teste si une journ�e de formation est supprimable (OLD)
	@param integer : formation
	@param date : date de la journ�e
	@return bool : true supprimable,false sinon
*/
function check_formation_evenements_supprimable_old($formation,$date){
	//$num_date=get_num_date($date);
	$sql="select count(*) from notation where num_cal=$formation";
	$res=pg_query($sql);
	
	if(pg_fetch_result($res,0,0)>0)
		return 'f';
	return 't';
}
 /**
	Teste si une journ�e de formation est supprimable 
	@param integer : formation
	@param date : date de la journ�e
	@return bool : true supprimable,false sinon
*/
function check_formation_evenements_supprimable($formation,$date){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_is_formation_evenements_supprimable($formation,'$date')";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
		return pg_fetch_result($res, 0, 0);
	}
	else
		$res=check_formation_evenements_supprimable_old($formation,$date);
	return $res;
}

/**
	Teste si un evenement de formation est supprimable (OLD)
	@param integer : formation
	@param date : date de la journ�e
	@param string : debut cours
	@return bool : true supprimable,false sinon
*/
function check_formation_evenement_supprimable_old($formation,$date,$debut){
	//$num_date=get_num_date($date);
	$sql="select count(*) from notation where num_cal=$formation";
	$res=pg_query($sql);
	
	if(pg_fetch_result($res,0,0)>0)
		return 'f';
	return 't';
}
/**
	Teste si un evenement de formation est supprimable 
	@param integer : formation
	@param date : date de la journ�e
	@param string : debut cours
	@return bool : true supprimable,false sinon
*/
function check_formation_evenement_supprimable($formation,$date,$debut){
	if (VERSION_BASE > VERSION) {
		$dt=$date." ".$debut;
		//echo "dt=$dt<br>";
		$date_deb=dbstamp2tstamp($dt);
		$sql="select * from contraintes.af_is_formation_evenement_supprimable($formation,cast(to_timestamp($date_deb) as timestamp without time zone))";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
		return pg_fetch_result($res, 0, 0);
	}
	else
		return check_formation_evenement_supprimable_old($formation,$date,$debut);
}

/**
	Obtenir les salles utilisees par une formation (OLD)
	@param integer : formation
*/
function get_formation_salles_old($formation){
	$sql="select distinct s.num_salle,s.nom_salle  from salle s,gest_cal g 
				where s.num_salle=g.num_salle and gnum_cal=$formation";
	return pg_query($sql);
}
/**
	Obtenir les salles utilisees par une formation 
	@param integer : formation
*/
function get_formation_salles($formation){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_salles($formation) as (id int4,libelle varchar)";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_salles_old($formation);
	return $res;
}

/**
	Obtenir les intervenants de la formation  (OLD)
	@param integer : formation
	@return integer : resultat requete ( )
*/
function get_formation_intervenants_old($formation){
	$sql="select distinct num_user,nom,prenom from utilisateurs,gestionnaire,gestionutil 
				where num_user=gu_num_user 
				and g_num_gest_g=gu_num_gest_gr 
				and g_num_cal=$formation and categorie like 'f%'";
	return pg_query($sql);
}
/**
	Obtenir les intervenants de la plateforme  
	@param integer : formation
	@return integer : resultat requete ( )
*/
function get_formation_intervenants($formation){
	 if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_intervenants($formation) as (id int4,nom varchar,prenom varchar)";
		$res=pg_query($sql);
	 }
	 else
		$res=get_formation_intervenants_old($formation);
	 return $res;
}

/**
	Obtenir l'avancement des individus d'une formation (OLD)
	@param integer : formation
	@return integer :   resultat requete (parcours int4, etape int4, formation int4, classe int4, stagiaire int4)
*/
function get_formation_avancement_old($formation){

}
/**
	Obtenir l'avancement des individus d'une formation 
	@param integer : formation
	@return integer :   resultat requete (parcours int4, etape int4, formation int4, classe int4, stagiaire int4)
*/
function get_formation_avancement($formation){
	 if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_avancement($formation) 
					as (parcours int4, etape int4, formation int4, stagiaire int4, type varchar)";
		$res=pg_query($sql);
	 }
	 else
		return  get_formation_avancement_old($formation);
	 return $res;

}

/**
	Obtenir les evenements presentiels d'une formation (OLD)

	@param integer : le numero de formation
 	@return integer : le resultat de la requete
*/
function get_formation_evenements_presentiels_old($formation){
	$sql="select jour from dates where num_date in
			(select num_date from gest_cal where gnum_cal=$formation and numtype=3) order by jour";
   return (pg_query($sql));
}
/**
	Obtenir lesevenements presentiels d'une formation

	@param integer : le numero de formation
 	@return integer : le resultat de la requete
*/
function get_formation_evenements_presentiels($formation){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_evenements_presentiels($formation) 
				as (formation int4,libelle varchar,date timestamp,duree interval,modalite int4,intervention int4,intervenant int4,salle int4) order by date";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
    }
    else
        $res = get_formation_evenements_presentiels_old($formation);
    return $res;
}

/**
	Obtenir les individus d'une formations (OLD)
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_individus_old($formation){

}
/**
	Obtenir les individus d'une formations 
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_individus($formation,$ordre="ASC"){
	if (VERSION_BASE > VERSION) {
		
		if($ordre=="ASC")$v_odre=" ORDER BY nom ASC";
		else $v_odre=" ORDER BY nom DESC";
		$sql="select * from contraintes.af_get_formation_individus($formation) as (id int4,nom varchar,prenom varchar,organisme int4)$v_odre  ";
//		echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		return get_formation_individus_old($formation);
	return $res;
}

/**
	Obtenir les individus d'une formations qui ne sont pas de certains types
	@param integer $formation formation
	@param integer $except_individus_types Tableau de types d'individus a exclure
	@return integer $ordre le resultat de la requete (id int4,nom varchar,prenom varchar,organisme int4)
*/
function get_formation_individus_except_individus_types($formation, $except_individus_types, $ordre="ASC"){
	if (VERSION_BASE > VERSION) {
		
		if($ordre=="ASC")$v_odre=" ORDER BY nom ASC";
		else $v_odre=" ORDER BY nom DESC";
		$sql="select * from contraintes.af_get_formation_individus_except_individus_types($formation,$except_individus_types) as (id int4,nom varchar,prenom varchar,organisme int4) $v_odre  ";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	return $res;
}


/**
	Obtenir les grilles presentielles pour une formation (OLD)
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_grilles_old($formation){
	$sql="select distinct num_grille,libelle_grille from grille where num_grille in (select num_grille from config_grille where num_cal=$formation)";
	return pg_query($sql); 
}
/**
	Obtenir les grilles presentielles pour une formation
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_grilles($formation){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_grilles($formation) as (id int4,libelle varchar)";
		$res=pg_query($sql);
	}
	else
		return get_formation_grilles_old($formation);
	return $res;
}

/**
	Obtenir les etapes d'une formation  (OLD)
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_etapes_old($formation){
	// recup du parcours de la formation
}
/**
	Obtenir les etapes d'une formation  
	@param integer : formation
	@return integer : le resultat de la requete (id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice integer)
*/
function get_formation_etapes($formation){
	if (VERSION_BASE > VERSION) {
		/*
		$sql="select * from contraintes.af_get_formation_etapes($formation) 
					as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar, type int4,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool)";
		*/
		$sql="select * from contraintes.af_get_formation_etapes($formation) 
			as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice int4)";
		//echo $sql."<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_etapes_old($formation);
	return $res;
}

/**
	Obtenir les etapes d'une formation pour un individu
	(cad : l'individu peut faire des �tapes qui lui son propres (en fonction des ses resultats)  
	@param integer : formation
	@return integer : le resultat de la requete (id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice integer)
*/
function get_formation_etapes_individu($formation,$individu){
	if (VERSION_BASE > VERSION) {
		/*
		$sql="select * from contraintes.af_get_formation_etapes($formation) 
					as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar, type int4,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool)";
		*/
		$sql="select * from contraintes.af_get_formation_etapes_individu($formation,$individu) 
			as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice int4)";
		//echo $sql."<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_etapes_old($formation);
	return $res;
}

/**
	Obtenir les etapes r�elles d'une formation pour un individu
	(cad : l'individu peut faire des �tapes qui lui son propres (en fonction des ses resultats)  
	@param integer : formation
	@return integer : le resultat de la requete (id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice integer)
*/
function get_formation_etapes_reelles_individu($formation,$individu){
	if (VERSION_BASE > VERSION) {
		/*
		$sql="select * from contraintes.af_get_formation_etapes($formation) 
					as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar, type int4,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool)";
		*/
		$sql="select * from contraintes.af_get_formation_etapes_reelles_individu($formation,$individu) 
			as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice int4)";
		//echo $sql."<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_etapes_old($formation);
	return $res;
}


/**
	Obtenir les etapes d'un type d'une formation  (OLD)
	@param integer : formation
	@param integer :type
	@return integer : le resultat de la requete
*/
function get_formation_etapes_type_old($formation,$type){

}
/**
	Obtenir les etapes d'un type d'une formation  
	@param integer : formation
	@param integer :type
	@return integer : le resultat de la requete
*/
function get_formation_etapes_type($formation,$type){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_etapes_type($formation,'$type') 
					as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool,indice int4)";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_etapes_type_old($formation,$type);
	return $res;
}

/**
	Obtenir les series d'une formation (OLD)
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_etapes_series_old($formation){
	// recup parcours de la formation
	$sql="select distinct g_num_parcours from gestionnaire where g_num_cal=$formation";
	$res=pg_query($sql);
	$parcours=pg_fetch_result($res,0,0);
	// recup des serie du parcours
	$sql="select distinct num_c_q_e_d,titre,nom_etape,type_etape,type_etape,num_etape,g_num_parcours,obligation,temps_etape,num_etape_suivante from
				gestparcours,serie_exo where g_num_parcours=$parcours and type_etape='B' and id=num_c_q_e_d and obligation='o'";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}
/**
	Obtenir les series d'une formation
	@param integer : formation
	@return integer : le resultat de la requete (id int4, libelle_pour_parcours varchar, libelle_etape  varchar, type int4,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool, libelle_type_etape varchar, modifiable bool)
*/
function get_formation_etapes_series($formation){
	if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_etapes_series($formation) as  (id int4, libelle_alias varchar, libelle varchar, libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool, libelle_type_original varchar, modifiable bool) order by parcours desc";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	}
	else
		$res=get_formation_etapes_series_old($formation);
	return $res;
}

/**
	Modifier la configuration d'une ressource de formation
	@param integer : formation
	@param integer : ressource
	@param integer : ancien espace
	@param integer : ancienne section
	@param string : titre de la ressource
	@param date : la date de d�but de disponibilite
	@param date : la date de fin de disponibilite
	@param integer : la dur�e de nouveaut�
	@param text : mots cles
	@param integer :  espace
	@param integer : section
	@return integer : le resultat de la requete
*/
function maj_formation_ressource($formation,$ressource,$old_espace,$old_section,$titre,$debut,$fin,$nouveau,$motcles,$espace,$section){
	//echo "$debut,$fin";
	$date_deb=dbstamp2tstamp($debut);
	$date_fin=dbstamp2tstamp($fin);
	$sql="select * from contraintes.af_maj_formation_ressource($formation,$ressource,$old_espace,$old_section,$titre,cast(to_timestamp($date_deb) as timestamp without time zone),cast(to_timestamp($date_fin) as timestamp without time zone),'$nouveau days','$motcles',$espace,$section);";
//	echo "sql=$sql<br>";
	return pg_query($sql);
}

/**
	Supprimes une ressource d'une formation
	@param integer : formation
	@param integer : ressource
	@param integer :  espace
	@param integer : section
	@return integer : le resultat de la requete
*/
function del_formation_ressource($formation,$ressource,$espace,$section){
	$sql="select * from contraintes.af_del_formation_ressource($formation,$ressource,$espace,$section)";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}

/**
	Supprime un espace (ressource) d'une formation
	@param integer : formation
	@param integer :  espace
	@return integer : le resultat de la requete
*/
function del_formation_espace($formation,$espace){
	$sql="select * from contraintes.af_del_formation_espace($formation,$espace)";
	return pg_query($sql);
}

/**
	Supprime une section d' espace (ressource) d'une formation
	@param integer : formation
	@param integer :  espace
	@param integer : section
	@return integer : le resultat de la requete
*/
function del_formation_espace_section($formation,$espace,$section){
	$sql="select * from contraintes.af_del_formation_espace_section($formation,$espace,$section)";
	return pg_query($sql);
}

/**
	Obtenir les forums disponibles pour une formation
	@param integer : formation
	@return integer : le resultat de la requete
*/
function get_formation_forums_disponibles($formation){
	$sql="select * from contraintes.af_get_formation_forums_disponibles($formation) as (id int4,libelle varchar)";
	return pg_query($sql);
}

/**
	Associer un forum � une formation
	@param integer : formation
	@param integer : forum
	@return integer : le resultat de la requete
	
*/
function set_formation_forum($formation,$forum){
	$sql="select * from contraintes.af_set_formation_forum($formation,$forum)";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}

/**
	Supprimer une association formation / forum
	@param integer : formation
	@param integer : forum
	@return integer : le resultat de la requete
	
*/
function del_formation_forum($formation,$forum){
	$sql="select * from contraintes.af_del_formation_forum($formation,$forum)";
	//echo "sql=$sql<br>";
	return pg_query($sql);
}

/**
	Obtenir les stagiaires d'une formation et leur etat de correction pour un exotlc (OLD)
	@param integer : formation
	@param integer : exotlc
	@param integer : classe
	@param string : ordre by ...
	@return integer : res req
*/
function get_formation_stagiaires_correction_old($formation,$exotlc,$classe,$orederby){
	$sql="SELECT num_user, stagiaires_non_corriges.nom, stagiaires_non_corriges.prenom, date_correction , rcpt_date, nom_fic
		FROM(
		(
			(SELECT   num_user, nom, prenom
				FROM utilisateurs
				WHERE categorie like '%u%'
				AND num_user in     
				(SELECT  gu_num_user
					FROM gestionutil
					WHERE gu_num_gest_gr = $classe)
			)  AS stagiaires_non_corriges

            LEFT OUTER JOIN

            (SELECT DISTINCT R.num_user, nom, prenom, date_correction
				FROM    resexouser R, utilisateurs U
                WHERE R.num_cal=$formation
                AND R.num_exo=$exotlc
                AND U.categorie like '%u%'
                AND R.num_user in     
				(SELECT  DISTINCT gu_num_user
					FROM gestionutil
                    WHERE gu_num_gest_gr = $classe)
				AND R.num_user = U.num_user
			) AS stagiaires_corriges

            USING (num_user)
        )

        LEFT OUTER JOIN

        (SELECT rcpt_date, nom_fic, num_stag
			FROM reception_exostlc
            WHERE num_exo=$exotlc
            AND num_cal=$formation
		) AS stagiaires_exos_rendus ON num_user=num_stag
		) $orderby";
	return (pg_query($sql));
}
/**
	Obtenir les stagiaires et leur etat de correction pour un exotlc
	@param integer : formation
	@param integer : exotlc
	@param integer : classe
	@param string : ordre by ...
	@return integer : res req
*/
function get_formation_stagiaires_correction($formation,$exotlc,$classe,$orederby){
	if(VERSION_BASE>VERSION){
		$sql="select * from contraintes.af_get_formation_stagiaires_correction($formation,$exotlc,$classe) 
				as (id int4,nom varchar,prenom varchar,date_correction timestamp)"; 
		$res=pg_query($sql);
	}
	else
		$res=get_formation_stagiaires_correction_old($formation,$exotlc,$classe,$orederby);
	return $res;
}

 /**
    Obtenir les exostlc avec fichier rendu pour une formation donn�e (OLD)
*
*   @param integer : le numero de formation
*   @return interger : le resultat de la requete
*/
function get_formation_exostlc_fichier_rendu_old($formation){
    $sql="select num_exo,lower(titre) as minnom from exotlc where num_exo in (select distinct num_exo from reception_exostlc where num_cal=$formation)
                order by minnom";
    return (pg_query($sql));
}
/**
    Obtenir les exostlc avec fichier rendu pour une formation donn�e
*
*   @param integer : le numero de formation
*   @return interger : le resultat de la requete
*/
function get_formation_exostlc_fichier_rendu($formation){
     if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_exostlc_fichier_rendu($formation) 
				as (id int4,commentaire varchar,formation int4,parcours int4)";
		$res=pg_query($sql);
    }
    else
        $res = get_formation_exostlc_fichier_rendu_old($formation);
    return $res;
}

/**
	Obtenir les forums d'une formation
	@param integer : formation
	@return interger : le resultat de la requete
*/
function get_formation_forums($formation){
	$sql="select * from contraintes.af_get_formation_forums($formation) as 
	(id int4,libelle varchar,description varchar,moderation boolean,moderateur varchar,fils int4,messages int4,upload boolean,theme varchar)";
//	echo "sql=$sql<br>";
	return pg_query($sql);
}

/**
 *   Obtenir les fichiers ressources d'une section d'un espace pour une formation (OLD)
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 *   @param integer : le numero de section
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_espace_section_ressources_old($formation, $espace, $section) {
	$sql = "select num_gest_res, libelle, num_fic, num_cal, num_espace, num_section, datedeb, datefin, nouveau, motcle, type from ressource_gest  where num_cal=$formation and num_espace=$espace and num_section=$section ";
	$res = pg_query($sql);
	return $res;
}

/**
 *   Obtenir les fichiers ressources d'une section d'un espace pour une formation
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 *   @param integer : le numero de section
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_espace_section_ressources($formation, $espace, $section) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_espace_section_ressources($formation,$espace, $section) AS (ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut date,fin date,nouveau interval,icone int4) order by libelle_espace,libelle_section";
//		echo "sql=$sql<br>";
		$res = pg_query($sql);
		return $res;
	} else
		return get_formation_espace_section_ressources_old($formation, $espace, $section);
}

/**
 *   Obtenir les fichiers ressources d'une section d'un espace d'une formation fusionn�es avec les etapes_ressources faites par l'individu
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 *   @param integer : le numero de section
 *   @param integer : le numero de l'individu
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_espace_section_all_ressources_individu($formation, $espace, $section,$individu) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_espace_section_all_ressources_individu($formation,$espace, $section,$individu) AS (ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut date,fin date,nouveau interval,icone int4) order by libelle_espace,libelle_section";
		//echo "sql=$sql<br>";
		$res = pg_query($sql);
		return $res;
	} else
		return get_formation_espace_section_ressources_old($formation, $espace, $section);
}



/**
 *   Obtenir les fichiers ressources d'une section d'un espace pour une formation
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 *   @param integer : le numero de section
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_espace_section_ressources_no_date($formation, $espace, $section) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_espace_section_ressources_no_date($formation,$espace, $section) AS (ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut date,fin date,nouveau interval,icone int4) order by libelle_espace,libelle_section";
//echo "sql=$sql<br>";
		$res = pg_query($sql);
		return $res;
	} else
		return get_formation_espace_section_ressources_old($formation, $espace, $section);
}


/**
 *   Obtenir les fichiers ressources  d'un espace pour une formation (OLD)
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_espace_ressources_old($formation, $espace) {
	$sql = "select num_gest_res, libelle, num_fic, num_cal, num_espace, num_section, datedeb, datefin, nouveau, motcle, type from ressource_gest  where num_cal=$formation and num_espace=$espace  ";
	$res = pg_query($sql);
	return $res;
}

/**
 *   Obtenir les fichiers ressources  d'un espace pour une formation
 *
 *   @param integer : l'identifiant de la formation
 *   @param integer : le numero d'espace
 
 *   @return integer : le resultat de la requete (id int4, libelle varchar, fichier int4, formation int4, espace int4, section int4, debut int4, fin int4, nouveaute int4, mots_cles text, type int4 )
 */
function get_formation_espace_ressources($formation, $espace) {
	if (VERSION_BASE > VERSION) {
		$sql = "SELECT * FROM contraintes.af_get_formation_espace_ressources($formation,$espace) AS (ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut date,fin date,nouveau interval,icone int4) order by libelle_espace,libelle_section";
		$res = pg_query($sql);
		return $res;
	} else
		return get_formation_espace_section_ressources_old($formation, $espace);
}

/**
	Supprimer les resultats d'un individu pour une formation (old)
	@param integer : formation
	@param integer : individu
	@return res req
*/
function del_formation_resultat_individu_old($formation,$individu){

}
/**
	Supprimer les resultats d'un individu pour une formation 
	@param integer : formation
	@param integer : individu
	@return res req
*/
function del_formation_resultat_individu($formation,$individu){
	 if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_del_formation_resultat_individu($formation,$individu)";
		return pg_query($sql);
	 }
	 else
		return del_formation_resultat_individu_old($formation,$individu);
}

/**
   Retourne le score pour un parcours pour une formation et un individu
   @param integer : formation
   @param integer : individu
   @param integer : parcours
   @return real : resultat
*/
function get_formation_individu_parcours_score($formation,$individu,$parcours){
	$sql="select * from contraintes.af_get_formation_individu_parcours_score($formation,$individu,$parcours)";
	return pg_query($sql);
}

/**
   Retourne le score pour les series d'un parcours pour une formation et un individu
   @param integer : formation
   @param integer : individu
   @param integer : parcours
   @return real : resultat
*/
function get_formation_individu_parcours_series_score($formation,$individu,$parcours){
	$sql="select * from contraintes.af_get_formation_individu_parcours_series_score($formation,$individu,$parcours)";
	return pg_query($sql);
}

/**
   Retourne le score pour les exotlcs d'un parcours pour une formation et un individu
   @param integer : formation
   @param integer : individu
   @param integer : parcours
   @return real : resultat
*/
function get_formation_individu_parcours_exotlcs_score($formation,$individu,$parcours){
	$sql="select * from contraintes.af_get_formation_individu_parcours_exotlcs_score($formation,$individu,$parcours)";
	return pg_query($sql);
}

/**
	Obtenir l'avancement d'un individu d'une formation pour un parcours
	@param integer : formation
	@param integer : individu
	@param integer : parcours
	@return integer :   resultat requete (parcours int4, etape int4, formation int4, classe int4, stagiaire int4)
*/
function get_formation_avancement_parcours_individu($formation,$individu,$parcours){
	 if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_avancement_parcours_individu($formation,$individu,$parcours) 
					as (parcours int4, etape int4, formation int4, stagiaire int4, type varchar)";
		//echo "sql=$sql<br>";
		$res=pg_query($sql);
	 }
	 else
		return  get_formation_avancement_parcours_individu_old($formation);
	 return $res;

}

/**
	Obtenir l'avancement d'un individu d'une formation 
	@param integer : formation
	@param integer : individu
	
	@return integer :   resultat requete (parcours int4, etape int4, formation int4, classe int4, stagiaire int4)
*/
function get_formation_avancement_individu($formation,$individu){
	 if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_avancement_individu($formation,$individu) 
					as (parcours int4, etape int4, formation int4, stagiaire int4, type varchar)";
//		echo "sql=$sql<br>";
		$res=pg_query($sql);
	 }
	 else
		return  get_formation_avancement_individu_old($formation);
	 return $res;

}

/**
	Obtenir l'avancement r�el d'un individu d'une formation 
	@param integer : formation
	@param integer : individu
	
	@return integer :   resultat requete (parcours int4, etape int4, formation int4, classe int4, stagiaire int4)
*/
function get_formation_avancement_reel_individu($formation,$individu){
	 if (VERSION_BASE > VERSION) {
		$sql="select * from contraintes.af_get_formation_avancement_reel_individu($formation,$individu) 
					as (parcours int4, etape int4, formation int4, stagiaire int4, type varchar)";
//		echo "sql=$sql<br>";
		$res=pg_query($sql);
	 }
	 else
		return  get_formation_avancement_individu_old($formation);
	 return $res;

}

/**
   Modifiier le nom d'une formation
   @param integer : formation
   @param string : libelle
   @return bool 
*/
function maj_formation_libelle($formation,$libelle){
	$sql="select * from contraintes.af_maj_formation_libelle($formation,'$libelle')";
	return pg_query($sql);
}



function get_formation_etape_individu_encadrants($formation,$individu,$parcours,$etape){
	$sql="select * from contraintes.af_get_formation_etape_individu_encadrants($formation,$individu,$parcours,$etape)
	as(id int4, libelle_pour_parcours varchar, libelle_etape  varchar,libelle_type_etape  varchar, type varchar,  pere int4, parcours int4, obligatoire bool, chrono int4, fin bool,libelle_type_original varchar, modifiable bool, indice int4)";
	 return pg_query($sql);
}

/** Obtenir les infos des sections d'une formation
	@param integer : numero de formation
	@return integer: le resultat de la requete :(espace int4,libelle_espace varchar,section int4,libelle_section varchar,icone int4,formation int4) order by libelle_espace,libelle_section
*/
function get_formation_sections($formation){
	$sql="select * from contraintes.af_get_formation_sections($formation) 
	as(espace int4,libelle_espace varchar,section int4,libelle_section varchar,icone int4,formation int4) order by libelle_espace,libelle_section";
	//echo $sql;
	return pg_query($sql);
}

/** Obtenir les infos des sections d'un espace
	@param integer : numero de formation
	@param integer :numero d'espace
	@return integer: le resultat de la requete :(espace int4,libelle_espace varchar,section int4,libelle_section varchar,icone int4,formation int4) order by libelle_espace,libelle_section
*/
function get_formation_espace_sections($formation,$espace){//espace, libelle_espace, section ,libelle_section ,icone ,formation ,ressource
	$sql="select * from contraintes.af_get_formation_espace_sections($formation,$espace)as(ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut timestamp,fin timestamp,nouveau interval,icone int4) order by libelle_espace,libelle_section";
			//"as(espace int4,libelle_espace varchar,section int4,libelle_section varchar,icone int4,formation int4,ressource int4) order by libelle_espace,libelle_section";
	//echo $sql;
	return pg_query($sql);
}

/**
 * Obtenir les ressources des parcours d'une formation
 * @param integer formation
 * @return integer resultat req
 * 
 */
function get_formation_parcourss_ressources($formation){
$sql="select * from contraintes.af_get_formation_parcourss_ressources($formation) 
	as(ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut timestamp,fin timestamp,nouveau interval,icone int4) order by libelle_espace,libelle_section";
return pg_query($sql);
	 	
}

/**
 * Obtenir toutes les ressources d'une formation (parcours compris)
 * @param integer formation
 * @return resultat req
 * 
 */
function get_formation_all_ressources_individu($formation,$all,$individu){
	$sql="select * from contraintes.af_get_formation_all_ressources_individu($formation,$all,$individu) 
		as(ressource int4,fichier int4,espace int4,libelle_espace varchar,section int4,libelle_section varchar,titre varchar,formation int4,type int4,mots_cles text,debut timestamp,fin timestamp,nouveau interval,icone int4) order by libelle_espace,libelle_section";
		//echo "sql=$sql<br>";
	return pg_query($sql);	
}

/**
 * Obtenir un boolean pour connaitre la nouveaute ou non d'une ressource
 * @param integer: le numero de la formation
 * @param integer: le numero d'un espace
 * @param integer: le numero d'une section
 * 
 *@return boolean: true si la ressource est nouvelle, sinon false.
 */
function get_formation_ressource_nouveaute($formation,$espace,$section,$ressource){
	$sql="select * from contraintes.af_get_formation_ressource_nouveaute($formation,$espace,$section,$ressource) ";
//	echo "$sql<br>";
	return pg_query($sql);
}


/**
 * Obtenir un boolean pour savoir si il y a au moin une ressource nouvelle
 * @param integer: le numero de la formation
 * 
 *@return boolean: true si il y a une ressource nouvelle, sinon false.
 */
function get_formation_ressources_nouveaute($formation){
	$sql="select * from contraintes.af_get_formation_ressources_nouveaute($formation) as valeur";
	return pg_query($sql);
}

/**
 * Enregistre les temps de session pour unr formation
 * @param integer : formation
 * @param time : dur�e d'inactivit�
 * @param time : dur�e max de session
 */
 function set_formation_session($formation,$duree_inactif,$duree_session){
 	$sql="select * from contraintes.af_set_formation_session($formation,'$duree_inactif','$duree_session')";
 	return pg_query($sql);
 }
 
 /**
  * Obtenir les temps de session pour une formation
  * @param integer : formation
  * @return : res req
  * 
  */
  function get_formation_session($formation){
  	$sql="select * from contraintes.af_get_formation_session($formation) as (formation int4,inactivite time ,duree_session time)";
  	return pg_query($sql);	
  }
  
  /**
   * Supprime les temps de session pour une formation
   * @param integer : formation
   */
   function del_formation_session($formation){
   		$sql="select * from contraintes.af_del_formation_session($formation)";
   		return pg_query($sql);	
   }
   
   /**
    * Mettre � jour le booleen pour la fusion des ressource
    * @param integer : formation
    * @param bool : true=fusion false=pas de fusion 
    * @return : bool
    */
    function maj_formation_ressources_fusionnees($formation,$fusion){
    	$sql="select * from contraintes.af_maj_formation_ressources_fusionnees($formation,$fusion)";
    	return pg_query($sql);
    }
    
   /**
    * Retourne l'evolution reelle d'un individu pour une formation
    * @param integer : formation
    * @param integer : individu
    * @return evolution (nb-etapes faites,total)
    */
    function get_formation_evolution_reelle_individu($formation,$individu){
    	$sql="select * from contraintes.af_get_formation_evolution_reelle_individu($formation,$individu) ";
    	//echo "sql=$sql<br>";
    	return pg_query($sql); 
    }

   /**
    * Supprime une formation
    * @param integer $formation formation
    * @return boolean
    */
    function del_formation($formation){
    	$sql="select * from contraintes.af_del_formation($formation) ";
    	//echo "sql=$sql<br>";
    	return sql2Bool(pg_fetch_result(pg_query($sql),0,0)); 
    }
?>