<?php
/*
	This file is part of Astairs.

    Astairs is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    Astairs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Astairs.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2006, 2007, 2008, 2009 AFIDE
    
	Auteurs:

	Said Ladjani <sladjani@yahoo.fr>
	Remi Cocquet <remi.cocquet@gmail.com>

*/

/**
 * Objet de gestion des Modules Astairs.
 * Cet objet permet la detection de Modules dans un repertoire
 * ainsi que leur stockage et r�cup�ration en base de donn�es.
 */
 class ModuleBuilder{

	/**
	 * @var object[] Module Les Modules d�tect�s.
	 */
	var $modules_detectes;
	
	/**
	 * @var object[] Module Les Modules actifs.
	 */	
	var $modules_actifs;	

	/**
	 * Constructeur.
	 * Initialise les attibuts.
	 */
	function ModuleBuilder(){
		$this->modules_detectes=array();
		$this->modules_actifs=array();
	}
	
	
	//---------------------------------------------------------------
	// 				MODULES DETECTES
	//---------------------------------------------------------------
	/**
	 * Charger un module � partir de son emplacement.
	 * L'emplacement doit bien �tre un Module.
	 * @param string $root Chemin racine dans lequel est localis� le repertoire.
	 * @param string $repertoire Nom du r�pertoire du module � charger. 
	 * @return object Module L'objet Module correspondant au repertoire. 
	 */
	function loadModule($root,$repertoire){
		if(!ModuleBuilder::isModule($root."/".$repertoire))
			die("ModuleBuilder::loadModule: Le r�pertoire $repertoire dans $root n'est pas un module.");
		//creation du parseur xml
		$xml_ml= new ModuleLoader($root,$repertoire);
		return $xml_ml->getModule();
	}

	/**
	 * Analyse les r�pertoires et cr�e les Modules associ�s.
	 * Cette m�thode permet l'affectation de l'attribut $modules_detectes
	 * @param string Chemin � partir duquel on souhaite detecter les Modules.
	 * @param string[] Tableau contenant une liste de repertoire � ignorer lors de la detection.
	 */
	function &detectModules($root,$a_ignorer){
		$handle=opendir($root);
		
		if ($handle){
			while ($fichier = readdir($handle)){
	
				if(is_dir($root."/".$fichier)&&!in_array($fichier,$a_ignorer)){
				
	  				if($this->isModule($root."/".$fichier)){
	  					$this->modules_detectes[]=$this->loadModule($root,$fichier);
	  				}else{
	  					die("Erreur: Le repertoire $root/$fichier n'est pas un module valide (manifest.xml manquant)");
	  				}
			    }
			}	
	
		closedir($handle);
		}
	}
	
	/**
	 * R�organisation des Modules d�tect�s.
	 * Cette m�thode permet, � partir d'un ensemble de Modules detect� sans sous Modules, de les r�organiser
	 * en Modules avec sous Modules.
	 * Les Modules correspondants � des sous Modules sont supprim�s des modules detect�s puis associ�s en tant que sous Modules aux Modules correspondants.
	 */
	function &buildModules(){
		if(is_array($this->modules_detectes)){
			foreach($this->modules_detectes as $i => $module){
				$sub_modules=&$module->getSubModules();
				foreach($sub_modules as $k => $sub_module){
					foreach($this->modules_detectes as $j => $module_a_comparer){
						if($sub_module->getCode()==$module_a_comparer->getCode()){
							$module->setSubModule($k,$module_a_comparer);
							unset($this->modules_detectes[$j]);
						}
					}
				}
			}
		}
		//réindexation
		$this->modules_detectes=array_values($this->modules_detectes);
	}

	/**
	 * Obtenir les objets Modules qui ont été détectés.
	 * Affecte l'identifiant de BDD aux modules détectés installés. 
	 * @param integer $strict Si true, retourne uniquement les detectés non installés
	 * @return object[] Module Les modules détectés
	 */
	function getModulesDetectes($strict=false){
		if($strict){
			$tmp=array();			
			$tmp=array_filter($this->modules_detectes,array($this,'isModuleDetectedNotInstalled'));
		}else{
			$tmp=$this->modules_detectes;
			array_walk($tmp,array($this,'initModuleFromDB'));
			
		}
		return $tmp;
	}
	
	/**
	 * Enregistrer en base les modules d�tect�s.
	 * @access public
	 */
	function saveModules(){
		if(is_array($this->modules_detectes)){
			foreach($this->modules_detectes as $i => $module){
				$this->saveModule(&$module);
			}
		}else{
			die("ModuleBuilder::saveModules(): Aucun module n'a �t� detect�.");
		}	
	}
	
	
	/**
	 * Enregistrer en base un Module.
	 * Enregistre l'ensemble des propri�t� du Module, cad ses pages et ses sous Modules.
	 * En cas d'erreur, affiche un message d'erreur et arrete l'ex�cution.
	 * @access private
	 * @param object Module $module Le Module � enregistrer en base de donn�es.
	 * @return integer L'identifiant en base de donn�es du Module ainsi cr�e. 
	 */
	function saveModule(&$module){
		//enregistrement en base du module
		$id=set_module(addslashes($module->getLibelleCode()),$module->getCode(),$module->getType(),$module->getVersion(), $module->getDirectory(), bool2Sql(false),$module->getDescription());
		if(!$id){
			die("ModuleBuilder::saveModule(): Erreur cr�ation Module.");	
		}
		
		//enregistement de ses pages
		$pages=$module->getPages();
		$module->setId($id);
		foreach($pages as $j => $page){
			set_module_page($id,addslashes($page->getLibelle()),$page->getType(),$page->getURL(),$page->getDescription());
		}
		
		//enregistrement de ses sous modules
		$sub_modules=$module->getSubModules();
		if(is_array($sub_modules)){
			foreach($sub_modules as $k => $sub_module){
				$id_sub=ModuleBuilder::saveModule($sub_module);
				set_module_submodule($id,$id_sub,$sub_module->getCode(),$sub_module->getVersion());
			}
		}
		return $id;	
	}
	
	//---------------------------------------------------------------
	// 				MODULES ACTIFS
	//---------------------------------------------------------------
	
	/**
	 * Chargement des Modules actifs presents en base de donnees.
	 * Affecte l'attribut $modules_actifs.
	 */
	function &loadModulesActifs(){
		$res_modules= get_modules('true');
		while($row_module=pg_fetch_row($res_modules)){ 
			$this->modules_actifs[]=new Module($row_module[0],$row_module[2],$row_module[4],$row_module[5],$row_module[1],$row_module[3],$row_module[7]); //($id,$code,$version,$directory, $libelle="",$type=""){
		}
	}
	
	function isModuleDetectedInstalled($module){
		$id=ModuleBuilder::getModuleId($module);
		die ($id);
		if(empty($id))return false;
		return true;
	}
	function isModuleDetectedNotInstalled($module){
		return !$this->isModuleDetectedInstalled($module);
	}
	
	function getModuleId($module){
		$res=get_module($module->getCode());
		return pg_fetch_result($res,0,0);
	}
	
	function initModuleFromDB(&$module){
		$res=get_module($module->getCode());
		$row=pg_fetch_row($res);
		if(!empty($row[0])){
			$module->setId($row[0]);
			$module->setActive(sql2Bool($row[6]));
		}
		$tmp=&$module->getSubModules();
		array_walk($tmp,array($this,'initModuleFromDB'));
	}
	
	/**
	 * Obtenir tous les objets Modules install�s (actifs et inactifs)
	 * 
	 * @return object[] Module Les modules . 
	 */
	function getModulesInstalled(){
		$res_modules= get_modules();
		$modules=array();
		while($row_module=pg_fetch_row($res_modules)){
			$modules[]=new Module($row_module[0],$row_module[2],$row_module[4],$row_module[5],$row_module[1],$row_module[3],$row_module[7],sql2Bool($row_module[6])); //($id,$code,$version,$directory, $libelle="",$type=""){
		}
		usort($modules,array($this,'compareTo'));
		return $modules;
	}

	function unsaveModule($module){
		$res=del_module($module->getId());
		return $res;
	}
	
	
	/**
	 * Obtenir tous les objets Modules install�s (actifs et inactifs)
	 * 
	 * @return object[] Module Les modules . 
	 */
	function getModulesNotInstalled(){
		$res_modules= get_modules();
		while($row_module=pg_fetch_row($res_modules)){
			$modules[]=new Module($row_module[0],$row_module[2],$row_module[4],$row_module[5],$row_module[1],$row_module[3],$row_module[7],sql2Bool($row_module[6])); //($id,$code,$version,$directory, $libelle="",$type=""){
		}
		usort($modules,array($this,'compareTo'));
		return $modules;
	}	

		
	/**
	 * Obtenir les objets Modules qui sont actifs.
	 * @return object[] Module Les modules actifs. 
	 */
	function &getModulesActifs(){
		return $this->modules_actifs;
	}
	
	/**
	 * R�organisation des Modules actifs.
	 * Cette m�thode permet, � partir d'un ensemble de Modules detect� sans sous Modules, de les r�organiser
	 * en Modules avec sous Modules.
	 * Les Modules correspondants � des sous Modules sont supprim�s des modules detect�s puis associ�s en tant que sous Modules aux Modules correspondants.
	 */
	function &buildModulesActifs(){
		if(is_array($this->modules_actifs)){
			foreach($this->modules_actifs as $i => $module){
//				$module->loadSubModules();
				$sub_modules=&$module->getSubModules();
				foreach($sub_modules as $k => $sub_module){
					foreach($this->modules_actifs as $j => $module_a_comparer){
						if($sub_module->getCode()==$module_a_comparer->getCode()){
							$module->setSubModule($k,$module_a_comparer);
							unset($this->modules_actifs[$j]);
						}
					}
				}
			}
		}
		//réindexation
		$this->modules_actifs=array_values($this->modules_actifs);
	}

	//---------------------------------------------------------------
	// 				DIVERS
	//---------------------------------------------------------------
	
	/**
	 * Fonction de comparaison de modules.
	 * Compare les lib�ll�s des 2 modules pass�s en parametres.
	 * @param object Module $a
	 * @param object Module $b
	 * @return integer 1 si le Module $a est sup�rieur au module $b, -1 si inf�rieur et 0 si egaux/
	 */
	function compareTo($a,$b){
		return strcmp($a->getLibelle(),$b->getLibelle());
	}
		
	/**
	 * Savoir si un r�pertoire est bien un module.
	 * Un r�pertoire est un r�pertoire de module si celui-ci contient un fichier manifest.xml
	 * @param string $repertoire Le r�pertoire � tester.
	 * @return boolean true si le repertoire est un modulen false sinon.
	 */
	function isModule($repertoire){
		if(file_exists($repertoire."/manifest.xml")){
			return true;
		}
		else{
			return false;
		}
	}
}
?>
