<?php
/*
	This file is part of Astairs.

    Astairs is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    Astairs is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Astairs.  If not, see <http://www.gnu.org/licenses/>.

	Copyright (C) 2006, 2007, 2008, 2009 AFIDE
    
	Auteurs:

	Said Ladjani <sladjani@yahoo.fr>
	Remi Cocquet <remi.cocquet@gmail.com>

*/

/**
Fichier de d�finition de la classe Serie

@package serie
*/
class Serie extends Liste{

	/**
	 * @var integer Identifiant du Bareme de la S�rie
	 */
	var $bareme;

	/**
	 * @var integer Chrono global, en secondes, de la s�rie
	 */	
	var $chrono;

	/**
	 * Construit une Serie.
	 * La Serie peut �tre construit � partir d'un identifiant (1 seul parametre), auquel cas les donn�es sont r�cup�r�es en base de donn�es,
	 * ou � partir de l'ensemble des donn�es (tous les parametres), pour ensuite �tre enregistr� en base.
	 *  
	 * @param integer $id identifiant de l'Etape
	 * @param string $libelle Libell�
	 * @param string $date_creation date de cr�ation (format SQL)
	 * @param integer $bareme Identifiant du Bareme de la S�rie
	 * @param integer $chrono Chrono global, en secondes, de la s�rie
	 */
	function Serie($id, $libelle="Etape Serie", $date_creation="27.11.2006",$bareme=0,$chrono=0){
		if(func_num_args()==1){
			$res_etape=get_serie($id);
			$row_etape=pg_fetch_row($res_etape);
			//print_rr($row_etape);
			$libelle=$row_etape[1];
			$date_creation=$row_etape[4];
			$bareme=$row_etape[2];
			$chrono=$row_etape[3];
		}else{
	        //echo "+ 1 arg<br>";
		}
		parent::Liste($id, $libelle,$date_creation);
		$this->initSerie($bareme,$chrono);
	}
	
	/**
	 * Initialise les attributs propres � l'Etape.
	 * @param integer $bareme Identifiant du Bareme de la S�rie
	 * @param integer $chrono Chrono global, en secondes, de la s�rie
	 */			
	function initSerie($bareme,$chrono){
		parent::initListe(TYPE_LISTE_SERIE);
		$this->bareme=$bareme;
		$this->chrono=$chrono;
	}
	
	/**
	 * Savoir si la s�rie est al�atoire.
	 * @return boolean true si la s�rie est al�atoire, false sinon.
	 */
	function isAleatoire(){
		$t_type=explode('_',parent::getType());
		if($t_type[1]!=1)return true;
		else return false;
	}

	/**
	 * Savoir si la s�rie est bloquante.
	 * @param integer Identifiant de la formation
	 * @param integer Identifiant du parcours
	 * @return boolean true si la s�rie est bloquante, false sinon.
	 */	
	function isBloquante($formation, $parcours){
		$res=get_serie_bloquante_configuration($this->id, $formation, $parcours);
		
		if( pg_num_rows($res)>0)
			if(pg_fetch_result($res,0,0)  )
				return true;
		return false;
	}
	
	
	
	/**
	 * Obtenir la configuration de la S�rie Bloquante.
	 * @param integer Identifiant de la formation
	 * @param integer Identifiant du parcours
	 * @return integer Resultat de la requete (serie int4,formation int4,parcours int4,seuil int4,etape_retour int4,tentatives int4,correction bool,blocage_parcours bool)
	 */
	function getBloquageConfiguration($formation, $parcours){
	//print_rr($this);
		return get_serie_bloquante_configuration($this->id, $formation, $parcours);
	}
	
	/**
	 * Obtenir le type de la S�rie.
	 * @return integer Le type de S�rie
	 */
	function getTypeSerie(){
		$t_type=explode('_',parent::getType());
		return $t_type[1];
	}
	
	/**
	 * Obtenir le type (complet) de la S�rie
	 * @return string Le type de S�rie
	 */
	function getType(){
		return parent::getType().'_'.TYPE_LISTE_SERIE;
	}
	
	/**
	 * Obtenir le type entier de la serie (en fonction de ses �valuations)
	 * @return string le type entier de la serie
	 */
	 function getTypeCalcule(){
	 	
	 	$res=get_etape_type_sous_type($this->id,pg_fetch_result(get_etape($this->id),0,2),'');
	 	return pg_fetch_result($res,0,0);	
	 }	
	
	/**
	 * Obtenir les Evaluations de la S�rie
	 * @return integer Le resultat de la requete :(id int4, libelle varchar, serie int4, pere int4, commentaire varchar, bareme int4, chrono int4, points int4, type int4 )
	 */
	function getEvaluations(){
		return get_serie_evaluations($this->id);
	}

	function getObjEvaluation($indice){
		$res=$this->getEvaluations();
		$row_evaluation=pg_fetch_row($res,$indice -1);
	 	return EvaluationFactory::createEvaluation($row_evaluation[0],$row_evaluation[8]);
	}

	/**
	 * Obtenir les types des Evaluations de la S�rie.
	 * Retourne les identifiants, lib�ll�s et le nombre d'Evaluation de chacuns de types.
	 * @return integer : resultat de la requete (id int4, libelle  varchar, code varchar,nb integer))
	 */	
	function getEvaluationsTypes(){
		return get_serie_types_evaluations_type($this->id,'FALSE');
	}
	
	/**
	 * Obtenir le chrono de la S�rie.
	 * @return integer: le chrono en secondes. 
	 */
	function getChrono() { return $this->chrono ; }

	/**
	 * Affecter le chrono de la S�rie.
	 * @param integer $chrono Le chrono en secondes. 
	 */
	function setChrono($chrono) { $this->chrono=$chrono ; }
	
	/**
	 * Obtenir le bareme de la S�rie.
	 * @return integer: l'identifiant du bareme. 
	 */
	function getBareme() {return $this->bareme ; }
	
	/**
	 * Affecter le bareme de la S�rie.
	 * @return integer $bareme L'identifiant du bareme. 
	 */
	function setBareme($bareme) { $this->bareme=$bareme ; }
	
	/**
	 * Obtenir l'objet correspondant au bareme de la S�rie.
	 * @return Object: l'identifiant du bareme. 
	 */
	
	function getObjBareme(){
		
		if(!empty($this->bareme)){
			return pg_fetch_object(get_bareme($this->bareme));
		}else{
			return null;
		}
	}
	 	
    /**
    * Fonction qui retourne le nombre total de secondes de la s�rie (global ou individuel)
    *
    * Cette fonction calcule le nombre total de seconde de la s�rie.
    * Si il n'y a pas de chrono, on retourne -1, si c'est un chrono global, on donne sa valeur
    * et si ce sont des chrono individuels, on calcule leur somme. 
    * 
    * NB: L'entier retourn� est toujours positif, sauf s'il n'y a pas de chrono.
    * 
    * @return integer : Entier positif �gal au nombre total de secondes de la s�rie (-1 pas de chrono -seul cas ou la valeur retourn�e est n�gative-)
    */
    function getChronoTotal()
    {
    	if(!empty($this->chrono)) return $this->chrono;
    	
    	if(is_null($this->chrono)){
    		$result=get_serie_somme_chronos($this->id);
            $temps= (pg_result($result,0,0)) ;
            //echo "\$temps=$temps";
            if(!empty($temps))
            	return $temps;
    	}	
            switch($this->chrono)
            {
                    // Pas de chrono
                    case '-1' :
                   		return -1 ;

                    // Chrono global, on retourne la valeur de $this->chrono
                    default :
                            return $this->chrono ;
            }
    }
    
    function getChronoTotalFormate(){
    	if(is_null($this->chrono))return null;	
    	switch($this->chrono){
    		// Pas de chrono
			case -1 :
	    		return null;
           // Chrono individuel
            case 0 :
	    		return sec_to_txt(($this->getChronoTotal()*-1));
	    	default:
	    		return sec_to_txt($this->getChronoTotal());
    	}
    }
    
    /**
	 * Obtenir l'icone associ�e � l'Etape.
	 * @return string Le nom du fichier icone
	 */
	function getIcone(){
		return "ico_serie.png";
	}
	
	/**
	 * Obtenir le nombre de points attribu�s � une �valuataion pour la S�rie.
	 * @param integer $evaluation L'identifiant de l'Evaluation
	 * @return integer le nombre de points attribu�s
	 */
	function getPoints($evaluation){
		
		$res=get_serie_evaluation($this->id,$evaluation);
		return pg_fetch_result($res,0,7);
	}
	
	function getCommentaire($evaluation){
		$res=get_serie_evaluation($this->id,$evaluation);
		return pg_fetch_result($res,0,4);
	}
		
	/**
	 * Obtenir le bareme d'une Evaluation pour la S�rie.
	 * @param integer $evaluation L'identifiant de l'Evaluation
	 * @return integer: l'identifiant du bareme. 
	 */
	function getBaremeEvaluation($evaluation){
		
		if(!empty($this->bareme)){
			return $this->getBareme();
		}else{
			$res=get_serie_evaluation($this->id,$evaluation);
			return pg_fetch_result($res,0,5);
		}
	}
	
	
	function getChronoEvaluation($evaluation){
		
		$res=get_serie_evaluation($this->id,$evaluation);
		$chrono_evaluation=pg_fetch_result($res,0,6);
		
		if(empty($chrono_evaluation)){
			return $this->getChrono();
		} 
		
		return $chrono_evaluation;
	}
	
	/**
	 * Obtenir l'objet correspondant au bareme d'une Evaluation pour la S�rie.
	 * @param integer $evaluation L'identifiant de l'Evaluation
	 * @return Object: l'identifiant du bareme. 
	 */
	function getObjBaremeEvaluation($evaluation){
	
		if(empty($this->bareme)){
			$bareme_evaluation=$this->getBaremeEvaluation($evaluation);
			if(empty($bareme_evaluation)){
				return null;
			}
			else{
				return pg_fetch_object(get_bareme($bareme_evaluation));
			}
		}else{
			//print_rr($this->getObjBareme());
			return $this->getObjBareme();
		}
	}
	
}
?>