//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <gtkmm.h>

#include "document-dialog.h"
#include "document.h"
#include "main.h"
#include "ucompose.hpp"

std::string DocumentDialog::default_tag_name = "type your tag";
DocumentDialog::DocumentDialog(Tapioca::Document *d)
    : name_column("Name", name_renderer)
{
  document = d;
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path()
                                     + "/document-dialog.gtk");

  xml->get_widget("dialog", dialog);
  dialog->set_icon_from_file(Main::get_data_path() + "/icon.png");

  xml->get_widget("title_entry", title_entry);
  xml->get_widget("description_textview", description_textview);
  xml->get_widget("thumbnail_image", thumbnail_image);
  xml->get_widget("lat_spinbutton", lat_spinbutton);
  xml->get_widget("lon_spinbutton", lon_spinbutton);

  if (document)
    {
      title_entry->set_text(d->get_title());
      description_textview->get_buffer()->set_text(d->get_description());
      lat_spinbutton->set_value(d->get_latitude());
      lon_spinbutton->set_value(d->get_longitude());
      thumbnail_image->set(d->get_thumbnail());
    }

  title_entry->signal_changed().connect
    (sigc::mem_fun(this,&DocumentDialog::on_title_changed));
  description_textview->get_buffer()->signal_changed().connect
    (sigc::mem_fun(this, &DocumentDialog::on_description_changed));
  lat_spinbutton->signal_value_changed().connect
    (sigc::mem_fun(this,&DocumentDialog::on_lat_changed));
  lon_spinbutton->signal_value_changed().connect
    (sigc::mem_fun(this,&DocumentDialog::on_lon_changed));

    xml->get_widget("tags_count_label", tags_count_label);
    xml->get_widget("treeview", treeview);
    tags_list = Gtk::ListStore::create(tags_columns);
    treeview->set_model(tags_list);
    treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
    treeview->get_selection()->signal_changed().connect
      (sigc::mem_fun(*this, &DocumentDialog::on_tag_selected));
    name_renderer.property_editable() = true;
    name_renderer.signal_edited().connect
      (sigc::mem_fun(*this, &DocumentDialog::on_tag_edited));
    name_column.set_cell_data_func
      (name_renderer, sigc::mem_fun(*this, &DocumentDialog::cell_data_tag));
    treeview->append_column(name_column);
    treeview->get_selection()->signal_changed().connect
      (sigc::mem_fun(*this, &DocumentDialog::on_tag_selected));
    fill_tags();
    xml->get_widget("add_button", add_button);
    add_button->signal_clicked().connect
      (sigc::mem_fun(this,&DocumentDialog::on_add_clicked));
    xml->get_widget("remove_button", remove_button);
    remove_button->signal_clicked().connect
      (sigc::mem_fun(this,&DocumentDialog::on_remove_clicked));
}

DocumentDialog::~DocumentDialog()
{
  delete dialog;
}

void DocumentDialog::set_parent_window(Gtk::Window &parent)
{
    dialog->set_transient_for(parent);
    //dialog->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
}

void DocumentDialog::hide()
{
  dialog->hide();
}

void DocumentDialog::run()
{
  int response = dialog->run();
  if (response == Gtk::RESPONSE_ACCEPT && document)
    {
      document->set_lat_long(lat_spinbutton->get_value(), lon_spinbutton->get_value());
      document->set_title(title_entry->get_text());
      document->set_description(description_textview->get_buffer()->get_text());
    }
  if (document)
    document->remove_tag(default_tag_name);
}

void DocumentDialog::on_title_changed()
{
  if (document)
    document->set_title(title_entry->get_text());
}

void DocumentDialog::on_description_changed()
{
  if (document)
    document->set_description(description_textview->get_buffer()->get_text());
}

void DocumentDialog::on_lat_changed()
{
  if (document)
    document->set_lat_long(lat_spinbutton->get_value(), 
                           lon_spinbutton->get_value());
}

void DocumentDialog::on_lon_changed()
{
  if (document)
    document->set_lat_long(lat_spinbutton->get_value(), 
                           lon_spinbutton->get_value());
}

void DocumentDialog::fill_tags()
{
  if (document)
    {
      std::list<std::string> tags = document->get_tags();
      for (std::list<std::string>::iterator it = tags.begin(); 
           it != tags.end(); it++)
        add_tag(*it);
      tags_count_label->set_text(String::ucompose("%1", tags.size()));
    }
}

void DocumentDialog::add_tag(std::string tag)
{
  Gtk::TreeIter i = tags_list->append();
  (*i)[tags_columns.name] = tag;
}

std::string DocumentDialog::get_selected_tag()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring tag = row[tags_columns.name];
      return tag;
    }
  return "";
}

void DocumentDialog::on_tag_selected()
{
  if (document->has_tag(default_tag_name) == true)
    {
      Gtk::TreeModel::Path path;
      std::list<std::string> tags = document->get_tags();
      path = Gtk::TreeModel::Path(String::ucompose("%1", tags.size()-1));
      treeview->get_selection()->select(path);
      on_remove_clicked();
    }
}

void DocumentDialog::on_add_clicked()
{
  std::string tag = default_tag_name;
  if (document->has_tag(tag) == false)
    {
      Gtk::TreeIter i = tags_list->append();
      (*i)[tags_columns.name] = tag;
      Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
      selection->select(i);
      document->add_tag(tag);
    }
  Gtk::TreeViewColumn *column = NULL;
  Gtk::TreeModel::Path path;
  treeview->get_cursor(path, column);
  std::list<std::string> tags = document->get_tags();
  path = Gtk::TreeModel::Path(String::ucompose("%1", tags.size()-1));
  treeview->set_cursor(path, name_column, true);
  tags_count_label->set_text(String::ucompose("%1", tags.size()));
}

void DocumentDialog::on_remove_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow) 
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring tag = row[tags_columns.name];
      document->remove_tag(tag);
      tags_list->erase(iterrow);
      std::list<std::string> tags = document->get_tags();
      tags_count_label->set_text(String::ucompose("%1", tags.size()));
    }
}

void
DocumentDialog::cell_data_tag(Gtk::CellRenderer *renderer, const Gtk::TreeIter &i)
{
  Gtk::CellRendererText*cell = dynamic_cast<Gtk::CellRendererText*>(renderer);
  cell->property_text() = (*i)[tags_columns.name];
}

void
DocumentDialog::on_tag_edited(const Glib::ustring &path,
                             const Glib::ustring &new_text)
{
  if (new_text == default_tag_name)
    {
      on_remove_clicked();
      return;
    }
  Gtk::TreeModel::iterator iterrow = tags_list->get_iter(Gtk::TreePath(path));
  Gtk::TreeModel::Row row = *iterrow;
  Glib::ustring old_tag = row[tags_columns.name];
  row[tags_columns.name] = new_text;
  document->update_tag(old_tag, new_text);
}

void DocumentDialog::on_cancel_tag_edit()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow) 
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring tag = row[tags_columns.name];
      if (tag == default_tag_name)
        on_remove_clicked();
    }
}
