//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <gtkmm.h>

#include "gallery-dialog.h"
#include "gallery.h"
#include "main.h"
#include "ucompose.hpp"

GalleryDialog* GalleryDialog::create(Tapioca::Gallery *g, const Tapioca::Gallery *orig, bool uneditable)
{
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path()
                                     + "/gallery-dialog.gtk");
  GalleryDialog *dialog;
  xml->get_widget_derived("dialog", dialog);
  dialog->on_dialog_loaded(g, orig, uneditable);
  return dialog;
}

GalleryDialog::GalleryDialog(BaseObjectType* baseObject, 
                                     const Glib::RefPtr<Gtk::Builder>& xml)
 : Gtk::Dialog(baseObject)
{
  gallery = NULL;
  original = NULL;
  readonly = false;
  set_icon_from_file(Main::get_data_path() + "/icon.png");

  signal_response().connect
    (sigc::mem_fun(*this, &GalleryDialog::on_response_received));
  xml->get_widget("title_entry", title_entry);
  xml->get_widget("description_textview", description_textview);
  xml->get_widget("thumbnail_image", thumbnail_image);
  xml->get_widget("lat_spinbutton", lat_spinbutton);
  xml->get_widget("lon_spinbutton", lon_spinbutton);
  xml->get_widget("item_tagging_widget_container", 
                  item_tagging_widget_container);
  xml->get_widget("doc_count_label", doc_count_label);
  xml->get_widget("title_label", title_label);
  xml->get_widget("description_label", description_label);
  xml->get_widget("latitude_label", latitude_label);
  xml->get_widget("longitude_label", longitude_label);
  xml->get_widget("private_audience_checkbutton", private_audience_checkbutton);

}

void GalleryDialog::on_dialog_loaded(Tapioca::Gallery *g,
                                     const Tapioca::Gallery *orig, 
                                     bool uneditable)
{
  gallery = g;
  original = orig;
  readonly = uneditable;
  if (gallery)
    {
      title_entry->set_text(gallery->get_title());
      description_textview->get_buffer()->set_text(gallery->get_description());
      lat_spinbutton->set_value(gallery->get_latitude());
      lon_spinbutton->set_value(gallery->get_longitude());
      thumbnail_image->set(gallery->get_thumbnail_image());
      private_audience_checkbutton->set_active(gallery->get_private_audience());
    }

  item_tagging_widget = new ItemTaggingWidget();
  item_tagging_widget->get_widget()->reparent(*item_tagging_widget_container);
  item_tagging_widget->fill_tags(gallery->get_tags());
  if (gallery)
    doc_count_label->set_text(String::ucompose("%1", gallery->size()));
  if (readonly)
    {
      title_entry->set_sensitive(false);
      description_textview->set_sensitive(false);
      lat_spinbutton->set_sensitive(false);
      lon_spinbutton->set_sensitive(false);
      item_tagging_widget->get_widget()->set_sensitive(false);
    }
  on_title_changed();
  on_description_changed();
  on_lat_changed();
  on_lon_changed();
  on_private_audience_toggled();
  title_entry->signal_changed().connect
    (sigc::mem_fun(this,&GalleryDialog::on_title_changed));
  description_textview->get_buffer()->signal_changed().connect
    (sigc::mem_fun(this, &GalleryDialog::on_description_changed));
  lat_spinbutton->signal_value_changed().connect
    (sigc::mem_fun(this,&GalleryDialog::on_lat_changed));
  lon_spinbutton->signal_value_changed().connect
    (sigc::mem_fun(this,&GalleryDialog::on_lon_changed));
  private_audience_checkbutton->signal_toggled().connect
    (sigc::mem_fun(this,&GalleryDialog::on_private_audience_toggled));
}

GalleryDialog::~GalleryDialog()
{
}

void GalleryDialog::on_response_received(int response)
{
  if (response == Gtk::RESPONSE_ACCEPT && gallery)
    {
      gallery->set_lat_long(lat_spinbutton->get_value(), lon_spinbutton->get_value());
      gallery->set_title(title_entry->get_text());
      gallery->set_description(description_textview->get_buffer()->get_text());
      gallery->set_tags(item_tagging_widget->get_tags());
    }
}

void GalleryDialog::on_title_changed()
{
  if (gallery)
    gallery->set_title(title_entry->get_text());
  Pango::AttrList attrs;
  if (original && gallery &&
      original->get_title() != gallery->get_title())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  title_label->set_attributes(attrs);
}

void GalleryDialog::on_description_changed()
{
  if (gallery)
    gallery->set_description(description_textview->get_buffer()->get_text());
  Pango::AttrList attrs;
  if (original && gallery && 
      original->get_description() != gallery->get_description())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  description_label->set_attributes(attrs);
}

void GalleryDialog::on_lat_changed()
{
  if (gallery)
    gallery->set_lat_long(lat_spinbutton->get_value(), 
                          lon_spinbutton->get_value());
  Pango::AttrList attrs;
  if (original && gallery &&
      original->get_latitude() != gallery->get_latitude())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  latitude_label->set_attributes(attrs);
}

void GalleryDialog::on_lon_changed()
{
  if (gallery)
    gallery->set_lat_long(lat_spinbutton->get_value(), 
                          lon_spinbutton->get_value());
  Pango::AttrList attrs;
  if (original && gallery &&
      original->get_longitude() != gallery->get_longitude())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  longitude_label->set_attributes(attrs);
}
    
void GalleryDialog::on_private_audience_toggled()
{
  if (gallery)
    {
      gallery->set_private_audience(private_audience_checkbutton->get_active());
    }
  Gtk::Label *l = dynamic_cast<Gtk::Label*>(private_audience_checkbutton->get_child());
  Pango::AttrList attrs;
  if (original && gallery &&
      original->get_private_audience() != gallery->get_private_audience())
    {
      Pango::Attribute weight = 
        Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  l->set_attributes(attrs);
}
