//  Copyright (C) 2010, 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <gtkmm.h>

#include "document-dialog.h"
#include "document.h"
#include "main.h"
#include "ucompose.hpp"

DocumentDialog::Tab DocumentDialog::last_tab_opened;
Gtk::Allocation DocumentDialog::last_window_size;

DocumentDialog* DocumentDialog::create(Tapioca::Document *d, const Tapioca::Document *orig, std::list<std::string> tags, bool uneditable)
{
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path()
                                     + "/document-dialog.gtk");
  DocumentDialog *dialog;
  xml->get_widget_derived("dialog", dialog);
  dialog->on_dialog_loaded(d, orig, tags, uneditable);
  return dialog;
}

void DocumentDialog::on_dialog_loaded(Tapioca::Document *d,
                                      const Tapioca::Document *orig, 
                                      std::list<std::string> tags,
                                      bool uneditable)
{
  document = d;
  readonly = uneditable;
  original = orig;
  if (document)
    {
      title_entry->set_text(d->get_title());
      description_textview->get_buffer()->set_text(d->get_description());
      lat_spinbutton->set_value(d->get_latitude());
      lon_spinbutton->set_value(d->get_longitude());
      thumbnail_image->set(d->get_thumbnail());
    }

  item_tagging_widget = new ItemTaggingWidget();
  item_tagging_widget->get_widget()->reparent(*item_tagging_widget_container);
  item_tagging_widget->tags_changed.connect
    (sigc::mem_fun(tags_changed, &sigc::signal<void, std::list<std::string> >::emit));
  item_tagging_widget->fill_tags(document->get_tags(), tags);

  if (readonly)
    {
      title_entry->set_sensitive(false);
      description_textview->set_sensitive(false);
      lat_spinbutton->set_sensitive(false);
      lon_spinbutton->set_sensitive(false);
      item_tagging_widget->get_widget()->set_sensitive(false);
    }
  on_title_changed();
  on_description_changed();
  on_lat_changed();
  on_lon_changed();
  title_entry->signal_changed().connect
    (sigc::mem_fun(this,&DocumentDialog::on_title_changed));
  description_textview->get_buffer()->signal_changed().connect
    (sigc::mem_fun(this, &DocumentDialog::on_description_changed));
  lat_spinbutton->signal_value_changed().connect
    (sigc::mem_fun(this,&DocumentDialog::on_lat_changed));
  lon_spinbutton->signal_value_changed().connect
    (sigc::mem_fun(this,&DocumentDialog::on_lon_changed));
  page_switcher = notebook->signal_switch_page().connect
    (sigc::hide<0>(sigc::mem_fun(*this, 
                                 &DocumentDialog::on_notebook_page_switched)));
}

DocumentDialog::DocumentDialog(BaseObjectType* baseObject, 
                               const Glib::RefPtr<Gtk::Builder>& xml)
 : Gtk::Dialog(baseObject)
{
  set_icon_from_file(Main::get_data_path() + "/icon.png");

  signal_response().connect
    (sigc::mem_fun(*this, &DocumentDialog::on_response_received));
  xml->get_widget("notebook", notebook);
  xml->get_widget("title_entry", title_entry);
  xml->get_widget("description_textview", description_textview);
  xml->get_widget("thumbnail_image", thumbnail_image);
  xml->get_widget("lat_spinbutton", lat_spinbutton);
  xml->get_widget("lon_spinbutton", lon_spinbutton);
  xml->get_widget("item_tagging_widget_container", 
                  item_tagging_widget_container);
  xml->get_widget("title_label", title_label);
  xml->get_widget("description_label", description_label);
  xml->get_widget("latitude_label", latitude_label);
  xml->get_widget("longitude_label", longitude_label);
  xml->get_widget("view_document_button", view_document_button);
  view_document_button->signal_clicked().connect
    (sigc::mem_fun(*this, &DocumentDialog::on_view_document_clicked));
  set_tab(last_tab_opened);
  if (last_window_size.get_width() != 0)
    set_size_request(last_window_size.get_width(), 
                     last_window_size.get_height());
}

DocumentDialog::~DocumentDialog()
{
}

void DocumentDialog::on_response_received(int response)
{
  last_window_size = get_allocation();
  page_switcher.disconnect();
  if (response == Gtk::RESPONSE_ACCEPT && document)
    {
      document->set_lat_long(lat_spinbutton->get_value(), 
                             lon_spinbutton->get_value());
      document->set_title(title_entry->get_text());
      document->set_description(description_textview->get_buffer()->get_text());
      document->set_tags(item_tagging_widget->get_tags());
    }
}

void DocumentDialog::on_title_changed()
{
  if (document)
    document->set_title(title_entry->get_text());
  Pango::AttrList attrs;
  if (original && document &&
      original->get_title() != document->get_title())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  title_label->set_attributes(attrs);
}

void DocumentDialog::on_description_changed()
{
  if (document)
    document->set_description(description_textview->get_buffer()->get_text());
      
  Pango::AttrList attrs;
  if (original && document &&
      original->get_description() != document->get_description())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  description_label->set_attributes(attrs);
}

void DocumentDialog::on_lat_changed()
{
  if (document)
    document->set_lat_long(lat_spinbutton->get_value(), 
                           lon_spinbutton->get_value());
  Pango::AttrList attrs;
  if (original && document &&
      original->get_latitude() != document->get_latitude())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  latitude_label->set_attributes(attrs);
}

void DocumentDialog::on_lon_changed()
{
  if (document)
    document->set_lat_long(lat_spinbutton->get_value(), 
                           lon_spinbutton->get_value());
  Pango::AttrList attrs;
  if (original && document && 
      original->get_longitude() != document->get_longitude())
    {
      Pango::Attribute weight = Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
    }
  longitude_label->set_attributes(attrs);
}

void DocumentDialog::on_notebook_page_switched(guint page_num)
{
  last_tab_opened = Tab(page_num);
}
    
void DocumentDialog::on_view_document_clicked()
{
  if (document)
    view_document.emit(document);
}
