// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <fstream>
#include "session-connector.h"
#include "main.h"
#include "ucompose.hpp"
#include <argz.h>
#include <libsoupmm/session.h>
#include <libsoupmm/message.h>
#include <libsoupmm/uri.h>
#include <libsoupmm/message-body.h>
#include <libsoupmm/cookie-jar.h>
#include <libsoup/soup.h>
#include "session.h"
#include "gallery.h"
#include "document.h"
#include "changed-document.h"
#include "changed-gallery.h"
#include "item.h"
using namespace Tapioca;
double
  SessionConnector::timeout = 10.0;

SessionConnector::SessionConnector (Session * s)
{
  session = s;
  web = Soup::Session::create (true);
  web->property_timeout () = 5;
  web->property_idle_timeout () = 5;
  web->property_max_conns_per_host () = 1;
  web->add_feature_by_type (SOUP_TYPE_COOKIE_JAR);
  last_login = 0;
}

SessionConnector::~SessionConnector ()
{
}

void
SessionConnector::on_pull_documents_attempted (Glib::RefPtr < Soup::Message >
					       &msg, Gallery * gallery,
					       const SessionConnector::
					       SlotPullGallery slot,
					       std::list < std::string >
					       &thumbnail_urls)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len == 0)
	{
	  pull_galleries_failed.emit (session);
	  return;
	}

      get_documents_from_document_listing_page (gallery, std::string (data),
						thumbnail_urls);
      slot (gallery, thumbnail_urls);
    }
  else
    {
      pull_galleries_failed.emit (session);
      return;
    }
}

std::string SessionConnector::extract_document_id_from_thumbnail_url (std::
								      string
								      u) const
{
  // -->http://88.198.21.167/atpic2/8119/38089/0/1977121/160.jpg
  std::vector < Glib::ustring > a = Glib::Regex::split_simple ("/", u);
  if (a.size () < 8)
    return "";
  return a[7];
}

void
SessionConnector::on_thumbnail_downloaded (Glib::RefPtr < Soup::Message >
					   &msg, Document * document,
					   const SessionConnector::
					   SlotPullThumbnail slot)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();

      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len != 0)
	{
	  process_thumbnail (document, data, len);
	  msg.reset ();
	  slot (document);
	}
      else
	{
	  msg.reset ();
	  pull_galleries_failed.emit (session);
	  return;
	}
    }
  else if (msg->property_status_code() == 403)
    {
      printf("forbidden!\n");
    slot (document);
    }
  else
    {
      msg.reset ();
      pull_galleries_failed.emit (session);
      return;
    }
}

void
SessionConnector::on_pull_galleries_attempted (Glib::RefPtr < Soup::Message >
					       &msg)
{
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len == 0)
	{
	  pull_galleries_failed.emit (session);
	  return;
	}
      get_galleries_from_gallery_listing_page (std::string (data));
      //go get the docs for each gallery.
      pulled_galleries.emit (session->get_profile ()->get_galleries ());
    }
  else
    {
    pull_galleries_failed.emit (session);
    }
}

void
SessionConnector::pull_galleries ()
{
  //thumbnail_urls.clear();
  Glib::ustring homepage = session->get_profile ()->get_url ();
  Glib::RefPtr < Soup::Message > msg =
    Soup::Message::create ("POST", homepage + "/home.php");
  web->queue_message (msg,
		      sigc::mem_fun (*this,
				     &SessionConnector::
				     on_pull_galleries_attempted));
  return;
}


bool
SessionConnector::on_download_document_timeout (Glib::RefPtr < Soup::Message >
						&msg)
{
  web->cancel_message (msg, 998);
  return false;
}

void
SessionConnector::download_document (Document * document,
				     const SessionConnector::
				     SlotDownloadDocument slot)
{
  if (document)
    {
      std::map < std::string, std::string > formdata;
      formdata["galleryid"] = document->get_gallery_id ();
      formdata["format"] = "txt";
      Glib::RefPtr < Soup::Message > msg;
      msg =
	Soup::Message::create_form_request ("GET",
					    session->get_profile ()->
					    get_url () + "/pic_list_urls.php",
					    formdata);
      guint status = web->send_message (msg);
      if (status != 200)
	{
	  download_document_failed.emit (document);
	  return;
	}

      if (status == 200)
	{
	  std::string uri =
	    get_document_link_from_link_listing (msg, document);
	  msg.reset ();
	  if (uri != "")
	    {
	      Glib::RefPtr < Soup::Message > m =
		Soup::Message::create ("GET", uri);
	      m->signal_wrote_chunk ().
		connect (sigc::
			 bind (sigc::
			       mem_fun (downloaded_chunk_of_document,
					&sigc::signal < void,
					Document * >::emit), document));
	      sigc::connection timer =
		Glib::signal_timeout ().
		connect (sigc::
			 bind (sigc::
			       mem_fun (*this,
					&SessionConnector::
					on_download_document_timeout), m),
			 10 * 1000);
	      web->queue_message (m, sigc::bind < Document *,
				  const SessionConnector::
				  SlotDownloadDocument,
				  sigc::connection >
				  (sigc::
				   mem_fun (this,
					    &SessionConnector::
					    on_document_downloaded), document,
				   slot, timer));
	    }
	  else
	    download_document_failed.emit (document);
	}
      else
	msg.reset ();
    }
  return;
}

bool
SessionConnector::is_login_necessary ()
{
  double now = (double) time (NULL);
  if ((double) last_login + (timeout * 60.0) < now)
    return true;
  else
    return false;
}

Glib::RefPtr < Soup::Message > SessionConnector::login ()
{
  std::map < std::string, std::string > formdata;
  formdata["login"] = session->get_profile ()->get_username ();
  formdata["password"] = session->get_profile ()->get_password ();
  Glib::ustring homepage = session->get_profile ()->get_url ();
  Glib::RefPtr < Soup::Message > msg =
    Soup::Message::create_form_request ("POST", homepage +
					"/artist_login_check.php", formdata);
  //web->queue_message(msg, sigc::mem_fun(*this, &SessionConnector::on_login_attempted));
  web->send_message (msg);
  return msg;
}

void
SessionConnector::login (sigc::slot < void, bool > slot)
{
  if (session->get_profile ()->get_username () == "" ||
      session->get_profile ()->get_password () == "")
    {
      slot (false);
      return;
    }
  if (is_login_necessary () == false)
    {
      slot (true);
      return;
    }
  Glib::RefPtr < Soup::Message > msg = login ();
  if (msg->property_status_code () == 200)
    on_login_attempted (msg, slot);
  else
    slot (false);
}

void
SessionConnector::on_login_attempted (Glib::RefPtr < Soup::Message > &msg,
				      sigc::slot < void, bool > slot)
{
  bool success = false;
  if (msg->property_status_code () == 200)
    {
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();;
      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len == 0)
	{
	  slot (success);
	  return;
	}
      if (strstr (data, "class=\"error\"") == NULL)
	{
	  last_login = time (NULL);
	  success = true;
	}
    }
  slot (success);
}

std::string SessionConnector::get_document_link_from_link_listing (Glib::
								   RefPtr <
								   Soup::
								   Message >
								   msg,
								   Document *
								   document)
  const
{
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *
    d = buffer.gobj ()->data;
  size_t
    len = buffer.gobj ()->length;
  if (len == 0)
    return "";
  std::string needle = "/" + document->get_id () + "/";
  char *
    argz = NULL;
  size_t
    argz_len = 0;
  argz_create_sep (d, '\n', &argz, &argz_len);
  char *
    entry = NULL;
  char *
    link = NULL;
  while ((entry = argz_next (argz, argz_len, entry)))
    {
      if (strstr (entry, needle.c_str ()) != NULL)
	link = entry;
    }
  if (!link)
    return "";
  return std::string (link);
}

void
SessionConnector::on_document_downloaded (Glib::RefPtr < Soup::Message > msg,
					  Document * document,
					  const SessionConnector::
					  SlotDownloadDocument slot,
					  sigc::connection timer)
{
  int status = msg->property_status_code ();
  if (status == 998)
    {
      msg.reset ();
      download_document_failed.emit (document);
      return;
    }
  timer.disconnect ();
  if (msg->property_status_code () == 200)
    {
      web->cancel_message (msg, msg->property_status_code ());
      downloaded_chunk_of_document.emit (document);
      Soup::MessageBody body (msg->gobj ()->response_body);
      body.set_accumulate (true);
      Soup::Buffer buffer = body.flatten ();

      const char *data = buffer.gobj ()->data;
      guint len = buffer.gobj ()->length;
      if (len == 0)
	{
	  msg.reset ();
	  download_document_failed.emit (document);
	  return;
	}

      downloaded_chunk_of_document.emit (document);
      std::string temp_filename = "/tmp/" PACKAGE_NAME ".XXXXXXX";
      int fd = Glib::mkstemp (temp_filename);
      close (fd);

      downloaded_chunk_of_document.emit (document);
      std::ofstream outfile (temp_filename.c_str (), std::ofstream::binary);
      if (outfile.is_open ())
	{
	  outfile.write (data, len);
	  outfile.close ();
	  msg.reset ();
	  downloaded_document.emit (document);
	  slot (document, temp_filename);
	}
      else
	{
	  msg.reset ();
	  downloaded_document.emit (document);
	}
    }
  else
    {
      msg.reset ();
      download_document_failed.emit (document);
    }
}

void
SessionConnector::pull_documents_for_gallery (Gallery * gallery,
					      const SessionConnector::
					      SlotPullGallery slot,
					      std::list < std::string >
					      &thumbnail_urls)
{
  Glib::RefPtr < Soup::Message > msg = Soup::Message::create
    ("POST", String::ucompose ("%1/gallery_view.php?galleryid=%2",
			       session->get_profile ()->get_url (),
			       gallery->get_id ()));
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_pull_documents_attempted),
				  gallery, slot, thumbnail_urls));
}

void
SessionConnector::pull_thumbnail (std::string url,
				  const SessionConnector::
				  SlotPullThumbnail slot)
{
  std::string doc_id = extract_document_id_from_thumbnail_url (url);
  Document *document = session->get_profile ()->find_doc_by_id (doc_id);

  if (!document)
    {
      pull_galleries_failed.emit (session);
      return;
    }

  Glib::RefPtr < Soup::Message > msg = Soup::Message::create ("GET", url);
  web->queue_message (msg, sigc::bind (sigc::mem_fun
				       (*this,
					&SessionConnector::
					on_thumbnail_downloaded), document,
				       slot));
}

bool
SessionConnector::revert_moved_document (ChangedDocument & added_document,
					 ChangedDocument & removed_document)
{
  //remove the removed_document from it's gallery.
  //put it back in the gallery belonging to added_document
  Gallery *original_gallery =
    session->find_by_id (removed_document.first.get_gallery_id ());
  if (original_gallery)
    {
      Gallery *current_gallery =
	session->find_by_id (added_document.first.get_gallery_id ());
      if (current_gallery)
	{
	  Document *document =
	    session->find_doc_by_id (added_document.first.get_id ());
	  if (document)
	    {
	      current_gallery->remove_document (document);
	      delete document;
	    }
	  original_gallery->
	    add_document (new Document (removed_document.first));
	  return true;
	}
    }
  return false;
}

guint
SessionConnector::revert_moved_documents (std::list < std::pair <
					  ChangedDocument,
					  ChangedDocument > >&moved_docs)
{
  guint count = 0;
  for (std::list < std::pair < ChangedDocument,
       ChangedDocument > >::iterator k = moved_docs.begin ();
       k != moved_docs.end (); k++)
    if (revert_moved_document ((*k).first, (*k).second) == true)
      count++;
  return count;
}

void
SessionConnector::revert (SessionChanges reverts)
{
  guint count = 0;
  count = revert_removed_galleries (reverts.removed_galleries);
  if (count)
    printf ("reverted %d removed galleries\n", count);
  if (count != reverts.removed_galleries.size ())
    printf ("%d removed galleries failed to revert\n",
	    count - reverts.removed_galleries.size ());


  reverts.find_moved_docs ();
  count = revert_moved_documents (reverts.moved_documents);
  if (count)
    printf ("reverted %d moved documents\n", count);
  if (count != reverts.moved_documents.size ())
    printf ("%d moved documents failed to revert.\n",
	    reverts.moved_documents.size () - count);

  count = revert_removed_documents (reverts.removed_documents);
  if (count)
    printf ("reverted %d removed documents\n", count);
  if (count != reverts.removed_documents.size ())
    printf ("%d removed documents failed to revert\n",
	    count - reverts.removed_documents.size ());

  count = revert_modified_galleries (reverts.modified_galleries);
  if (count)
    printf ("reverted %d modified galleries\n", count);
  if (count != reverts.modified_galleries.size ())
    printf ("%d modified galleries failed to revert\n",
	    count - reverts.modified_galleries.size ());

  count = revert_modified_documents (reverts.modified_docs);
  if (count)
    printf ("reverted %d modified documents\n", count);
  if (count != reverts.modified_docs.size ())
    printf ("%d modified documents failed to revert\n",
	    count - reverts.modified_docs.size ());

  count = revert_added_galleries (reverts.added_galleries);
  if (count)
    printf ("reverted %d added galleries\n", count);
  if (count != reverts.added_galleries.size ())
    printf ("%d added galleries failed to revert\n",
	    count - reverts.added_galleries.size ());

  count = revert_added_documents (reverts.added_documents);
  if (count)
    printf ("reverted %d added documents\n", count);
  if (count != reverts.added_documents.size ())
    printf ("%d added documents failed to revert\n",
	    count - reverts.added_documents.size ());
}

guint
SessionConnector::revert_modified_galleries (std::list < ChangedGallery > l)
{
  guint count = 0;
  for (std::list < ChangedGallery >::iterator i = l.begin (); i != l.end ();
       i++)
    if (revert_modified_gallery (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::revert_modified_documents (std::list < ChangedDocument > l)
{
  guint count = 0;
  for (std::list < ChangedDocument >::iterator i = l.begin (); i != l.end ();
       i++)
    if (revert_modified_document (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::revert_added_galleries (std::list < ChangedGallery > l)
{
  guint count = 0;
  for (std::list < ChangedGallery >::iterator i = l.begin (); i != l.end ();
       i++)
    if (revert_added_gallery (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::revert_removed_galleries (std::list < ChangedGallery > l)
{
  guint count = 0;
  for (std::list < ChangedGallery >::iterator i = l.begin (); i != l.end ();
       i++)
    if (revert_removed_gallery (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::revert_added_documents (std::list < ChangedDocument > l)
{
  guint count = 0;
  for (std::list < ChangedDocument >::iterator i = l.begin (); i != l.end ();
       i++)
    if (revert_added_document (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::revert_removed_documents (std::list < ChangedDocument > l)
{
  guint count = 0;
  for (std::list < ChangedDocument >::iterator i = l.begin (); i != l.end ();
       i++)
    if (revert_removed_document (*i) == true)
      count++;
  return count;
}

bool
SessionConnector::revert_modified_gallery (ChangedGallery g)
{
  Gallery *gallery = session->find_by_id (g.second.get_id ());
  if (gallery)
    {
      *gallery = g.second;
      return true;
    }
  return false;
}

bool
SessionConnector::revert_modified_document (ChangedDocument d)
{
  Document *document = session->find_doc_by_id (d.second.get_id ());
  if (document)
    {
      *document = d.second;
      return true;
    }
  return false;
}

bool
SessionConnector::revert_added_gallery (ChangedGallery g)
{
  Gallery *gallery = session->find_by_id (g.first.get_id ());
  if (gallery)
    {
      session->remove (gallery);
      delete gallery;
      return true;
    }
  return false;
}

bool
SessionConnector::revert_removed_gallery (ChangedGallery g)
{
  Gallery *gallery = session->get_profile ()->find_by_id (g.first.get_id ());
  if (gallery)
    {
      session->push_back (new Gallery (*gallery));
      session->back ()->remove_all_documents ();
      return true;
    }
  return false;
}

bool
SessionConnector::revert_added_document (ChangedDocument d)
{
  Document *document = session->find_doc_by_id (d.first.get_id ());
  if (document)
    {
      Gallery *gallery = session->find_by_id (document->get_gallery_id ());
      if (gallery)
	{
	  gallery->remove_document (document);
	  delete document;
	  return true;
	}
    }
  return false;
}

bool
SessionConnector::revert_removed_document (ChangedDocument d)
{
  Document *document =
    session->get_profile ()->find_doc_by_id (d.first.get_id ());
  if (document)
    {
      Gallery *gallery = session->find_by_id (d.first.get_gallery_id ());
      if (gallery)
	{
	  if (session->find_doc_by_id (d.first.get_id ()) == NULL)
	    gallery->add_document (new Document (*document));
	  return true;
	}
    }
  return false;
}

guint
SessionConnector::update_modified_galleries (std::list < ChangedGallery > l)
{
  guint count = 0;
  for (std::list < ChangedGallery >::iterator i = l.begin (); i != l.end ();
       i++)
    if (update_modified_gallery (*i) == true)
      count++;
  if (count != l.size ())
    printf ("%d modified galleries failed to update\n", l.size () - count);
  return count;
}

guint
SessionConnector::update_modified_documents (std::list < ChangedDocument > l)
{
  guint count = 0;
  for (std::list < ChangedDocument >::iterator i = l.begin (); i != l.end ();
       i++)
    if (update_modified_document (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::update_added_galleries (std::list < ChangedGallery > l)
{
  guint count = 0;
  for (std::list < ChangedGallery >::iterator i = l.begin (); i != l.end ();
       i++)
    if (update_added_gallery (*i) == true)
      count++;
  return count;
}

guint
SessionConnector::update_added_documents (std::list < ChangedDocument > l)
{
  guint count = 0;
  for (std::list < ChangedDocument >::iterator i = l.begin (); i != l.end ();
       i++)
    if (update_added_document (*i) == true)
      count++;
  return count;
}

bool
SessionConnector::update_modified_document (ChangedDocument d)
{
  Document *document = session->find_doc_by_id (d.second.get_id ());
  if (document)
    {
      *document = d.first;
      return true;
    }
  return false;
}

bool
SessionConnector::update_modified_gallery (ChangedGallery g)
{
  Gallery *gallery = session->find_by_id (g.second.get_id ());
  if (gallery)
    {
      *gallery = g.first;
      return true;
    }
  return false;
}

bool
SessionConnector::update_added_document (ChangedDocument d)
{
  Document *document = session->find_doc_by_id (d.first.get_id ());
  if (document)
    {
      *document = d.first;
      return true;
    }
  return false;
}

bool
SessionConnector::update_added_gallery (ChangedGallery g)
{
  Gallery *gallery = session->find_by_id (g.first.get_id ());
  if (gallery)
    {
      Gallery *new_gallery = new Gallery (g.first);
      new_gallery->add (gallery);
      session->remove (gallery);
      session->push_back (new_gallery);
      delete gallery;
      return true;
    }
  return false;
}

std::string SessionConnector::get_session_id_from_page (std::string data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("input name=\"sessionid\" type=\"hidden\" value=\"", data);
  if (a.size () == 1)
    {
      a = Glib::Regex::split_simple
	("input type=\"hidden\" name=\"sessionid\" value=\"", data);
      if (a.size () == 1)
	return "";
    }

  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\"", a[1]);
  return b[0];
}

void
SessionConnector::add_gallery (Gallery * gallery,
			       const SessionConnector::SlotAddGallery slot,
			       bool retry)
{
  Glib::RefPtr < Soup::Message > msg;
  msg =
    Soup::Message::create ("GET",
			   session->get_profile ()->get_url () +
			   "/gallery_insert.php");
  //go get our session id.
  int status = web->send_message (msg);
  if (status != 200)
    {
      add_gallery_failed.emit (gallery);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      add_gallery_failed.emit (gallery);
      return;
    }
  std::string session_id = get_session_id_from_page (std::string (data));
  if (session_id == "")
    {
      add_gallery_failed.emit (gallery);
      return;
    }

  std::map < std::string, std::string > formdata;
  formdata["submitted"] = "1";
  formdata["sessionid"] = session_id;
  formdata["dir"] = "0";
  if (retry)
    formdata["file"] = gallery->get_id ();
  else
    {
      if (gallery->get_title () != "")
	formdata["file"] = gallery->get_title ();
      else
	formdata["file"] = gallery->get_id ();
    }

  formdata["thetitle"] = gallery->get_title ();
  formdata["thetext"] = gallery->get_description ();
  formdata["priority"] = gallery->get_order ();
  if (gallery->get_private_audience() == false)
    formdata["mode"] = "b";
  else
    formdata["mode"] = "v";
  formdata["rows"] = "3";
  formdata["cols"] = "3";
  formdata["lat"] = String::ucompose ("%1", gallery->get_latitude ());
  formdata["long"] = String::ucompose ("%1", gallery->get_longitude ());
  formdata["select_skin_gallery"] = "0";
  formdata["select_spin_pic"] = "0";
  formdata["select_css_gallery_0"] = "0";
  formdata["select_css_pic_0"] = "0";
  formdata["select_template_gallery"] = "0";
  formdata["select_template_pic"] = "0";
  formdata["submit"] = "Submit";
  msg =
    Soup::Message::create_form_request ("POST",
					session->get_profile ()->get_url () +
					"/gallery_insert.php", formdata);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_add_gallery_attempted), gallery,
				  slot, retry));
}

std::string SessionConnector::get_gallery_id_from_gallery_insert_page (std::
								       string
								       data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("refparent=", data);
  if (a.size () == 1)
    return "";

  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("&", a[1]);
  return b[0];
}

std::string SessionConnector::get_document_id_from_document_insert_page (std::
									 string
									 data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("refparent=", data);
  if (a.size () == 1)
    return "";

  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("&", a[1]);
  return b[0];
}

void
SessionConnector::on_add_gallery_attempted (Glib::RefPtr < Soup::Message >
					    &msg, Gallery * gallery,
					    const SessionConnector::
					    SlotAddGallery slot, bool retried)
{
  if (msg->property_status_code () != 200)
    {
      add_gallery_failed.emit (gallery);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      add_gallery_failed.emit (gallery);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      if (retried == false)
	{
	  add_gallery (gallery, slot, true);
	  return;
	}
      add_gallery_failed.emit (gallery);
      return;
    }

  std::string gallery_id =
    get_gallery_id_from_gallery_insert_page (std::string (data));
  if (gallery_id == "")
    {
      add_gallery_failed.emit (gallery);
      return;
    }
  gallery->set_id(gallery_id);
  if (gallery->get_tags ().size () > 0)
    {
      update_item_tags (gallery,
			sigc::bind (sigc::
				    mem_fun (this,
					     &SessionConnector::
					     on_new_gallery_updated_tag),
				    slot, gallery), true);
    }
  else
    slot (gallery);
}

void
SessionConnector::on_new_gallery_updated_tag (Item * item,
					      const SessionConnector::
					      SlotAddGallery slot,
					      Gallery * gallery)
{
  slot (gallery);
}


void
SessionConnector::update_item_tags (Item * item,
				    const SessionConnector::
				    SlotUpdateTags slot, bool gallery)
{
//http://master.atpic.com/tag_insert.php?type=g&refparent=38558&secret=&tags=
  Glib::RefPtr < Soup::Message > msg;
  Glib::ustring url = String::ucompose
    ("%1/tag_insert.php?type=%2&refparent=%3&secret=%4&tags=",
     session->get_profile ()->get_url (), gallery ? "g" : "p", item->get_id (),
     gallery ? "" : "0");
  msg = Soup::Message::create ("GET", url);
  //go get our session id.
  int status = web->send_message (msg);
  if (status != 200)
    {
      update_tags_failed.emit (item);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      update_tags_failed.emit (item);
      return;
    }
  std::string session_id = get_session_id_from_page (std::string (data));
  if (session_id == "")
    {
      update_tags_failed.emit (item);
      return;
    }

  std::map < std::string, std::string > formdata;
  formdata["submitted"] = "1";
  if (gallery)
    formdata["secret"] = "";
  else
    formdata["secret"] = "0";
  formdata["sessionid"] = session_id;
  formdata["refparent"] = item->get_id ();
  if (gallery)
    formdata["type"] = "g";
  else
    formdata["type"] = "p";
  formdata["tags"] = item->get_tags_as_string ();
  formdata["submit"] = "Submit";
  msg =
    Soup::Message::create_form_request ("POST",
					session->get_profile ()->get_url () +
					"/tag_insert.php", formdata);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_update_tags_attempted), item,
				  slot));
}

void
SessionConnector::on_update_tags_attempted (Glib::RefPtr < Soup::Message >
					    &msg, Item * item,
					    const SessionConnector::
					    SlotUpdateTags slot)
{
  if (msg->property_status_code () != 200 &&
      msg->property_status_code () != 302)
    {
      update_tags_failed.emit (item);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      update_tags_failed.emit (item);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      update_tags_failed.emit (item);
      return;
    }

  slot (item);
}

void
SessionConnector::remove_gallery (Gallery * gallery,
				  const SessionConnector::
				  SlotRemoveGallery slot)
{
  Glib::RefPtr < Soup::Message > msg;
  Glib::ustring url =
    String::ucompose ("%1/gallery_delete.php?galleryid=%2&amp;confirmed=1",
		      session->get_profile ()->get_url (),
		      gallery->get_id ());
  msg = Soup::Message::create ("GET", url);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_remove_gallery_attempted),
				  gallery, slot));
}

void
SessionConnector::on_remove_gallery_attempted (Glib::RefPtr < Soup::Message >
					       &msg, Gallery * gallery,
					       const SessionConnector::
					       SlotRemoveGallery slot)
{
  if (msg->property_status_code () != 200)
    {
      remove_gallery_failed.emit (gallery);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      remove_gallery_failed.emit (gallery);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      remove_gallery_failed.emit (gallery);
      return;
    }

  slot (gallery);
}

void
SessionChanges::find_moved_docs ()
{
  for (std::list < ChangedDocument >::iterator i = added_documents.begin ();
       i != added_documents.end (); i++)
    {
      for (std::list < ChangedDocument >::iterator j =
	   removed_documents.begin (); j != removed_documents.end (); j++)
	{
	  if ((*i).first.get_id () == (*j).first.get_id ())
	    {
	      moved_documents.push_back
		(std::pair < ChangedDocument, ChangedDocument > (*i, *j));
	    }
	}
    }
  for (std::list < std::pair < ChangedDocument,
       ChangedDocument > >::iterator i = moved_documents.begin ();
       i != moved_documents.end (); i++)
    {
      added_documents.remove ((*i).first);
      removed_documents.remove ((*i).second);
    }
}

void
SessionConnector::move_document (Document * src, Document * dest,
				 const SessionConnector::
				 SlotMoveDocument slot)
{
  //lots to do here!
  //download the src document.
  download_document (src,
		     sigc::bind (sigc::
				 mem_fun (this,
					  &SessionConnector::
					  on_document_downloaded_for_move),
				 dest, slot));
  //make a new document with it
  //remove the old document.

  return;
}

void
SessionConnector::on_document_downloaded_for_move (Document * src,
						   std::string file,
						   Document * dest,
						   const SessionConnector::
						   SlotMoveDocument slot)
{
  dest->set_image_filename (file);	//does it matter if this is video??
  add_document (dest,
		sigc::bind (sigc::
			    mem_fun (this,
				     &SessionConnector::
				     on_document_added_for_move), src, slot));
}


Glib::RefPtr < Soup::Message >
create_file_upload_form_request (Glib::ustring method, Glib::ustring url,
				 std::map < std::string,
				 std::string > formdata,
				 std::string upload_key_name,
				 std::string filename)
{
  SoupMultipart *
    multipart = soup_multipart_new ("multipart/form-data");
  for (std::map < std::string, std::string >::iterator i = formdata.begin ();
       i != formdata.end (); i++)
    soup_multipart_append_form_string (multipart, (*i).first.c_str (),
				       (*i).second.c_str ());

  std::string mime_type;
  Glib::RefPtr < Gio::File > file = Gio::File::create_for_path (filename);
  Glib::RefPtr < Gio::FileInfo > file_info = file->query_filesystem_info ();
  mime_type =
    file_info->get_attribute_string ("FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE");
  if (mime_type == "")
    mime_type =
      file_info->
      get_attribute_string ("FILE_ATTRIBUTE_STANDARD_FAST_CONTENT_TYPE");
  char *
    data;
  size_t
    data_len;
  file->load_contents (data, data_len);
  Soup::Buffer buffer (Soup::MEMORY_TEMPORARY, data, data_len);
  soup_multipart_append_form_file (multipart, upload_key_name.c_str (),
				   url.c_str (), mime_type.c_str (),
				   buffer.gobj ());
  SoupMessage *
    msg = soup_form_request_new_from_multipart (url.c_str (), multipart);
  soup_multipart_free (multipart);	//maybe should be deleted after the send?
  return Glib::wrap (msg);
}

void
SessionConnector::add_document (Document * document,
				const SessionConnector::SlotAddDocument slot)
{
  Glib::RefPtr < Soup::Message > msg;
  Glib::ustring url = String::ucompose ("%1/pic_insert.php?galleryid=%2",
					session->get_profile ()->get_url (),
					document->get_gallery_id ());
  msg = Soup::Message::create ("GET", url);
  //go get our session id.
  int status = web->send_message (msg);
  if (status != 200)
    {
      add_document_failed.emit (document);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      add_document_failed.emit (document);
      return;
    }
  std::string session_id = get_session_id_from_page (std::string (data));
  if (session_id == "")
    {
      add_document_failed.emit (document);
      return;
    }

  std::map < std::string, std::string > formdata;
  //formdata["userfile"] = document->get_filename();
  formdata["submitted"] = "1";
  formdata["sessionid"] = session_id;
  formdata["galleryid"] = document->get_gallery_id ();
  formdata["thetitle"] = document->get_title ();
  formdata["thetext"] = document->get_description ();
  formdata["priority"] = document->get_order ();
  formdata["lat"] = String::ucompose ("%1", document->get_latitude ());
  formdata["long"] = String::ucompose ("%1", document->get_longitude ());
  formdata["MAX_FILE_SIZE"] = "5000000";
  formdata["ftp"] = "";
  formdata["submit"] = "Submit";
  //maybe if i append to the body in the right way?
  //nope.  you have to make a multipart message.
  //then wrap that in a soup::message.
  msg =
    create_file_upload_form_request ("POST",
				     session->get_profile ()->get_url () +
				     "/pic_insert.php", formdata, "userfile",
				     document->get_filename ());

  //msg = Soup::Message::create_form_request("POST", session->get_profile()->get_url() + "/pic_insert.php", formdata);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_add_document_attempted),
				  document, slot));
}

void
SessionConnector::on_add_document_attempted (Glib::RefPtr < Soup::Message >
					     &msg, Document * document,
					     const SessionConnector::
					     SlotAddDocument slot)
{
  if (msg->property_status_code () != 200)
    {
      add_document_failed.emit (document);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      add_document_failed.emit (document);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      add_document_failed.emit (document);
      return;
    }

  std::string document_id =
    get_document_id_from_document_insert_page (std::string (data));
  if (document_id == "")
    {
      add_document_failed.emit (document);
      return;
    }
  document->set_id(document_id);
  if (document->get_tags ().size () > 0)
    {
      update_item_tags (document,
			sigc::bind (sigc::
				    mem_fun (this,
					     &SessionConnector::
					     on_new_document_updated_tag),
				    slot, document), false);
    }
  else
    slot (document);
  //need to ask the server what the priority is.
  //basically i need to fill up the values of this document again.
  //alternatively i can just pull all galleries after a publish.
}

double
SessionConnector::get_longitude_from_document_page (std::string data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("\\<input name=\"long\" value=\"", data);
  if (a.size () <= 1)
    return 0.0;
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\"", a[1]);
  std::string lon = b[0];
  return atof (lon.c_str ());
}

double
SessionConnector::get_latitude_from_document_page (std::string data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("\\<input name=\"lat\" value=\"", data);
  if (a.size () <= 1)
    return 0.0;
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\"", a[1]);
  std::string lat = b[0];
  return atof (lat.c_str ());
}

void
SessionConnector::fill_in_gallery_values_from_gallery_page (Gallery * gallery,
							    std::string data)
{
  gallery->set_title (get_title_from_document_page (data));
  gallery->set_description (get_description_from_document_page (data));
  gallery->set_order (get_priority_from_document_page (data));
  gallery->set_lat_long (get_latitude_from_document_page (data),
			 get_longitude_from_document_page (data));
}

void
SessionConnector::fill_in_document_values_from_document_page (Document *
							      document,
							      std::
							      string data)
{
  document->set_title (get_title_from_document_page (data));
  document->set_description (get_description_from_document_page (data));
  document->set_order (get_priority_from_document_page (data));
  document->set_lat_long (get_latitude_from_document_page (data),
			  get_longitude_from_document_page (data));
}

std::string SessionConnector::get_title_from_document_page (std::string data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("\\<input name=\"thetitle\" value=\"", data);
  if (a.size () <= 1)
    return "";
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\"", a[1]);
  return b[0];
}

std::string SessionConnector::get_description_from_document_page (std::
								  string data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("\\<textarea name=\"thetext\" rows=\"8\" cols=\"50\"\\>", data);
  if (a.size () <= 1)
    return "";
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\\<", a[1]);
  return b[0];
}

std::string SessionConnector::get_priority_from_document_page (std::
							       string data)
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple
    ("\\<input name=\"priority\" value=\"", data);
  if (a.size () <= 1)
    return "";
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\"", a[1]);
  return b[0];
}

void
SessionConnector::on_new_document_updated_tag (Item * item,
					       const SessionConnector::
					       SlotAddDocument slot,
					       Document * document)
{
  slot (document);
}


void
SessionConnector::on_document_added_for_move (Document * dest,
					      Document * src,
					      const SessionConnector::
					      SlotMoveDocument slot)
{
  //and we can remove the file now because it's uploaded.
  Document::remove_file (dest->get_filename ());
  dest->set_image_filename ("");
  //and now we remove the original document from the server.
  remove_document (src,
		   sigc::bind (sigc::
			       mem_fun (this,
					&SessionConnector::
					on_document_removed_for_move), dest,
			       slot));
}

void
SessionConnector::remove_document (Document * document,
				   const SessionConnector::
				   SlotRemoveDocument slot)
{
  Glib::RefPtr < Soup::Message > msg;
  Gallery *gallery = session->find_by_id (document->get_gallery_id ());
  if (!gallery)
    return;
  Glib::ustring url =
    String::ucompose ("%1/pic_delete.php?picid=%2&galleryid=%3",
		      session->get_profile ()->get_url (),
		      document->get_id (), gallery->get_id ());
  msg = Soup::Message::create ("GET", url);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_remove_document_attempted),
				  document, slot));
}

void
SessionConnector::on_remove_document_attempted (Glib::RefPtr < Soup::Message >
						&msg, Document * document,
						const SessionConnector::
						SlotRemoveDocument slot)
{
  if (msg->property_status_code () != 200)
    {
      remove_document_failed.emit (document);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      remove_document_failed.emit (document);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      remove_document_failed.emit (document);
      return;
    }

  slot (document);
}

void
SessionConnector::on_document_removed_for_move (Document * src,
						Document * dest,
						const SessionConnector::
						SlotMoveDocument slot)
{
  slot (src, dest);
}

void
SessionConnector::modify_gallery (Gallery * gallery,
				  const SessionConnector::
				  SlotModifyGallery slot)
{
  Glib::ustring url = String::ucompose ("%1/gallery_update.php?galleryid=%2",
					session->get_profile ()->get_url (),
					gallery->get_id ());
  Glib::RefPtr < Soup::Message > msg = Soup::Message::create ("GET", url);
  int status = web->send_message (msg);
  if (status != 200 && status != 302)
    {
      modify_gallery_failed.emit (gallery);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      modify_gallery_failed.emit (gallery);
      return;
    }
  std::string session_id = get_session_id_from_page (std::string (data));
  if (session_id == "")
    {
      modify_gallery_failed.emit (gallery);
      return;
    }

  std::map < std::string, std::string > formdata;
  formdata["submitted"] = "1";
  formdata["galleryid"] = gallery->get_id ();
  formdata["sessionid"] = session_id;
  formdata["dir"] = "0";
  if (gallery->get_title () != "")
    formdata["file"] = gallery->get_title ();
  else
    formdata["file"] = gallery->get_id ();
  formdata["thetitle"] = gallery->get_title ();
  formdata["thetext"] = gallery->get_description ();
  formdata["priority"] = gallery->get_order ();
  if (gallery->get_private_audience() == false)
    formdata["mode"] = "b";
  else
    formdata["mode"] = "v";
  formdata["rows"] = "3";
  formdata["cols"] = "3";
  formdata["lat"] = String::ucompose ("%1", gallery->get_latitude ());
  formdata["long"] = String::ucompose ("%1", gallery->get_longitude ());
  formdata["select_skin_gallery"] = "0";
  formdata["select_spin_pic"] = "0";
  formdata["select_css_gallery_0"] = "0";
  formdata["select_css_pic_0"] = "0";
  formdata["select_template_gallery"] = "0";
  formdata["select_template_pic"] = "0";
  formdata["submit"] = "Submit";
  msg =
    Soup::Message::create_form_request ("POST",
					session->get_profile ()->get_url () +
					"/gallery_update.php", formdata);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_modify_gallery_attempted),
				  gallery, slot));
}

void
SessionConnector::on_modify_gallery_attempted (Glib::RefPtr < Soup::Message >
					       &msg, Gallery * gallery,
					       const SessionConnector::
					       SlotModifyGallery slot)
{
  if (msg->property_status_code () != 200)
    {
      modify_gallery_failed.emit (gallery);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      modify_gallery_failed.emit (gallery);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      modify_gallery_failed.emit (gallery);
      return;
    }

  bool retval = fill_in_gallery_values (gallery);
  if (retval == false)
    {
      modify_gallery_failed.emit (gallery);
      return;
    }

  if (gallery->get_tags ().size () > 0)
    {
      update_item_tags (gallery,
			sigc::bind (sigc::
				    mem_fun (this,
					     &SessionConnector::
					     on_modify_gallery_updated_tag),
				    slot, gallery), true);
    }
  else
    slot (gallery);
}

void
SessionConnector::on_modify_gallery_updated_tag (Item * item,
						 const SessionConnector::
						 SlotModifyGallery slot,
						 Gallery * gallery)
{
  slot (gallery);
}


void
SessionConnector::modify_document (Document * document,
				   const SessionConnector::
				   SlotModifyDocument slot)
{
  Glib::ustring url =
    String::ucompose ("%1/pic_update.php?picid=%2&galleryid=%3",
		      session->get_profile ()->get_url (),
		      document->get_id (), document->get_gallery_id ());
  Glib::RefPtr < Soup::Message > msg = Soup::Message::create ("GET", url);
  //go get our session id.
  int status = web->send_message (msg);
  if (status != 200)
    {
      modify_document_failed.emit (document);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      modify_document_failed.emit (document);
      return;
    }
  std::string session_id = get_session_id_from_page (std::string (data));
  if (session_id == "")
    {
      modify_document_failed.emit (document);
      return;
    }

  std::map < std::string, std::string > formdata;
  formdata["submitted"] = "1";
  formdata["picid"] = document->get_id ();
  formdata["sessionid"] = session_id;
  formdata["galleryid"] = document->get_gallery_id ();
  formdata["thetitle"] = document->get_title ();
  formdata["thetext"] = document->get_description ();
  formdata["priority"] = document->get_order ();
  formdata["lat"] = String::ucompose ("%1", document->get_latitude ());
  formdata["long"] = String::ucompose ("%1", document->get_longitude ());
  formdata["submit"] = "Submit";
  msg =
    Soup::Message::create_form_request ("POST",
					session->get_profile ()->get_url () +
					"/pic_update.php", formdata);
  web->queue_message (msg,
		      sigc::bind (sigc::
				  mem_fun (*this,
					   &SessionConnector::
					   on_modify_document_attempted),
				  document, slot));
}

void
SessionConnector::on_modify_document_attempted (Glib::RefPtr < Soup::Message >
						&msg, Document * document,
						const SessionConnector::
						SlotModifyDocument slot)
{
  if (msg->property_status_code () != 200)
    {
      modify_document_failed.emit (document);
      return;
    }
  web->cancel_message (msg, msg->property_status_code ());
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    {
      modify_document_failed.emit (document);
      return;
    }
  if (strstr (data, "div class=\"error\""))
    {
      modify_document_failed.emit (document);
      return;
    }

  bool retval = fill_in_document_values (document);
  if (retval == false)
    {
      modify_document_failed.emit (document);
      return;
    }

  if (document->get_tags ().size () > 0)
    {
      update_item_tags (document,
			sigc::bind (sigc::
				    mem_fun (this,
					     &SessionConnector::
					     on_modify_document_updated_tag),
				    slot, document), false);
    }
  else
    slot (document);
}

bool
SessionConnector::fill_in_document_values (Document * document)
{
  Glib::ustring url =
    String::ucompose ("%1/pic_update.php?picid=%2&galleryid=%3",
		      session->get_profile ()->get_url (),
		      document->get_id (), document->get_gallery_id ());
  Glib::RefPtr < Soup::Message > msg;
  msg = Soup::Message::create ("GET", url);
  guint status = web->send_message (msg);
  if (status != 200)
    return false;
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();;
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    return false;
  fill_in_document_values_from_document_page (document, std::string (data));
  return true;
}

bool
SessionConnector::fill_in_gallery_values (Gallery * gallery)
{
  Glib::ustring url =
    String::ucompose ("%1/gallery_update.php?galleryid=%2",
		      session->get_profile ()->get_url (),
		      gallery->get_id ());
  Glib::RefPtr < Soup::Message > msg;
  msg = Soup::Message::create ("GET", url);
  guint status = web->send_message (msg);
  if (status != 200)
    return false;
  Soup::MessageBody body (msg->gobj ()->response_body);
  body.set_accumulate (true);
  Soup::Buffer buffer = body.flatten ();;
  const char *data = buffer.gobj ()->data;
  guint len = buffer.gobj ()->length;
  if (len == 0)
    return false;
  fill_in_gallery_values_from_gallery_page (gallery, std::string (data));
  return true;
}

void
SessionConnector::on_modify_document_updated_tag (Item * item,
						  const SessionConnector::
						  SlotModifyDocument slot,
						  Document * document)
{
  slot (document);
}

void
SessionConnector::apply (SessionChanges changes)
{
  guint count = 0;
  count = update_modified_galleries (changes.modified_galleries);
  if (count)
    printf ("updated %d modified galleries\n", count);
  if (count != changes.modified_galleries.size ())
    printf ("%d modified galleries failed to update\n",
	    changes.modified_galleries.size () - count);
  count = update_modified_documents (changes.modified_docs);
  if (count)
    printf ("updated %d modified documents\n", count);
  if (count != changes.modified_docs.size ())
    printf ("%d added galleries failed to update\n",
	    changes.modified_docs.size () - count);
  count = update_added_galleries (changes.added_galleries);
  if (count)
    printf ("updated %d added galleries\n", count);
  if (count != changes.added_galleries.size ())
    printf ("%d added galleries failed to update\n",
	    changes.added_galleries.size () - count);
  count = update_added_documents (changes.added_documents);
  if (count)
    printf ("updated %d added documents\n", count);
  if (count != changes.added_documents.size ())
    printf ("%d added documents failed to update\n",
	    changes.added_documents.size () - count);
  //removed docs and galleries can't have been edited.
}

std::string SessionConnector::get_uid_from_gallery_listing_page (std::
								 string data)
  const
{
  std::vector < Glib::ustring > c = Glib::Regex::split_simple ("UID", data);
  std::vector < Glib::ustring > d = Glib::Regex::split_simple ("\\)", c[1]);
  return d[0];
}

Gallery *
SessionConnector::process_gallery_chunk (std::string data) const
{
  Gallery *gallery = new Gallery ();

  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("span class=\"field\"", data);
  std::string gallery_id = get_gallery_id_from_gallery_listing_page (a[0]);
  gallery->set_id (gallery_id);

  std::string gallery_thumbnail_id =
    get_gallery_thumbnail_id_from_gallery_listing_page (a[0]);
  gallery->set_thumbnail (gallery_thumbnail_id);

  std::string title = get_gallery_title_from_gallery_listing_page (a[2]);
  if (title != " ")
    gallery->set_title (title);

  std::list < std::string > tags =
    get_gallery_tags_from_gallery_listing_page (a[4]);
  gallery->set_tags (tags);

  std::string description =
    get_gallery_description_from_gallery_listing_page (a[5]);
  gallery->set_description (description);

  std::string order = get_gallery_priority_from_gallery_listing_page (a[6]);
  gallery->set_order (order);

  double lat, lon;
  get_gallery_lat_lon_from_gallery_listing_page (a[6], lat, lon);
  gallery->set_lat_long (lat, lon);

  bool private_audience = get_gallery_audience_from_gallery_listing_page(a[7]);
  gallery->set_private_audience(private_audience);
  return gallery;
}

bool 
SessionConnector::get_gallery_audience_from_gallery_listing_page(std::
                                                                 string data)
  const
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple ("\\(", data);
  if (a.size() <= 1)
    return false;
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\\)", a[1]);
  if (b[0] == "b")
    return false;
  else if (b[0] == "v")
    return true;
  return false;
}

void
SessionConnector::get_gallery_lat_lon_from_gallery_listing_page (std::
								 string data,
								 double &lat,
								 double &lon)
  const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple (" \\<\\/span\\>", data);
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\<td class=\"geo\"\\>", a[1]);
  std::vector < Glib::ustring > c = Glib::Regex::split_simple ("\\<", b[2]);
  lat = atof (std::string (c[0]).c_str ());
  std::vector < Glib::ustring > d = Glib::Regex::split_simple ("\\<", b[4]);
  lon = atof (std::string (d[0]).c_str ());
}

std::string SessionConnector::
get_gallery_priority_from_gallery_listing_page (std::string data) const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple (" \\<\\/span\\>", data);
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\\<", a[1]);
  std::string s = b[0];
  if (b[0] == "Not set")
    return "";
  return b[0];
}

std::string SessionConnector::get_gallery_id_from_gallery_listing_page (std::
									string
									data)
  const
{
  std::list < std::string > tags;
  std::vector < Glib::ustring > a = Glib::Regex::split_simple ("/", data);
  return a[5];
}

std::string SessionConnector::
get_gallery_thumbnail_id_from_gallery_listing_page (std::
						    string data) const
{
  std::list < std::string > tags;
  std::vector < Glib::ustring > a = Glib::Regex::split_simple ("/", data);
  return a[7];
}

std::list < std::string >
  SessionConnector::get_gallery_tags_from_gallery_listing_page (std::
								string data)
  const
{
  std::list < std::string > tags;
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple (" \\<\\/span\\>", data);
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\\<", a[1]);
  std::vector < Glib::ustring > c = Glib::Regex::split_simple (" ", b[0]);
  for (unsigned int i = 0; i < c.size (); i++)
    {
      if (c[i] != " " && c[i] != "")
	tags.push_back (c[i]);
    }
  return tags;
}

std::string SessionConnector::
get_gallery_title_from_gallery_listing_page (std::string data) const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<\\/span\\> ", data);
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\\<", a[1]);
  return decode_xml (b[0]);
}

std::string SessionConnector::
get_gallery_description_from_gallery_listing_page (std::
						   string data) const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<\\/span\\> ", data);
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\</td\\>", a[1]);
  return decode_xml (b[0]);
}

void
SessionConnector::get_galleries_from_gallery_listing_page (std::string data)
{
  session->get_profile ()->remove_all_galleries ();
  std::vector < Glib::ustring > chunks =
    Glib::Regex::split_simple ("class=\"homegalimage\"", data);
  std::vector < Glib::ustring >::iterator it = chunks.begin ();
  it++;
  if (session->get_profile ()->get_id () == "")
    session->get_profile ()->set_id (get_uid_from_gallery_listing_page (*it));
  it++;
  for (; it != chunks.end (); it++)
    {
      Gallery *gallery = process_gallery_chunk (*it);
      if (gallery)
	{
	  session->get_profile ()->push_back (gallery);
	  processed_gallery.emit (gallery);
	}
    }
  return;
}

std::string SessionConnector::get_document_id_from_gallery_listing_page (std::
									 string
									 data)
  const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("picid=", data);
  if (a.size () <= 1)
    return "";
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("&", a[1]);
  return b[0];
}

std::string SessionConnector::
get_document_order_from_gallery_listing_page (std::string data) const
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple (": ", data);
  if (a.size () <= 1)
    return "";
  return a[1];
}

std::string SessionConnector::
get_document_title_from_gallery_listing_page (std::string data) const
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple (": ", data);
  if (a.size () <= 1)
    return "";
  return decode_xml (a[1]);
}

std::string SessionConnector::
get_original_document_filename_from_gallery_listing_page (std::
							  string data) const
{
  std::vector < Glib::ustring > a = Glib::Regex::split_simple (": ", data);
  if (a.size () <= 1)
    return "";
  std::vector < Glib::ustring > b = Glib::Regex::split_simple ("\n", a[1]);
  return decode_xml (b[0]);
}

std::string SessionConnector::
get_document_description_from_gallery_listing_page (std::
						    string data) const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<div class=\"center\"\\>", data);
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\<br /\\>", a[0]);
  if (b.size () <= 4)
    return "";
  Glib::ustring desc;
  for (unsigned int i = 4; i < b.size (); i++)
    desc += b[i];
  Glib::ustring::size_type colon = desc.find (":");
  if (colon == Glib::ustring::npos)
    return "";
  Glib::ustring::size_type space = desc.find (" ", colon);
  if (space == Glib::ustring::npos)
    return "";
  desc = desc.substr (space + 1, desc.length ());
  return decode_xml (desc);
}

std::list < std::string >
  SessionConnector::get_document_tags_from_gallery_listing_page (std::
								 string data)
  const
{
  std::list < std::string > tags;
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<div class=\"tags\"\\>", data);
  if (a.size () <= 1)
    return tags;
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\<i\\>", a[1]);
  if (b.size () <= 1)
    return tags;
  std::vector < Glib::ustring > c =
    Glib::Regex::split_simple ("\\</i\\>", b[1]);
  std::vector < Glib::ustring > d =
    Glib::Regex::split_simple ("\\</i\\>", c[0]);
  for (unsigned int i = 0; i < d.size (); i++)
    {
      if (d[i] == " " || d[i] == "\n" || d[i] == "")
	continue;
      tags.push_back (String::utrim(d[i]));
    }
  return tags;
}

void
SessionConnector::get_document_lat_long_from_gallery_listing_page (std::
								   string
								   data,
								   double
								   &lat,
								   double
								   &lon) const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<td class=\"technical\"\\>", data);
  if (a.size () <= 1)
    return;
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\</td\\>", a[1]);
  std::vector < Glib::ustring > c =
    Glib::Regex::split_simple ("\\<br /\\>", b[0]);
  for (unsigned int i = 0; i < c.size (); i++)
    {
      if (c[i].find ("latitude") != Glib::ustring::npos)
	{
	  std::vector < Glib::ustring > d =
	    Glib::Regex::split_simple (" ", c[i]);
	  if (d.size () > 1)
	    {
	      std::string latstr = d[1];
	      lat = atof (latstr.c_str ());
	    }
	}
      if (c[i].find ("longitude") != Glib::ustring::npos)
	{
	  std::vector < Glib::ustring > d =
	    Glib::Regex::split_simple (" ", c[i]);
	  if (d.size () > 1)
	    {
	      std::string lonstr = d[1];
	      lon = atof (lonstr.c_str ());
	    }
	}
    }
}

Tapioca::Document *
  SessionConnector::get_document_from_gallery_listing_page (std::
							    string data) const
{
  Document *document = new Document ();
  std::string id = get_document_id_from_gallery_listing_page (data);
  document->set_id (id);
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<td valign=\"top\"\\>", data);
  if (a.size () <= 1)
    {
      delete document;
      return NULL;
    }
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\<br /\\>", a[1]);
  std::string order = get_document_order_from_gallery_listing_page (b[1]);
  document->set_order (order);

  std::string orig =
    get_original_document_filename_from_gallery_listing_page (b[2]);
  document->set_original_filename (orig);

  std::string title = get_document_title_from_gallery_listing_page (b[3]);
  document->set_title (title);

  std::string desc =
    get_document_description_from_gallery_listing_page (a[1]);
  document->set_description (desc);

  std::list < std::string > tags =
    get_document_tags_from_gallery_listing_page (a[1]);
  document->set_tags (tags);

  double lat, lon = 0.0;
  get_document_lat_long_from_gallery_listing_page (a[1], lat, lon);
  document->set_lat_long (lat, lon);

  return document;
}


std::string SessionConnector::
get_thumbnail_url_from_gallery_listing_page (std::string data) const
{
  std::vector < Glib::ustring > a =
    Glib::Regex::split_simple ("\\<td class=\"rotate\"\\>", data);
  if (a.size () <= 2)
    return "";
  std::vector < Glib::ustring > b =
    Glib::Regex::split_simple ("\\<img src=\"", a[2]);
  if (b.size () <= 1)
    return "";
  std::vector < Glib::ustring > c = Glib::Regex::split_simple ("\"", b[1]);
  return c[0];
}

void
SessionConnector::get_documents_from_document_listing_page (Gallery * gallery,
							    std::string data,
							    std::list <
							    std::string >
							    &thumbnail_urls)
{
  gallery->remove_all_documents ();
  std::vector < Glib::ustring > chunks =
    Glib::Regex::split_simple ("\\<table class=\"rotate\"\\>", data);
  if (chunks.size () == 1)
    return;
  for (unsigned int i = 1; i < chunks.size (); i++)
    {
      Document *document = get_document_from_gallery_listing_page (chunks[i]);
      if (document)
	{
	  document->set_gallery_id (gallery->get_id ());
	  gallery->add_document (document);
	  std::string url =
	    get_thumbnail_url_from_gallery_listing_page (chunks[i]);
	  thumbnail_urls.push_back (url);
	  processed_document_for_gallery (gallery, document);
	}
    }
  return;
}

void
SessionConnector::process_thumbnail (Document * document, const char *data,
				     guint len)
{
  std::string temp_filename = "/tmp/" PACKAGE_NAME ".XXXXXXX";
  int fd = Glib::mkstemp (temp_filename);
  close (fd);

  std::ofstream outfile (temp_filename.c_str (), std::ofstream::binary);
  if (outfile.is_open ())
    {
      outfile.write (data, len);
      outfile.close ();
    }
  document->set_thumbnail (temp_filename);
  Document::remove_file (temp_filename);
  processed_thumbnail_for_document.emit (document);
}

void
SessionConnector::replace (Glib::ustring & data, const Glib::ustring src,
			   const Glib::ustring dest)
{
  Glib::RefPtr < Glib::Regex > p =
    Glib::Regex::create (Glib::Regex::escape_string (src));
  data =
    p->replace (data, 0, dest, static_cast < Glib::RegexMatchFlags > (0));
}

Glib::ustring SessionConnector::decode_xml (Glib::ustring data)
{
  if (data.find ("&") == Glib::ustring::npos)
    return data;
  replace (data, "&gt;", ">");
  replace (data, "&lt;", "<");
  replace (data, "&apos;", "'");
  replace (data, "&amp;", "&");
  replace (data, "&quot;", "\"");
  return data;
}
