//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef TAPIOCA_WINDOW_H
#define TAPIOCA_WINDOW_H

#include <sigc++/sigc++.h>
#include <memory>
#include <gtkmm.h>
#include "gallery.h"
#include "session.h"
#include "item-tagging-widget.h"

#define TAPIOCA_EXT ".tao"
    
class ProgressDialog;
class TapiocaWindow: public Gtk::Window
{
 public:
    static TapiocaWindow* create(std::string file = "");

    TapiocaWindow(BaseObjectType* baseObject, 
                  const Glib::RefPtr<Gtk::Builder>& xml);

    virtual ~TapiocaWindow();
 private:

    //for the treeview
    class GalleriesColumns: public Gtk::TreeModelColumnRecord {
  public:
      GalleriesColumns()
	{add(name); add(data);}
      Gtk::TreeModelColumn<Glib::ustring> name;
      Gtk::TreeModelColumn<Tapioca::Gallery*> data;
    };
    const GalleriesColumns galleries_columns;
    Glib::RefPtr<Gtk::ListStore> galleries_list;
    
    sigc::signal<void, Tapioca::Document*> downloaded_document;
    //for the iconview
    class DocumentsColumns: public Gtk::TreeModelColumnRecord {
  public:
      DocumentsColumns() {add(name); add(image); add(data);}
      Gtk::TreeModelColumn<Glib::ustring> name;
      Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > image;
      Gtk::TreeModelColumn<Tapioca::Document*> data;
    };
    const DocumentsColumns documents_columns;
    Glib::RefPtr<Gtk::ListStore> documents_list;

    //widgets
    Gtk::HBox *contents;
    Gtk::TreeView *treeview;
    Gtk::IconView *iconview;
    Gtk::MenuItem *connect_menuitem;
    Gtk::MenuItem *open_offline_menuitem;
    Gtk::MenuItem *save_offline_menuitem;
    Gtk::MenuItem *save_offline_as_menuitem;
    Gtk::MenuItem *save_changes_menuitem;
    Gtk::MenuItem *export_menuitem;
    Gtk::MenuItem *exit_menuitem;
    Gtk::MenuItem *preferences_menuitem;
    Gtk::MenuItem *image_properties_menuitem;
    Gtk::MenuItem *gallery_properties_menuitem;
    Gtk::MenuItem *add_document_menuitem;
    Gtk::MenuItem *remove_document_menuitem;
    Gtk::MenuItem *add_gallery_menuitem;
    Gtk::MenuItem *remove_gallery_menuitem;
    Gtk::CheckMenuItem *fullscreen_menuitem;
    Gtk::CheckMenuItem *show_tags_menuitem;
    Gtk::MenuItem *view_document_menuitem;
    Gtk::MenuItem *about_menuitem;
    Gtk::Label *gallery_label;
    Gtk::ToolButton *connect_toolbutton;
    Gtk::ToolButton *open_toolbutton;
    Gtk::ToolButton *save_toolbutton;
    Gtk::ToolButton *publish_toolbutton;

    Gtk::Menu treeview_context_menu;
    Gtk::Menu iconview_context_menu;
    Tapioca::ItemTaggingWidget *item_tagging_widget;

    //callbacks
    void on_window_loaded(std::string filename);
    void on_connect_clicked();
    void on_open_offline_changes_clicked();
    void on_save_offline_changes_clicked();
    void on_save_offline_changes_as_clicked();
    void on_save_changes_to_atpic_clicked();
    void on_export_clicked();
    void on_exit_clicked();

    void on_preferences_clicked();
    void on_image_properties_clicked();
    void on_gallery_properties_clicked();

    void on_about_clicked();

    void on_gallery_selected();
    void on_document_selected();
    void on_gallery_activated(const Gtk::TreeModel::Path& path,
                              Gtk::TreeViewColumn *col);
    void on_document_activated(const Gtk::TreeModel::Path& path);
    void on_treeview_clicked(GdkEventButton* event);
    void on_iconview_clicked(GdkEventButton* event);

    void on_fullscreen_toggled();
    void on_show_tags_toggled();

    void on_add_gallery_clicked();
    void on_remove_gallery_clicked();
    void on_add_document_clicked();
    void on_remove_document_clicked();
    void on_view_document_clicked();
    void on_iconview_reordered(const Glib::RefPtr<Gdk::DragContext>&context);
    void on_document_drag_data_get(const Glib::RefPtr<Gdk::DragContext> &drag_context, Gtk::SelectionData &data, guint info, guint time);
    void on_galleries_drop_drag_data_received(const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData& selection_data, guint c, guint time);
    void on_documents_drop_drag_data_received(const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData& selection_data, guint c, guint time);
    void view_downloaded_document(Tapioca::Document *document, std::string f);
    void view_local_document(Tapioca::Document *document, std::string file);
    void export_downloaded_document(Tapioca::Document *document, std::string file, std::string export_dir, std::list<Tapioca::Document*> docs, ProgressDialog &d);
    void on_connect_attempted(bool login_successful, Tapioca::Session *new_session);
    void on_download_document_attempted(bool login_successful, Tapioca::Document *document, const Tapioca::Session::SlotDownloadDocument slot);
    void on_document_downloaded_progress(Tapioca::Document *document, ProgressDialog &d);
    void on_document_downloaded(Tapioca::Document *document, ProgressDialog &d);
    void on_document_download_failed(Tapioca::Document *document, ProgressDialog &d, const Tapioca::Session::SlotDownloadDocument slot);
    void on_export_document_download_failed(Tapioca::Document *document, ProgressDialog &d);
    void on_export_documents_attempted(bool logged_in, ProgressDialog &d, std::string export_dir, std::list<Tapioca::Document*> docs, const Tapioca::Session::SlotDownloadDocument slot);
    void on_progress_aborted();
    void on_pull_galleries_failed(Tapioca::Session *s, ProgressDialog &d);
    void on_pulled_gallery_listing(std::list<Tapioca::Gallery*> galleries, Tapioca::Session *s, ProgressDialog &d);
    void on_galleries_pulled(Tapioca::Session *s);
    void on_pulled_documents_for_gallery(Tapioca::Gallery *gallery, std::list<std::string> &thumbnail_urls, std::list<Tapioca::Gallery*> galleries, Tapioca::Session *s, ProgressDialog &d);
    void on_thumbnail_downloaded(Tapioca::Document *document, Tapioca::Session *s, std::list<std::string> thumbnail_urls, ProgressDialog &d);

    // helpers
    Tapioca::Gallery* get_selected_gallery();
    Tapioca::Document* get_selected_document();
    void update_menuitems();
    void fill_in_documents();
    void add_document(Tapioca::Document *document);
    void fill_in_galleries();
    void add_gallery(Tapioca::Gallery *gallery);
    void update_selected_gallery_name();
    void update_selected_document_name();
    void save_session();
    void load_session();
    void remove_deleted_documents();
    void update_window_title();
    void load_session_in_the_background(ProgressDialog &d);
    void save_session_in_the_background(ProgressDialog &d);
    void export_documents_in_the_background(ProgressDialog &d, std::string export_dir, std::list<Tapioca::Document*> docs);
    void view_document_in_the_background(ProgressDialog &d, Tapioca::Document *document, const Tapioca::Session::SlotDownloadDocument slot);
    void pull_galleries_in_the_background(Tapioca::Session *s, ProgressDialog &d);

    void on_published_gallery_added(Tapioca::Gallery *gallery, sigc::slot<void> slot, ProgressDialog &d);
    void on_published_gallery_removed(Tapioca::Gallery *gallery, sigc::slot<void> slot, ProgressDialog &d);
    void on_published_document_added(Tapioca::Document *document, sigc::slot<void> slot, ProgressDialog &d);
    void on_published_document_removed(Tapioca::Document *document, sigc::slot<void> slot, ProgressDialog &d);
    void on_published_gallery_modified(Tapioca::Gallery *gallery, sigc::slot<void> slot, ProgressDialog &d);
    void on_published_document_modified(Tapioca::Document *document, sigc::slot<void> slot, ProgressDialog &d);
    void on_published_galleries_added(ProgressDialog &d);
    void on_published_galleries_removed(ProgressDialog &d);
    void on_published_documents_added(ProgressDialog &d);
    void on_published_documents_removed(ProgressDialog &d);
    void on_published_galleries_modified(ProgressDialog &d);
    void on_published_documents_modified(ProgressDialog &d);
    void on_published_documents_moved(ProgressDialog &d);


    void add_published_galleries(sigc::slot<void> slot, ProgressDialog &d);
    void remove_published_galleries(sigc::slot<void> slot, ProgressDialog &d);
    void add_published_documents(sigc::slot<void> slot, ProgressDialog &d);
    void remove_published_documents(sigc::slot<void> slot, ProgressDialog &d);
    void modify_published_galleries(sigc::slot<void> slot, ProgressDialog &d);
    void modify_published_documents(sigc::slot<void> slot, ProgressDialog &d);
    void move_published_documents(sigc::slot<void> slot, ProgressDialog &d);
    void on_published_document_moved(Tapioca::Document *src, Tapioca::Document *dest, sigc::slot<void> slot, ProgressDialog &d);
    void on_tags_updated(Tapioca::Item *item);
    std::string add_ext_if_necessary(std::string file, std::string ext) const;
    void publish(Tapioca::SessionChanges &changes, ProgressDialog &d);
    bool on_delete_event(GdkEventAny *event);
    bool quit();
    void view_document(Tapioca::Document *document);
    void show_tagging_widget();
    void unshow_tagging_widget();
    void on_tags_changed(std::list<std::string> tags);
    void on_gallery_tags_changed(std::list<std::string> tags, Tapioca::Gallery *gallery);
    void on_document_tags_changed(std::list<std::string> tags, Tapioca::Document*gallery);

    //DATA
    std::string current_save_filename;
    Tapioca::Session *session;
    bool needs_saving;
    std::string temporary_directory;
    bool progress_aborted;
    Tapioca::SessionChanges publish_changes;
    std::list<std::string> d_thumbnail_urls;
};

#endif // TAPIOCA_WINDOW_H
