#! /usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright (C) 2005, 2006  Jonás Melián García

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

jonasmg --AT-- SoftHome --DOT-- net
"""

import fileinput
import glob
import gzip
import os
import os.path
import platform
import popen2
import re
import sys


def show(messageType, message):
    """It shows a different ``message`` according to the
    ``messageType``.
    
    ``messageType`` can be:
    
    * ``d`` -- when lacks some dependency.
    * ``w`` -- for warning.
    * ``e`` -- for error.
    * ``x`` -- for error also but it leaves immediately the program.
    *  `null` -- for the rest of messages.
    """
    if messageType == "w":
        print >>sys.stderr, "Warning:", message
    elif messageType == "e":
        print >>sys.stderr, "Error!", message
    elif messageType == "x":
        sys.exit("Error! %s" % message)
    elif messageType == "d":
        print >>sys.stderr, "Dependency: %s\n" % message
    elif messageType == "":
        print >>sys.stderr, "\t", message


def get_flavour_linux():
    """It checks if the system is Debian or Gentoo or another.
    
    To return ``linux_flavour`` and its installation message in
    ``linux_install``.
    """
    global flag_run, linux_flavour, linux_install
    flag_run = True  # To know if this function has been executed.
    
    if os.path.isfile('/etc/gentoo-release'):
        linux_flavour = "gentoo"
        linux_install = "# emerge "
    elif os.path.isfile('/etc/debian_version'):
        linux_flavour = "debian"
        linux_install = "# apt-get install "
    else:
        linux_flavour = ""
        linux_install = ""


def check():
    """It checks that:
    
    1. this tool has been executed by root.
    2. python version must be 2.2 or higher.
    3. linux kernel must be 2.6 or higher.
    4. PS/2 mouse is supported.
    5. virtual directories */proc* and */sys* are working.
    6. *pci.ids* file [1]_ exists.
    7. *ddcxinfo* tool [2]_ is installed.
    8. *Fontconfig* tool [3]_ is installed.
    
    Return ``flagError`` as True if there is some error.
    
    .. [1]: `Linux PCI ID Repository`. It has all known ID's used in
    PCI devices, and it is used by this tool to identify graphical cards.
    .. [2]: To get data of EDID (plug'n'play) monitors like modes,
    timings, etc.
    .. [3]: Fontconfig_ is used by XFT for configuring and customizing
    font access. XFT lets to use anti-aliased fonts.
    
    .. _Linux PCI ID Repository: http://pciids.sourceforge.net/
    .. _Fontconfig: http://freedesktop.org/Software/fontconfig
    """
    global flag_run, linux_flavour, linux_install, has_fontconfig
    flagError = flag_run = False
    
    # 1) Root
    if os.geteuid() != 0:
        flagError = True
        show('x', "You must be root")
    
    # 2) Python version
    if sys.version_info < (2, 2):
        show('x', "Sorry, this tool requires Python version 2.2 or higher.")
    
    # 3) Kernel
    kernel = platform.release().split('-', 1)[0].split('.')  # 2.6.12-gentoo-r9
    kernel = tuple( map(int, kernel) )
    
    if kernel < (2, 6):
        flagError = True
        show('e', "you need a kernel 2.6 or higher")
    # 4) Mouse
    else:
        if not os.path.exists('/dev/input/mice'):
            flagError = True
            show('e', "To set 'PS/2 mouse (MOUSE_PS2)' in kernel")
    
    # 5) Virtual directories
    virtualDir = ['/proc', '/sys']
    for d in virtualDir:
        try:
            # If the directory is created but there are not files,
            # then it is not mounted.
            if not os.listdir(d):
                flagError = True
                show( 'e', "%r not mounted" % d)
        # If the directory does not exist, then the kernel
        # has not been built with support for that virtual directory.
        except OSError:
            flagError = True
            
            if d == virtualDir[0]:
                var = "PROC_FS"
            else:
                var = "SYSFS"
            
            show('e', "To set '%s file system (%s)' in kernel" % (d, var))
    
    # 6) Linux PCI ID Repository file
    file = "/usr/share/misc/pci.ids"
    if not os.path.isfile(file):
        flagError = True
        dload = "http://pciids.sourceforge.net/pci.ids.gz"
        file = os.path.dirname(file)
        get_flavour_linux()
        
        if linux_flavour == 'gentoo' or linux_flavour == 'debian':
            package = linux_install + "pciutils"
        else:
            package = linux_install + "http://atrey.karlin.mff.cuni.cz/~mj/pciutils.html"
        
        show('d', "To install 'pciutils' package.\n\t%s \
         \nOr uncompress the linux PCI ID in %s \
         \n\t%s" % (package, file, dload))
    
    # 7) ddcxinfo tool
    flagExist = False
    # It gets the directories that are in 'PATH' environment variable
    path = str.split( os.getenv('PATH'), ':' )
    
    # to checking if exist 'ddcprobe' tool.
    for d in path:
        file = os.path.join(d, 'ddcprobe')
        if os.path.isfile(file):
            flagExist = True
            break
    
    if not flagExist:
        flagError = True
        if not flag_run: get_flavour_linux()
        
        if linux_flavour == 'gentoo':
            package = linux_install + "sys-apps/ddcxinfo-knoppix"
        else:
            package = linux_install + "ftp://ftp.ussg.iu.edu/pub/linux/" + \
                              "gentoo/distfiles/ddcxinfo-knoppix_0.6-5.tar.gz"
        
        show('d', "To install 'ddcxinfo-knoppix' tool.\n\t%s" % package)
    
    # 8) fontconfig tool
    has_fontconfig = False
    
    for d in path:
        file = os.path.join(d, 'fc-cache')
        if os.path.isfile(file):
            has_fontconfig = True
            break
    
    if not has_fontconfig:
        if not flag_run: get_flavour_linux()
        
        if linux_flavour == 'gentoo':
            package = linux_install + "media-libs/fontconfig"
        else:
            package = linux_install + \
                              "http://freedesktop.org/Software/fontconfig"
        
        show('w', "You should install 'Fontconfig' tool to let XFT.\n\t%s" \
                  % package)
    
    # It exits if there was some error.
    if flagError:
        sys.exit(1)


def section_files():
    """It looks for the fonts directories in */usr/share/fonts/*,
    check that they have fonts, and place them according to:
    
    1. The first path on our list is */usr/share/fonts/misc* because
    is where all-important fixed and cursor fonts are kept.
    These are crucial for the proper operation of X Window.
    2. It must place the fonts with better quality, beginning with
    outline fonts, and those that more prefers.
    3. If it uses a greater resolution than 1024x768, it must place
    100dpi fonts before of 75dpi.
    4. No longer it is necessary to use *:unscaled*.
    """
    global has_fontconfig
    pathFonts = "/usr/share/fonts/"
    dirBaseMain = ['misc', 'OTF', 'TTF', 'Type1', 'CID', '100dpi', '75dpi']
    dirFonts = list(dirBaseMain)
    # Subdirectories that are omitted because they are not fonts directories.
    dirSkip = dirBaseMain + ['encodings', 'util', 'default']
    
    # It builds directories path from 'pathFonts'.
    for root, dirs, files in os.walk(pathFonts):
        # In the first, it deletes all directories in 'dirSkip'.
        for name in dirs[:]:  # It uses a copy.
            if name in dirSkip:
                dirs.remove(name)
        
        # It gets the directory base name:
        # /usr/share/fonts/corefonts -> corefonts
        dirBase = os.path.basename(root)
        
        # It checks the file type.
        if glob.glob1(root, '*.pcf*'):  # bitmap
            # bitmap directories go to the end.
            dirFonts.append(dirBase)
        elif glob.glob1(root, '*.ot[fc]'):  # OpenType
            # It gets the place where inserting it.
            # OpenType directories go after 'OTF' and before of 'TTF'.
            pos = dirFonts.index('TTF')
            dirFonts.insert(pos, dirBase)
        elif glob.glob1(root, '*.tt[fc]'):  # TrueType
            # TrueType directories go after 'TTF' and before of 'Type1'.
            pos = dirFonts.index('Type1')
            dirFonts.insert(pos, dirBase)
        elif glob.glob1(root, '*.pf[ab]'):  # Type1
            # Type1 directories go after 'Type1' and before of 'CID'.
            pos = dirFonts.index('CID')
            dirFonts.insert(pos, dirBase)
        elif glob.glob1(root, '*.cid'):  # CID
            # CID directories go after 'CID' and before of '100dpi'.
            pos = dirFonts.index('100dpi')
            dirFonts.insert(pos, dirBase)
    
    # It 'fontconfig' tool is installed, then
    # it gets only outline fonts (from 'OTF' until 'CID' - 1) for
    # place them in fontconfig.
    if has_fontconfig:
        global fonts_outline
        pos = dirFonts.index('CID')
        fonts_outline = dirFonts[1:pos]
    
    """ Mozilla - no cesario si hay xft
    # It checks that 'firefox' and/or 'thunderbird' are installed
    # ls -lF /{usr/lib,opt}/*{firefox,thunderbird}*/defaults/pref/*{firefox,thunderbird}.js 2>/dev/null
    fileFirefox = fileThunderbird = ""
    for d in ['/usr/lib/', '/opt/']:
        for root, dirs, files in os.walk(d):
            if root.endswith('/defaults/pref'):
                if re.search('(?i)firefox', root):
                    for file in files:
                        if file.endswith('firefox.js'):
                            fileFirefox = os.path.join(root, file)
                elif re.search('(?i)thunderbird', root):
                    for file in files:
                        if file.endswith('thunderbird.js'):
                            fileThunderbird = os.path.join(root, file)
        # to get only truetype fonts (TTF and OTF).
        if fileFirefox or fileThunderbird:
            global fonts_truetype
            pos = dirFonts.index('Type1')
            fonts_truetype = dirFonts[1:pos]
            break
    """
    
    # It checks if the base directories (dirBaseMain) exist or
    # they have files else those directories are deleted.
    for d in dirBaseMain:
        dirCheck = os.path.join(pathFonts, d)  # /usr/share/fonts/ + misc
        try:
            # If the directory exists and there are files.
            if os.listdir(dirCheck):
                pass
            # If it exists but there are not files.
            else:
                dirFonts.remove(d)
        # If the directory does not exist.
        except OSError:
            dirFonts.remove(d)
    
    # It deletes fonts base directories that are empty.
    if has_fontconfig:
        for x in ['OTF', 'TTF', 'Type1']:
            if x not in dirFonts:
                fonts_outline.remove(x)
    
    """ Mozilla - no necesario si hay xft
    if fonts_truetype:
        for x in ['OTF', 'TTF']:
            if x not in dirFonts:
                fonts_truetype.remove(x)
        
        if fileFirefox:
        
        if fileThunderbird:
    """
    
    # It builds the line for each fonts directory:
    # FontPath "/usr/share/fonts/misc/"
    fonts = []
    for x in dirFonts:
        fonts.append('\tFontPath    "'\
         + os.path.join(pathFonts, x) + '/"')
    fonts = '\n'.join(fonts)
    
    file_tmp.append('\n\
Section "Files"\n\
\tRgbPath     "/usr/lib/X11/rgb"\n\
\tModulePath  "/usr/lib/modules"\n\
' + fonts + '\n\
EndSection')


def section_module():
    """It adds a section for loading modules.
    
    The 3D modules are added only if the driver supports 3d.
    """
    # 'freetype' module supports all fonts types in X.org .
    file_tmp.append('''
Section "Module"
\tLoad  "dbe"
\tLoad  "freetype"''')
    
    # It adds a tag to indicate where are the 3d modules.
    # Thus, if it checks that 3D is not supported by the graphic card,
    # then it will can delete easily 'dri' and 'glx' modules.
    file_tmp.append('#tag_3d')
    
    file_tmp.append('''\tLoad  "dri"
\tLoad  "glx"''')
    
    file_tmp.append('''
\tSubSection "extmod"
\t\tOption  "omit xfree86-dga"
\tEndSubSection
EndSection''')


def search_lang(file):
    '''It looks for the variables about language in file.
    '''
    global listLang, flagLang
    flagLang = flagFile = False
    listTmpLang = []
    
    if os.path.exists(file):
        try:
            for ln in open(file):
                for i in ['LANG', 'LC_ALL']:
                    if not ln.lstrip().startswith('#') and i in ln.upper():
                        getLang = re.findall(r'[a-z]{2}_[A-Z]{2}', ln)
                        if getLang:
                            flagLang = True
                            # It adds the file name once.
                            if not flagFile:
                                flagFile = True
                                listTmpLang.append(os.path.dirname(file))
                            
                            listTmpLang.append(i)
                            listTmpLang.append(''.join(getLang))
            
        except IOError:
            pass
        
        if flagLang:
            listLang.append(listTmpLang)


def get_xkeymap(lang):
    '''It gets the X key map of a locale.
    '''
    # ls -lF /usr/lib/X11/xkb/symbols/
    # more /usr/share/locale/locale.alias
    # Xkeymap of these countries is the country in lower.
    xmapSame = ['BE', 'BR', 'DE', 'DK', 'ES', 'FI', 'FR', 'GB', 'IS', 'IT',
                            'JP', 'NO', 'NL', 'PT', 'SE', 'TR']
    
    xmapDifferent = {
        'CA': 'ca_enhanced',
        'US': 'en_US',
    }
    
    xmapSeveral = (
        ('CZ', 'cz', 'cz_qwerty'),
        ('HU', 'hu', 'hu_qwerty'),
        ('PL', 'pl', 'pl2'),
        ('SK', 'sk', 'sk_qwerty'),
    )
    
    xKeymap = []
    countryLang = lang.split('_')[-1]
    
    if countryLang in xmapSame:
        xKeymap.append(countryLang.lower())
    elif lang.startswith('es_'):  # Latin keyboard.
        xKeymap.append("la")
    elif countryLang in xmapDifferent:
        xKeymap.append(xmapDifferent[countryLang])
    else:
        for i in xmapSeveral:
            if countryLang in i:
                xKeymap = list(i)
                del xKeymap[0]
                break
    
    return xKeymap


def get_locale():
    '''It looks for the locales.
    '''
    global listLang, flagLang, xmap
    flagLangGlobal = False
    listLang = []
    dirHome = "/home"
    
    # Configuration files for system-wide settings:
    # /etc/environment is read by login
    # Gentoo: variables in files of '/etc/env.d/' are set
    # in /etc/{profile,csh}.env
    # Debian '/etc/sysconfig/'
    cfgGlobal = ['environment', 'profile.env', 'sysconfig/i18n',
                        'profile',
                        'csh.cshrc', 'csh.login',
                        'zshenv', 'zprofile', 'zshrc', 'zlogin']
    # For user-specific settings.
    cfgLocal = ['.bash_profile', '.bash_login', '.profile', '.bashrc',
                        '.cshrc', '.tcshrc', '.login',
                        '.zshenv', '.zprofile', '.zshrc', '.zlogin']
    
    # It's going to looking for language variables in
    # global configuration files.
    for f in cfgGlobal:
        search_lang(os.path.join('/etc', f))
        if flagLang:
            flagLangGlobal = True
            break
    
    # It's going to looking for in local configuration files too.
    # It discards hidden directories and another.
    home = [os.path.join(dirHome, d) for d in os.listdir(dirHome) \
                   if not d.startswith('.') and not 'lost+found' in d]
    
    for d in home:
        for f in cfgLocal:
            search_lang(os.path.join(d, f))
            if flagLang:
                break
        
        if not flagLangGlobal and not flagLang:
            for f in os.listdir(d):
                path = os.path.join(d, f)
                # It looks for hidden files.
                if f.startswith('.') and os.path.isfile(path) and \
                   not os.path.islink(path):
                    # It is looking for executables files that
                    # could export variables.
                    line = open(path).readline().strip()
                    if line.startswith('#') and '/bin/' in line:
                        search_lang(path)
                        if flagLang:
                            break
    
    if not listLang:
        show('w', "Language variable not found. Setting 'en_US' Xmap.")
        lang = "en_US"
    else:
        # It lets only 'LANG' variable if possible.
        for i in listLang:
            if 'LC_ALL' in i and 'LANG' in i:
                pos = i.index('LC_ALL')
                del i[pos + 1]
                del i[pos]
                pos = i.index('LANG')
                del i[pos]
            elif not 'LC_ALL' in i:
                pos = i.index('LANG')
                del i[pos]
        
        # It looks for a global language.
        if len(listLang) > 1 and not flagLangGlobal:
            numLang = {}  # It stores the number of each language.
            for i in listLang:
                s = i[-1]
                if not numLang.has_key(s):
                    numLang[s] = 1
                else:
                    numLang[s] += 1
            
            # numLang = {'de_DE': 2, 'en_US': 1}
            # So it needs change the items order for get
            # the maxium value.
            lang = max([(value,key) for key,value in numLang.items()])[-1]
    #lang = max((value,key) for key,value in numLang.iteritems())[-1] #2.4
            
        else:
            lang = listLang[0][-1]  # Global language is on the first list.
        
        # It builds a X keymap configuration file for each user
        # with different locale.
        if len(listLang) > 1:
            flagSkipGlobal = flagShow = False
            for i in listLang:
                # It skips the first item that is the global configuration.
                if flagLangGlobal and not flagSkipGlobal:
                    flagSkipGlobal = True
                    continue
                
                # If global locale is the same that for local user
                # then skip it.
                if lang == i[-1]:
                    continue
                
                xmap = get_xkeymap(i[-1])
                if not xmap:
                    continue
                
                fileTmp = []
                fileXkbmap = os.path.join(i[0], '.Xkbmap')
                fileTmp.append("%s\n" % xmap[0])
                write_file(fileTmp, fileXkbmap)
                
                if not flagShow:
                    print "\n User's X key map"
                    flagShow = True
                print "\t%s\t%s" % (fileXkbmap, xmap[0])
        
    
    xmap = get_xkeymap(lang)
    if not xmap:
        show('w', "'%s' locale Xkeymap not found. Using 'en_US'." % lang)
        xmap.append('en_US')
        
    print "\n Global X key map: %s" % xmap[0]
    '''
    # To know if a file has some character non printable: od -c file
    pattern = re.compile(r'/usr/.*/[xwgk]dm\0')
    
    xStarted=''
    for p in glob.glob("/proc/[0-9]*/cmdline"):
        line = open(p).readline()
        if line:
            if re.match(pattern, line):
                xStarted = "dm"
                break
            elif '/usr/bin/startx' in line:
                xStarted = "startx"
                break
    '''


def section_input():
    '''It adds mouse and keyboard and it also detects touchpad.
    
    If touchpad is detected then checks to exist a specialized driver
    for to have advanced features of it. Although that driver isn't
    necessary for using it.
    '''
    global linux_flavour, linux_install, flagTpad, flagTpadOK, xmap
    flagTpad = flagEvent = flagDriver = flagTpadOK = False
    
    get_locale()
    
    # It checks if exists a touchpad.
    pattern = re.compile('(?i)(synaptics|alps)')
    for ln in open('/proc/bus/input/devices'):
        # It looks for if exist touchpad name 'synaptics or 'alps'.
        if ln.startswith('N:') and re.search(pattern, ln):
            flagTpad = True
        # If it exists, it checks if there is an event for that device.
        if flagTpad and ln.startswith('H:'):
            if 'event' in ln:
                flagEvent = True
                eventTouchpad = ln.split('event')[-1][0]  # The event number.
                break
            else:
                break
    
    if flagTpad:
        moduleTouchpad = "/usr/lib/modules/input/synaptics_drv.o"
        
        # It checks if the touchpad driver is installed.
        if os.path.isfile(moduleTouchpad):
            flagDriver = True
        else:
            get_flavour_linux()
            
            if linux_flavour == 'gentoo':
                package = linux_install + "x11-drivers/synaptics"
            else:
                package = linux_install + "http://w1.894.telia.com" + \
                                   "/~u89404340/touchpad/"
            
            show('d', "You must install a specialized XFree86 driver \
to take advantage of the advanced features of the touchpad.\n\t%s" % package)
        
        if flagEvent:
            show('w', "To set 'Event interface (INPUT_EVDEV)' in kernel \
to use touchpad driver.")
    
    if flagEvent and flagDriver:
        flagTpadOK = True
    
    # It puts options for all XKeymaps.
    flagFirst = True
    tmp = []
    
    for x in xmap:
        if flagFirst:
            tmp.append('\tOption      "XkbLayout" "%s"' % x)
            flagFirst = False
        else:
            tmp.append('\t#Option      "XkbLayout" "%s"' % x)
    
    if len(xmap) > 1:
        show('', "'%s' X keymap is set as default." % (xmap[0]))
    
    xmap = '\n'.join(tmp)
    
    # In the first it adds keyboard and mouse.
    file_tmp.append('\n\
Section "InputDevice"\n\
\tIdentifier  "Keyboard"\n\
\tDriver      "kbd"\n\
\tOption      "XkbModel" "pc105"\n\
' + xmap + '\n\
EndSection')
    
    file_tmp.append('''
Section "InputDevice"
\tIdentifier  "Mouse"
\tDriver      "mouse"
\tOption      "Protocol" "auto"
\tOption      "Device" "/dev/input/mice"
\tOption      "ZAxisMapping" "4 5"
EndSection''')
    
    # Later it adds touchpad if it exists, its driver is installed and
    # there is an event.
    if flagTpadOK:
        file_tmp.append('\n\
Section "InputDevice"\n\
\tIdentifier  "Touchpad"\n\
\tDriver      "synaptics"\n\
\tOption      "SendCoreEvents"\n\
\tOption      "Device" "/dev/input/event'\
 + eventTouchpad + '"\n\
\tOption      "Protocol" "event"\n\
\tOption      "SHMConfig" "on"\n\
\tOption      "LeftEdge" "1900"\n\
\tOption      "RightEdge" "5400"\n\
\tOption      "TopEdge" "1400"\n\
\tOption      "BottomEdge" "4500"\n\
\tOption      "FingerLow" "25"\n\
\tOption      "FingerHigh" "30"\n\
\tOption      "MaxTapTime" "180"\n\
\tOption      "MaxTapMove" "220"\n\
\tOption      "VertScrollDelta" "100"\n\
\tOption      "MinSpeed" "0.02"\n\
\tOption      "MaxSpeed" "0.18"\n\
\tOption      "AccelFactor" "0.0010"\n\
EndSection')


def get_card_ids(directory):
    '''It gets the graphic card identifiers.
    '''
    
    # The vendor identifier is in:
    # /sys/bus/pci/devices/[disp_gráfico]/vendor
    file = os.path.join(directory, 'vendor')
    # It reads the firs line, deletes the last character (\n)
    # and deletes both first characters (0x).
    g_card['idVendor'] = open(file).readline().rstrip()[2:]
    
    # /sys/bus/pci/devices/[disp_graphic]/device
    file = os.path.join(directory, 'device')
    g_card['idDevice'] = open(file).readline().rstrip()[2:]
    
    #/sys/bus/pci/devices/[disp_graphic]/subsystem_vendor
    file = os.path.join(directory, 'subsystem_vendor')
    g_card['idSubVendor'] = open(file).readline().rstrip()[2:]
    
    #/sys/bus/pci/devices/[disp_graphic]/subsystem_device
    file = os.path.join(directory, 'subsystem_device')
    g_card['idSubDevice'] = open(file).readline().rstrip()[2:]
    
    # It gets the directory base (0000:02:00.0),
    # that it is the graphic card bus,
    g_card['Bus'] = os.path.basename(directory)
    # and it is turned in the format for configuration file (PCI:2:0:0).
    g_card['idBus'] = 'PCI:%i:%i:%i' % \
    tuple(map(int, directory.split(':' ,1)[-1].replace('.', ':').split(':')))
    # Another way:
    #g_card['idBus'] = "PCI:%d:%d:%d" % \
    #tuple(map(int, re.split('[:.]', directory)[1:]))


def get_card_data():
    '''It looks for identifiers in pci data base and it gets its names.
    '''
    filePci = "/usr/share/misc/pci.ids"
    flagVendor = flagDevice = False
    idDevice = re.compile(r'^\t%s' % g_card['idDevice'])
    idSubDevice = re.compile(r'^\t\t%s %s'
                             % (g_card['idSubVendor'], g_card['idSubDevice']))
    
    try:
        for ln in open(filePci):
            if not flagVendor and ln.startswith(g_card['idVendor']):
                # It deletes the first column (identifier).
                g_card['Vendor'] = ln.split(None, 1)[-1].rstrip()
                flagVendor = True
            if not flagDevice and flagVendor and re.search(idDevice, ln):
                g_card['Device'] = ln.split(None, 1)[-1].rstrip()
                flagDevice = True
            if flagDevice and re.search(idSubDevice, ln):
                g_card['Model'] = ln.split(None, 2)[-1].rstrip()
                break
    except IOError, err:
        print "Error! %s: %r" % (err.strerror, err.filename)
        sys.exit(1)


def get_card_driver():
    '''It gets the graphic card driver and it checks if it exists.
    '''
    flagDriver = flagOwner = flagFree = False
    drvFree2Owner = { 'radeon': 'fglrx', 'nv': 'nvidia' }
    
    driver = {
        '1142': 'apm',
        'edd8': 'apm',
        '102c': 'chips',
        '3d3d': 'glint',
        '105d': 'i128',
        '102b': 'mga',
        '10c8': 'neomagic',
        '10de': 'nv',
        '1163': 'rendition',
        '126f': 'siliconmotion',
        '1039': 'sis',
        '121a': 'tdfx',
        '1023': 'trident',
        '100c': 'tseng',
        '1106': 'via',
        '15ad': 'vmware',
    }
    
    drivers = (
        ('1002', 'ati', 'r128:rage 128,rage fury,xpert 128,xpert 99',
                 'radeon:radeon,firegl,fire gl'),
        ('1013', 'cirrus', 'cirrus_alpine:alpine', 'cirrus_laguna:laguna'),
        ('8086', 'i740:827\d{2},7\d{2}', 'i810:82[89]\d{2},[89]\d{2}'),
        ('12d2', 'nv', 'riva128:riva128'),
        ('5333', 's3', 's3virge:virge', 'savage:savage'),
    )
    
    if driver.has_key(g_card['idVendor']):
        flagDriver = True
        driverCard = driver[g_card['idVendor']]
    else:
        for drv in drivers:
            if flagDriver: break
            driverBase = ""
            # It compares the first column with 'idVendor'.
            if drv[0] == g_card['idVendor']:
                for column in drv[1:]:
                    if ':' in column:
                        # It splits driver and strings.
                        driverTmp, string = column.split(':')
                        string = string.split(',')
                        nameCard = g_card['Device'].lower()
                        
                        for s in string:
                            if re.search(s, nameCard):
                                flagDriver = True
                                driverCard = driverTmp
                                break
                    else:
                        driverBase = column
                
                if not flagDriver and driverBase:
                    flagDriver = True
                    driverCard = driverBase
    
    # If the graphic card driver is not detected, then it shows
    # the card information for that it can be sent by the user
    # through e-mail.
    if not flagDriver:
        show('e', "Graphic card driver doesn't detected.")
        print "Please send the next information to %s\n\n%s" % (email, g_card)
        sys.exit(1)
    
    # It checks if the free driver has an owner alternative and
    # if it is installed.
    if drvFree2Owner.has_key(driverCard) and \
       exist_driver(drvFree2Owner[driverCard]):
        flagOwner = True
        g_card['DriverOwner'] = drvFree2Owner[driverCard]
    
    # It checks if exist the free driver.
    if exist_driver(driverCard):
        flagFree = True
        g_card['Driver'] = driverCard
    
    # If does not exist any of detected drivers then
    # it checks the basic drivers.
    if not flagFree and not flagOwner:
        for x in ['vesa', 'fbdev', 'vga']:
            if exist_driver(x):
                flagFree = True
                g_card['Driver'] = x
                break
        if not flagFree:
            show('x', "Don't found needed drivers.")


def exist_driver(driver):
    '''It checks if exist a video driver.
    '''
    path = "/usr/lib/modules/drivers/"
    driverPath = "%s%s" % (path, driver)
    
    if os.path.isfile('%s.o' % driverPath) or \
       os.path.isfile('%s_drv.o' % driverPath):
        return True


def get_card_options():
    '''It looks for the driver options in its manual, if it exists.
    '''
    pathManual = "/usr/share/man/man4/%s.4x.gz" % (g_card['Driver'])
    
    if os.path.isfile(pathManual):
        options = []
        fileManual = gzip.GzipFile(pathManual, 'r:gz')
        for ln in fileManual:
            if ln.startswith('.BI "Option'):
                lineTemp = ln.lstrip('.BI  ').replace('\*q','') \
                                    .replace('"','').split()
                
                lineTemp.insert(0, '\t#')
                lineTemp.insert(2, '      "')
                lineTemp.insert(4, '"')
                optionLen = len(lineTemp[3])
                
                if optionLen < 9:
                    lineTemp.insert(5, '\t\t\t# ')
                elif optionLen > 25:
                    lineTemp.insert(5, '\t# ')
                else:
                    lineTemp.insert(5, '\t\t# ')
                
                options.append(''.join(lineTemp))
        
        options = '\n'.join(options)
        g_card['options'] = options


def check_driver_3d():
    '''It checks if X driver supports 3D.
    '''
    global graph_cards
    flag3d = False
    driverNo3d = ['vmware', 'vesa', 'fbdev', 'vga']
    
    # It checks if 3D is supported by some graphic card detected.
    for card in graph_cards:
        if card['Driver'] not in driverNo3d:
            flag3d = True
    
    # If 3D is not supported then it deletes 3D modules and the tag.
    index3d = file_tmp.index('#tag_3d')  # It gets the 'tag_3d' position.
    if not flag3d:
        del file_tmp[index3d+1] # In the first it deletes after tag.
        del file_tmp[index3d]   # Later it deletes the tag.
    else:
        del file_tmp[index3d]  # It only deletes the tag.
        file_tmp.append('''
Section "DRI"
\tMode  0666
EndSection

#Section "Extensions"
#\tOption  "Composite"
#EndSection''')


def section_device():
    '''It detects all graphic cards, and it adds its options from man.
    '''
    global graph_cards
    
    # Devices classes are defined in '/usr/include/linux/pci_ids.h'
    # grep 'CLASS.*0x03' /usr/include/linux/pci_ids.h
    # 0x03 corresponds to graphic devices so it looks for it in 'class' files.
    fileClass = fileinput.input(glob.glob('/sys/bus/pci/devices/*/class'))
    
    dirDevices = []
    # It gets the path of all graphic cards.
    for ln in fileClass:
        if ln.startswith('0x030'):  # It discards 0x038000
            dirDevices.append(os.path.dirname(fileClass.filename()))
    
    # For all graphic cards that it have found ...
    graph_cards = []
    for x in range(len(dirDevices)):
        g_card['id'] = "Card_%u" % (x+1)
        get_card_ids(dirDevices[x])
        get_card_data()
        get_card_driver()
        get_card_options()
        graph_cards.append(g_card)  #It adds information of each graphic card.
        
        file_tmp.append('\n\
Section "Device"\n\
\tIdentifier  "' + graph_cards[x]['id'] + '"\n\
\tBusID       "' + graph_cards[x]['idBus'] + '"\n\
\tVendorName  "' + graph_cards[x]['Vendor'] + '"\n\
\tBoardName   "' + graph_cards[x]['Device'] + '"\n\
\tDriver      "' + graph_cards[x]['Driver'] + '"')
        
        if graph_cards[x].has_key('options'):
            file_tmp.append(graph_cards[x]['options'])
        
        if graph_cards[x].has_key('DriverOwner'):
            file_tmp.append('\n\
\t#Driver      "' + graph_cards[x]['DriverOwner'] + '"')
        
        file_tmp.append('EndSection')
    
    check_driver_3d()


def rsplitx(s): #2.3
    try:
        s = s.strip()
        i = s.rindex('x')
        return s[:i], s[i+1:]
    except ValueError:
        return s


def get_data_ddcprobe():
    '''It uses *ddcprobe* tool for get information about monitor and
    modes in graphic card.
    '''
    
    global is_digital
    frecuency = []; modes = []
    video_monitors = []  # To Storing several monitors.
    # String to find the video modes of the graphical card.
    pattern = re.compile(r'^\s*\d*x\d*x\d')  # 320x200x64k
    
    # Number of colors with its corresponding number of color bits.
    colour2bits = { '2': 1, '16': 4, '256': 8, '32k': 15,
                              '64k': 16, '16m': 24 }
    
    # It runs this tool to get monitor information.
    p = popen2.Popen3('ddcprobe')
    if p.wait() != 0:
        show('x', "'ddcprobe' has failed in your system.")
    
    ddc = p.fromchild.read().split('\n')  #It stores the output of 'ddcprobe'
    lastline =  filter(str.strip, ddc)[-1]
    
    # Old monitors and laptops does not show information about EDID.
    if lastline.startswith('EDID'):
        v_monitor['edid'] = False
        if flagTpad:
            v_monitor['laptop'] = True
        else:
            v_monitor['laptop'] = False  # An old monitor.
    else:
        v_monitor['edid'] = True
    
    
    flagModes = True; flagMonitor = flagRun = is_digital = False
    for ln in ddc:
        # Modes with its colour number supported by the graphical card.
        if flagModes and re.match(pattern, ln):
            # It splits by the last 'x', so it splits mode and colours
            # (800x600x16)
            #mode, num = ln.strip().rsplit('x', 1) #2.4
            mode, num = rsplitx(ln)
            mode = map(int, mode.split('x'))
            # It builds the list with colour bits and video mode.
            modes.append( (colour2bits[num], mode) )
        
        if v_monitor['edid']:  # For plug'n'play monitors.
            # It gets the manufacturer.
            if ln.startswith('Manufacturer:'):
                flagModes = False
                v_monitor['vendor'] = ln.split(': ')[-1]
            # For to know if monitor is LCD.
            elif not flagRun and ln.startswith('Input signal type:') and \
                  not 'analog' in ln:
                is_digital = flagRun = True
            # The screen size
            # Screen size max 36 cm horizontal, 27 cm vertical
            elif ln.startswith('Screen size'):
                horiz = float(ln.split(' cm')[0].split()[-1])
                vert = float(ln.split(' cm')[1].split()[-1])
                # It turns from cm. to mm.
                horiz *= 10; vert *= 10
                v_monitor['screen'] = str(horiz) + '  ' + str(vert)
            # (Gamma: 2.760000.)
            elif ln.startswith('Gamma:'):
                v_monitor['gamma'] = float(ln.split(': ')[-1].rstrip('.'))
            elif ln.startswith('DPMS'):
                v_monitor['pixel'] = ln.split(': ')[-1].split(',')[0].lower()
                if 'standby' in ln:
                    v_monitor['dpms'] = True
                else:
                    v_monitor['dpms'] = False
            # Video modes and monitor frecuency.
            elif ln.startswith('Standard'):
                # The list is built in the format needed for to make
                # a correct arrangement.
                # (Standard timing 0: 85 Hz, 640x480)
                _, _, _, freq, _, mode = ln.split()
                # It splits mode (horiz,vert) and is transformed in decimal.
                mode = map(int, mode.split('x'))
                frecuency.append((freq, mode))
            # It indicates when begins the monitor characteristics.
            elif ln.startswith('Monitor details 1'):
                flagMonitor = True
            
            if flagMonitor:
                if ln.startswith('\tName:'):
                    v_monitor['model'] = ln.split(': ')[-1]
                elif ln.startswith('\tTiming ranges:'):
                    v_monitor['sync'] = ln.split('= ')[1].split(',')[0]
                    v_monitor['refresh'] = ln.split('= ')[-1]
    
    # It sorts of greater to minor (better to worse) the video modes.
    #modes.sort(reverse=True) #2.4
    modes.sort();modes.reverse()  # By colour bits.
    v_monitor['modes'] = modes
    
    if v_monitor['edid']:
        #frecuency.sort(reverse=True) #2.4
        frecuency.sort();frecuency.reverse()  # By refresh rate.
        # It only adds resolutions (of better to worse
        # ('85', [1280, 1024]) -> [1280, 1024]
        frecuency = [mode for (refresh,mode) in frecuency]
        v_monitor['frecuency'] = frecuency


def choose_modes():
    '''It looks for the best modes and depth bits.
    '''
    
    # It builds a list with colour bits supported in the graphical card.
    depthSupport=[]
    for ln in v_monitor['modes']:  # (24, [1280, 1024])
        if ln[0] not in depthSupport:
            depthSupport.append(ln[0])
    
    v_monitor['depthSupport'] = depthSupport
    
    # It gets the best bits colour number (as minium 8 bits),
    # for a resolution like minimum of 1024x768 else 800x600
    for resolution in [1024, 860]:
        depth = ''
        # It checks in the card mode,
        for ln in v_monitor['modes']:
            # if resolution is greater or equal than 1024 or 800,
            # positive if x>=resolution
            if cmp(ln[1][0], resolution):  # (24, [_1280_, 1024])
                # and depth bits.
                depth = ln[0]
                if depth >= 8: break
        if depth >= 8:
            v_monitor['depth'] = str(depth)
            break
    
    if not v_monitor['edid']:
        modes_non_edid()
    
    # It sorts the video modes of graphical card according to
    # order of modes in frecuency 
    modes = []
    for bits in v_monitor['depthSupport']:
    # It looks each monitor resolution,
        for mode in v_monitor['frecuency']:  # [1280, 1024]
            # to checking if it exists in the card modes with that bits number
            for ln in v_monitor['modes']:  # (24, [1280, 1024])
                # It only adds supported modes by monitor sorted of
                # better to worse for each bits number.
                if bits == ln[0] and mode == ln[1]:
                    modes.append(ln)
                    break
        
    del v_monitor['frecuency']
    v_monitor['modes'] = modes


def modes_non_edid():
    '''It adds VESA modes supported in graphic card.
    '''
    
    # cvs.freedesktop.org/xorg/xc/programs/Xserver/hw/xfree86/etc/vesamodes
    vesaModes = (
        ( [640, 350],
'# 640x350 @ 85Hz (VESA) hsync: 37.9kHz\n\
ModeLine "640x350"    31.5  640  672  736  832    350  382  385  445 \
+hsync -vsync' ),
        ( [640, 400],
'# 640x400 @ 85Hz (VESA) hsync: 37.9kHz\n\
ModeLine "640x400"    31.5  640  672  736  832    400  401  404  445 \
-hsync +vsync' ),
        ( [720, 400],
'# 720x400 @ 85Hz (VESA) hsync: 37.9kHz\n\
ModeLine "720x400"    35.5  720  756  828  936    400  401  404  446 \
-hsync +vsync' ),
        ( [640, 480],
'# 640x480 @ 60Hz (Industry standard) hsync: 31.5kHz\n\
ModeLine "640x480"    25.2  640  656  752  800    480  490  492  525 \
-hsync -vsync' ),
        ( [640, 480],
'# 640x480 @ 72Hz (VESA) hsync: 37.9kHz\n\
ModeLine "640x480"    31.5  640  664  704  832    480  489  491  520 \
-hsync -vsync' ),
        ( [640, 480],
'# 640x480 @ 75Hz (VESA) hsync: 37.5kHz\n\
ModeLine "640x480"    31.5  640  656  720  840    480  481  484  500 \
-hsync -vsync' ),
        ( [640, 480],
'# 640x480 @ 85Hz (VESA) hsync: 43.3kHz\n\
ModeLine "640x480"    36.0  640  696  752  832    480  481  484  509 \
-hsync -vsync' ),
        ( [800, 600],
'# 800x600 @ 56Hz (VESA) hsync: 35.2kHz\n\
ModeLine "800x600"    36.0  800  824  896 1024    600  601  603  625 \
+hsync +vsync' ),
        ( [800, 600],
'# 800x600 @ 60Hz (VESA) hsync: 37.9kHz\n\
ModeLine "800x600"    40.0  800  840  968 1056    600  601  605  628 \
+hsync +vsync' ),
        ( [800, 600],
'# 800x600 @ 72Hz (VESA) hsync: 48.1kHz\n\
ModeLine "800x600"    50.0  800  856  976 1040    600  637  643  666 \
+hsync +vsync' ),
        ( [800, 600],
'# 800x600 @ 75Hz (VESA) hsync: 46.9kHz\n\
ModeLine "800x600"    49.5  800  816  896 1056    600  601  604  625 \
+hsync +vsync' ),
        ( [800, 600],
'# 800x600 @ 85Hz (VESA) hsync: 53.7kHz\n\
ModeLine "800x600"    56.3  800  832  896 1048    600  601  604  631 \
+hsync +vsync' ),
        ( [1024, 768],
'# 1024x768i @ 43Hz (industry standard) hsync: 35.5kHz\n\
ModeLine "1024x768"   44.9 1024 1032 1208 1264    768  768  776  817 \
+hsync +vsync Interlace' ),
        ( [1024, 768],
'# 1024x768 @ 60Hz (VESA) hsync: 48.4kHz\n\
ModeLine "1024x768"   65.0 1024 1048 1184 1344    768  771  777  806 \
-hsync -vsync' ),
        ( [1024, 768],
'# 1024x768 @ 70Hz (VESA) hsync: 56.5kHz\n\
ModeLine "1024x768"   75.0 1024 1048 1184 1328    768  771  777  806 \
-hsync -vsync' ),
        ( [1024, 768],
'# 1024x768 @ 75Hz (VESA) hsync: 60.0kHz\n\
ModeLine "1024x768"   78.8 1024 1040 1136 1312    768  769  772  800 \
+hsync +vsync' ),
        ( [1024, 768],
'# 1024x768 @ 85Hz (VESA) hsync: 68.7kHz\n\
ModeLine "1024x768"   94.5 1024 1072 1168 1376    768  769  772  808 \
+hsync +vsync' ),
        ( [1152, 864],
'# 1152x864 @ 75Hz (VESA) hsync: 67.5kHz\n\
ModeLine "1152x864"  108.0 1152 1216 1344 1600    864  865  868  900 \
+hsync +vsync' ),
        ( [1280, 960],
'# 1280x960 @ 60Hz (VESA) hsync: 60.0kHz\n\
ModeLine "1280x960"  108.0 1280 1376 1488 1800    960  961  964 1000 \
+hsync +vsync' ),
        ( [1280, 960],
'# 1280x960 @ 85Hz (VESA) hsync: 85.9kHz\n\
ModeLine "1280x960"  148.5 1280 1344 1504 1728    960  961  964 1011 \
+hsync +vsync' ),
        ( [1280, 1024],
'# 1280x1024 @ 60Hz (VESA) hsync: 64.0kHz\n\
ModeLine "1280x1024" 108.0 1280 1328 1440 1688   1024 1025 1028 1066 \
+hsync +vsync' ),
        ( [1280, 1024],
'# 1280x1024 @ 75Hz (VESA) hsync: 80.0kHz\n\
ModeLine "1280x1024" 135.0 1280 1296 1440 1688   1024 1025 1028 1066 \
+hsync +vsync' ),
        ( [1280, 1024],
'# 1280x1024 @ 85Hz (VESA) hsync: 91.1kHz\n\
ModeLine "1280x1024" 157.5 1280 1344 1504 1728   1024 1025 1028 1072 \
+hsync +vsync' ),
        ( [1600, 1200],
'# 1600x1200 @ 60Hz (VESA) hsync: 75.0kHz\n\
ModeLine "1600x1200" 162.0 1600 1664 1856 2160   1200 1201 1204 1250 \
+hsync +vsync' ),
        ( [1600, 1200],
'# 1600x1200 @ 65Hz (VESA) hsync: 81.3kHz\n\
ModeLine "1600x1200" 175.5 1600 1664 1856 2160   1200 1201 1204 1250 \
+hsync +vsync' ),
        ( [1600, 1200],
'# 1600x1200 @ 70Hz (VESA) hsync: 87.5kHz\n\
ModeLine "1600x1200" 189.0 1600 1664 1856 2160   1200 1201 1204 1250 \
+hsync +vsync' ),
        ( [1600, 1200],
'# 1600x1200 @ 75Hz (VESA) hsync: 93.8kHz\n\
ModeLine "1600x1200" 202.5 1600 1664 1856 2160   1200 1201 1204 1250 \
+hsync +vsync' ) ,
        ( [1600, 1200],
'# 1600x1200 @ 85Hz (VESA) hsync: 106.3kHz\n\
ModeLine "1600x1200" 229.5 1600 1664 1856 2160   1200 1201 1204 1250 \
+hsync +vsync' ),
        ( [1792, 1344],
'# 1792x1344 @ 60Hz (VESA) hsync: 83.6kHz\n\
ModeLine "1792x1344" 204.8 1792 1920 2120 2448   1344 1345 1348 1394 \
-hsync +vsync' ),
        ( [1792, 1344],
'# 1792x1344 @ 75Hz (VESA) hsync: 106.3kHz\n\
ModeLine "1792x1344" 261.0 1792 1888 2104 2456   1344 1345 1348 1417 \
-hsync +vsync' ),
        ( [1856, 1392],
'# 1856x1392 @ 60Hz (VESA) hsync: 86.3kHz\n\
ModeLine "1856x1392" 218.3 1856 1952 2176 2528   1392 1393 1396 1439 \
-hsync +vsync' ),
        ( [1856, 1392],
'# 1856x1392 @ 75Hz (VESA) hsync: 112.5kHz\n\
ModeLine "1856x1392" 288.0 1856 1984 2208 2560   1392 1393 1396 1500 \
-hsync +vsync' ),
        ( [1920, 1440],
'# 1920x1440 @ 60Hz (VESA) hsync: 90.0kHz\n\
ModeLine "1920x1440" 234.0 1920 2048 2256 2600   1440 1441 1444 1500 \
-hsync +vsync' ),
        ( [1920, 1440],
'# 1920x1440 @ 75Hz (VESA) hsync: 112.5kHz\n\
ModeLine "1920x1440" 297.0 1920 2064 2288 2640   1440 1441 1444 1500 \
-hsync +vsync' ),
    )
    
    # It only selects Vesa video modes supported in the graphic card.
    frecuency = []; modeLines = []
    for vesa in vesaModes:
        for mode in v_monitor['modes']:
            if mode[1] == vesa[0]:
                modeLines.append(vesa[1] + '\n')
                if vesa[0] not in frecuency:
                    frecuency.append(vesa[0])
                break
    
    modeLines = '\n'.join(modeLines).rstrip()  # It deletes the last line.
    v_monitor['modeLines'] = modeLines
    v_monitor['frecuency'] = frecuency


def write_data_monitor():
    '''It adds information about monitor.
    '''
    flagSection = False
    
    file_tmp.append('''
Section "Monitor"
\tIdentifier   "Monitor_1"''')
    
    if v_monitor['edid']:
        file_tmp.append('\
\tVendorName   "' + v_monitor['vendor'] +'"\n\
\tModelName    "' + v_monitor['model'] +'"\n\
\tHorizSync     ' + v_monitor['sync'] +'\n\
\tVertRefresh   ' + v_monitor['refresh'] +'\n\
\tDisplaySize   ' + v_monitor['screen'] +'\n\
\tGamma         ' + str(v_monitor['gamma']) +'')
        
        # Options for DPMS.
        if v_monitor['dpms']:
            flagSection = True
            file_tmp.append('''\
\tOption       "DPMS"
EndSection

Section "ServerFlags"
\tOption  "StandbyTime" "15"
\tOption  "SuspendTime" "25"
\tOption  "OffTime"     "35"''')
    
    else:  # Laptos and old monitors.
        # If it is not a laptop then it adds minimum frequencies.
        if not v_monitor['laptop']:
            sync = "\
\tHorizSync    28 - 96  # Warning: This may fry old Monitors\n\
\tVertRefresh  50 - 75  # Very conservative. May flicker.\n\
\t#HorizSync    28 - 78  # Warning: This may fry very old Monitors\n\
\t#VertRefresh  50 - 62  # Extreme conservative. Will flicker."
            
            file_tmp.append(sync)
            
            # It shows frecuency configuration and warning message.
            show('w', "No DDC-capable monitor attached?")
            print "BY SPECIFYING AN OUT-OF-RANGE FREQUENCY CAN CAUSE \
YOUR MONITOR TO FRY.\n\n\
For non-EDID monitors has been used this configuration:\n%s\n\n\
But YOU SHOULD REFER TO YOUR MONITOR'S USER MANUAL FOR THE CORRECT \
NUMBERS." % sync
        
        file_tmp.append('\n' + v_monitor['modeLines'] +'')
    
    
    if not flagSection:
        file_tmp.append('''\
EndSection

Section "ServerFlags"''')
    
    # It checks if the system has been run with option
    # for deactivating power manager.
    if 'noapm' in open('/proc/cmdline').readline():
        file_tmp.append('\tOption  "NoPM"')
        show('w', "Your system has been run with 'noapm' option.\n\
So X.org has been configured with that option too.")
    
    file_tmp.append('\
\tOption  "AllowMouseOpenFail"\n\
EndSection\n\
\n\
Section "Screen"\n\
\tIdentifier  "Screen_1"\n\
\tDevice      "Card_1"\n\
\tMonitor     "Monitor_1"\n\
\tDefaultColorDepth  '+ v_monitor['depth'] +'\n')
    
    
    for bits in v_monitor['depthSupport']:  # 24, 16, 15, 8, 4
        modes = []
        file_tmp.append('\
\tSubSection "Display"\n\
\t\tDepth   '+ str(bits) +'')
        
        for ln in v_monitor['modes']:
            if bits == ln[0]:
                modes.append(' "%sx%s"' % (ln[1][0], ln[1][1]))
        
        file_tmp.append('\
\t\tModes ' + ''.join(modes) + '\n\
\tEndSubSection')
    
    file_tmp.append('EndSection')


def section_monitor():
    get_data_ddcprobe()
    choose_modes()
    write_data_monitor()


def section_layout():
    global flagTpadOK
    
    file_tmp.append('''
Section "ServerLayout"
\tIdentifier   "Layout_1"
\tScreen     1 "Screen_1"
\tInputDevice  "Keyboard" "CoreKeyboard"''')
    
    if not flagTpadOK:
        file_tmp.append('\
\tInputDevice  "Mouse"    "CorePointer"\n\
EndSection\n')
    else:
        file_tmp.append('\
\tInputDevice  "Touchpad" "CorePointer"\n\
\tInputDevice  "Mouse"    "SendCoreEvents"\n\
EndSection\n')


def fontconfig():
    '''It configures the Xft font system.
    '''
    
    global fonts_outline
    pathFonts = "/usr/share/fonts/"
    fileLocalFontconfig = "/etc/fonts/local.conf"
    fileTmp = []
    fonts = []
    
    for x in fonts_outline:
        fonts.append('\t<dir>' + os.path.join(pathFonts, x) + '</dir>')
    fonts.append('\t<dir>~/.fonts</dir>')
    fonts = '\n'.join(fonts)
    
    fileTmp.append('\
<?xml version="1.0"?>\n\
<!DOCTYPE fontconfig SYSTEM "fonts.dtd">\n\
<!-- /etc/fonts/local.conf file for local customizations -->\n\
<fontconfig>\n\
\n\
<!-- Font directory list configured on  -->\n\
' + fonts + '')
    
    if is_digital:
        fileTmp.append('\n\
<!-- Enable sub-pixel rendering -->\n\
\t<match target="font">\n\
\t\t<test qual="all" name="rgba">\n\
\t\t\t<const>unknown</const>\n\
\t\t</test>\n\
\t\t<edit name="rgba" mode="assign"><const>\
' + v_monitor['pixel'] + '\
</const></edit>\n\
\t</match>')
    
    # If Bitstream fonts are installed, then they are used by default.
    if 'corefonts' in fonts_outline:
        fileTmp.append('''
<!-- Use Bitstream Vera fonts by default -->
\t<alias>
\t\t<family>serif</family>
\t\t<prefer>
\t\t\t<family>Bitstream Vera Serif</family>
\t\t</prefer>
\t</alias>
\t<alias>
\t\t<family>sans-serif</family>
\t\t<prefer>
\t\t\t<family>Bitstream Vera Sans</family>
\t\t</prefer>
\t</alias>
\t<alias>
\t\t<family>monospace</family>
\t\t<prefer>
\t\t\t<family>Bitstream Vera Sans Mono</family>
\t\t</prefer>
\t</alias>''')
    
    fileTmp.append('''
<!-- Use the Autohinter -->
\t<match target="font">
\t\t<edit name="autohint" mode="assign"><bool>true</bool></edit>
\t</match>

<!-- Disable Autohinting for bold fonts -->
\t<match target="font">
\t\t<test name="weight" compare="more">
\t\t\t<const>medium</const>
\t\t</test>
\t\t<edit name="autohint" mode="assign"><bool>false</bool></edit>
\t</match>

<!-- Exclude/Include a range of fonts for Anti Aliasing -->
<!--
\t<match target="font">
\t\t<test qual="any" name="size" compare="more">
\t\t\t<double>9</double>
\t\t</test>
\t\t<test qual="any" name="size" compare="less">
\t\t\t<double>14</double>
\t\t</test>
\t\t<edit name="antialias" mode="assign">
\t\t\t<bool>true</bool>
\t\t</edit>
\t</match>
-->

<!-- And/Or disable Anti Aliasing for a range on pixel-based size.
     Disabling this using both methods seems to fix Firefox. -->
<!--
\t<match target="font">
\t\t<test compare="less" name="pixelsize" qual="any">
\t\t\t<double>20</double>
\t\t</test>
\t\t<edit mode="assign" name="antialias">
\t\t\t<bool>false</bool>
\t\t</edit>
\t</match>
-->

</fontconfig>
''')
    
    write_file(fileTmp, fileLocalFontconfig)


def write_file(fileSource, fileDest):
    '''It writes a file but if it exits then it creates a backup.
    '''
    global show_backup
    
    if os.path.isfile(fileDest):
        open(fileDest + '~', 'w').write(open(fileDest, 'r').read())
        show_backup.append(fileDest + '~')
    
    fileSource = '\n'.join(fileSource)
    open(fileDest, 'w').write(fileSource)


def main():
    global file_tmp, email, g_card, v_monitor, show_backup
    
    configFile = "/etc/X11/xorg.conf"
    # Data of each graphic card and monitor.
    g_card = {}; v_monitor = {}
    file_tmp = []
    show_backup = []
    
    email = "jonasmg --AT-- SoftHome --DOT-- net"
    email = re.sub(' --AT-- ', '@', email)
    email = re.sub(' --DOT-- ', '.', email)
    
    check()
    file_tmp.append("\
# Configuration created by '"+ os.path.basename(sys.argv[0]) +"'\n\
# https://savannah.nongnu.org/projects/autocfg-xorg/")
    
    section_files()
    section_module()
    section_input()
    section_device()
    section_monitor()
    section_layout()
    if has_fontconfig: fontconfig()
    
    write_file(file_tmp, configFile)
    
    if show_backup:
        if len(show_backup) > 1:
            print "\n Backups"
        else:
            print "\n Backup:"
        
        show_backup = '\n\t'.join(show_backup)
        print "\t", show_backup


if __name__ == '__main__':
    main()
