-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "dynamics"
require "physics"
require "moremath"
require "morejoints"
require "toon"
require "shading"
require "imaging"
require "switches"

local last
local cache = {}

local function callhooks (hooks, ...)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
         if ... and type(...) == "table" then
            hook (unpack(...))
         else
            hook (...)
         end
      end
   elseif type (hooks) == "function" then
      if ... and type(...) == "table" then
         hooks(unpack(...))
      else
         hooks (...)
      end
   end
end

-- Create a modulated version of the stripe texture.

local modulated = function (name)
	       return function (values)
			 local image

			 -- Search the cache first.

			 for cached, texture in pairs(cache) do
			    if cached.name == name and
			       cached.hue == values.hue and
			       cached.saturation == values.saturation and
			       cached.value == values.value then

			       return texture
			    end
			 end

			 -- If nothing was found modulate the base image.

			 image = resources.dofile(name)
			 image.pixels = imaging.modulate (image.pixels,
							  image.size[1],
							  image.size[2],
							  values.value,
							  values.saturation,
							  values.hue)
			 
			 image = textures.mirrored (image)

			 -- And cache it before returning it.

			 cache[{
			       name = name,
			       hue = values.hue,
			       saturation = values.saturation,
			       value = values.value
			    }] = image
			 
			 return image
		      end
	    end

dynamics.collision.gate = function (a, b)
   local pylon, gate, time

   -- Check if we crossed a pylon.

   if a.ispylon or b.ispylon then
      pylon = a.ispylon and a or b
      time = common.time

      -- Oooops, just wenth through a pylon.  If it
      -- its not already damaged tear it to pieces.

      if not pylon.istorn then
	 pylon.istorn = true

	 pylon.cloth.size = {5, 5, 30}
	 pylon.cloth.selected.mesh = resources.fabric "airrace/meshes/tornpylon.lc" {
	    granularity = 3,

	    density = 2,
	    mobility = 3e7,
	    drag = 10,
	 
	    stretch = {10000, 50},
	    bend = {50, 2},
	    shear = {50, 2},
	 }

	 callhooks (airrace.crossedpylon, pylon.parent)
      end

      return 0
   end

   -- Check if we crossed a gate.

   if a.isgate or b.isgate then
      gate = a.isgate and a or b
      time = common.time

      -- Since we're in here we haven't yet
      -- crossed the gate completely.

      if last ~= gate.parent then
	 callhooks (airrace.crossedgate, gate.ancestry[2])

	 last = gate.parent
      end

      return 0
   end
end

function airrace.volume (values)
   local self, meta, oldindex, oldnewindex

   self = bodies.box {
      isgate = true,
      classification = 2,
   }
 
   meta = getmetatable (self)

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     return meta[key] or oldindex (self, key)
		  end

   meta.__newindex = function (self, key, value)
      if key == "size" then
	 local delta

	 meta[key] = value

	 oldnewindex(self, key, {value[1], value[2], 15})
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			return "Volume"
		     end

   -- Set default values.

   self.position = {0, 0, 0}
   self.size = {5, 10, 15}

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end


function airrace.pylon (values)
   local self, meta, oldindex, oldnewindex

   self = springs.euler {
      axes = {{1, 0, 0}, {0, 1, 0}, {0, 0, 1}},
      
      stiffness = 1000,
      damping = 750,

      fan = bodies.capsule {
	 radius = 2.5,
	 length = 11,

	 classification = 1,
	 ispylon = true,

	 mass = physics.capsulemass (20, 2.5, 11),

	 -- link = function(self)
	 -- 	      print (self.mass[1])
	 -- 	   end,

	 cloth = switches.bound {
	    size = {5, 5, 15},

	    selected = (options.toon and toon.cel or shading.cloth) {
	       -- link = function (self) print (tostring(self) .. " in.") end,
	       -- unlink = function (self) print (tostring(self) .. " out.") end,

	       thickness = options.toon and 2,
	       color = options.toon and
		       resources.mirrored "airrace/imagery/stripes.lc",
	
	       parameter = not options.toon and {3.2, 0.6},
	       diffuse = not options.toon and
		             resources.mirrored "airrace/imagery/stripes.lc",
	    
	       mesh = resources.fabric "airrace/meshes/pylon.lc" {
		  granularity = 3,

		  density = 2,
		  mobility = 3e7,
		  drag = 10,

		  stretch = {10000, 50},
		  bend = {50, 2},
		  shear = {50, 2},
	       },
	    },
	 },

	 fan = options.toon and toon.cel {
	    color = math.scale ({1, 1, 1}, 0.3),
	    thickness = 2,
	    
	    mesh = resources.static "airrace/meshes/fan.lc" {
	       position = {0, 0, -1.2},
	    },
	 } or shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.3),
	    specular = resources.mirrored "airrace/imagery/scratched.lc",
	    parameter = {4, 0.3},
	    
	    mesh = resources.static "airrace/meshes/fan.lc" {
	       position = {0, 0, -1.2},
	    },
	 },
      },
   }
 
   meta = getmetatable (self)
   meta.position = {0, 0, 0}
   meta.color = {1, 1, 1}

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     return meta[key] or oldindex (self, key)
		  end

   meta.__newindex = function (self, key, value)
      if key == "position" then
	 meta[key] = value

	 self.fan.position = value
	 self.anchor = math.add (value, {0, 0, 1})
      elseif key == "color" then
	 local new

	 meta[key] = value

	 new = modulated "airrace/imagery/stripes.lc" {
	    value = value[1],
	    saturation = value[2],
	    hue = value[3],
	 }

	 self.fan.cloth.selected.diffuse = not options.toon and new
	 self.fan.cloth.selected.color = options.toon and new
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			return "Pylon"
		     end

   -- Set default values.

   self.position = {0, 0, 0}
   self.color = {1, 1, 1}

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function airrace.gate (values)
   local meta, self, oldindex, oldnewindex

   self = frames.transform {
      volumes = bodies.environment {
	 center = airrace.volume {},
      },

      pylons = bodies.system {
      	 left = airrace.pylon {},
      	 right = airrace.pylon {},
      },
   }
 
   meta = getmetatable (self)

   meta.width = 5
   meta.position = {0, 0, 0}
   meta.rotation = 0
   meta.color = {1, 1, 1}

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     return meta[key] or oldindex (self, key)
		  end

   meta.__newindex = function (self, key, value)
      if key == "position" or key == "rotation" or
         key == "width" or key == "color" then
	 local delta
	 
	 meta[key] = value

	 -- Configure the base transform as well
	 -- for any nodes we might hang below it.

	 if key == "position" then
	    oldnewindex(self, key, value)
	 elseif key == "rotation" then
	    oldnewindex(self, "orientation",
			transforms.euler (0, 0, meta.rotation))
	 end

	 delta = math.transform (transforms.euler (0, 0, meta.rotation),
				 {0, 0.5 * meta.width, 0})

	 -- Position, scale and orient the volumes.

	 self.volumes.center.position = meta.position
	 self.volumes.center.orientation = transforms.euler (0, 0, meta.rotation)
	 self.volumes.center.size = {5, meta.width - 5}

	 -- Position and colorize the pylons.

	 self.pylons.left.position = math.add(meta.position, delta)
	 self.pylons.right.position = math.subtract(meta.position, delta)
	 self.pylons.left.color = meta.color
	 self.pylons.right.color = meta.color
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			return "Airgate"
		     end

   -- Set default values.

   self.position = {0, 0, 0}
   self.color = {1, 1, 1}
   self.rotation = 0
   self.width = 20

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function airrace.halfgate (values)
   local meta, self, oldindex, oldnewindex

   self = frames.transform {
      volumes = bodies.environment {
	 center = airrace.volume{},
      },

      pylons = bodies.system {
      	 left = airrace.pylon {},
      },
   }
 
   meta = getmetatable (self)

   meta.width = 5
   meta.position = {0, 0, 0}
   meta.rotation = 0
   meta.color = {1, 1, 1}

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     return meta[key] or oldindex (self, key)
		  end

   meta.__newindex = function (self, key, value)
      if key == "position" or key == "rotation" or
         key == "width" or key == "color" then
	 local delta
	 
	 meta[key] = value

	 -- Configure the base transform as well
	 -- for any nodes we might hang below it.

	 if key == "position" then
	    oldnewindex(self, key, value)
	 elseif key == "rotation" then
	    oldnewindex(self, "orientation",
			transforms.euler (0, 0, meta.rotation))
	 end

	 delta = math.transform (transforms.euler (0, 0, meta.rotation),
				 {0, 0.5 * meta.width, 0})

	 -- Position, scale and orient the volumes.

	 self.volumes.center.position = meta.position
	 self.volumes.center.orientation = transforms.euler (0, 0, meta.rotation)
	 self.volumes.center.size = {5, meta.width - 5}

	 -- Position and colorize the pylons.

	 self.pylons.left.position = math.add(meta.position, delta)
	 self.pylons.left.color = meta.color
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			return "Halfgate"
		     end

   -- Set default values.

   self.position = {0, 0, 0}
   self.color = {1, 1, 1}
   self.rotation = 0
   self.width = 20

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function airrace.quadgate (values)
   local meta, self, oldindex, oldnewindex

   self = frames.transform {
      alpha = bodies.environment {
	 front = airrace.volume {},
	 back = airrace.volume {},
      },

      beta = bodies.environment {
	 front = airrace.volume {},
	 back = airrace.volume {},
      },

      pylons = bodies.system {
      	 frontleft = airrace.pylon {},
      	 frontright = airrace.pylon {},
      	 backleft = airrace.pylon {},
      	 backright = airrace.pylon {},
      },
   }
 
   meta = getmetatable (self)

   meta.position = {0, 0, 0}
   meta.size = {10, 10}
   meta.rotation = 0
   meta.color = {1, 1, 1}

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     return meta[key] or oldindex (self, key)
		  end

   meta.__newindex = function (self, key, value)
      if key == "position" or key == "rotation" or
         key == "size" or key == "color" then
	 local delta_x, delta_y
	 
	 meta[key] = value

	 -- Configure the base transform as well
	 -- for any nodes we might hang below it.

	 if key == "position" then
	    oldnewindex(self, key, value)
	 elseif key == "rotation" then
	    oldnewindex(self, "orientation",
			transforms.euler (0, 0, meta.rotation))
	 end

	 -- Transform the volumes into place.

	 delta_x = math.transform (transforms.euler (0, 0, meta.rotation),
				   {0.5 * meta.size[1] + 5, 0, 0})
	 delta_y = math.transform (transforms.euler (0, 0, meta.rotation),
				   {0, 0.5 * meta.size[2] + 5, 0})
	 
	 self.alpha.front.position = math.add(meta.position, 
					      math.scale (delta_x, -1))

	 self.alpha.back.position = math.add(meta.position, 
					     math.scale (delta_x, 1))

	 self.beta.front.position = math.add(meta.position, 
					     math.scale (delta_y, -1))

	 self.beta.back.position = math.add(meta.position,
					    math.scale (delta_y, 1))

	 -- And the pylons too.

	 delta_x = math.transform (transforms.euler (0, 0, meta.rotation),
				   {0.5 * meta.size[1], 0, 0})
	 delta_y = math.transform (transforms.euler (0, 0, meta.rotation),
				   {0, 0.5 * meta.size[2], 0})

	 self.pylons.frontleft.position = math.add(meta.position,
						   math.scale (delta_x, -1),
						   math.scale (delta_y, 1))

	 self.pylons.frontright.position = math.add(meta.position,
						    math.scale (delta_x, 1),
						    math.scale (delta_y, 1))

	 self.pylons.backleft.position = math.add(meta.position,
						  math.scale (delta_x, -1),
						  math.scale (delta_y, -1))

	 self.pylons.backright.position = math.add(meta.position, 
						   math.scale (delta_x, 1),
						   math.scale (delta_y, -1))
	 
	 -- Orient and scale the volumes.

	 self.alpha.front.orientation = transforms.euler (0, 0, meta.rotation)
	 self.alpha.back.orientation = transforms.euler (0, 0, meta.rotation)
	 self.beta.front.orientation = transforms.euler (0, 0, meta.rotation)
	 self.beta.back.orientation = transforms.euler (0, 0, meta.rotation)

	 self.alpha.front.size = {5, meta.size[2]}
	 self.alpha.back.size = {5, meta.size[2]}
	 self.beta.front.size = {meta.size[1], 5}
	 self.beta.back.size = {meta.size[1], 5}

	 -- And recolor them.

	 self.pylons.frontleft.color = meta.color
	 self.pylons.frontright.color = meta.color
	 self.pylons.backleft.color = meta.color
	 self.pylons.backright.color = meta.color
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			return "Quadgate"
		     end

   -- Set default values.

   self.position = {0, 0, 0}
   self.rotation = 0
   self.size = {20, 20}
   self.color = {1, 1, 1}

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end
