-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "instrumentation"
require "network"
require "pages"
require "resources"
require "derived"
require "meteorology"
require "widgets"
require "shapes"
require "frames"
require "physics"
require "units"
require "transforms"
require "recorder"

-- Load common configuration.

resources.dofile "common/atmosphere.lua"
resources.dofile "common/kolmogorov.lua"

local function callhooks (hooks, ...)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
         if ... and type(...) == "table" then
            hook (unpack(...))
         else
            hook (...)
         end
      end
   elseif type (hooks) == "function" then
      if ... and type(...) == "table" then
         hooks(unpack(...))
      else
         hooks (...)
      end
   end
end

-- These values are fixed.

network.http = 29176

graphics.title = "Aviation"
graphics.window = options.fullscreen and configuration.screen or {800, 600}
graphics.canvas = {0.07, 0.07, 0.075}
graphics.close = function ()
		    common.iterate = false
		 end

derived.gee = units.meters(-9.81)
derived.frustum = {52, units.meters (0.1), units.meters (100000)}

turbulence.scale = {0.4, 0.35}

dynamics.tolerance = {1e-6, 0.2}
dynamics.popvelocity = 10
dynamics.surfacelayer = 1e-2
dynamics.stepsize = 0.01
dynamics.ceiling = 0.5
dynamics.collision = {}

dynamics.collision.ground = function (a, b)
   if a.isground or b.isground then
      local other = a.isground and b or a

      if other.isairplane then
	 if not other.crashed then
	    other.crashed = true
	    other.engine.throttle = 0
	    other.avionics = nil
	    other.interface = nil
	    other.dynamics.lift = nil
	    other.dynamics.sideforce = nil
	    other.dynamics.roll = nil
	    other.dynamics.pitch = nil
	    other.dynamics.yaw = nil
	    other.dynamics.drag = nil

	    -- Avoid calling the callback from
	    -- within the collision hook.

	    other.finish = function (self)
			       callhooks (aviation.crashed)
			       self.finish = nil
			    end
	 end

	 return 3, 1, 0--, 3e-5, 0.003
      else
	 return 1, 0.6, 0--, 3e-5, 0.003
      end	 
   end
end

-- Create and initialize Aviation's configurations table.

aviation = {}
aviation.crashed = {}

aviation.initial = {units.meters {0, 0, 600}, units.knots (120), 0}

aviation.dataset = "columbia"
aviation.longitudes = {0, 10}
aviation.latitudes = {0, 8}

aviation.location = {120.5, 45.4}
aviation.time = {142, 12.5}
aviation.turbidity = 3

aviation.sensitivity = {0.001, 0.002, 0.001}

-- Load per-user configuration.

local rcscript = loadfile (os.getenv ("HOME") .. "/.aviation")

if rcscript then
   print ("  " .. os.getenv ("HOME") .. "/.aviation")
   rcscript ()
end

-- Set up the scene if requested.

if not (common.engineering or options.fullshading) and
   (not string.find (configuration.graphics[1], "NVIDIA") or
    not string.find (configuration.graphics[4], "fragment_shader")) then
   print ("I'm going to take a wild guess that your card doesn't " .. 
	  "support full\nshading so I'm falling back to toon mode. " ..
	  "Override with -Ofullshading.\n")
   options.toon = true
   options.poweroftwo = true
end

if type(options.run) ~= "table" then
   options.run = {options.run}
end

for _, line in pairs(options.run) do
   assert(loadstring(line))()
end

if not options.stall then
   resources.dofile "aviation/restart.lua"

   graphics.hide = false
end
