-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "aviation"
require "director"
require "shading"
require "toon"
require "shapes"
require "frames"
require "transforms"
require "moremath"
require "moreframes"
require "moreparticles"
require "units"
require "switches"

resources.dofile "aviation/avionics.lua"
resources.dofile "aviation/display.lua"

local bitonal = resources.pipeline {resources.quantize(2), textures.mirrored}
local octonal = resources.pipeline {resources.quantize(8), textures.clamped}

local x_0 = aviation.initial[1]
local R_0 = transforms.euler(0, 0, aviation.initial[3])
local v_0 = math.transform (R_0, {aviation.initial[2], 0, 0})

aviation.nodes.plane = resources.polyhedron "aviation/polyhedra/fuselage.lc" {
   position = x_0,
   orientation = R_0,
   velocity = v_0,

   mass = {
      units.pounds (1180) - 67.96,
      {0, 0, 0},
      {units.slugsquarefeet (362) - 311.22, 0, 0,
       0, units.slugsquarefeet (460) - 8.42, 0,
       0, 0, units.slugsquarefeet (746) - 319.64},
   },
   
   isairplane = true,

   origin = frames.origin {},

   belt = springs.euler {
      axes = {math.transform (R_0, {1, 0, 0}),
   	      math.transform (R_0, {0, 1, 0}),
   	      {0, 0, 1},},

      tolerance = math.scale ({1, 1, 1}, 5e-5),
      stiffness = math.scale({1, 1, 1}, 7000),
      damping = math.scale({1, 1, 1}, 600),
      
      anchor = math.add (x_0, 
   			 math.transform (R_0, units.meters {-0.2, 0, 0})),

      torso = bodies.point {
   	 position = math.add (x_0,
   			      math.transform (R_0,
   					      units.meters {-0.2, 0, 0})),
   	 orientation = options.simple and
   	               math.concatenate (R_0, transforms.euler (0, -9, 0)) or
   		       R_0,
   	 velocity = v_0,

   	 mass = physics.spheremass (15, 1),

	 neck = springs.euler {
	    axes = {math.transform (R_0, {1, 0, 0}),
		    math.transform (R_0, {0, 1, 0}),
		    {0, 0, 1},},

	    tolerance = math.scale ({1, 1, 1}, 5e-5),
	    stiffness = math.scale({1, 1, 1}, 700),
	    damping = math.scale({1, 1, 1}, 60),
      
	    anchor = math.add (x_0, 
			       math.transform (R_0, units.meters {-0.2, 0, 0.5})),

	    head = bodies.point {
	       position = math.add (x_0,
				    math.transform (R_0,
						    units.meters {-0.2, 0, 0.5})),
	       orientation = options.simple and
		             math.concatenate (R_0, transforms.euler (0, -9, 0)) or
			     R_0,

	       velocity = v_0,
	       mass = physics.spheremass (1, 1),
	    
	       eyes = frames.observer {
		  position = {0, 0, 0},
		  orientation = { 0, 0, -1,
				  0, 1,  0,
				  -1, 0,  0},
	       },
	    },
	 },

   	 -- link = function(self) 
   	 -- 	   print (self.mass[1]);
   	 -- 	   abort()
   	 -- 	end,
      },
   },
   
   -- The wing rigs.
   
   rightwing = joints.spherical {
      anchor = x_0,
      axes = {math.transform (R_0, {1, 0, 0}),
   	      math.transform (R_0, {0, 1, 0}),
   	      {0, 0, 1}},

      motor = {{0, 35e3}, {0, 100e3}, {0, 100e3}},
      
      tip = bodies.box {
   	 position = math.add (x_0,
   			      math.transform (R_0, units.feet {0, -6.4, 0})),
   	 orientation = R_0,
   	 velocity = v_0,

   	 mass = physics.boxmass (3000,
   				 units.feet(4),
   				 units.feet(10),
   				 units.feet(0.01)),
 
   	 -- link = function(self) 
   	 -- 	   local m = physics.translatemass (self.mass, units.feet {0, -6.4, 0})
   	 -- 	   print (self.ancestry[2].mass[1], unpack(self.ancestry[2].mass[3]))
   	 -- 	   print (m[1], unpack(m[3]));
   	 -- 	   abort()
   	 -- 	end,

   	 size = units.feet {4, 10, 0.1},
      },
   },
   
   leftwing = joints.spherical {
      anchor = x_0,
      axes = {math.transform (R_0, {1, 0, 0}),
   	      math.transform (R_0, {0, 1, 0}),
   	      {0, 0, 1}},
      
      motor = {{0, 35e3}, {0, 100e3}, {0, 100e3}},
      
      tip = bodies.box {
    	 position = math.add (x_0,
   			      math.transform (R_0, units.feet {0, 6.4, 0})),
   	 orientation = R_0,
   	 velocity = v_0,

   	 mass = physics.boxmass (3000,
   				 units.feet(4),
   				 units.feet(10),
   				 units.feet(0.01)),
   	 size = units.feet {4, 10, 0.1},
      },
   },

   avionics = frames.node {
      ailerons = 0,
      elevators = 0,
      rudder = 0,
      throttle = 1,

      blackbox = {},
      script = nil,

      link = function (self)
	 graphics.cursor = true

	 if options.replay then
	    self.script = assert(loadfile(options.replay))()
	    print ("Read " .. #self.script .. " entries into " ..
		   "the black box.")
	 end
      end,

      unlink = function (self)
	 if options.record then
	    local log

	    log = io.open (options.record, "w")
	    log:write ("-- Captain's log, stardate " ..
		       os.date ("%a %b %d, %X %Y") .. "\n\n" ..
		       "return {\n")

	     for _, entry in ipairs (self.blackbox) do
		log:write (string.format("        {%g, %g, %g, %g},\n",
					 unpack(entry)))
	     end

	     log:write ("}\n")
	     log:close ()

	     print ("Dumped " .. #self.blackbox .. " entries from " .. 
		    "the black box.")
	  end
 
	 graphics.cursor = true
      end,

      step = function (self)
         local sums = {
	    ailerons = 0,
	    elevators = 0,
	    rudder = 0,
	    throttle = 0
	 }

	 if self.script and #self.script > 0 then
	    -- If replaying from the black box ignore
	    -- the avionics and set the sums from memory.

	    sums.ailerons = self.script[1][1]
	    sums.elevators = self.script[1][2]
	    sums.rudder = self.script[1][3]
	    sums.throttle = self.script[1][4]
	    
	    table.remove (self.script, 1)
	 else
	    -- Sum up the results of each system.

	    for command, _ in pairs(sums) do
	       for _, child in children (self) do
		  sums[command] = sums[command] + child[command]
	       end
	    end
	 end

	 table.insert(self.blackbox, {sums.ailerons,
				      sums.elevators,
				      sums.rudder,
				      sums.throttle})
	 
	 -- Apply the sums to the dynamics and engine models.

	 self.parent.dynamics.ailerons = sums.ailerons
	 self.parent.dynamics.elevators = sums.elevators
	 self.parent.dynamics.rudder = sums.rudder
	 self.parent.engine.throttle = sums.throttle
      end,
   },

   interface = switches.button {
      selected = frames.event {
   	 buttonpress = function (self, button, x, y)
            if not self.pressed then
   	       self.pressed = button

   	       if self.button == nil then
   		  self.button = button
   		  self.origin = nil
   	       elseif self.button ~= button then
   		  self.subbutton = button
   		  self.origin = nil
   	       elseif self.button == button then
   		  self.button = nil
   		  self.origin = nil
   	       end
   	    end

   	    self.motion (self, button, x, y)
   	 end,
	    
   	 buttonrelease = function (self, button, x, y)
   	    if self.pressed == button then
   	       self.pressed = nil

   	       if self.subbutton == button then
   		  self.subbutton = nil
   		  self.origin = nil
   	       end
   	    end

   	    self.motion (self, button, x, y)
   	 end,

         motion = function (self, button, x, y)
            if self.button == 1 then
   	       if self.subbutton == 3 then
   		  if not self.origin then
   		     self.origin = {x + self.ancestry[2].avionics.rudder /
   				    aviation.sensitivity[3],
   				    y + self.ancestry[2].avionics.elevators /
   				    aviation.sensitivity[2]}
   		  end
		  
   		  self.ancestry[2].avionics.rudder = -(x - self.origin[1]) *
   		                                aviation.sensitivity[3]
   		  self.ancestry[2].avionics.elevators = (self.origin[2] - y) *
   		                                   aviation.sensitivity[2]
   	       else
   		  if not self.origin then
   		     self.origin = {x - self.ancestry[2].avionics.ailerons /
   				    aviation.sensitivity[1],
   				    y + self.ancestry[2].avionics.elevators /
   				    aviation.sensitivity[2]}
   		  end

   		  self.ancestry[2].avionics.ailerons = (x - self.origin[1]) *
   		                                  aviation.sensitivity[1]
   		  self.ancestry[2].avionics.elevators = (self.origin[2] - y) *
   		                                   aviation.sensitivity[2]
   		  self.ancestry[2].avionics.rudder = 0
   	       end
   	    else
   	       self.ancestry[2].avionics.ailerons = 0
   	       self.ancestry[2].avionics.elevators = 0
   	       self.ancestry[2].avionics.rudder = 0
   	    end

   	    if self.button == 3 then
   	       local theta, phi

   	       if not self.origin then
   	       	  self.origin = {x, y}
   	       end

   	       theta = math.clamp((x - self.origin[1]) * 0.01,
   	       			  units.degrees (-70),
   	       			  units.degrees (70))

   	       phi = math.clamp((y - self.origin[2]) * 0.01,
   	       			units.degrees (0),
   	       			units.degrees (40 * math.abs (math.cos(theta))))

   	       self.ancestry[2].belt.torso.neck.preload = {0, phi, theta}
   	    else
   	       self.ancestry[2].belt.torso.neck.preload = {0, 0, 0}	    
   	    end
   	 end,
	       
   	 scroll = function (self, direction)
      	    if direction == "up" then
   	       self.ancestry[2].avionics.throttle =
   		  math.clamp(self.ancestry[2].avionics.throttle + 0.1, 0, 1)
   	    elseif direction == "down" then
   	       self.ancestry[2].avionics.throttle =
   		  math.clamp(self.ancestry[2].avionics.throttle - 0.1, 0, 1)
   	    end
   	 end,

   	 keypress = function (self, key)
   	 end,
      }
   },

   -- Mechanical specs.
   
   dynamics = aircraft.airplane {
      area = units.squarefeet (97.96),
      span = units.feet (24.35),
      chord = units.feet (4.02),
      
      -- traverse = function (self)
      -- 	       print (self.elevators, self.ailerons, self.rudder)
      -- 	    end,
   
      drag = {
   	 attack = {
   	     -math.pi, 1.5,
             -1.57,    1.5,
             -0.26,    0.026,
              0,       0.02,
              0.26,    0.026,
              1.57,    1.5,
   	      math.pi, 1.5,
   	 },
   
   	 sideslip = {
   	      -math.pi, 1.23,
              -1.57,    1.23,
              -0.26,    0.05,
               0,       0,
               0.26,    0.05,
               1.57,    1.23,
   	       math.pi, 1.23,
   	 },
   	 
   	 elevators = {
   	      -math.pi, -math.pi * 0.04,
   	       math.pi,  math.pi * 0.04
   	 } 
      },
      
      sideforce = {
   	 sideslip = {
   	      -math.pi, -math.pi * -1,
   	       math.pi,  math.pi * -1
   	 },
      },
      
      lift = {
   	 attack = {
   	    -math.pi, -0.730,
   	    -0.2,     -0.73,
   	     0,        0.17,
             0.23,     1.2,
             0.6,      0.582,
   	     math.pi,  0.582,
   	 },
   	 
   	 elevators = {
   	    -math.pi, -math.pi * 0.2,
   	     math.pi,  math.pi * 0.2
   	 }
      },
      
      roll = {
   	 sideslip = {
   	    -math.pi, -math.pi * -0.1,
   	     math.pi,  math.pi * -0.1
   	 },
   	 
   	 roll = {
   	    -math.pi, -math.pi * -0.37,
   	     math.pi,  math.pi * -0.37
   	 },
   	 
   	 yaw = {
   	    -math.pi, -math.pi * 0.15,
   	     math.pi,  math.pi * 0.15
   	 },
   	 
   	 ailerons = {
   	    -math.pi, -math.pi * 0.22,
   	    math.pi,  math.pi * 0.22
   	 },
   	 
   	 rudder = {
   	    -math.pi, -math.pi * 0.01,
   	     math.pi,  math.pi * 0.01
   	 },
      },
      
      pitch = {
   	 attack = {
   	    -math.pi, -math.pi * -0.5,
   	     math.pi,  math.pi * -0.5
   	 },
   	 
   	 attackrate = {
   	    -math.pi, -math.pi * -7,
   	     math.pi,  math.pi * -7
   	 },
   	 
   	 pitch = {
   	    -math.pi, -math.pi * -15,
   	     math.pi,  math.pi * -15
   	 },
   	 
   	 elevators = {
   	    -math.pi, -math.pi * -0.9,
   	     math.pi,  math.pi * -0.9
   	 },
      },
      
      yaw = {
   	 sideslip = {
   	    -math.pi, -math.pi * 0.12,
   	     math.pi,  math.pi * 0.12
   	 },	 
   	 
   	 yaw = {
   	    -math.pi, -math.pi * -0.15,
   	     math.pi,  math.pi * -0.15
   	 },
   	 
   	 ailerons = {
   	    -math.pi, -math.pi * -0.003,
   	     math.pi,  math.pi * -0.003
   	 },
   	 
   	 rudder = {
   	    -math.pi, -math.pi * -0.1,
   	     math.pi,  math.pi * -0.1
   	 }
      },
   },

   engine = propulsion.piston {
      position = {1.9, 0, 0},
      orientation = transforms.euler (0, 90, 0),
      
      diameter = units.feet (6.25),
      inertia = units.slugsquarefeet(1.67),
      throttle = 1.0,
      ratio = 1.3,
      idle = units.rotationsperminute (550);
      
      --        traverse = function (self)
      -- 		     print (self.speed)
      -- 		  end,

      brakepower = {
	 units.rotationsperminute(0)   , units.metrichorsepower(0), 
	 units.rotationsperminute(400) , units.metrichorsepower(0), 
	 units.rotationsperminute(1000), units.metrichorsepower(0.438 * 264), 
	 units.rotationsperminute(2000), units.metrichorsepower(0.854 * 264), 
	 units.rotationsperminute(2750), units.metrichorsepower(1 * 264), 
	 units.rotationsperminute(3000), units.metrichorsepower(0.876 * 264), 
	 units.rotationsperminute(3500), units.metrichorsepower(0)
      },
      
      thrust = {0.0,   0.068,
		0.1,   0.068,
		0.2,   0.067,
		0.3,   0.066,
		0.4,   0.064,
		0.5,   0.061,
		0.6,   0.057,
		0.7,   0.050,
		0.8,   0.040,
		0.9,   0.029,
		1.0,   0.019,
		1.1,   0.008,
		1.2,   -0.001,
		1.3,   -0.008,
		1.4,   -0.019,
		1.5,   -0.029,
		1.6,   -0.040,
		1.7,   -0.050,
		1.8,   -0.057,
		1.9,   -0.061,
		2.0,   -0.064,
		2.1,   -0.066,
		2.2,   -0.067,
		2.3,   -0.068,
		5.0,   -0.068,
		10.0,   -0.068},
      
      power = {0.0,   0.0580,
	       0.1,   0.0620,
	       0.2,   0.0600,
	       0.3,   0.0580,
	       0.4,   0.0520,
	       0.5,   0.0450,
	       0.6,   0.0421,
	       0.7,   0.0389,
	       0.8,   0.0346,
	       0.9,   0.0280,
	       1.0,   0.0202,
	       1.1,   0.0111,
	       1.2,   0.0075,
	       1.3,   0.0111,
	       1.4,   0.0202,
	       1.5,   0.0280,
	       1.6,   0.0346,
	       1.7,   0.0389,
	       1.8,   0.0421,
	       1.9,   0.0436,
	       2.0,   0.0445,
	       2.1,   0.0445,
	       2.2,   0.0442,
	       2.3,   0.0431,
	       2.4,   0.0424,
	       5.0,   0.0413,
	       10.0,  0.0413},

      exhaust = resources.loop "aviation/waves/engine.lc" {
      	 reference = 1 / 0,

      	 prepare = function (self)
      		      local uvw, alpha

      		      -- Calculate the angle of attack.

      		      uvw = transforms.tonode(self.ancestry[2], self.ancestry[2].velocity)
      		      alpha = math.deg (math.abs(math.atan2 (uvw[3], uvw[1]))) + 1e-3

      		      self.gain = 0.1 + 0.5 * self.parent.throttle *
      			 math.clamp(math.log10(alpha), 1, 1.5)

      		      self.pitch = 0.25 + math.clamp(0.004 * self.parent.speed, 0.5, 2.3)
      		   end
      },

      propeller = resources.loop "aviation/waves/propeller.lc" {
      	 reference = 1 / 0,

      	 prepare = function (self)
      		      local uvw, alpha

      		      -- Calculate the angle of attack.

      		      uvw = transforms.tonode(self.ancestry[2], self.ancestry[2].velocity)
      		      alpha = math.deg (math.abs(math.atan2 (uvw[3], uvw[1]))) + 1e-3

      		      self.gain = math.clamp(math.log10(alpha), 0.6, 0.9)
      		      self.pitch = 0.5 + math.clamp(0.005 * self.parent.speed, 0.3, 1.5)
      		   end
      },
   },
    
   wind = resources.loop "aviation/waves/wind.lc" {
      reference = 1 / 0,

      prepare = function (self)
		   local uvw

		   -- Calculate the local speed.
		   
		   uvw = transforms.tonode(self.parent,
		   			   self.parent.velocity)
		   
		   self.gain = 0.3 + math.clamp(0.003 * uvw[1], 0.03, 0.25)
		   self.pitch = 1.25 + 0.007 * uvw[1]
		end
   },

   -- The cockpit.
   
   cockpit = frames.transform {
      orientation = transforms.relue (90, 0, -90),
      position = units.meters{-0.11, 0, 0.17},
     
      section = options.toon and toon.cel {
      	 color = math.scale({0.35, 0.45, 0.07}, 1),
      	 thickness = 2,
	 
      	 mesh = resources.static "aviation/meshes/section.lc" {}
      } or shading.cook {
      	 diffuse = resources.mirrored "aviation/imagery/section.lc",
      	 specular = resources.periodic "aviation/imagery/sectiongloss.lc",
      	 parameter = {1.5, 0.1},
	 
      	 mesh = resources.static "aviation/meshes/section.lc" {}
      },
      
      star = options.toon and toon.cel {
      	 color = math.scale({1, 0.3, 0.2}, 0.6),
      	 thickness = 2,
	 
      	 mesh = resources.static "aviation/meshes/star.lc" {}
      } or shading.cook {
      	 diffuse = math.scale({1, 0.3, 0.2}, 0.6),
      	 specular = resources.periodic "aviation/imagery/sectiongloss.lc",
      	 parameter = {1.5, 0.1},
	 
      	 mesh = resources.static "aviation/meshes/star.lc" {}
      },
      
      propeller = frames.top {
	 position = units.meters{0, -0.12, -2.574},
	 axis = {0, 0, 1},
	 
	 -- The trails.
	 
	 [1] = shapes.pie {
	    orientation = transforms.euler (0, 0, 60),
	    color = {0.5, 0.5, 0.5},
	    opacity = 0.4,
	    radius = 0.9,
	    segments = 16,
	 },
	 
	 [2] = shapes.pie {
	    orientation = transforms.euler (0, 0, 180),
	    color = {0.5, 0.5, 0.5},
	    opacity = 0.4,
	    radius = 0.9,
	    segments = 16,
	 },
	 
	 [3] = shapes.pie {
	    orientation = transforms.euler (0, 0, 300),
	    color = {0.5, 0.5, 0.5},
	    opacity = 0.4,
	    radius = 0.9,
	    segments = 16,
	 },
	 
	 -- The blades.
	 
	 blades = options.toon and toon.cel {
	    color = math.scale({1, 1, 1}, 0.7),
	 thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/propeller.lc" {}
	 } or shading.anisotropic {
	    diffuse = math.scale({1, 1, 1}, 0.7),
	    specular = math.scale({1, 1, 1}, 0.7),
	    parameter = {16, 0.1},
	    
	    mesh = resources.static "aviation/meshes/propeller.lc" {},
	 },
	 
	 prepare = function (self)
			local w = self.ancestry[2].engine.speed /
			   (2 * math.pi)
			local w_0 = 24
			local N = 3
			
			self.spin = math.mod(2 * math.pi * w / w_0,
					     2 * math.pi / N) * w_0
			
			if self.spin > math.pi / N * w_0 then
			   self.spin = self.spin - 2 * math.pi / N * w_0
			end
			
			for i = 1, 3 do
			   self[i].arc = -self.spin / w_0
			end
			
-- 		      print (self.parent.engine.throttle,
-- 			     self.parent.engine.speed / (2 * math.pi) * 60,
-- 		             self.parent.engine.speed / (2 * math.pi) * 60 *
-- 			     self.parent.engine.ratio,
-- 		             self.parent.engine.thrust)
		     end
      },
      
      cabin = options.toon and toon.cel {
      	 color = bitonal "aviation/imagery/scratches.lc",
      	 thickness = 2,
	 
      	 mesh = resources.static "aviation/meshes/cabin.lc" {}
      } or shading.cook {
      	 diffuse = math.scale ({1, 1, 1}, 0.01),
      	 specular = resources.mirrored "aviation/imagery/scratches.lc",
      	 parameter = {0.75, 0.2},
	 
      	 mesh = resources.static "aviation/meshes/cabin.lc" {},
      },
      
      border = shapes.lines {
	 position = units.meters {-0.19, 0.103, -0.52},
	 
	 color = {1, 1, 0.7},
	 opacity = 0.25,
	 width = 2,
	 
	 units.meters {0.09, 0.05, 0}, 
	 units.meters {0.1, 0.04, 0}, 
	 units.meters {-0.1, 0.04, 0}, 
	 units.meters {-0.09, 0.05, 0}, 
	 units.meters {-0.09, -0.05, 0}, 
	 units.meters {-0.1, -0.04, 0}, 
	 units.meters {0.1, -0.04, 0}, 
	 units.meters {0.09, -0.05, 0}, 
	 units.meters {-0.09, -0.05, 0}, 
	 units.meters {0.09, -0.05, 0}, 
	 units.meters {0.1, 0.04, 0}, 
	 units.meters {0.1, -0.04, 0}, 
	 units.meters {0.09, 0.05, 0}, 
	 units.meters {-0.09, 0.05, 0}, 
	 units.meters {-0.1, -0.04, 0}, 
	 units.meters {-0.1, 0.04, 0}, 
      },

      smallborder = shapes.lines {
	 position = units.meters {0.175, 0.113, -0.52},
	 
	 color = {1, 1, 0.7},
	 opacity = 0.25,
	 width = 2,

	 units.meters {0.063, 0.034369, 0}, 
	 units.meters {0.07, 0.027369, 0}, 
	 units.meters {-0.07, 0.027369, 0}, 
	 units.meters {-0.063, 0.034369, 0}, 
	 units.meters {-0.063, -0.035631, 0}, 
	 units.meters {-0.07, -0.028631, 0}, 
	 units.meters {0.07, -0.028631, 0}, 
	 units.meters {0.063, -0.035631, 0}, 
	 units.meters {-0.063, -0.035631, 0}, 
	 units.meters {0.063, -0.035631, 0}, 
	 units.meters {0.07, 0.027369, 0}, 
	 units.meters {0.07, -0.028631, 0}, 
	 units.meters {0.063, 0.034369, 0}, 
	 units.meters {-0.063, 0.034369, 0}, 
	 units.meters {-0.07, 0.027369, 0}, 
	 units.meters {-0.07, -0.028631, 0}, 
      },

      switchborder = shapes.lines {
	 position = units.meters {0.175, 0.064, -0.52},
	 
	 color = {1, 1, 0.7},
	 opacity = 0.25,
	 width = 2,

	 {0.063, 0.009784, 0}, 
	 {0.07, 0.002784, 0}, 
	 {-0.07, 0.002784, 0}, 
	 {-0.063, 0.009784, 0}, 
	 {-0.063, -0.011046, 0}, 
	 {-0.07, -0.004046, 0}, 
	 {0.07, -0.004046, 0}, 
	 {0.063, -0.011046, 0}, 
	 {-0.063, -0.011046, 0}, 
	 {0.063, -0.011046, 0}, 
	 {0.07, 0.002784, 0}, 
	 {0.07, -0.004046, 0}, 
	 {0.063, 0.009784, 0}, 
	 {-0.063, 0.009784, 0}, 
	 {-0.07, 0.002784, 0}, 
	 {-0.07, -0.004046, 0}, 
      },
      
      buttonborder = shapes.lines {
	 position = units.meters {0.29, 0.085, -0.52},
	 
	 color = {1, 1, 0.7},
	 opacity = 0.25,
	 width = 2,

	 units.meters {0.027104, 0.0308, 0}, 
	 units.meters {0.0308, 0.027104, 0}, 
	 units.meters {-0.0308, 0.027104, 0}, 
	 units.meters {-0.027104, 0.0308, 0}, 
	 units.meters {-0.027104, -0.0308, 0}, 
	 units.meters {-0.0308, -0.027104, 0}, 
	 units.meters {0.0308, -0.027104, 0}, 
	 units.meters {0.027104, -0.0308, 0}, 
	 units.meters {-0.027104, -0.0308, 0}, 
	 units.meters {0.027104, -0.0308, 0}, 
	 units.meters {0.0308, 0.027104, 0}, 
	 units.meters {0.0308, -0.027104, 0}, 
	 units.meters {0.027104, 0.0308, 0}, 
	 units.meters {-0.027104, 0.0308, 0}, 
	 units.meters {-0.0308, 0.027104, 0}, 
	 units.meters {-0.0308, -0.027104, 0}, 
     },
      
      display = frames.transform {
	 position = units.meters {0.008, 0.105, -0.513},
	 
	 bezel = options.toon and toon.cel {
	    color = math.scale({1, 1, 1}, 0.35),
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/display.lc" {}
	 } or shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.05),
	    specular = resources.mirrored "aviation/imagery/scratches.lc",
	    parameter = {0.75, 0.2},
	    
	    mesh = resources.static "aviation/meshes/display.lc" {},
	 },
	 
	 screen = options.toon and toon.cel {
	    color = math.scale ({0.2805, 0.6223, 0.0164}, 0.7),
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/screen.lc" {}
	 } or shading.cook {
	    diffuse = math.scale ({0.2805, 0.6223, 0.0164}, 0.7),
	    specular = resources.mirrored "aviation/imagery/specularity.lc",
	    parameter = {1.5, 0.25},
	    
	    mesh = resources.static "aviation/meshes/screen.lc" {},
	 },
      },
      
      altimeter = frames.transform {
	 position = units.meters {-0.24, 0.103, -0.513},
	 
	 bezel = options.toon and toon.cel {
	    color = math.scale({1, 1, 1}, 0.15),
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/bezel.lc" {
	       position = units.meters {0, 0, -0.007}
	    }
	 } or shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.05),
	    specular = resources.mirrored "aviation/imagery/scratches.lc",
	    parameter = {0.75, 0.2},
	    
	    mesh = resources.static "aviation/meshes/bezel.lc" {
	       position = units.meters {0, 0, -0.007},
	    },
	 },
	 
	 face = options.toon and toon.cel {
	    color = octonal "aviation/imagery/altimeter.lc",
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/altimeter.lc" {}
	 } or shading.cook {
	    diffuse = resources.clamped "aviation/imagery/altimeter.lc",
	    specular = resources.mirrored "aviation/imagery/specularity.lc",
	    parameter = {1.5, 0.25},
	    
	    mesh = resources.static "aviation/meshes/altimeter.lc" {},
	 },
	 
	 hundreds = (options.toon and toon.cel or shading.lambert) {
	    diffuse = not options.toon and {1, 1, 1},
	    color = options.toon and {1, 1, 1},
	    thickness = options.toon and 2,

	    mesh = resources.static "aviation/meshes/hundreds.lc" {
	       traverse = function (self)
			     local h = self.ancestry[4].position[3] 
			     self.orientation = transforms.euler (0, 0,
								  -(h % 1e3) /
							       1e2 * 36)
			  end
	    },
	 },
	 
	 thousands = (options.toon and toon.cel or shading.lambert) {
	    diffuse = not options.toon and {1, 1, 1},
	    color = options.toon and {1, 1, 1},
	    thickness = options.toon and 2,

	    mesh = resources.static "aviation/meshes/thousands.lc" {	       
	       traverse = function (self)
			     local h = self.ancestry[4].position[3] 
			     self.orientation = transforms.euler (0, 0,
								  -(h % 1e4) /
							       1e3 * 36)
			  end
	    },
	 },
	 
	 tenthousands = (options.toon and toon.cel or shading.lambert) {
	    diffuse = not options.toon and {1, 1, 1},
	    color = options.toon and {1, 1, 1},
	    thickness = options.toon and 2,

	    mesh = resources.static "aviation/meshes/tenthousands.lc" {
	       traverse = function (self)
			     local h = self.ancestry[4].position[3] 
			     self.orientation = transforms.euler (0, 0,
								  -(h % 1e5) /
							       1e4 * 36)
			  end
	    },
	 }
      },
      
      airspeed = frames.transform {
	 position = units.meters {-0.14, 0.103, -0.513},
	 
	 bezel = options.toon and toon.cel {
	    color = math.scale({1, 1, 1}, 0.15),
	    thickness = options.toon and 2,
	 
	    mesh = resources.static "aviation/meshes/bezel.lc" {
	       position = units.meters {0, 0, -0.007}
	    }
	 } or shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.05),
	    specular = resources.mirrored "aviation/imagery/scratches.lc",
	    parameter = {0.75, 0.2},
	    
	    mesh = resources.static "aviation/meshes/bezel.lc" {
	       position = units.meters {0, 0, -0.007},
	    },
	 },
	 
	 face = options.toon and toon.cel {
	    color = octonal "aviation/imagery/airspeed.lc",
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/airspeed.lc" {}
	 } or shading.cook {
	    diffuse = resources.clamped "aviation/imagery/airspeed.lc",
	    specular = resources.mirrored "aviation/imagery/specularity.lc",
	    parameter = {1.5, 0.25},
	    
	    mesh = resources.static "aviation/meshes/airspeed.lc" {},
	 },
	 
	 knots = (options.toon and toon.cel or shading.lambert) {
	    diffuse = not options.toon and {1, 1, 1},
	    color = options.toon and {1, 1, 1},
	    thickness = options.toon and 2,

	    mesh = resources.static "aviation/meshes/knots.lc" {	       
	       traverse = function (self)
	          local v = transforms.tonode(self.ancestry[4],
					      self.ancestry[4].velocity)[1] /
		            units.milesperhour (1)
			  
		  self.orientation = transforms.euler (0, 0, -(v % 260) /
						             260 * 360)
	       end
	    },
	 },
      },
      
      tachometer = frames.transform {
	 position = units.meters {0.14, 0.113, -0.513},
	 
	 bezel = options.toon and toon.cel {
	    color = math.scale({1, 1, 1}, 0.15),
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/smallbezel.lc" {
	       position = units.meters {0, 0, -0.008}
	    }
	 } or shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.05),
	    specular = resources.mirrored "aviation/imagery/scratches.lc",
	    parameter = {0.75, 0.2},
	    
	    mesh = resources.static "aviation/meshes/smallbezel.lc" {
	       position = units.meters {0, 0, -0.008},
	    },
	 },
	 
	 face = options.toon and toon.cel {
	    color = octonal "aviation/imagery/tachometer.lc",
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/tachometer.lc" {}
	 } or shading.cook {
	    diffuse = resources.clamped "aviation/imagery/tachometer.lc",
	    specular = resources.mirrored "aviation/imagery/specularity.lc",
	    parameter = {1.5, 0.25},
	    
	    mesh = resources.static "aviation/meshes/tachometer.lc" {},
	 },
	 
	 needle = (options.toon and toon.cel or shading.lambert) {
	    diffuse = not options.toon and {1, 1, 1},
	    color = options.toon and {1, 1, 1},
	    thickness = options.toon and 2,

	    mesh = resources.static "aviation/meshes/rpms.lc" {	       
	       traverse = function (self)
			     local nu = self.ancestry[4].engine.speed /
			        (2 * math.pi) * 60
			     
			     self.orientation = transforms.euler (0, 0, 121 - nu / 14)
			  end
	    },
	 },
      },
      
      compass = frames.transform {
	 position = units.meters {0.21, 0.113, -0.513},
	 
	 bezel = options.toon and toon.cel {
	    color = math.scale({1, 1, 1}, 0.15),
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/smallbezel.lc" {
	       position = units.meters {0, 0, -0.008}
	    }
	 } or shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.05),
	    specular = resources.mirrored "aviation/imagery/scratches.lc",
	    parameter = {0.75, 0.2},
	    
	    mesh = resources.static "aviation/meshes/smallbezel.lc" {
	       position = units.meters {0, 0, -0.008},
	    },
	 },
	 
	 face = options.toon and toon.cel {
	    color = octonal "aviation/imagery/compassface.lc",
	    thickness = 2,
	 
	    mesh = resources.static "aviation/meshes/compassface.lc" {}
	 } or shading.cook {
	    diffuse = resources.clamped "aviation/imagery/compassface.lc",
	    specular = resources.mirrored "aviation/imagery/specularity.lc",
	    parameter = {1.5, 0.25},
	    
	    mesh = resources.static "aviation/meshes/compassface.lc" {},
	 },
	 
	 dial = (options.toon and toon.cel or shading.lambert) {
	    color = options.toon and octonal "aviation/imagery/compassdial.lc",
	    thickness = options.toon and 2,

	    diffuse = not options.toon and resources.clamped "aviation/imagery/compassdial.lc",
	    specular = not options.toon and math.scale ({0.12, 0.17, 0.19}, 1), 
	    
	    mesh = resources.static "aviation/meshes/compassdial.lc" {
	       position = units.meters {0, 0, -0.022},
	       
	       traverse = function (self)
			     local v = math.normalize(
				math.project (transforms.fromnode (
						 self.ancestry[4],
						 {1, 0, 0}),
					      {0, 0, 1}))
			     
			     if v[2] > 0 then
				-- print (math.deg(math.acos(v[1])))
				self.orientation = transforms.rotation (
				   math.deg(math.acos(v[1])),
				   {0, 1, 0}
				)
			     else
				-- print (360 - math.deg(math.acos(v[1])))
				self.orientation = transforms.rotation (
				   360 - math.deg(math.acos(v[1])),
				   {0, 1, 0}
				)
			     end
			  end
	    },
	 },
      },
      
      switchgear = frames.node {
	 switches = arrays.linear {
	    position = units.meters {0.116, 0.065, -0.52},
	    
	    size = 3,
	    spacing = 0.02375,

	    mold = resources.dofile "aviation/switch.lua",

	    spawn = function (self, i)
               local labels = {"YDS", "SWI", "APS"}
	       local actions = {
		  function (self)
		     aviation.nodes.yawdamper.parent = self.on and
			aviation.nodes.plane.avionics
		  end,

		  function (self)
		     aviation.nodes.stallindicator.parent = self.on and
			aviation.nodes.plane.cockpit.switchgear.light
		  end,

		  function (self)
		     aviation.nodes.autopilot.parent = 
			self.on and aviation.nodes.plane.avionics
		     aviation.nodes.aileroncontrol.parent =
			not self.on and aviation.nodes.plane.avionics
		     aviation.nodes.elevatorcontrol.parent =
			not self.on and aviation.nodes.plane.avionics
		     aviation.nodes.ruddercontrol.parent =
			not self.on and aviation.nodes.plane.avionics
		  end,
	       }

	       self[i].label = labels[i] .. " disabled"
	       self[i].flick = function (self)
				  actions[i](self)

				  self.label = labels[i] .. (self.on and
							     " enabled" or
							     " disabled")
			       end
	    end
	 },
      
	 light = resources.dofile "aviation/indicator.lua" {
	    position = units.meters {0.23, 0.065, -0.52},
	    label = "SWI"
	 },
      
	 button = resources.dofile "aviation/button.lua" {
	    position = units.meters {0.29, 0.085, -0.52},
	    label = "SIG",
	    momentary = true,
	 
	    bezel = options.toon and toon.cel {
	       color = octonal "aviation/imagery/yellowstripes.lc",
	       thickness = 2,
	 
	       mesh = resources.static "aviation/meshes/stripedbezel.lc" {
		  position = units.meters {0, 0, 0}
	       }
	    } or shading.cook {
	       diffuse = resources.clamped "aviation/imagery/yellowstripes.lc",
	       specular = resources.mirrored "aviation/imagery/scratches.lc",
	       parameter = {0.75, 0.2},
	       
	       mesh = resources.static "aviation/meshes/stripedbezel.lc" {
		  position = units.meters {0, 0, 0},
	       },
	    },
	 },
      },
   },
}

aviation.nodes.plane.parent = aviation.nodes.sky.system
aviation.nodes.screen.parent = aviation.nodes.plane.cockpit.display
aviation.nodes.display.parent = options.simple and aviation.nodes.plane

aviation.nodes.aileroncontrol.parent = aviation.nodes.plane.avionics
aviation.nodes.elevatorcontrol.parent = aviation.nodes.plane.avionics
aviation.nodes.ruddercontrol.parent = aviation.nodes.plane.avionics
aviation.nodes.throttlecontrol.parent = aviation.nodes.plane.avionics
aviation.nodes.rubberband.parent = aviation.nodes.plane.avionics

if not options.noyawdamping then
   aviation.nodes.plane.cockpit.switchgear.switches[1].on = true
end

if not options.nostallwarning then
   aviation.nodes.plane.cockpit.switchgear.switches[2].on = true
end
