/* AVLDB - AVL DataBase library, tavl functions for "relative" address space

   Copyright (C) 2002,2003 Petr Silhavy <silhavy@mef.cz>
   
   Based on Ben Pfaff's libavl tavl.c with dizzy modifications
   to fit for my silly needs. Original copyright follows.
*/

/* This file was formerly part of Ben Pfaff's libavl */

/* Produced by texiweb from libavl.w on 2002/03/14 at 12:47. */

/* libavl - library for manipulation of binary trees.
   Copyright (C) 1998-2002 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

   The author may be contacted at <blp@gnu.org> on the Internet, or
   as Ben Pfaff, 12167 Airport Rd, DeWitt MI 48820, USA through more
   mundane means.
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#define  __ADB 1
#include "adb.h"


#ifndef DATA
#define DATA(p) ((void *)p - offsetof( struct ADB_RECORD, node ) )
#endif

/* Creates and returns a new table
   with comparison function |compare| using parameter |param|
   and memory allocator |allocator|.
   Returns |NULL| if memory allocation failed. */
adb_tavl_table_t *
adb_tavl_create (adb_tavl_table_t * const tree)
{

  tree->tavl_root = NULL; 
  tree->tavl_count = 0 ; 

  return tree;
}

/* Search |tree| for an item matching |item|, and return it if found.
   Otherwise return |NULL|. */
void *
adb_tavl_find (const adb_bh_t *bh, const void *item)
{
  struct adb_tavl_table *tree = bh->tree;
  unsigned int sb = bh->addr ;
  const struct adb_tavl_node *p;

  assert (tree != NULL && item != NULL);

  p = tree->tavl_root;
  if (p == NULL /* NULL */)
    return NULL;

  p = (void *)p + sb ;
  for (;;)
    {
      int cmp, dir;

      cmp = bh->cmp (item, DATA(p), bh, 0);
      if (cmp == 0)
        return DATA(p);

      dir = cmp > 0;
      if (p->tavl_tag[dir] == TAVL_CHILD)
        p = (void *)p->tavl_link[dir] + sb;
      else
        return NULL;
    }
}

/* Search |tree| for an item close to |item|, and return it if found.
   This function will return a null pointer only if TREE is empty. */

void *
adb_tavl_find_close (const adb_bh_t *bh, const void *item)
{
  struct adb_tavl_table *tree = bh->tree;
  unsigned int sb = bh->addr ;
  const struct adb_tavl_node *p;

  assert (tree != NULL && item != NULL);

  p = tree->tavl_root;
  if (p == NULL)
    return NULL;

  p = (void *)p + sb ;
  for (;;)
    {
      int cmp, dir;

      cmp = bh->cmp (item, DATA(p), bh, 0);
      if (cmp == 0)
        return DATA(p);

      dir = cmp > 0;
      if (p->tavl_tag[dir] == TAVL_CHILD)
        p = (void *)p->tavl_link[dir] + sb;
      else
        return DATA(p);
    }
}

/* Inserts |item| into |tree| and returns a pointer to |item|'s address.
   If a duplicate item is found in the tree,
   returns a pointer to the duplicate without inserting |item|.
   Returns |NULL| in case of memory allocation failure. */
void *
adb_tavl_probe (const adb_bh_t *bh, adb_tavl_node_t *n, void *item)
{
  adb_tavl_table_t *tree = bh->tree ;
  unsigned int sb = bh->addr ;

  struct adb_tavl_node *y, *z; /* Top node to update balance factor, and parent. */
  struct adb_tavl_node *p, *q; /* Iterator, and parent. */
/*   struct adb_tavl_node *n; */     /* Newly inserted node. */
  struct adb_tavl_node *w = NULL ;     /* New root of rebalanced subtree. */
  int dir;                /* Direction to descend. */

  unsigned char da[TAVL_MAX_HEIGHT]; /* Cached comparison results. */
  int k = 0;              /* Number of cached results. */

  assert (tree != NULL && item != NULL);

  z = (void *) &tree->tavl_root;
  y = (void *) tree->tavl_root + sb ;
  if (y != (void *)sb)
    {
      for (q = z, p = y; ; q = p, p = (void *)p->tavl_link[dir] + sb)
        {
          int cmp = bh->cmp (item, DATA(p), bh, 0);
          if (cmp == 0)
            return DATA(p) ;

          if (p->tavl_balance != 0)
            z = q, y = p, k = 0;
          da[k++] = dir = cmp > 0;

          if (p->tavl_tag[dir] == TAVL_THREAD)
            break;
        }
    }
  else
    {
      p = z;
      dir = 0;
    }

/*   n = tree->tavl_alloc->libavl_malloc (tree->tavl_alloc, sizeof *n); */
/*   if (n == NULL) */
/*     return NULL; */

/*   _bip(tree); _bip(p); */

  tree->tavl_count++;

/* BUG?  n->tavl_data = (void *)item - sb ;  */
  n->tavl_tag[0] = n->tavl_tag[1] = TAVL_THREAD; 
  n->tavl_link[dir] = p->tavl_link[dir]; 

  if (tree->tavl_root != NULL)
    {
      p->tavl_tag[dir] = TAVL_CHILD; 
      n->tavl_link[!dir] = (void *)p - sb; 
    }
  else
    {
      n->tavl_link[1] = NULL ;
    }
  p->tavl_link[dir] = (void *)n - sb; 
  n->tavl_balance = 0;

  if ( ((void *)tree->tavl_root + sb) == n)
    return DATA(n) ;

  /* Step 3 Update balance factors after AVL insertion */
  for (p = y, k = 0; p != n; p = (void *)p->tavl_link[da[k]] + sb, k++)
    if (da[k] == 0)
      {
	p->tavl_balance--;
      }
    else
      {
	p->tavl_balance++;
      }
  
  /* Step 4: Rebalance after TAVL insertion 304 */
  if (y->tavl_balance == -2)
    {
      struct adb_tavl_node *x = (void *)y->tavl_link[0] + sb;
      if (x->tavl_balance == -1)
        {
          w = x;

          if (x->tavl_tag[1] == TAVL_THREAD)
            {
              x->tavl_tag[1] = TAVL_CHILD; 
              y->tavl_tag[0] = TAVL_THREAD;
              y->tavl_link[0] = (void *)x - sb; 
            }
          else
	    {
	      y->tavl_link[0] = x->tavl_link[1]; 
	    }
          x->tavl_link[1] = (void *)y - sb; 
          x->tavl_balance = y->tavl_balance = 0; 
        }
      else
        { /* Rotate left at |x| then right at |y| in AVL tree:: 156 */
          assert (x->tavl_balance == +1);

          w = (void *)x->tavl_link[1] + sb;

          x->tavl_link[1] = w->tavl_link[0];
          w->tavl_link[0] = (void *)x - sb;
          y->tavl_link[0] = w->tavl_link[1];
          w->tavl_link[1] = (void *)y - sb;

          if (w->tavl_balance == -1)
	      x->tavl_balance = 0, y->tavl_balance = +1;
          else if (w->tavl_balance == 0)
	      x->tavl_balance = y->tavl_balance = 0;
          else /* |w->tavl_balance == +1| */
	      x->tavl_balance = -1, y->tavl_balance = 0;
          w->tavl_balance = 0;
	  /* Rebalance for + balance factor in TAVL insertion in left subtree 307 */
          if (w->tavl_tag[0] == TAVL_THREAD)
            {
              x->tavl_tag[1] = TAVL_THREAD;
              x->tavl_link[1] = (void *)w - sb;
              w->tavl_tag[0] = TAVL_CHILD;
            }
          if (w->tavl_tag[1] == TAVL_THREAD)
            {
              y->tavl_tag[0] = TAVL_THREAD;
              y->tavl_link[0] = (void *)w - sb;
              w->tavl_tag[1] = TAVL_CHILD;
            }
        }
    }
  else if (y->tavl_balance == +2)
    { /* Rebalance TAVL tree after insertion in right subtree 308 */
      struct adb_tavl_node *x = (void *)y->tavl_link[1] + sb;
      if (x->tavl_balance == +1)
        {
          w = x;
          if (x->tavl_tag[0] == TAVL_THREAD)
            {
              x->tavl_tag[0] = TAVL_CHILD;
              y->tavl_tag[1] = TAVL_THREAD;
              y->tavl_link[1] = (void *)x - sb;
            }
          else
	    y->tavl_link[1] = x->tavl_link[0];

          x->tavl_link[0] = (void *)y - sb;
          x->tavl_balance = y->tavl_balance = 0;

        }
      else
        {
          assert (x->tavl_balance == -1);

          w = (void *)x->tavl_link[0] + sb;

          x->tavl_link[0] = w->tavl_link[1];
          w->tavl_link[1] = (void *)x - sb;
          y->tavl_link[1] = w->tavl_link[0];
          w->tavl_link[0] = (void *)y - sb ;

          if (w->tavl_balance == +1)
	    x->tavl_balance = 0, y->tavl_balance = -1;
          else if (w->tavl_balance == 0)
	    x->tavl_balance = y->tavl_balance = 0;
          else /* |w->tavl_balance == -1| */
	    x->tavl_balance = +1, y->tavl_balance = 0;

          w->tavl_balance = 0;
          if (w->tavl_tag[0] == TAVL_THREAD)
            {
              y->tavl_tag[1] = TAVL_THREAD;
              y->tavl_link[1] = (void *)w - sb;
              w->tavl_tag[0] = TAVL_CHILD;
            }
          if (w->tavl_tag[1] == TAVL_THREAD)
            {
              x->tavl_tag[0] = TAVL_THREAD;
              x->tavl_link[0] = (void *)w - sb;
              w->tavl_tag[1] = TAVL_CHILD;
            }
        }
    }
  else
    return DATA(n);

  z->tavl_link[y != ( (void *)z->tavl_link[0] + sb ) ] = (void *)w - sb;

  return DATA(n) ;
}

/* Inserts |item| into |table|.
   Returns |NULL| if |item| was successfully inserted
   or if a memory allocation error occurred.
   Otherwise, returns the duplicate item. */
void *
adb_tavl_insert (const adb_bh_t *bh, adb_tavl_node_t *n, void *item)
{
  void *p = adb_tavl_probe (bh, n, item);
  return p == NULL || p == item ? NULL : p;
}

/* Inserts |item| into |table|, replacing any duplicate item.
   Returns |NULL| if |item| was inserted without replacing a duplicate,
   or if a memory allocation error occurred.
   Otherwise, returns the item that was replaced. */
void *
adb_tavl_replace (adb_bh_t *bh, adb_tavl_node_t *n, void *item)
{
  void *p = adb_tavl_probe (bh, n, item);
  if (p == NULL || p == item)
    return NULL;
  else
    {
      void *r = p;
      p = item;
      return r;
    }
}

/* Returns the parent of |node| within |tree|,
   or a pointer to |tavl_root| if |s| is the root of the tree. */
/* !DONE */
struct adb_tavl_node *
adb_find_parent (unsigned int sb , struct adb_tavl_table *tree, struct adb_tavl_node *node)
{
  if (node != (void *)tree->tavl_root + sb)
    {
      struct adb_tavl_node *x, *y;

      for (x = y = node; 
	   ; 
	   x = (void *)x->tavl_link[0] + sb, y = (void *)y->tavl_link[1] + sb)
        if (y->tavl_tag[1] == TAVL_THREAD)
          {
            struct adb_tavl_node *p = (void *)y->tavl_link[1] + sb;
            if (p == (void *)sb || ((void *)p->tavl_link[0] + sb) != node)
              {
                while (x->tavl_tag[0] == TAVL_CHILD)
                  x = (void *)x->tavl_link[0] + sb;
                p = (void *)x->tavl_link[0] + sb;
              }
            return p;
          }
        else if (x->tavl_tag[0] == TAVL_THREAD)
          {
            struct adb_tavl_node *p = (void *)x->tavl_link[0] + sb;
            if (p == (void *)sb || ( (void *)p->tavl_link[1] + sb) != node)
              {
                while (y->tavl_tag[1] == TAVL_CHILD)
                  y = (void *)y->tavl_link[1] + sb;
                p = (void *)y->tavl_link[1] + sb;
              }
            return p;
          }
    }
  else
    return (struct adb_tavl_node *) &tree->tavl_root;
}

/* Deletes from |tree| and returns an item matching |item|.
   Returns a null pointer if no matching item found. */
#if 0
void *
1_4_adb_tavl_delete (const adb_bh_t *bh, const void *item)
{
  /* Uses my Algorithm DT, which can be found at
     http://www.msu.edu/user/pfaffben/avl.  Algorithm DT is based on
     Knuth's Algorithms 6.2.2D (Tree deletion), 6.2.3A (Balanced tree
     search and insertion), 2.3.1I (Insertion into a threaded binary
     trees), and the notes on pages 465-466 of Vol. 3. */

  /* D1. */
  adb_tavl_table_t *tree = bh->tree ;
  unsigned int sb = bh->addr ;

/*   void *item = (void *)_item + sb ; */
  adb_tavl_node_t *pa[AVL_MAX_HEIGHT];	/* Stack P: Nodes. */
  unsigned char a[AVL_MAX_HEIGHT];	/* Stack P: Bits. */
  int k = 1;				/* Stack P: Pointer. */
  
  adb_tavl_node_t *p;

  assert (tree != NULL);

  if ( tree->tavl_root == NULL)
    /* Empty tree. */
    return NULL;

  a[0] = 0;
  pa[0] = (adb_tavl_node_t *)&tree->tavl_root;
  p = (void *)tree->tavl_root + sb ;
  for (;;)
    {
      /* D2. */
      int diff = bh->cmp (item, (void *)p->tavl_data + sb, bh);

      if (diff == 0)
	break;

      /* D3, D4. */
      pa[k] = p;
      if (diff < 0)
	{
	  if (p->tavl_tag[0] == TAVL_CHILD )
	    {
	      p = (void *)p->tavl_link[0] + sb ;
	      a[k] = 0;
	    }
	  else
	    return NULL;
	}
      else if (diff > 0)
	{
	  if (p->tavl_tag[1] == TAVL_CHILD )
	    {
	      p = (void *)p->tavl_link[1] + sb; 
	      a[k] = 1;
	    }
	  else
	    return NULL;
	}

      k++;
    }
  tree->tavl_count--;
  
  item = (void *)p->tavl_data + sb;

  {
    adb_tavl_node_t *t = p;
    adb_tavl_node_t **q = (void *)&pa[k - 1]->tavl_link[(int) a[k - 1]] ;
/*     adb_tavl_node **_q = (void *)&pa[k - 1]->tavl_link[(int) a[k - 1]] ; */
/*     adb_tavl_node **q = (void *) _q + sb ; */

    /* D5. */
    if (t->tavl_tag[1] == TAVL_THREAD)
      {
	if (t->tavl_tag[0] == TAVL_CHILD )
	  {
	    adb_tavl_node_t *const x = (void *)t->tavl_link[0] + sb;
	    adb_tavl_node_t *qq = (void *) *q + sb ;

	    qq  = x; /* *q = x ; */
	    qq->tavl_balance = 0; /* *q->tavl_balance = 0 ; */
	    if (x->tavl_tag[1] == TAVL_THREAD)
	      {
		if (a[k - 1] == 1)
		  x->tavl_link[1] = t->tavl_link[1];
		else
		  x->tavl_link[1] = (void *)pa[k - 1] - sb ;
	      }
	  }
	else
	  {
	    adb_tavl_node_t *qq = (void *) *q + sb ;
	    qq = (void *)t->tavl_link[a[k - 1]] + sb;
	    pa[k - 1]->tavl_tag[a[k - 1]] = TAVL_THREAD;
	  }
      }
    else
      {
	/* D6. */
	adb_tavl_node_t *r = (void *)t->tavl_link[1]  + sb , *qq = (void *) *q + sb;
	if (r->tavl_tag[0] == TAVL_THREAD)
	  {
	    r->tavl_link[0] = t->tavl_link[0];
	    r->tavl_tag[0] = t->tavl_tag[0];
	    r->tavl_balance = t->tavl_balance;
	    if (r->tavl_tag[0] == TAVL_CHILD )
	      {
		adb_tavl_node_t *s = (void *)r->tavl_link[0] + sb;
		while (s->tavl_tag[1] == TAVL_CHILD )
		  s = (void *)s->tavl_link[1] + sb;
		assert (s->tavl_tag[1] == TAVL_THREAD);
		s->tavl_link[1] = (void *) r - sb;
	      }
	    qq = r;
	    a[k] = 1;
	    pa[k++] = r;
	  }
	else
	  {
	    /* D7. */
	    adb_tavl_node_t *s = (void *)r->tavl_link[0] + sb ;

	    a[k] = 1;
	    pa[k++] = t;

	    a[k] = 0;
	    pa[k++] = r;
	    
	    /* D8. */
	    while (s->tavl_tag[0] != TAVL_THREAD)
	      {
		r = s;
		s = (void *)r->tavl_link[0] + sb;
		a[k] = 0;
		pa[k++] = r;
	      }

	    /* D9. */
	    t->tavl_data = s->tavl_data;
	    if (s->tavl_tag[1] == TAVL_THREAD)
	      {
		r->tavl_tag[0] = TAVL_THREAD;
		r->tavl_link[0] = (void *)t - sb ;
	      }
	    else
	      {
		adb_tavl_node_t *s1 = (void *)s->tavl_link[1] + sb ;
		r->tavl_link[0] = s->tavl_link[1];
		if (s1->tavl_tag[0] == TAVL_THREAD)
		  s1->tavl_link[0] = (void *)t - sb;
	      }
	    p = s;
	  }
      }
  }
#if 0
  ta->free ((void *)p, bh->m, sizeof(*p));
#endif /* 0 */
  assert (k > 0);
  /* D10. */
  while (--k)
    {
      adb_tavl_node_t *const s = pa[k];

      if (a[k] == 0)
	{
	  adb_tavl_node_t *const r = (void *)s->tavl_link[1] + sb ;
	  
	  /* D10. */
	  if (s->tavl_balance == -1)
	    {
	      s->tavl_balance = 0;
	      continue;
	    }
	  else if (s->tavl_balance == 0)
	    {
	      s->tavl_balance = +1;
	      break;
	    }

	  assert (s->tavl_balance == +1);
	  if (s->tavl_tag[1] == TAVL_THREAD || r->tavl_balance == 0)
	    {
	      /* D11. */
	      s->tavl_link[1] = r->tavl_link[0];
	      r->tavl_link[0] = (void *)s - sb ;
	      r->tavl_balance = -1;
	      pa[k - 1]->tavl_link[(int) a[k - 1]] = (void *)r - sb ;
	      break;
	    }
	  else if (r->tavl_balance == +1)
	    {
	      /* D12. */
	      if (TAVL_CHILD  == (s->tavl_tag[1] = r->tavl_tag[0]))
		s->tavl_link[1] = r->tavl_link[0];
	      r->tavl_link[0] = (void *)s - sb;
	      r->tavl_tag[0] = TAVL_CHILD ;
 	      s->tavl_balance = r->tavl_balance = 0;
	      pa[k - 1]->tavl_link[a[k - 1]] = (void *)r - sb ;
	    }
	  else 
	    {
	      /* D13. */
	      assert (r->tavl_balance == -1);
	      p = (void *)r->tavl_link[0] + sb ;
	      if (TAVL_CHILD  == (r->tavl_tag[0] = p->tavl_tag[1]))
		r->tavl_link[0] = p->tavl_link[1];
	      p->tavl_link[1] = (void *) r - sb;
	      p->tavl_tag[1] = TAVL_CHILD ;
	      if (TAVL_THREAD == (s->tavl_tag[1] = p->tavl_tag[0]))
		s->tavl_link[1] = (void *)p - sb ;
	      else
		s->tavl_link[1] = p->tavl_link[0];
	      p->tavl_link[0] = (void *)s - sb ;
	      p->tavl_tag[0] = TAVL_CHILD ;
	      if (p->tavl_balance == +1)
		s->tavl_balance = -1, r->tavl_balance = 0;
	      else if (p->tavl_balance == 0)
		s->tavl_balance = r->tavl_balance = 0;
	      else
		{
		  assert (p->tavl_balance == -1);
		  s->tavl_balance = 0, r->tavl_balance = +1;
		}
	      p->tavl_balance = 0;
	      pa[k - 1]->tavl_link[(int) a[k - 1]] = (void *)p - sb ;
	      pa[k - 1]->tavl_tag[(int) a[k - 1]] = TAVL_CHILD ;
	    }
	}
      else
	{
	  adb_tavl_node_t *const r = (void *)s->tavl_link[0] + sb ;
	  
	  /* D10. */
	  if (s->tavl_balance == +1)
	    {
	      s->tavl_balance = 0;
	      continue;
	    }
	  else if (s->tavl_balance == 0)
	    {
	      s->tavl_balance = -1;
	      break;
	    }

	  assert (s->tavl_balance == -1);
	  if (s->tavl_tag[0] == TAVL_THREAD || r->tavl_balance == 0)
	    {
	      /* D11. */
	      s->tavl_link[0] = r->tavl_link[1];
	      r->tavl_link[1] = (void *)s - sb ;
	      r->tavl_balance = +1;
	      pa[k - 1]->tavl_link[(int) a[k - 1]] = (void *)r - sb ;
	      break;
	    }
	  else if (r->tavl_balance == -1)
	    {
	      /* D12. */
	      if (TAVL_CHILD  == (s->tavl_tag[0] = r->tavl_tag[1]))
		s->tavl_link[0] = r->tavl_link[1];
	      r->tavl_link[1] = (void *)s - sb ;
	      r->tavl_tag[1] = TAVL_CHILD ;
	      s->tavl_balance = r->tavl_balance = 0;
	      pa[k - 1]->tavl_link[a[k - 1]] = (void *)r - sb ;
	    }
	  else 
	    {
	      /* D13. */
	      assert (r->tavl_balance == +1);
	      p = (void *) r->tavl_link[1] + sb ;
	      if (TAVL_CHILD  == (r->tavl_tag[1] = p->tavl_tag[0]))
		r->tavl_link[1] = p->tavl_link[0];
	      p->tavl_link[0] = (void *)r - sb ;
	      p->tavl_tag[0] = TAVL_CHILD ;
	      if (TAVL_THREAD == (s->tavl_tag[0] = p->tavl_tag[1]))
		s->tavl_link[0] = (void *)p - sb;
	      else
		s->tavl_link[0] = p->tavl_link[1];
	      p->tavl_link[1] = (void *)s - sb ;
	      p->tavl_tag[1] = TAVL_CHILD ;
	      if (p->tavl_balance == -1)
		s->tavl_balance = +1, r->tavl_balance = 0;
	      else if (p->tavl_balance == 0)
		s->tavl_balance = r->tavl_balance = 0;
	      else
		{
		  assert (p->tavl_balance == +1);
		  s->tavl_balance = 0, r->tavl_balance = -1;
		}
	      p->tavl_balance = 0;
	      pa[k - 1]->tavl_link[(int) a[k - 1]] = (void *)p - sb ;
	      pa[k - 1]->tavl_tag[(int) a[k - 1]] = TAVL_CHILD ;
	    }
	}
    }
      
  return (void *) item;
}
#endif /* 1 */

const void *
adb_tavl_delete (const adb_bh_t *bh, const void *item)
{
  struct adb_tavl_table *tree = bh->tree ;
  unsigned int sb = bh->addr ;
  struct adb_tavl_node *p; /* Traverses tree to find node to delete. */
  struct adb_tavl_node *q; /* Parent of |p|. */
  int dir;             /* Index into |q->tavl_link[]| to get |p|. */
  int cmp;             /* Result of comparison between |item| and |p|. */
  int i = 0 ;

  assert (tree != NULL && item != NULL);

  if (tree->tavl_root == NULL)
    return NULL;

  p = (struct adb_tavl_node *) &tree->tavl_root ;
/*   q = (struct tavl_node *) &tree->tavl_root; */
/*   p = tree->tavl_root; */

  for (cmp = -1 ; cmp != 0; 
       cmp = bh->cmp (item, DATA(p), bh , 0))

    {
      dir = cmp > 0;

      q = p;

      if ( ! i )
	++i ;
      else
	if (p->tavl_tag[dir] == TAVL_THREAD)
	  return NULL;
      p = (void *)p->tavl_link[dir] + sb ;
    }

  item = DATA(p);

  /* Step 2: Delete item from TAVL tree 313 */
  if (p->tavl_tag[1] == TAVL_THREAD)
    {
      if (p->tavl_tag[0] == TAVL_CHILD)
        {
          struct adb_tavl_node *t = (void *)p->tavl_link[0] + sb;
          while (t->tavl_tag[1] == TAVL_CHILD)
            t = (void *)t->tavl_link[1] + sb;
          t->tavl_link[1] = p->tavl_link[1];
          q->tavl_link[dir] = p->tavl_link[0];
        }
      else
        {
          q->tavl_link[dir] = p->tavl_link[dir]; 
          if (q != (struct adb_tavl_node *) &tree->tavl_root)
            q->tavl_tag[dir] = TAVL_THREAD;
        }
    }
  else
    {
      struct adb_tavl_node *r = (void *)p->tavl_link[1] + sb;
      if (r->tavl_tag[0] == TAVL_THREAD)
        {
          r->tavl_link[0] = p->tavl_link[0];
          r->tavl_tag[0] = p->tavl_tag[0];
          if (r->tavl_tag[0] == TAVL_CHILD)
            {
              struct adb_tavl_node *t = (void *)r->tavl_link[0] + sb;
              while (t->tavl_tag[1] == TAVL_CHILD)
                t = (void *)t->tavl_link[1] + sb;
              t->tavl_link[1] = (void *)r - sb;
            }
          q->tavl_link[dir] = (void *)r - sb;
          r->tavl_balance = p->tavl_balance;
          q = r;
          dir = 1;
        }
      else /* <Case 4 in TBST deletion 263 */
        {
          struct adb_tavl_node *s;
	  
          for (;;)
            {
              s = (void *) r->tavl_link[0] + sb;
              if (s->tavl_tag[0] == TAVL_THREAD)
                break;

              r = s;
            }

          if (s->tavl_tag[1] == TAVL_CHILD)
            r->tavl_link[0] = s->tavl_link[1];
          else
            {
              r->tavl_link[0] = (void *)s - sb;
              r->tavl_tag[0] = TAVL_THREAD;
            }

          s->tavl_link[0] = p->tavl_link[0];
          if (p->tavl_tag[0] == TAVL_CHILD)
            {
              struct adb_tavl_node *t = (void *)p->tavl_link[0] + sb;
              while (t->tavl_tag[1] == TAVL_CHILD)
                t = (void *) t->tavl_link[1] + sb;

              t->tavl_link[1] = (void *)s - sb;

              s->tavl_tag[0] = TAVL_CHILD;
            }

          s->tavl_link[1] = p->tavl_link[1];
          s->tavl_tag[1] = TAVL_CHILD;

/*  	  if ( q == (adb_tavl_node_t *)&tree->tavl_root ) */
/* 	    tree->tavl_root = NULL ; */
/* 	  else  */
	  q->tavl_link[dir] = (void *)s - sb;
          s->tavl_balance = p->tavl_balance;
	  
          q = r;
          dir = 0;
        }
    }

/*   tree->tavl_alloc->libavl_free (tree->tavl_alloc, p); */

  while (q != (struct adb_tavl_node *) &tree->tavl_root)  
    {
      struct adb_tavl_node *y = q;

      q = adb_find_parent (sb, tree, y);

      if (dir == 0)
        {
          dir = q->tavl_link[0] != ((void *)y - sb);
          y->tavl_balance++;
          if (y->tavl_balance == +1)
	    {
	      break;
	    }
          else if (y->tavl_balance == +2)
            {
              adb_tavl_node_t *x = (void *)y->tavl_link[1] + sb;

              assert (x != NULL);
              if (x->tavl_balance == -1)
                {
                  adb_tavl_node_t *w;

                  assert (x->tavl_balance == -1);
                  w = (void *)x->tavl_link[0] + sb;

                  x->tavl_link[0] = w->tavl_link[1];
                  w->tavl_link[1] = (void *)x - sb;
                  y->tavl_link[1] = w->tavl_link[0];
                  w->tavl_link[0] = (void *)y - sb;
                  if (w->tavl_balance == +1)
                    x->tavl_balance = 0, y->tavl_balance = -1;
                  else if (w->tavl_balance == 0)
                    x->tavl_balance = y->tavl_balance = 0;
                  else /* |w->tavl_balance == -1| */
                    x->tavl_balance = +1, y->tavl_balance = 0;
                  w->tavl_balance = 0;
                  if (w->tavl_tag[0] == TAVL_THREAD)
                    {
                      y->tavl_tag[1] = TAVL_THREAD;
                      y->tavl_link[1] = (void *)w - sb;
                      w->tavl_tag[0] = TAVL_CHILD;
                    }
                  if (w->tavl_tag[1] == TAVL_THREAD)
                    {
                      x->tavl_tag[0] = TAVL_THREAD;
                      x->tavl_link[0] = (void *)w - sb;
                      w->tavl_tag[1] = TAVL_CHILD;
                    }
                  q->tavl_link[dir] = (void *)w - sb;
		  
                }
              else
                {
                  q->tavl_link[dir] = (void *)x - sb;

                  if (x->tavl_balance == 0)
                    {
                      y->tavl_link[1] = x->tavl_link[0];
                      x->tavl_link[0] = (void *)y - sb;
                      x->tavl_balance = -1;
                      y->tavl_balance = +1;
                      break;
                    }
                  else /* |x->tavl_balance == +1| */
                    {
                      if (x->tavl_tag[0] == TAVL_CHILD)
                        y->tavl_link[1] = x->tavl_link[0];
                      else
                        {
                          y->tavl_tag[1] = TAVL_THREAD;
                          x->tavl_tag[0] = TAVL_CHILD;
                        }
                      x->tavl_link[0] = (void *)y - sb;
                      y->tavl_balance = x->tavl_balance = 0;
                    }
                }
            }
        }
      else
        {
          dir = q->tavl_link[0] != ( (void *)y - sb);
          y->tavl_balance--;
          if (y->tavl_balance == -1)
	    {
	      break;
	    }
          else if (y->tavl_balance == -2)
            {
              struct adb_tavl_node *x = (void *)y->tavl_link[0] + sb;
              assert (x != NULL);

              if (x->tavl_balance == +1)
                {
                  struct adb_tavl_node *w;

                  assert (x->tavl_balance == +1);
                  w = (void *)x->tavl_link[1] + sb;

                  x->tavl_link[1] = w->tavl_link[0];
                  w->tavl_link[0] = (void *)x - sb;
                  y->tavl_link[0] = w->tavl_link[1];
                  w->tavl_link[1] = (void *)y - sb;
                  if (w->tavl_balance == -1)
                    x->tavl_balance = 0, y->tavl_balance = +1;
                  else if (w->tavl_balance == 0)
                    x->tavl_balance = y->tavl_balance = 0;
                  else /* |w->tavl_balance == +1| */
                    x->tavl_balance = -1, y->tavl_balance = 0;
                  w->tavl_balance = 0;
                  if (w->tavl_tag[0] == TAVL_THREAD)
                    {
                      x->tavl_tag[1] = TAVL_THREAD;
                      x->tavl_link[1] = (void *)w - sb;
                      w->tavl_tag[0] = TAVL_CHILD;
                    }
                  if (w->tavl_tag[1] == TAVL_THREAD)
                    {
                      y->tavl_tag[0] = TAVL_THREAD;
                      y->tavl_link[0] = (void *)w - sb;
                      w->tavl_tag[1] = TAVL_CHILD;
                    }
                  q->tavl_link[dir] = (void *)w - sb;
                }
              else
                {
                  q->tavl_link[dir] = (void *)x - sb;

                  if (x->tavl_balance == 0)
                    {
                      y->tavl_link[0] = x->tavl_link[1];
                      x->tavl_link[1] = (void *)y - sb;
                      x->tavl_balance = +1;
                      y->tavl_balance = -1;
                      break;
                    }
                  else /* |x->tavl_balance == -1| */
                    {
                      if (x->tavl_tag[1] == TAVL_CHILD)
                        y->tavl_link[0] = x->tavl_link[1];
                      else
                        {
                          y->tavl_tag[0] = TAVL_THREAD;
                          x->tavl_tag[1] = TAVL_CHILD;
                        }
                      x->tavl_link[1] = (void *)y - sb;
                      y->tavl_balance = x->tavl_balance = 0;
                    }
                }
            }
        }
    }

  tree->tavl_count--;

  return item;
}

/* Initializes |trav| for use with |tree|
   and selects the null node. */
void
adb_tavl_t_init (const adb_bh_t *bh,struct adb_tavl_traverser *trav)
{
  trav->tavl_table = bh->tree;
  trav->tavl_node = NULL;
}

/* Initializes |trav| for |tree|.
   Returns data item in |tree| with the least value,
   or |NULL| if |tree| is empty. */
void *
adb_tavl_t_first (const adb_bh_t *bh, struct adb_tavl_traverser *trav)
{
  unsigned int sb = bh->addr ;

  assert (trav != NULL);

  trav->tavl_table = bh->tree;
  trav->tavl_node = (void *)bh->tree->tavl_root + sb;
  if (trav->tavl_node != (adb_tavl_node_t *)bh->m ) /* != NULL */
    {
      while (trav->tavl_node->tavl_tag[0] == TAVL_CHILD)
        trav->tavl_node = (void *)trav->tavl_node->tavl_link[0] + sb ;
      return DATA(trav->tavl_node);
    }
  else
    return NULL;
}

/* Initializes |trav| for |tree|.
   Returns data item in |tree| with the greatest value,
   or |NULL| if |tree| is empty. */
void *
adb_tavl_t_last (const adb_bh_t *bh, struct adb_tavl_traverser *trav)
{
  unsigned int sb = bh->addr ;

  assert (trav != NULL);

  trav->tavl_table = bh->tree;
  trav->tavl_node = (void *)bh->tree->tavl_root + sb;
  if (trav->tavl_node != NULL)
    {
      while (trav->tavl_node->tavl_tag[1] == TAVL_CHILD)
        trav->tavl_node = (void *)trav->tavl_node->tavl_link[1] + sb;
      return DATA(trav->tavl_node);
    }
  else
    return NULL;
}

/* Searches for |item| in |tree|.
   If found, initializes |trav| to the item found and returns the item
   as well.
   If there is no matching item, initializes |trav| to the null item
   and returns |NULL|. */
void *
adb_tavl_t_find (const adb_bh_t *bh, struct adb_tavl_traverser *trav, void *item)
{
  unsigned int sb = bh->addr ;
  struct adb_tavl_node *p;
  adb_tavl_table_t *tree = bh->tree ;

  assert (trav != NULL && item != NULL);

  trav->tavl_table = tree;
  trav->tavl_node = NULL;

  p = (void *)tree->tavl_root + sb ;
  if ( tree->tavl_root == NULL )
    return NULL;

  for (;;)
    {
      int cmp, dir;

      cmp = bh->cmp (item, DATA(p), bh, 0);
      if (cmp == 0)
        {
          trav->tavl_node = p;
          return DATA(p);
        }

      dir = cmp > 0;
      if (p->tavl_tag[dir] == TAVL_CHILD)
        p = (void *)p->tavl_link[dir] + sb;
      else
        return NULL;
    }
}

/* Searches for |item| in |tree|.
   If found, initializes |trav| to the item found and returns the item
   as well.
   If there is no matching item, initializes |trav| to the null item
   and returns |NULL|. */
void *
adb_tavl_t_find_close (const adb_bh_t *bh, struct adb_tavl_traverser *trav, void *item)
{
  unsigned int sb = bh->addr ;
  struct adb_tavl_node *p;
  adb_tavl_table_t *tree = bh->tree ;

  assert (trav != NULL && item != NULL);

  trav->tavl_table = tree;
  trav->tavl_node = NULL;

  p = (void *)tree->tavl_root + sb ;
  if ( tree->tavl_root == NULL )
    return NULL;

  for (;;)
    {
      int cmp, dir;

      cmp = bh->cmp (item, DATA(p), bh, 0);
      if (cmp == 0)
        {
          trav->tavl_node = p;
          return DATA(p);
        }

      dir = cmp > 0;
      if (p->tavl_tag[dir] == TAVL_CHILD)
        p = (void *)p->tavl_link[dir] + sb;
      else
	{
	  trav->tavl_node = p;
	  return DATA(p);
	}
    }
}

/* Attempts to insert |item| into |tree|.
   If |item| is inserted successfully, it is returned and |trav| is
   initialized to its location.
   If a duplicate is found, it is returned and |trav| is initialized to
   its location.  No replacement of the item occurs.
   If a memory allocation failure occurs, |NULL| is returned and |trav|
   is initialized to the null item. */
void *
adb_tavl_t_insert (const adb_bh_t *bh, struct adb_tavl_traverser *trav,adb_tavl_node_t *n, void *item)
{
/*   unsigned int sb = ta->addr ; */
  void *p;

  assert (trav != NULL && item != NULL);

  p = adb_tavl_probe (bh, n, item);
  if (p != NULL)
    {
      trav->tavl_table = bh->tree;
      trav->tavl_node = n ;
/*         ((struct tavl_node *) */
/*          ((char *) p - offsetof (struct tavl_node, tavl_data))); */
      return DATA(p);
    }
  else
    {
      adb_tavl_t_init (bh, trav);
      return NULL;
    }
}

/* Initializes |trav| to have the same current node as |src|. */
void *
adb_tavl_t_copy (const adb_bh_t *bh UNUSED , struct adb_tavl_traverser *trav, const struct adb_tavl_traverser *src)
{
  assert (trav != NULL && src != NULL);

  trav->tavl_table = src->tavl_table;
  trav->tavl_node = src->tavl_node;

  return trav->tavl_node != NULL ? DATA(trav->tavl_node) : NULL;
}

/* Returns the next data item in inorder
   within the tree being traversed with |trav|,
   or if there are no more data items returns |NULL|. */
void *
adb_tavl_t_next (const adb_bh_t *bh, adb_tavl_traverser_t *trav)
{
  const unsigned int sb = bh->addr ;

  assert (trav != NULL);

  if (trav->tavl_node == NULL)
    return adb_tavl_t_first (bh, trav);
  else if (trav->tavl_node->tavl_tag[1] == TAVL_THREAD)
    {
      trav->tavl_node = (void *)trav->tavl_node->tavl_link[1] + sb ;
      return trav->tavl_node != (void *)sb ? DATA(trav->tavl_node) : NULL;
    }
  else
    {
      trav->tavl_node = (void *)trav->tavl_node->tavl_link[1] + sb;
      while (trav->tavl_node->tavl_tag[0] == TAVL_CHILD)
        trav->tavl_node = (void *)trav->tavl_node->tavl_link[0] + sb;
      return DATA(trav->tavl_node) ;
    }
}

/* Returns the previous data item in inorder
   within the tree being traversed with |trav|,
   or if there are no more data items returns |NULL|. */
void *
adb_tavl_t_prev (const adb_bh_t *bh,struct adb_tavl_traverser *trav)
{
  const unsigned int sb = bh->addr ;

  assert (trav != NULL);

  if (trav->tavl_node == NULL)
    return adb_tavl_t_last (bh, trav);
  else if (trav->tavl_node->tavl_tag[0] == TAVL_THREAD)
    {
      trav->tavl_node = (void *)trav->tavl_node->tavl_link[0] + sb;
      return trav->tavl_node != (void *)sb ? DATA(trav->tavl_node) : NULL;
    }
  else
    {
      trav->tavl_node = (void *)trav->tavl_node->tavl_link[0] + sb;
      while (trav->tavl_node->tavl_tag[1] == TAVL_CHILD)
        trav->tavl_node = (void *)trav->tavl_node->tavl_link[1] + sb;
      return DATA(trav->tavl_node);
    }
}

/* Returns |trav|'s current item. */
void *
adb_tavl_t_cur (const adb_bh_t *bh, struct adb_tavl_traverser *trav)
{
  assert (trav != NULL);

  return trav->tavl_node != (void *)bh->addr ? DATA(trav->tavl_node) : NULL;
}

/* Replaces the current item in |trav| by |new| and returns the item replaced.
   |trav| must not have the null item selected.
   The new item must not upset the ordering of the tree. */
void *
adb_tavl_t_replace (const adb_bh_t *bh UNUSED ,struct adb_tavl_traverser *trav, void *new)
{
  struct adb_tavl_node *old;

  assert (trav != NULL && trav->tavl_node != NULL && new != NULL);
  old = DATA(trav->tavl_node) ;
  assert("Don't work" == 0 );
#if 0
  trav->tavl_node->tavl_data = (void *)new - bh->addr;
#endif
  return old;
}

/* ****************************************************************************** */

#if 0
/* Creates a new node as a child of |dst| on side |dir|.
   Copies data and |tavl_balance| from |src| into the new node,
   applying |copy()|, if non-null.
   Returns nonzero only if fully successful.
   Regardless of success, integrity of the tree structure is assured,
   though failure may leave a null pointer in a |tavl_data| member. */
static int
copy_node (struct tavl_table *tree,
           struct adb_tavl_node *dst, int dir,
           const struct adb_tavl_node *src, tavl_copy_func *copy)
{
  struct adb_tavl_node *new =
    tree->tavl_alloc->libavl_malloc (tree->tavl_alloc, sizeof *new);
  if (new == NULL)
    return 0;

  new->tavl_link[dir] = dst->tavl_link[dir];
  new->tavl_tag[dir] = TAVL_THREAD;
  new->tavl_link[!dir] = dst;
  new->tavl_tag[!dir] = TAVL_THREAD;
  dst->tavl_link[dir] = new;
  dst->tavl_tag[dir] = TAVL_CHILD;

  new->tavl_balance = src->tavl_balance;
  if (copy == NULL)
    new->tavl_data = src->tavl_data;
  else
    {
      new->tavl_data = copy (src->tavl_data, tree->tavl_param);
      if (new->tavl_data == NULL)
        return 0;
    }

  return 1;
}

static void
copy_error_recovery (struct adb_tavl_node *p,
                     struct tavl_table *new, tavl_item_func *destroy)
{
  new->tavl_root = p;
  if (p != NULL)
    {
      while (p->tavl_tag[1] == TAVL_CHILD)
        p = p->tavl_link[1];
      p->tavl_link[1] = NULL;
    }
  tavl_destroy (new, destroy);
}

/* Copies |org| to a newly created tree, which is returned.
   If |copy != NULL|, each data item in |org| is first passed to |copy|,
   and the return values are inserted into the tree,
   with |NULL| return values taken as indications of failure.
   On failure, destroys the partially created new tree,
   applying |destroy|, if non-null, to each item in the new tree so far,
   and returns |NULL|.
   If |allocator != NULL|, it is used for allocation in the new tree.
   Otherwise, the same allocator used for |org| is used. */
struct tavl_table *
tavl_copy (const struct tavl_table *org, tavl_copy_func *copy,
          tavl_item_func *destroy, struct libavl_allocator *allocator)
{
  struct tavl_table *new;

  const struct adb_tavl_node *p;
  struct adb_tavl_node *q;
  struct adb_tavl_node rp, rq;

  assert (org != NULL);
  new = tavl_create (org->tavl_compare, org->tavl_param,
                     allocator != NULL ? allocator : org->tavl_alloc);
  if (new == NULL)
    return NULL;

  new->tavl_count = org->tavl_count;
  if (new->tavl_count == 0)
    return new;

  p = &rp;
  rp.tavl_link[0] = org->tavl_root;
  rp.tavl_tag[0] = TAVL_CHILD;

  q = &rq;
  rq.tavl_link[0] = NULL;
  rq.tavl_tag[0] = TAVL_THREAD;

  for (;;)
    {
      if (p->tavl_tag[0] == TAVL_CHILD)
        {
          if (!copy_node (new, q, 0, p->tavl_link[0], copy))
            {
              copy_error_recovery (rq.tavl_link[0], new, destroy);
              return NULL;
            }

          p = p->tavl_link[0];
          q = q->tavl_link[0];
        }
      else
        {
          while (p->tavl_tag[1] == TAVL_THREAD)
            {
              p = p->tavl_link[1];
              if (p == NULL)
                {
                  q->tavl_link[1] = NULL;
                  new->tavl_root = rq.tavl_link[0];
                  return new;
                }

              q = q->tavl_link[1];
            }

          p = p->tavl_link[1];
          q = q->tavl_link[1];
        }

      if (p->tavl_tag[1] == TAVL_CHILD)
        if (!copy_node (new, q, 1, p->tavl_link[1], copy))
          {
            copy_error_recovery (rq.tavl_link[0], new, destroy);
            return NULL;
          }
    }
}

#endif /* 0 */ /* *********************************************************************** */

void *
tavl_root(adb_bh_t *bh)
{
  struct adb_tavl_table *tree = bh->tree ;
  struct adb_tavl_node *p; /* Current node. */
  p = tree->tavl_root;
  return DATA(p) ;
}

#if 0
/* useless bugware -- no MM etc. */
void
adb_tavl_split(adb_bh_t *left /* old */, adb_bh_t *right /* new */)
{
  struct tavl_table *left_tree = &left->m->tree , *right_tree = &right->m->tree ;
  unsigned int sb = left->addr ;
  struct adb_tavl_node *p ; /* Current node. */
  adb_tavl_traverser_t tr ;
  size_t left_count , right_count ;
  void *root = tavl_root(left);
  adb_rec_t *rr ;

  assert( left_tree->tavl_count );

  
  p = left_tree->tavl_root;
  
  adb_tavl_t_init( left, &tr );
  for ( left_count = 0 ;  ; ++left_count )
    if (adb_tavl_t_next( left, &tr ) == root )
	break ;

  for ( right_count = 0 ;  ; ++right_count )
    if ( ! adb_tavl_t_next( left, &tr ))
      break ;

  assert( left_tree->tavl_count == left_count + right_count + 1 );
  
  left_tree->tavl_root = p->tavl_link[0] ;
  right_tree->tavl_root = p->tavl_link[1] ;

  rr = (adb_rec_t *)root ;
  adb_tavl_insert(left, &rr->node, rr );
}
#endif /* 0 */

/* Frees storage allocated for |tree|.
   If |destroy != NULL|, applies it to each data item in inorder. */
void
adb_tavl_destroy (adb_bh_t *bh, tavl_item_func *destroy, void *param)
{
  struct adb_tavl_table *tree = bh->tree ;
  struct adb_tavl_node *p; /* Current node. */
  struct adb_tavl_node *n; /* Next node. */

  p = tree->tavl_root;
  if (p != NULL)
    while (p->tavl_tag[0] == TAVL_CHILD)
      p = p->tavl_link[0];

  while (p != NULL)
    {
      n = p->tavl_link[1];
      if (p->tavl_tag[1] == TAVL_CHILD)
        while (n->tavl_tag[0] == TAVL_CHILD)
          n = n->tavl_link[0];

      if (destroy != NULL && DATA(p) != NULL)
        destroy (DATA(p), param);
/*       tree->tavl_alloc->libavl_free (tree->tavl_alloc, p); */

      p = n;
    }

/*   tree->tavl_alloc->libavl_free (tree->tavl_alloc, tree); */
}

#undef NDEBUG
#include <assert.h>

/* Asserts that | tavl_insert | succeeds at inserting |item| into |table|. */
void
adb_tavl_assert_insert (adb_bh_t *bh ,adb_tavl_node_t *n, void *item)
{
  void *p = adb_tavl_probe (bh, n, item);
  assert (p != NULL && p == item);
}

/* Asserts that |tavl_delete()| really removes |item| from |table|,
   and returns the removed item. */
const void *
adb_tavl_assert_delete (adb_bh_t *bh , void *item)
{
  const void *p = adb_tavl_delete (bh, item);
  assert (p != NULL);
  return p;
}


int
adb_tavl_tree_height(const adb_bh_t *bh)
{
  int height = 1 ;
  adb_tavl_table_t *tree = bh->tree ;
  register adb_tavl_node_t *n ;
  register const unsigned int sb = bh->addr ;

  n = (void *)tree->tavl_root + sb;

  while ( n->tavl_tag[0] == TAVL_CHILD)
    {
      n = (void *)n->tavl_link[0] + sb ;
      ++height ;
    }
  return height ;
}

void 
adb_tavl_walk (const adb_bh_t *bh, adb_tavl_walk_func_t func, void *param)
{
#if 1
  register const unsigned int sb = bh->addr ;
  register void *null = (void *)sb ;
  register adb_tavl_node_t *n ;
  adb_tavl_table_t *tree = bh->tree ;

  /* tavl_t_first */

  n = (void *)tree->tavl_root + sb;
  if ( tree->tavl_root == NULL )
    return/*  NULL */ ;

  while ( n->tavl_tag[0] == TAVL_CHILD)
    n = (void *)n->tavl_link[0] + sb ;

  func( bh, DATA(n) , param );

  for ( ; ; )
    {
      if (n->tavl_tag[1] == TAVL_THREAD)
	{
	  n = (void *)n->tavl_link[1] + sb ;
	  /*       return trav->tavl_node != (void *)sb ? (void *)trav->tavl_node->tavl_data + sb : NULL; */
	}
      else
	{
	  n = (void *)n->tavl_link[1] + sb;
	  while (n->tavl_tag[0] == TAVL_CHILD)
	    n = (void *)n->tavl_link[0] + sb ;
	  /*       return (void *)trav->tavl_node->tavl_data + sb; */
	}
      if ( n == null )
	return ;
      func( bh, DATA(n) , param );
    }
#else /* 0 */
  adb_tavl_traverser_t tr ;
  void *data ;
  adb_tavl_t_init(bh, &tr); 
  while ( ( data = adb_tavl_t_next( bh, &tr )) != NULL )
    func(data, param); 

#endif /* 0 */
}
