/* AVLDB - AVL DataBase library, file functions + API

   Copyright (C) 2002,2003 Petr Silhavy <silhavy@mef.cz>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

#define  __ADB 1
#include "adb.h"

/* adb_file_t *adb_file_open
 *   adb_tnx_atom_t *adb_insert_by_name
 *
 */

extern adb_rec_t *Last ;
/*
 * === Record layout ===
 *
 * 
 *
 */

static void
Adb_show_field(void *a, void *param unused)
{
  adb_field_t *field = ( adb_field_t * ) a ;
  assert( field->magic == ADB_MAGIC_FIELD );
  printf("==== Field : %s =====\n", field->name);
}

static void
Adb_show_file(void *a, void *param , void *_db)
{
  adb_bh_t fbh ;
  adb_meta_file_t *mfile = param ;
  adb_db_t *db = _db ;

/*   assert ( file->magic == ADB_MAGIC_FILE ); */

  fbh.addr = db->addr ;
  fbh.tree = &mfile->indexes ;
  printf("== File : %s ======== \n", mfile->name );
  if ( mfile->magic == ADB_MAGIC_META_FILE )
    adb_tavl_walk ( &fbh, Adb_show_field , NULL );
}


void
adb_show_files(adb_db_t *db)
{
  adb_tavl_walk ( db->files_bh,  Adb_show_file , db );
}

#ifndef CMP
#define CMP(TYPE, A, B) (*(TYPE *)A > *(TYPE *)B) - (*(TYPE *)A < *(TYPE *)B)
#endif

int
Adb_data_cmp(const void *_a, const void *_b, const void *_param, 
	    int only_first_n /* fields */ )
{
  adb_rec_t *a_rec = ( adb_rec_t *) _a ;
  adb_rec_t *b_rec = ( adb_rec_t *) _b ;

  adb_bh_t *bh = ( adb_bh_t *) _param ;
  adb_file_t *file = ( adb_file_t *) bh->file ;


  adb_field_t *fld = NULL ;
  int i ;
  int j ;
  int r = 0 ;
  
  void *a , *b ;
  unsigned int sb = file->db->addr ;

  assert( file->magic == ADB_MAGIC_FILE );
  assert( a_rec->magic == ADB_MAGIC_RECORD );
  assert( b_rec->magic == ADB_MAGIC_RECORD );

/*   for ( i = 0 ; i < file->last_order ; ++i ) */
  for ( i = 0 , j = 0 ; i < file->meta->idx_size ; ++i )
    {
      if ( file->idx[i] )
	{
	  if ( only_first_n && j++ == only_first_n )
	    {
	      if ( Adb_debug & ADB_DEBUG_DATA_CMP )
		fprintf(stderr,"%s: Artifical break after %d fields\n", 
			__func__, only_first_n);

	      break ;
	    }
	  fld = (void *)file->idx[i] + sb ;
	  if ( fld->magic == ADB_MAGIC_FIELD )
	    {
	      a = (void *)_a + fld->offset ;
	      b = (void *)_b + fld->offset  ;

	      switch ( fld->type )
		{
		case ADB_FIELD_UNSIGNED_INT:
		  r = CMP(unsigned int, a, b);
		  break ;
		case ADB_FIELD_INT: /*   return (*a > *b) - (*a < *b); */
		  r = CMP(int , a, b);
		  if ( Adb_debug & ADB_DEBUG_DATA_CMP )
		    fprintf(stderr,"%s: %d = %d > %d - %d < %d)\n", 
			    __func__, r, 
			    (*(int *)(_a + fld->offset)), ( *(int *)(_b + fld->offset) ),
			    (*(int *)(_a + fld->offset)), (*(int *)(_b + fld->offset) )
			    );
		  break ;
		case ADB_FIELD_PTR_DIFF:
		  r = CMP( __PTRDIFF_TYPE__, a, b);
		  break ;
		case ADB_FIELD_LONG_LONG:
		  r = CMP( long long, a, b);
		  break ;
		case ADB_FIELD_UNSIGNED_LONG_LONG:
		  r = CMP( unsigned long long, a, b);
		  break ;
		case ADB_FIELD_DOUBLE:
		  r = CMP(long double, a, b);
		  break ;
		case ADB_FIELD_STRING:
		  r = strcmp( (char *)(_a + fld->offset), ((char *)(_b + fld->offset) ));
		  if ( Adb_debug & ADB_DEBUG_DATA_CMP )
		    fprintf(stderr,"%s: %d = strcmp(%s, %s)\n", 
			    __func__, r,
			    (char *)(_a + fld->offset), ((char *)(_b + fld->offset) ));
		  break ;
		case ADB_FIELD_VSTRING:
		  {
		    adb_vstring_t *avs = a , *bvs = b ;
		    a = (void *)avs + avs->offset ;
		    b = (void *)bvs + bvs->offset ;
		    r = strcmp( (char *)a , (char *)b );
		    if ( Adb_debug & ADB_DEBUG_DATA_CMP )
		      fprintf(stderr,"%s: %d = strcmp(%s, %s)\n", 
			      __func__, r,
			      (char *)a, (char *)b );
		  }
		  break ;
		case ADB_FIELD_WCHAR:
		  {
		    adb_wchar_t *aw = a , *bw = b ;
		    wchar_t *wa, *wb ;
		    wa = (void *)_a + aw->offset ;
		    wb = (void *)_b + bw->offset ;
		    r = wcscoll( wa, wb);
		    if ( Adb_debug & ADB_DEBUG_DATA_CMP )
		      {
			fprintf(stderr,"%s: %d = wcscoll(%ls,%ls)\n", 
				__func__,
				r, wa, wb);
		      }
		  }
		  break ;
		default:
		  assert("FIXME" == 0 );
		  break ;
		}
	      if ( r )
		{
		  /* FIXME : do this as HAVE_MAD_STRCMP */
		  if ( r > 1 ) r = 1 ;
		  else if ( r < -1 ) r = -1 ;
		  if ( Adb_debug & ADB_DEBUG_DATA_CMP )
		    fprintf(stderr,"%s: retval = %d, last IDX field [%s]\n",
			    __func__, 
			    r, fld->name);
		  return r ;
		}
	    }
	}
    }
  if ( Adb_debug & ADB_DEBUG_DATA_CMP )
    fprintf(stderr,"%s: retval = %d, All checked, last IDX field [%s]\n",
	    __func__,
	    r,fld->name);
  return 0 ;
}

adb_bh_t *
Adb_file_activate(adb_tnx_t *tnx, adb_file_t *file, adb_tnx_state_t state)
{
  adb_bh_t *bh ;


  if ( ! file->meta->data_start_type ) /* 1st data segment - creat() */
    {
      bh = Adb_seg_add( tnx, file, &file->meta->data_start ,
			ADB_MAGIC_TREE, 0 , -1LL );
      fprintf(stderr, "bh %p bh->m %p bh->m->pool %p\n",
	      bh , bh->m , (void *)bh->m + ADB_MAGIC_SIZE + bh->m->map_size /* pool */ );

      /* adb_avlt_create - this stuff does _adb_seg_init */
/*       bh->m->tree = adb_avlt_create( __data_cmp, SEG(bh->addr) , bh ); */
      Adb_tnx_set_data_start_type(file, &file->meta->data_start_type, ADB_MAGIC_TREE);
    }
  else
    {
      bh = Adb_seg_attach( tnx, file, file->meta->data_start, state );
      assert ( bh->m->content == file->meta->data_start_type );
    }

  return bh ;
}


static int 
Adb_ofile_cmp(const void *_a, const void *_b, void *_param)
{
  const adb_file_t *a = _a , *b = _b ;

  assert(a->magic == ADB_MAGIC_FILE );
  assert(b->magic == ADB_MAGIC_FILE );
  return ( ( a->meta > b->meta ) - ( a->meta < b->meta ));
}

void
Adb__file_open(adb_db_t *db, adb_meta_file_t *mfile, adb_file_t *file )
{
  file->fd = db->fds[mfile->nfd] ;
  file->db = db ;

  file->order = (void *)mfile->__order + db->addr ;
  file->idx = (void *)mfile->__idx + db->addr ;

  file->blob_fd = db->fds[0] ;

  file->fields = &mfile->fields ;
  file->indexes = &mfile->indexes ;

}

adb_file_t *
Adb_file_open_primary(adb_db_t *db, adb_meta_file_t *mfile , adb_file_t *file)
{
  void *ret ;

  if ( mfile->__parent ) /* don't open by secondary index */
    mfile = ( void *)mfile->__parent + db->addr ;
  
  ret = tavl_insert(&db->open_files, Adb_ofile_cmp, &file->node, file );
  assert( ret == NULL);
  
  Adb__file_open(db, mfile, file );

  {
    int i ;
    adb_tavl_traverser_t tr ;
    adb_file_t **s ;
    adb_meta_file_t *m ;
    adb_bh_t fbh = { .addr = db->addr , .magic = ADB_MAGIC_BH } ;


    file->parent = NULL ;
    file->sis = Malloc(file->indexes->tavl_count * sizeof(adb_file_t *));
      
    fbh.tree = file->indexes ;

    for ( s = file->sis , i = 0 ; i < file->indexes->tavl_count ; ++i , ++s )
      {
	if ( i )
	  m = adb_tavl_t_next(&fbh, &tr );
	else
	  m = adb_tavl_t_first(&fbh, &tr);

	*s = Malloc(sizeof(adb_file_t));

	(*s)->magic = ADB_MAGIC_FILE ;
	(*s)->parent = file ;
	(*s)->meta = m ;

	Adb__file_open(db, m, *s);

	fprintf(stderr,"%s sis[%d]: %s <%p>\n",__func__, i, (*s)->meta->name, m);
	}
    }

  file->xip = NULL ;

  return file ;
}

adb_file_t *
Adb_file_open(adb_db_t *db, adb_meta_file_t *mfile)
{
  adb_file_t *file = Malloc(sizeof(*file)) , *f ;


  file->magic = ADB_MAGIC_FILE ;
  file->meta = mfile ;
  file->parent = NULL ;
  
  if ( (f = tavl_find(&db->open_files, Adb_ofile_cmp, file )))
     {
       free(file);
       return f ;
     }
  else
    return Adb_file_open_primary(db, mfile, file );
}

adb_file_t *
adb_file_open(adb_db_t *db, char *filename)
{
  adb_meta_file_t *mfile ;
  
  adb_meta_file_t tmp ;

  strncpy( tmp.name , filename , ADB_NAME_MAX -1 );

  if ((mfile = (adb_meta_file_t *)adb_tavl_find(db->files_bh, &tmp)) == NULL)
    {
      ADB_WARNING( "Can't find such file `%s'", filename );
/*       va_end(ap); ???? */
      return NULL ;
    }

  return Adb_file_open(db, mfile) ;
}

int
adb_file_close(adb_file_t *file)
{
  free(file);
  return adb_errno ;
}

adb_file_t *
Adb_sidx_get(adb_file_t *f, adb_meta_file_t *msi)
{ 
  int i ;
  adb_file_t **s ;
  for ( s = f->sis , i = 0 ; i < f->indexes->tavl_count ; ++i , ++s )
    if ( (*s)->meta == msi )
      return *s ;

  abort();
}

adb_file_t *
adb_file_open_use_index(adb_db_t *db, char *filename, char *idx_name)
{

  adb_meta_file_t tmp , *msi , *mfile ;

  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_file_cmp, 
		   .addr = db->addr, 
  } ;
  adb_file_t *file = Malloc(sizeof(*file)) , *f ;

  strncpy( tmp.name , filename , ADB_NAME_MAX -1 );

  if ((mfile = adb_tavl_find(db->files_bh, &tmp)) == NULL)
    {
      ADB_WARNING( "Can't find such file `%s'", filename );
/*       va_end(ap); ??? */
      return NULL ;
    }

  assert(!mfile->__parent );

  fbh.tree = &mfile->indexes ;

  strncpy( tmp.name , idx_name , ADB_NAME_MAX -1 );

  if ( (msi = adb_tavl_find( &fbh, &tmp )) == NULL )
    {
      ADB_WARNING("Can't find index `%s' in file `%s' database `%s'",
		  idx_name, filename, db->name);
      return NULL ;
    }


  file->magic = ADB_MAGIC_FILE ;
  file->meta = mfile ;
  file->parent = NULL ;
  
  if ( ( f = tavl_find(&db->open_files, Adb_ofile_cmp, file )))
    {
      free(file);
    }
  else
    f = Adb_file_open_primary(db, mfile, file );
  

  return Adb_sidx_get(f, msi );

#if 0
  /* *********************************************************** */

  si = Malloc(sizeof(*si));
  si->magic = ADB_MAGIC_FILE ;
  si->meta = msi ;

  si->fields = &mfile->fields ;
  si->indexes = NULL ;

  si->fd = db->fds[msi->nfd] ;
  si->db = db ;

  si->order = (void *)mfile->__order + db->addr ;
  si->idx = (void *)mfile->__idx + db->addr ;

  si->blob_fd = db->fds[0] ;

  { /* debug */
    int i ;
    adb_field_t *fld ;
    for ( i = 0 ; i < si->meta->idx_size ; ++i )
      {
	fld = (void *)si->idx[i] + db->addr ;
	fprintf(stderr,"%s [%d] %s\n",__FUNCTION__, i, fld->name);
      }
  }
  
  return si ;
#endif /* 0 */
}

void
Adb_data_show(const void *data, adb_file_t *file)
{
  int i ;
  adb_field_t *fld ;
  const adb_rec_t *rec = (adb_rec_t *)data ; 
  
  unsigned int sb = file->db->addr ;

  assert( file->magic == ADB_MAGIC_FILE );
  /* Ranges carry magic stuff too */
  assert( rec->magic == ADB_MAGIC_RECORD );

  fprintf(stderr, "Record: ");
  for ( i = 0 ; i < file->meta->last_order ; ++i )
    {
      if ( (void *)file->order[i] + sb )
	{
	  fld = (void *)file->order[i] + sb ;
	  if ( fld->magic == ADB_MAGIC_FIELD )
	    {
	      fprintf(stderr,"[%s] ",fld->name );
	      switch (  fld->type )
		{
		case ADB_FIELD_INT:
		  {
		    int *p_int ;
		    p_int =  (void *)data + fld->offset ;
		    fprintf(stderr,"{%10d} ", *p_int );
		    /* 		  assert ( *p_int < 10001000 && *p_int >= 0); */ /* ET value 26361856 */
		  }
		  break ;
		case ADB_FIELD_STRING:
		  {
		    char *p_char ;
		    p_char = (void *)data + fld->offset ;
		    fprintf(stderr,"{%s} ", p_char );
		  }
		  break ;
		case ADB_FIELD_VSTRING:
		  {
		    adb_vstring_t *vs ;
		    char *p_char ;
		    vs = (void *)data + fld->offset ;
		    p_char = (void *) vs + vs->offset ;
		    fprintf(stderr,"{[%d@%d] %s} ", vs->size, vs->offset, p_char );
		  }
		  break ;
		case ADB_FIELD_WCHAR:
		  {
		    adb_wchar_t *awc ;
		    wchar_t *ws ;
		    awc = (void *)data + fld->offset ;
		    ws = (void *)awc + awc->offset ;
		    fprintf(stderr,"{[%d@%d] <%ls> } ", awc->size, awc->offset, ws );
		  }
		  break ;
		case ADB_FIELD_LONG_LONG:
		  {
		    long long *p_ll ;
		    p_ll = (void *)data + fld->offset ;
		    fprintf(stderr,"{%lld} ", *p_ll );
		  }
		  break ;
		case ADB_FIELD_UNSIGNED_LONG_LONG:
		  {
		    unsigned long long *p_ll ;
		    p_ll = (void *)data + fld->offset ;
		    fprintf(stderr,"{%llu} ", *p_ll );
		  }
		  break ;
		case ADB_FIELD_DOUBLE:
		  {
		      long double *p_d ;
		      p_d = (void *)data + fld->offset ;
		      fprintf(stderr, "{%Lf} ", *p_d );
		  }
		  break ;
		case ADB_FIELD_LINK:
		  {
		    adb_link_t *link ;
		    link = (void *)data + fld->offset ;
		    fprintf(stderr, "{Link %llu/%llu} ", link->id, link->count );
		  }
		  break; 
		default:
		  assert("FIXME" == 0);
		  break ;
		}
	    }
	}
    }
  fputc('\n',stderr);
}

static void
Adb_tr_init(adb_tr_t *tr)
{
  int i ;
  tr->bhs = Malloc( sizeof(*tr->bhs) * tr->height );
  tr->ranges = Malloc(  sizeof(*tr->ranges) * tr->height );
  tr->trs = Malloc( sizeof(adb_tavl_traverser_t *) * tr->height );
  for ( i = 0 ; i < tr->height ; ++i )
    tr->trs[i] = Malloc( sizeof(adb_tavl_traverser_t ) );
}

/* semi-interface */
void
Adb_tr_free(adb_tr_t *tr)
{
  int i ;
  free( tr->bhs );
  free(tr->ranges);
  for ( i = 0 ; i < tr->height ; ++i )
    free(tr->trs[i]);
  free(tr->trs);
}

void 
adb_tr_free(adb_tr_t *tr)
{
  Adb_tr_free(tr);
}

adb_bh_t *
Adb_bh_dig(adb_tnx_t *tnx , adb_file_t *file, adb_rec_t *_rec , 
	     adb_tr_t *tr, adb_bh_dig_t mode)
{
  adb_bh_t *bh_tmp ;
  adb_range_t *range = (void *)EOF;
  int i ;

  switch ( file->meta->data_start_type )
    {
    case ADB_MAGIC_TREE:
      bh_tmp =  Adb_seg_load( tnx, file, file->meta->data_start, ADB_TNX_NULL ) ;
#ifdef HAVE_MADVISE
      madvise(bh_tmp->m, bh_tmp->m->size, MADV_RANDOM);
#endif /* HAVE_MADVISE */
      if ( tr )
	{
	  tr->height = 1 ;
	  Adb_tr_init(tr);
	  tr->bhs[0] = bh_tmp ;
	}
      return bh_tmp ;
      break ;
    case ADB_MAGIC_RANGE:
      for (i = 0, 
	     bh_tmp = Adb_seg_load( tnx, file, file->meta->data_start, ADB_TNX_NULL ),
#ifdef HAVE_MADVISE
	     madvise(bh_tmp->m, bh_tmp->m->size, MADV_RANDOM) 
#endif /* HAVE_MADVISE */
	     ; ; ++i )
	{
	  range = (adb_range_t *)adb_tavl_find( bh_tmp, _rec);
	  if ( tr && !i )
	    {
	      tr->height = bh_tmp->m->level + 1 ;
	       Adb_tr_init(tr);
	    }

	  if ( tr )
	    {
	      adb_range_t *rg ;
	      tr->bhs[bh_tmp->m->level] = bh_tmp ;
	      tr->ranges[bh_tmp->m->level] = range ;
	      rg = adb_tavl_t_find( tr->bhs[bh_tmp->m->level], 
			       tr->trs[bh_tmp->m->level], 
			       tr->ranges[bh_tmp->m->level]);
	      assert(rg);
	      assert(rg == range);
	    }

	  if ( ! range )
	    return NULL ; /* dup */
	  assert(range->magic == ADB_MAGIC_RANGE );

	  bh_tmp = Adb_seg_load( tnx, file, range->offset , 
				  ADB_TNX_NULL ); /* not dirty - don't know if segment 
						     is last */
#ifdef HAVE_MADVISE
	  madvise(bh_tmp->m, bh_tmp->m->size, MADV_RANDOM);
#endif /* HAVE_MADVISE */
	  if ( bh_tmp->m->content == ADB_MAGIC_TREE ) 
	    { 
	      /* if got traverser, don't mark block dirty e.g.*/
	      if ( tr != NULL )
		tr->bhs[0] = bh_tmp ;

	      return bh_tmp ;
	    }
	}
      break ;
    case 0: /* uninitialized */
      bh_tmp = Adb_seg_add( tnx, file, &file->meta->data_start ,
			ADB_MAGIC_TREE, 0 , -1LL );
      fprintf(stderr, "bh %p bh->m %p bh->m->pool %p\n",
	      bh_tmp , bh_tmp->m , (void *)bh_tmp->m + ADB_MAGIC_SIZE + bh_tmp->m->map_size /* pool */ );

      /* adb_avlt_create - this stuff does _adb_seg_init */
/*       ta->m->tree = adb_avlt_create( __data_cmp, SEG(ta->addr) , ta ); */
      Adb_tnx_set_data_start_type(file, &file->meta->data_start_type, ADB_MAGIC_TREE) ;
      if ( mode == ADB_BH_DIG_FIND )
	return NULL ;

      return bh_tmp ;
      break ;
    default:
      assert("Bug" == 0 );
      break ;
    }
  return NULL ;
}

typedef enum
  {
    TR_FIRST ,
    TR_LAST  ,
  } adb_tr_first_t ;

adb_rec_t *
Adb_tr_first(adb_tnx_t *tnx , adb_file_t *file, adb_tr_t *tr, adb_tr_first_t op)
{
  adb_bh_t *bh_tmp ;
  int i ;
  void * (*first_func)(const adb_bh_t *, adb_tavl_traverser_t *);

  switch ( op )
    {
    case TR_FIRST:
      first_func = adb_tavl_t_first ;
      break ;
    case TR_LAST:
      first_func = adb_tavl_t_last ;
      break ;
    default:
      abort();
      break ;
    }

  switch ( file->meta->data_start_type )
    {
    case ADB_MAGIC_TREE:
      bh_tmp =  Adb_seg_load( tnx, file, file->meta->data_start, ADB_TNX_NULL ) ;
      tr->height = 1 ;
      Adb_tr_init(tr);
      tr->bhs[0] = bh_tmp ;
      return first_func(bh_tmp,tr->trs[0]); ;
      break ;
    case ADB_MAGIC_RANGE:
      for (i = 0, 
	     bh_tmp = Adb_seg_load( tnx, file, file->meta->data_start, ADB_TNX_NULL ) 
	     ; ; ++i )
	{
	  adb_range_t *rg ;
	  if ( !i )
	    {
	      tr->height = bh_tmp->m->level + 1 ;
	      Adb_tr_init(tr);
	    }

	  tr->bhs[bh_tmp->m->level] = bh_tmp ;
	  rg = first_func( tr->bhs[bh_tmp->m->level], 
			   tr->trs[bh_tmp->m->level]);
	  tr->ranges[bh_tmp->m->level] = rg ;
	  assert(rg);
	  assert(rg->magic == ADB_MAGIC_RANGE );

	  /* not dirty - don't know if segment is last */
	  bh_tmp = Adb_seg_load( tnx, file, rg->offset, ADB_TNX_NULL ); 

	  if ( bh_tmp->m->content == ADB_MAGIC_TREE ) 
	    { 
	      adb_rec_t *rec ;
	      tr->bhs[0] = bh_tmp ;
	      
	      rec = first_func(bh_tmp,tr->trs[0]) ;
#if 0
	      if ( ! rec )
		{
		  adb_range_zap_empty(tnx, file, bh_tmp);
		  adb_tr_free(tr);
		  goto again ;
		}
#endif
	      assert(rec);
	      assert(rec->magic == ADB_MAGIC_RECORD);
	      return rec ;
	    }
	}
      break ;
    case 0: /* uninitialized */
      adb_errno = ADB_ERR_EMPTY_DB ;
      return NULL ;
      break ;
    default:
      assert("Bug" == 0 );
      break ;
    }
  return NULL ;
}

int
adb_tr_first(adb_tnx_t *tnx , adb_file_t *file, adb_tr_t *tr, const adb_rec_t ** const _rec)
{
  adb_rec_t *rec ;

  ADB_ERROK();

  if ( ( rec = Adb_tr_first(tnx, file, tr, TR_FIRST)))
    {
      assert( rec->magic == ADB_MAGIC_RECORD );
      *_rec = rec ;
    }
  else
    *_rec = NULL ;


  return adb_errno ;
}

int
adb_tr_last(adb_tnx_t *tnx , adb_file_t *file, adb_tr_t *tr, const adb_rec_t ** const _rec)
{
  adb_rec_t *rec ;

  ADB_ERROK();

  if ( ( rec = Adb_tr_first(tnx, file, tr, TR_LAST)))
    {
      *_rec = rec ;
      assert( rec->magic == ADB_MAGIC_RECORD );
    }
  else
    *_rec = NULL ;


  return adb_errno ;
}


size_t
Adb_vsize_get(adb_file_t *file, adb_rec_t *rec)
{
  int i , size ;
  adb_vstring_t *vs ;
  adb_wchar_t *wc ;
  adb_field_t *fld ;

  
  char *p ;
  unsigned int sb = file->db->addr ;
  for ( i = 0 , size = 0 ; i < file->meta->last_order ; ++i )
    {
      if ( (void *)file->order[i] + sb )
	{
	  fld = (void *)file->order[i] + sb ;
	  if ( fld->magic == ADB_MAGIC_FIELD )
	    {
	      switch ( fld->type )
		{
		case ADB_FIELD_VSTRING:
		  vs = (void *)rec + fld->offset ;
		  p = (void *)vs + vs->offset ;
		  assert( (strlen(p) + 1) == vs->size );
		  size += vs->size ;
		  break ;
		case ADB_FIELD_WCHAR:
		  wc = (void *)rec + fld->offset ;
		  assert( wc->magic == ADB_MAGIC_WCHAR ) ;
		  size += wc->size ;
		  break ;
		default:
		  break ;
		}
	    }
	}
    }
  return size ;
}
#if 0
static void
Adb_vstring_insert(const adb_file_t *file, adb_rec_t *dst, const adb_rec_t *src )
{
  int i , pos ;
  adb_vstring_t *svs , *dvs ;
  adb_field_t *fld ;
  unsigned int sb = file->db->addr ;
  
   for ( i = 0 , pos = file->all_data_size ; i < file->last_order ; ++i )
    {
      if ( (void *)file->order[i] + sb )
	{
	  fld = (void *)file->order[i] + sb ;
	  if ( fld->magic == ADB_MAGIC_FIELD && fld->type == ADB_FIELD_VSTRING )
	    {
	      svs = (void *)src + fld->offset ;
	      dvs = (void *)dst + fld->offset ;
	      dvs->offset = svs->offset ;
	      dvs->size = svs->size ;

	      dvs->magic = svs->magic ;
	      strcpy( (void *) dst + pos , (void *)svs + svs->offset );
	      pos += svs->size ;
	    }
	}
    }
}
#endif /* 0 */

/* #define SAFE_MALLOC_DEBUG 1 */

void *
Adb_safe_malloc(adb_tnx_t *tnx, adb_file_t *file, adb_rec_t *_rec , adb_bh_t **bhp, size_t size , int *fail)
{
  adb_bh_t *bh ;
  void *ret ;
  static int rw = 0 ;
   
  if ( fail ) *fail = 0 ;
  
  while (1)
    {
      if ( ! (bh = Adb_bh_dig( tnx, file, _rec, NULL , ADB_BH_DIG_INSERT)) )
	{
	  adb_lock_free(file, 0LL, 0LL, ADB_LOCK_FILE, ADB_LOCK_WRITE );
	  adb_errno = ADB_ERR_DUPLICITY ;
	  return NULL ;
	}

      if ( ! ( bh->state & ADB_TNX_DIRTY )) /* RO segment */
	Adb_seg_dirty(tnx, bh);
      ret = Adb_malloc( file, size, bh->m, __FUNCTION__ );
      if ( ! ret )
	{
	  if ( fail ) *fail = 1 ;

	  if (setjmp(jb))
	    {
	      fprintf(stderr,"%s() Double malloc failure\n",__FUNCTION__);
	      /* kill bogus mallocs */
	      Adb_malloc_undo();
	    }
	  else
	    {
	      Adb_meta_rw(file->db); ++rw ;
	      Adb_cs_range(tnx, bh, file);
	    }
	}
      else
	{
#ifdef SAFE_MALLOC_DEBUG
	  size_t i ;
	  unsigned char *c ;
	  for ( i = 0 , c = ret ; i < size ; ++i )
	    if ( *c++ )
	      abort();
#endif /*  SAFE_MALLOC_DEBUG */
	  *bhp = bh ;
	  if ( rw )
	    Adb_meta_ro(file->db);
	  return ret ;
	}
    }
  /* clean locks */
}

adb_rec_t *
Adb_insert(adb_tnx_t *tnx, adb_file_t *file, 
	     adb_rec_t *_rec /* malloced */)
{
  adb_bh_t *bh_tmp ;
  adb_rec_t *rec ;
  int size = Adb_vsize_get(file, _rec) + file->meta->all_data_size ;

  if ( file->meta->__parent )
    {
      adb_file_t *f = (void *) file->meta->__parent + file->db->addr ;
      return Adb_insert(tnx, f, _rec );
    }

  rec = (adb_rec_t *)Adb_safe_malloc( tnx, file, _rec, &bh_tmp, size, NULL);

  fprintf(stderr,"%% New %p size %d next %p\n",rec, size, (void *)rec + size );
  if ( rec )
    {
      Adb_check_bh( bh_tmp, __FUNCTION__, FALSE );
      Last = NULL ; Adb_verify_tree_content(tnx, file, bh_tmp, &Last );
      memcpy( rec, _rec, size ); /* copy vstring part too */

      Adb_check_bh( bh_tmp, __FUNCTION__, FALSE );

      if ( ! adb_tavl_insert( bh_tmp , &rec->node, rec)  )
	{
	  adb_bh_t *sbh ;

	  Adb_blob_insert( file, rec );
	  Adb_check_bh( bh_tmp, __FUNCTION__, FALSE );
	  Last = NULL ; Adb_verify_tree_content(tnx, file, bh_tmp, &Last );

	  if ( file->indexes->tavl_count )
	    {
	      int i ;
	      adb_file_t *fsi ;
	      for ( i = 0 ; i < file->indexes->tavl_count ; ++i )
		{

		  fsi = file->sis[i] ;

		  rec = Adb_safe_malloc( tnx, fsi, _rec, &sbh, size, NULL);
		  memcpy( rec, _rec, size );
		  if ( adb_tavl_insert( sbh , &rec->node, rec) )
		    goto L_dup ;
		  
		}
	    }
	  assert( rec->magic == ADB_MAGIC_RECORD );
	  return rec ;
	}
    }

 L_dup:      
  fprintf(stderr,"\aDuplicity found !!! "); Adb_data_show( _rec, file );
  return NULL ;
}

static int 
Adb_check_all_idx(va_list ap, adb_file_t *file)
{
  int i ;
  adb_field_t *fld ;
  unsigned int sb = file->db->addr ;
  char *name ;
  char *c ;
  wchar_t *w ;
  int j ;
  long long ll ;
  unsigned long long ull ;
  long double ld ;
  void *v ;
  unsigned int u ;
  
  int idx_found ;

  return 0 ;

  for ( idx_found = 0 ; ( name = va_arg( ap, char * )) ; )
    {
      for ( i = 0 ; i < file->meta->idx_size ; ++i )
	if ( file->idx[i] )
	  {
	    fld = (void *)file->idx[i] + sb ;
	    if ( fld->magic == ADB_MAGIC_FIELD )
	      {
		if ( ! strcmp(name, fld->name ))
		  {
		    ++idx_found ;
		    switch (fld->type)
		      {
		      case ADB_FIELD_INT: j = va_arg(ap, int ); break ;
		      case ADB_FIELD_STRING: c = va_arg(ap, char *); break ;
		      case ADB_FIELD_LONG_LONG: ll = va_arg(ap, long long); break ;
		      case ADB_FIELD_UNSIGNED_LONG_LONG: ull = va_arg(ap, long long); break ;
		      case ADB_FIELD_DOUBLE: ld = va_arg(ap,long double ); break ;
		      case ADB_FIELD_UNSIGNED_INT: u = va_arg(ap, unsigned int); break ;
		      case ADB_FIELD_BLOB: v = va_arg(ap, void *); break ;
		      case ADB_FIELD_BLOB_STRING: v = va_arg(ap, void *); break ;
		      case ADB_FIELD_VSTRING: c = va_arg(ap, char *); break ;
		      case ADB_FIELD_WCHAR: w = va_arg(ap, wchar_t *); break ;
		      default: abort(); break ;
		      }
		    break ;
		  }
		if ( ! name )
		  {
		    ADB_WARNING("Missing index field `%s'",name);
		    adb_errno = ADB_ERR_EMPTY_IDX_FIELD ;
		    return adb_errno ;
		  }
	      }
	  }
      }
  return 0 ;
}

typedef enum
  {
    ADB_FIND_FIRST = 0, /* Unimplemented !!! */
    ADB_FIND_ZERO  = 1,
  } Adb_find_t ;

/* *result = field count */

static int 
Adb_va_parse_and_set(va_list ap, adb_file_t *file , adb_rec_t **NewRec, 
		     Adb_find_t which, int *result ) 
{
  char *name ;
  adb_field_t  *fld, tmp_fld ;
  adb_rec_t *rec ;
  int vsize = 0 ;
  int f /* , need_idx , found_idx = 0 */ ;
  adb_tavl_traverser_t tr ;
  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_field_cmp, 
		   .addr = file->db->addr, 
		   .tree = file->fields } ;
/*   adb_field_t *fields[file->order_size] ; */
  int n ;

  rec = Malloc0( file->meta->all_data_size );
  rec->magic = ADB_MAGIC_RECORD ;

  for ( f = 0 , n = 0 ; ( name = va_arg( ap , char * ) ) ; ++f )
    {

      strncpy( tmp_fld.name , name , ADB_NAME_MAX -1 );

      if ( ( fld = ( adb_field_t *) adb_tavl_find ( &fbh, &tmp_fld ) ) )
	{
	  assert( fld->magic == ADB_MAGIC_FIELD );
	  assert( rec->magic == ADB_MAGIC_RECORD );
	  
	  ++n ;
	  switch ( fld->type )
	    {
	    case ADB_FIELD_INT:
	      {
		int i_data ;
		i_data = va_arg( ap , int ) ;
		/* BUG: fld->offset gets magically ZERO !!!! */
		memcpy( (void *) rec + fld->offset, &i_data ,sizeof(int)) ;
	      }
	      break ;
	    case ADB_FIELD_UNSIGNED_INT:
	      {
		unsigned int data ;
		data = va_arg(ap, unsigned int);
		memcpy( (void *) rec + fld->offset, &data ,sizeof(data)) ;
	      }
	      break ;
	    case ADB_FIELD_PTR_DIFF:
	      {
		ptrdiff_t diff ;
		diff = va_arg(ap, ptrdiff_t );
		memcpy( (void *)rec + fld->offset, &diff, sizeof(diff));
	      }
	      break ;
	    case ADB_FIELD_STRING:
	      {
		char *s_data ;
		s_data = va_arg( ap , char *);
		strncpy( (void *)rec + fld->offset, s_data , fld->len );
	      }
	      break ;
	    case ADB_FIELD_VSTRING:
	      {
		char *vs_data ;
		adb_vstring_t vs ;
		adb_rec_t *new ;

		vs_data = va_arg( ap , char *);
		vs.size = strlen(vs_data) + 1 ;
		vs.offset = file->meta->all_data_size + vsize - fld->offset ;
		vs.magic = ADB_MAGIC_VSTRING ;

		memcpy( (void *) rec + fld->offset, &vs, sizeof(vs));

		new = Malloc0( file->meta->all_data_size + vs.size + vsize ); 
		/* BUG? : new = Malloc( file->meta->all_data_size + vs.size ); */
		memcpy( new , rec, file->meta->all_data_size + vsize );
		vsize += vs.size ;

		strcpy( (void *)new + vs.offset + fld->offset, vs_data );
		free(rec);

		rec = new ;
	      }
	      break ;
	    case ADB_FIELD_WCHAR:
	      {
		adb_wchar_t *wc = (void *) rec + fld->offset ;
		wchar_t *s = va_arg(ap, wchar_t *);
		adb_rec_t *new ;
		
		wc->size = (wcslen(s) + 1) * sizeof(wchar_t) ;
		wc->offset = file->meta->all_data_size + vsize - fld->offset ;
		wc->magic = ADB_MAGIC_WCHAR ;

		new = Malloc0( file->meta->all_data_size + wc->size + vsize );
		memcpy( new , rec, file->meta->all_data_size + vsize );
		vsize += wc->size ;

		memcpy( (void *)new + wc->offset + fld->offset, s , wc->size);
		free(rec);

		rec = new ;
	      }
	      break ;
	    case ADB_FIELD_LINK:
	    case ADB_FIELD_CONTAINER:
	      {
		adb_link_t *c ;
/* 		size_t size = va_arg(ap, int ); */

		c = (void *)rec + fld->offset ;
		c->magic = ADB_MAGIC_LINK ;
		c->count = 0 ;
	      }
	      break ;
	    case ADB_FIELD_LONG_LONG:
	      {
		long long ll_data ;
		ll_data = va_arg( ap , long long );
		memcpy( (void *) rec + fld->offset, &ll_data ,sizeof(long long)) ;
	      }
	      break ;
	    case ADB_FIELD_UNSIGNED_LONG_LONG:
	      {
		long long ull_data ;
		ull_data = va_arg( ap , unsigned long long );
		memcpy( (void *) rec + fld->offset, &ull_data ,sizeof ull_data ) ;
	      }
	      break ;
	    case ADB_FIELD_DOUBLE:
	      {
		long double d_data ;
		d_data = va_arg( ap, long double );
		memcpy( (void *) rec + fld->offset, &d_data ,sizeof(long double)) ;
	      }
	      break ;
	    case ADB_FIELD_BLOB:
	    case ADB_FIELD_BLOB_STRING:
	      {
		adb_blob_t *blob ;

		blob = va_arg( ap, void *);
		assert ( blob->magic == ADB_MAGIC_BLOB ||  
			 blob->magic == ADB_FIELD_BLOB_STRING );

		memcpy( (void *) rec + fld->offset, blob, sizeof(adb_blob_t ));
	      }
	      break ;
	    default:
	      va_end(ap);
	      assert("FIXME" == 0);
	      break ;
	    }
	}
      else
	{
/* 	  adb_tavl_walk ( file->fields_ta, _adb_show_field , NULL ); */
	  g_warning("Can't find such beast (or field) `%s'", name);
 	  va_end(ap);
	  return ADB_ERR_NOT_FIELD ;
	}
    }
  va_end (ap);

  for ( fld = ( adb_field_t *) adb_tavl_t_first( &fbh, &tr ) ;
	fld ;
	fld = ( adb_field_t *) adb_tavl_t_next( &fbh, &tr ))
    {
      switch( fld->type )
	{
	case ADB_FIELD_LINK:
	case ADB_FIELD_CONTAINER:
	  {
	    adb_link_t *c ;

	    c = (void *)rec + fld->offset ;
	    c->magic = ADB_MAGIC_LINK ;
	    c->count = 0 ;
	    c->id = 0ULL ;
	  }	  
	  break ;
	default:
	  break ;
	}
    }
  
  *NewRec = rec ;
  *result = n ;

  return 0 ;

}

int /* adb_rec_t * */
Adb_va_parse(va_list ap, adb_file_t *file , adb_rec_t **_rec) 
{
  char *name ;
  adb_field_t  *fld, tmp_fld ;
  adb_rec_t *rec ;
  int vsize = 0 ;
  int f /* , need_idx , found_idx = 0 */ ;
  adb_tavl_traverser_t tr ;
  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_field_cmp, 
		   .addr = file->db->addr, 
		   .tree = file->fields } ;

/*   adb_field_t *fields[file->order_size] ; */

  rec = Malloc( file->meta->all_data_size );
  rec->magic = ADB_MAGIC_RECORD ;

  for ( f = 0 ; ( name = va_arg( ap , char * ) ) ; ++f )
    {

      strncpy( tmp_fld.name , name , ADB_NAME_MAX -1 );

      if ( ( fld = ( adb_field_t *) adb_tavl_find ( &fbh, &tmp_fld ) ) )
	{
	  assert( fld->magic == ADB_MAGIC_FIELD );
	  assert( rec->magic == ADB_MAGIC_RECORD );

	  switch ( fld->type )
	    {
	    case ADB_FIELD_INT:
	      {
		int i_data ;
		i_data = va_arg( ap , int ) ;
		/* BUG: fld->offset gets magically ZERO !!!! */
		memcpy( (void *) rec + fld->offset, &i_data ,sizeof(int)) ;
	      }
	      break ;
	    case ADB_FIELD_UNSIGNED_INT:
	      {
		unsigned int u ;
		u = va_arg( ap, unsigned int );
		memcpy( (void *) rec + fld->offset, &u ,sizeof (unsigned int)) ;
	      }
	      break ;
	    case ADB_FIELD_PTR_DIFF:
	      {
		__PTRDIFF_TYPE__ p ;
		p = va_arg( ap, __PTRDIFF_TYPE__ );
		memcpy( (void *) rec + fld->offset, &p ,sizeof p) ;
	      }
	      break ;
	    case ADB_FIELD_STRING:
	      {
		char *s_data ;
		s_data = va_arg( ap , char *);
		strncpy( (void *)rec + fld->offset, s_data , fld->len );
	      }
	      break ;
	    case ADB_FIELD_VSTRING:
	      {
		char *vs_data ;
		adb_vstring_t vs ;
		adb_rec_t *new ;

		vs_data = va_arg( ap , char *);
		vs.size = strlen(vs_data) + 1 ;
		vs.offset = file->meta->all_data_size + vsize - fld->offset ;
		vs.magic = ADB_MAGIC_VSTRING ;

		memcpy( (void *) rec + fld->offset, &vs, sizeof(vs));

		new = Malloc( file->meta->all_data_size + vs.size + vsize ); 
		/* BUG? : new = Malloc( file->meta->all_data_size + vs.size ); */
		memcpy( new , rec, file->meta->all_data_size + vsize );
		vsize += vs.size ;

		strcpy( (void *)new + vs.offset + fld->offset, vs_data );
		free(rec);

		rec = new ;
	      }
	      break ;
	    case ADB_FIELD_WCHAR:
	      {
		adb_wchar_t *wc = (void *) rec + fld->offset ;
		wchar_t *s = va_arg(ap, wchar_t *);
		adb_rec_t *new ;
		
		wc->size = (wcslen(s) + 1) * sizeof(wchar_t) ;
		wc->offset = file->meta->all_data_size + vsize - fld->offset ;
		wc->magic = ADB_MAGIC_WCHAR ;

		new = Malloc( file->meta->all_data_size + wc->size + vsize );
		memcpy( new , rec, file->meta->all_data_size + vsize );
		vsize += wc->size ;

		memcpy( (void *)new + wc->offset + fld->offset, s , wc->size);
		free(rec);

		rec = new ;
	      }
	      break ;
	    case ADB_FIELD_LINK:
	    case ADB_FIELD_CONTAINER:
	      {
		adb_link_t *c ;
/* 		size_t size = va_arg(ap, int ); */

		c = (void *)rec + fld->offset ;
		c->magic = ADB_MAGIC_LINK ;
		c->count = 0 ;
	      }
	      break ;
	    case ADB_FIELD_LONG_LONG:
	      {
		long long ll_data ;
		ll_data = va_arg( ap , long long );
		memcpy( (void *) rec + fld->offset, &ll_data ,sizeof(long long)) ;
	      }
	      break ;
	    case ADB_FIELD_UNSIGNED_LONG_LONG:
	      {
		long long ull_data ;
		ull_data = va_arg( ap , unsigned long long );
		memcpy( (void *) rec + fld->offset, &ull_data ,sizeof ull_data ) ;
	      }
	      break ;
	    case ADB_FIELD_DOUBLE:
	      {
		long double d_data ;
		d_data = va_arg( ap, long double );
		memcpy( (void *) rec + fld->offset, &d_data ,sizeof(long double)) ;
	      }
	      break ;
	    case ADB_FIELD_BLOB:
	    case ADB_FIELD_BLOB_STRING:
	      {
		adb_blob_t *blob ;

		blob = va_arg( ap, void *);
		assert ( blob->magic == ADB_MAGIC_BLOB ||  
			 blob->magic == ADB_FIELD_BLOB_STRING );

		memcpy( (void *) rec + fld->offset, blob, sizeof(adb_blob_t ));
	      }
	      break ;
	    default:
	      va_end(ap);
	      assert("FIXME" == 0);
	      break ;
	    }
	}
      else
	{
/* 	  adb_tavl_walk ( file->fields_ta, _adb_show_field , NULL ); */
	  g_warning("Can't find such beast (or field) `%s'", name);
 	  va_end(ap);
	  return ADB_ERR_NOT_FIELD ;
	}
    }
  va_end (ap);

  for ( fld = ( adb_field_t *) adb_tavl_t_first( &fbh, &tr ) ;
	fld ;
	fld = ( adb_field_t *) adb_tavl_t_next( &fbh, &tr ))
    {
      switch( fld->type )
	{
	case ADB_FIELD_LINK:
	case ADB_FIELD_CONTAINER:
	  {
	    adb_link_t *c ;

	    c = (void *)rec + fld->offset ;
	    c->magic = ADB_MAGIC_LINK ;
	    c->count = 0 ;
	    c->id = 0ULL ;
	  }	  
	  break ;
	default:
	  break ;
	}
    }
  
  *_rec = rec ;
  return 0 ;

#if 0
  for ( i = 0 ; i < file->idx_size ; ++i )
    {
      adb_field_t *ff = (void *)file->idx[i] + sb ;
      if ( file->idx[i] && ff->magic == ADB_MAGIC_FIELD )
	{
	  int j ;
	  for ( j = 0 ; j <= f ; ++j )
	    if ( fields[j] == (void *)file->idx[i] + sb )
	      goto L_ok ;
	  ADB_ERROR("Missing index field '%s'\n", file->idx[i]->name);
	L_ok:
	}
    }

  adb_errno = ADB_ERR_EMPTY_IDX_FIELD ;
  return adb_errno ;
#endif /* 0 */
}


inline adb_bh_t *
Adb_init_bh(adb_tnx_t *tnx,adb_bh_t *bh, adb_file_t *file, adb_tnx_state_t state)
{
 if ( bh )
    assert( bh->magic == ADB_MAGIC_BH ) ;
  else
    bh = Adb_file_activate(tnx, file , state );
 return bh ;
}


void
Adb_data_walk(adb_bh_t *bh, void *a, void *param UNUSED )
{
  adb_rec_t *r = (adb_rec_t *) a ;

  assert( r->magic == ADB_MAGIC_RECORD );
  
  Adb_data_show( r, bh->file );
}

void
Adb_range_walk(adb_tnx_t *tnx, adb_bh_t *rbh, adb_file_t *file, 
		 adb_tavl_walk_func_t func,  void *param)
{
  adb_tavl_traverser_t tr ;
  adb_bh_t *bh ;
  adb_range_t *r ;

  assert(rbh->m->content == ADB_MAGIC_RANGE );
  
  adb_tavl_t_init(rbh, &tr);
  for ( r = adb_tavl_t_first(rbh, &tr) ; r ; r = adb_tavl_t_next(rbh, &tr))
    {
      bh = Adb_seg_load(tnx, file, r->offset, ADB_TNX_NULL );
/*       madvise(bh->m, bh->m->size, MADV_WILLNEED); */

      assert ( r->magic == ADB_MAGIC_RANGE );
      assert( bh->m->content == r->data_start_type );

      switch ( r->data_start_type )
	{
	case ADB_MAGIC_TREE:
	  if ( Adb_debug )
	    { fprintf(stderr,"^^@ Begin ");  Adb_show_one_range( r, rbh, NULL ); }
	  adb_tavl_walk( bh, func, param );
	  if ( Adb_debug )
	    { fprintf(stderr,"^^@ End   ");  Adb_show_one_range( r, rbh, NULL ); }
	  break ;
	case ADB_MAGIC_RANGE: /* here only if range->level > 0 */
	  if ( Adb_debug )
	    { fprintf(stderr,"^^^ Begin "); Adb_show_one_range( r, rbh, NULL ); }
	  Adb_range_walk( tnx, bh, file, func, param );
	  if ( Adb_debug )
	    { fprintf(stderr,"^^^ End   "); Adb_show_one_range( r, rbh, NULL ); }
	  break ;
	default:
	  assert("What a hell ?" == 0 );
	  break ;
	}
      Adb_seg_unload(tnx, bh , TRUE );
    }
}



void
adb_walk(adb_tnx_t *tnx, adb_file_t *file, void (*_func)() , void *param)
{
  /* load top level block */
  adb_bh_t *top_bh = Adb_init_bh(tnx, NULL ,file, ADB_TNX_NULL ); 
  void (*func)() = _func ? _func : Adb_data_walk ;
  ADB_ERROK();

  switch ( file->meta->data_start_type )
    {
    case ADB_MAGIC_TREE:
      adb_tavl_walk( top_bh, func, param );
      break ;
    case ADB_MAGIC_RANGE:
      Adb_range_walk( tnx, top_bh, file, func, param );
      break ;
    case 0:
      return ;
    default:
      assert("What a hell ?" == 0 );
      break ;
    }
   Adb_seg_unload(tnx, top_bh, TRUE );
}


int
adb_find_by_name(adb_tnx_t *tnx, adb_file_t *file, adb_tr_t *tr , 
		 const adb_rec_t ** const _rec, int exact, ... )
{
  va_list ap ;
  adb_rec_t *rec , *result ;
  adb_bh_t *bh ;

  ADB_ERROK();

  assert(file->magic == ADB_MAGIC_FILE );
  assert(tnx->magic == ADB_MAGIC_TNX );

  va_start(ap, exact);
  if ( Adb_check_all_idx(ap, file))
    return adb_errno ;

  va_start(ap, exact);

  /* this is strange but it makes sense, in rec are stored &of_user_vars */
  if ( Adb_va_parse( ap, file, &rec ))
    {
      va_end(ap);
      free(rec);
      return adb_errno ;
    }
       

  va_end(ap);

  bh = Adb_bh_dig(tnx, file, rec , tr , ADB_BH_DIG_FIND );
  if ( bh ) /* non empty file */
    {
      if ( exact )
	result = adb_tavl_t_find(bh, tr->trs[0], rec );
      else
	result = adb_tavl_t_find_close(bh, tr->trs[0], rec );
      
      if ( result )
	*_rec = result ;
      else
	{
	  adb_errno = ADB_ERR_NO_RECORD ;
	  Adb_tr_free(tr);
	}
    }
  else /* empty file */
    adb_errno = ADB_ERR_NO_RECORD ;

  return adb_errno ;
}

/*
 * Expects ` "field-name", field-value ' pairs terminated by NULL 
 */
int
adb_find_first_where(adb_tnx_t *tnx, adb_file_t *file, adb_tr_t *tr , 
	       const adb_rec_t ** const recp, ... )
{
  va_list ap ;
  adb_rec_t *rec /* , *result */ ;
  adb_rec_t *result ;
  adb_bh_t *bh ;
  int n ;

  ADB_ERROK();

  assert(file->magic == ADB_MAGIC_FILE );
  assert(tnx->magic == ADB_MAGIC_TNX );

  va_start(ap, recp);


  /* this is strange but it makes sense, in rec are stored &of_user_vars */
  if ( Adb_va_parse_and_set(ap, file , &rec, ADB_FIND_ZERO, &n ))
    {
      va_end(ap);
      free(rec);
      return adb_errno ;
    }
       

  va_end(ap);

  bh = Adb_bh_dig(tnx, file, rec , tr , ADB_BH_DIG_FIND );
  if ( bh ) /* non empty file */
    { /* inexact one */
      int r ;
      result = adb_tavl_t_find_close(bh, tr->trs[0], rec );

      if ( result )
	{
	  r = Adb_data_cmp(result, rec, bh, n) ;
	  fprintf(stderr, "%s: -----------------------------------------------------\n"
		  "%s: find_close %d == Adb_data_cmp()\n", __func__, __func__,
		  r);
	  fprintf(stderr,"%s: zero record : ", __func__ );
	  Adb_data_show(rec, file);
	  fprintf(stderr,"%s: first record : ", __func__);
	  Adb_data_show(result, file);
	  
	  switch ( r )
	    {
	    case 1: /* result > rec */
	      adb_errno = ADB_ERR_NO_RECORD ; return adb_errno ;
	      break ;
	    case -1: /* result < rec */
	      {
		const adb_rec_t *next ;
		if ( adb_next(tnx, file, tr, &next ) )
		  {
		    fprintf(stderr, "%s: Unsuccessful move to next\n", __func__ );
		    adb_errno = ADB_ERR_NO_RECORD ;
		    Adb_tr_free(tr);
		  }
		else
		  {
		    fprintf(stderr,"%s: next record : ", __func__);
		    r = Adb_data_cmp(next, rec, bh, n) ;
		    fprintf(stderr, "%s: -----------------------------------------------------\n"
			    "%s: find_close %d == Adb_data_cmp()\n", __func__, __func__,
			    r);
		    fprintf(stderr,"%s: zero record : ", __func__ );
		    Adb_data_show(rec, file);
		    fprintf(stderr,"%s: first record : ", __func__);
		    Adb_data_show(next, file);
		    
		    if ( ! r )
		      {
			*recp = next ;
/* 			Adb_data_show(next, file); */
			return adb_errno ;
		      }
		    else
		      {
			abort(); /* tmp */
			adb_errno = ADB_ERR_NO_RECORD ; return adb_errno ;
		      }
		  }
	      }
	      break ;
	    case 0:
	      *recp = result ;
	      return adb_errno ;
	    default:
	      abort();
	      break ;
	    }
	} /* inexact */
      else
	Adb_tr_free(tr);
    }

  adb_errno = ADB_ERR_NO_RECORD ;
  return adb_errno ;
}

typedef enum
  {
    NEXT = 1 ,
    PREV = 0 ,
  } adb_next_t ;

static int
Adb_next(adb_tnx_t *tnx, adb_file_t *file, adb_tr_t *tr , 
	   const adb_rec_t ** const _rec, adb_next_t op )
{
  adb_rec_t *rec ;
  void * (*next_func)(const adb_bh_t *, adb_tavl_traverser_t *);
  void * (*first_func)(const adb_bh_t *, adb_tavl_traverser_t *);
  switch ( op )
    {
    case NEXT:
      next_func = adb_tavl_t_next ;
      first_func = adb_tavl_t_first ;
      break ;
    case PREV:
      next_func = adb_tavl_t_prev ;
      first_func = adb_tavl_t_last ;
      break ;
    default:
      abort();
    }

  if ( ( rec = /* adb_tavl_t_next */ next_func(tr->bhs[0], tr->trs[0])) != NULL )
    *_rec = rec ;
  else
    {
      adb_range_t *next ;
      int i ;
      for ( i = 0 ; ; ++i )
	{
	  Adb_seg_unload(tnx, tr->bhs[i], TRUE);
	  /* traverser at higher is initialized by Adb_bhg_dig */
/* 	  range = adb_tavl_t_find( tr->bhs[i+1], tr->trs[i+1], tr->ranges[i+1]); */
/* 	  assert(range); */

	  /* load upper block to be sure */
	  Adb_seg_load(tnx, file, tr->bhs[i+1]->offset, ADB_TNX_NULL);
	  next = next_func( tr->bhs[i+1], tr->trs[i+1]);
	  if ( next )
	    break ;

	  if ( tr->bhs[i+1]->m->parent == EOF ) /* top level */
	    {
	     adb_errno = ADB_ERR_NO_RECORD ;
	     return adb_errno ;
	    }
	}

      for ( ; i >= 0 ; --i )
	{
	  tr->bhs[i] = Adb_seg_load(tnx, file, next->offset, ADB_TNX_NULL);
	  if ( i )
	    /* initialize traverser */
	    {
	      next = tr->ranges[i] = first_func(tr->bhs[i], tr->trs[i]);
	    }
	  else
	    {
	      assert(tr->bhs[i]->m->content == ADB_MAGIC_TREE );
	      rec = first_func(tr->bhs[i], tr->trs[i]);
	      assert( rec->magic == ADB_MAGIC_RECORD );
	      *_rec = rec ;
	      return adb_errno ;
	    }
	}
    }
  return adb_errno ;
}

int
adb_next(adb_tnx_t *tnx, adb_file_t *file, adb_tr_t *tr , const adb_rec_t ** const _rec)
{
  return Adb_next(tnx, file, tr, _rec, NEXT );
}

int
adb_prev(adb_tnx_t *tnx, adb_file_t *file, adb_tr_t *tr , const adb_rec_t ** const _rec)
{
  return Adb_next(tnx, file, tr, _rec, PREV );
}

adb_rec_t *
adb_insert_by_name(adb_tnx_t *tnx, adb_file_t *file,  ... )
{
  va_list ap;
  adb_rec_t *rec , *rv ;

  va_start(ap, file);
  if ( Adb_check_all_idx( ap, file))
    return NULL ;


  va_start (ap, file);         /* Initialize the argument list. */
  if ( Adb_va_parse( ap, /* tnx, ta, */ file, &rec))
    {
      va_end(ap);
      free(rec);
      return NULL ;
    }

  va_end (ap);                  /* Clean up. */

  rv = Adb_insert( tnx, file, rec );

  free( rec );

  Adb_tnx_clean(tnx, file);

  return rv ;
}

adb_rec_t * 
adb_insert(adb_tnx_t *tnx, adb_file_t *file, adb_rec_t *rec )
{
  adb_rec_t *rv ;
  rv = Adb_insert( tnx, file, rec );
  free( rec );

  Adb_tnx_clean(tnx, file);

  return rv ;
}

int
adb_delete(adb_tnx_t *tnx, adb_file_t *file, const adb_rec_t * const rec )
{
  adb_bh_t *bh , *sbh ;
  size_t size ;
  const void *to_free , *primary_to_free ;

  ADB_ERROK();

  assert(file->magic == ADB_MAGIC_FILE );
  assert(tnx->magic == ADB_MAGIC_TNX );

  if ( file->meta->__parent )
    {
      adb_file_t *f = (void *) file->meta->__parent + file->db->addr ;
      return adb_delete(tnx, f, rec );
    }

  bh = Adb_bh_dig(tnx, file, (adb_rec_t *)rec , NULL , ADB_BH_DIG_FIND);
  
  Adb_seg_dirty(tnx, bh); /* RW */

  if ( (primary_to_free = adb_tavl_delete(bh, (adb_rec_t *)rec )) )
    {
      size = Adb_vsize_get( file, (void *)primary_to_free) 
	+ file->meta->all_data_size ;

      /* empty blocks considered harmfull */
      adb_range_zap_empty(tnx, file, bh);

      if ( file->indexes->tavl_count )
	{
	  int i ; 
	  adb_file_t *fsi ;
	  for ( i = 0 ; i < file->indexes->tavl_count ; ++i )
	    {
	      fsi = file->sis[i] ;

	      sbh = Adb_bh_dig(tnx, fsi, (adb_rec_t *)rec, NULL, ADB_BH_DIG_FIND);
	      Adb_seg_dirty(tnx, sbh);
	      to_free = adb_tavl_delete(sbh, (adb_rec_t *)rec );
	      if ( ! to_free )
		ADB_ERROR("Can't delete record by *secondary* index `%s'",fsi->meta->name);
	      Adb_free(file, (void *)to_free, size, sbh->m, __FUNCTION__ );
	      adb_range_zap_empty(tnx, fsi, sbh);

	    }
	}
      Adb_free(file, (void *)primary_to_free, size, bh->m, __FUNCTION__ );

    }
  else
    {
      adb_errno = ADB_ERR_NO_RECORD ;
    }

  Adb_tnx_clean(tnx, file);

  return adb_errno ;
  
}

int
adb_fld_get(adb_file_t *file, const adb_rec_t * const rec, ... )
{
  va_list ap;
  char *name ;
  adb_field_t *fld, tmp_fld ;
  int f ;
  adb_field_t *fields[file->meta->order_size] ;
  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_field_cmp, 
		   .addr = file->db->addr, 
		   .tree = file->fields } ;

  va_start (ap, rec); 

  ADB_ERROK();

  if ( rec->magic != ADB_MAGIC_RECORD )
    {
      ADB_ERROR("Bad magic 0x%x\n",rec->magic );
      adb_errno = ADB_ERR_INVALID_DATA ;
      return adb_errno ;
    }
  
  for ( f = 0 ; ( name = va_arg( ap , char * ) ) ; ++f )
    {

      strncpy( tmp_fld.name , name , ADB_NAME_MAX - 1 );

      if ( ( fld = ( adb_field_t *) adb_tavl_find ( &fbh, &tmp_fld ) ) )
	{
	  assert( rec->magic == ADB_MAGIC_RECORD );

	  fields[f] = fld ;

	  switch ( fld->type )
	    {
	    case ADB_FIELD_INT:
	      {
		int *i_data ;
		i_data = va_arg( ap , int *) ;
		*i_data = *(int *)((void *) rec + fld->offset) ;
	      }
	      break ;
	    case ADB_FIELD_UNSIGNED_INT:
	      {
		unsigned int *u ;
		u = va_arg( ap, unsigned int *);
		*u = *(unsigned int *) ((void *) rec + fld->offset) ;
	      }
	      break ;
	    case ADB_FIELD_STRING:
	      {
		char **s_data ;
		s_data = va_arg( ap , char **);
		if ( *s_data == NULL )
		  *s_data = Malloc(fld->len);
		else
		  *s_data = realloc(*s_data,fld->len);

		strncpy( *s_data, (void *)rec + fld->offset, fld->len );
	      }
	      break ;
	    case ADB_FIELD_VSTRING:
	      {
		adb_vstring_t *vs ;
		char **s ;

		s = va_arg( ap , char **);
		vs = (void *) rec + fld->offset ;
		assert(vs->magic == ADB_MAGIC_VSTRING );

		if ( *s == NULL )
		  *s = Malloc(vs->size);
		else
		  *s = realloc(*s,vs->size);
		
		strncpy( *s, (void *)vs + vs->offset , vs->size);
	      }
	      break ;
	    case ADB_FIELD_WCHAR:
	      {
		adb_wchar_t *wc ;
		wchar_t **w ;

		w = va_arg( ap, wchar_t **);
		wc = (void *) rec + fld->offset ;
		assert( wc->magic == ADB_MAGIC_WCHAR);

		if ( *w == NULL )
		  *w = Malloc(wc->size);
		else
		  *w = realloc(*w, wc->size );

		memcpy(*w, (void *)rec + wc->offset + fld->offset, wc->size );
	      }
	      break ;
	    case ADB_FIELD_LINK:
	    case ADB_FIELD_CONTAINER:
	      {
		adb_link_t *l , **lp ;
		lp = va_arg(ap, adb_link_t ** );
		l = (void *)rec + fld->offset ;
		assert( l->magic == ADB_MAGIC_LINK );

		if ( *lp == NULL )
		  *lp = Malloc(sizeof(*l));
		else
		  *lp = realloc(*lp, sizeof(*l));

		memcpy( *lp, l , sizeof(*l));
	      }
	      break ;
	    case ADB_FIELD_LONG_LONG:
	      {
		long long *ll_data ;
		ll_data = va_arg( ap , long long *);
		memcpy( ll_data, (void *) rec + fld->offset, sizeof(long long)) ;
	      }
	      break ;
	    case ADB_FIELD_UNSIGNED_LONG_LONG:
	      {
		unsigned long long *ull_data ;
		ull_data = va_arg( ap , unsigned long long *);
		memcpy( ull_data, (void *) rec + fld->offset, sizeof ull_data ) ;
	      }
	      break ;
	    case ADB_FIELD_DOUBLE:
	      {
		long double *d_data ;
		d_data = va_arg( ap, long double *);
		memcpy( d_data, (void *) rec + fld->offset, sizeof(long double)) ;
	      }
	      break ;
	    case ADB_FIELD_BLOB:
	    case ADB_FIELD_BLOB_STRING:
	      {
		char **bs ;
		adb_blob_t *blob ;
		int ret ;

		bs = va_arg( ap, char **);

		blob = (void *) rec + fld->offset ;

		assert ( blob->magic == ADB_MAGIC_BLOB ||  
			 blob->magic == ADB_FIELD_BLOB_STRING );

		if ( *bs == NULL )
		  *bs = Malloc(blob->size);
		else
		  *bs = realloc(*bs,blob->size);

		ret = pread( file->blob_fd, *bs, blob->size , blob->offset);
		assert( ret != EOF );
	      }
	      break ;
	    default:
	      va_end(ap);
	      assert("FIXME" == 0);
	      break ;
	    }
	}
      else
	{
	  adb_tavl_walk ( &fbh, Adb_show_field , NULL );
	  g_warning("Can't find such beast (or field) `%s'", name);
 	  va_end(ap);
	  return ADB_ERR_NOT_FIELD ;
	}
    }
  va_end (ap);

  return adb_errno ;
}
