/* 
*  This file is part of BCC.
*
*  BCCsuite is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCCsuite is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "IO.h"

void IO_open_read(bcc_status* status, void** f, const char *name, const bcc_boolean from_frame)
{
  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(*f==NULL,status,IO_ENNULL,IO_MSGENNULL);

  if (from_frame==BCC_TRUE)
#ifdef HAVE_LIBFRAME
      TRY(IO_open_frame(status, (FrFile**) f, name),status);
#else
      ABORT(status,IO_EFRLIB,IO_MSGEFRLIB);
#endif // HAVE_LIBFRAME
  else
    TRY(IO_open_ascii(status, (FILE**) f, name),status);

  RETURN(status);
}

void IO_close_read(bcc_status* status, void** f, const bcc_boolean from_frame)
{

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);

  if (from_frame==BCC_TRUE)
#ifdef HAVE_LIBFRAME
      TRY(IO_close_frame(status, (FrFile **) f),status);
#else
      ABORT(status,IO_EFRLIB,IO_MSGEFRLIB);
#endif // HAVE_LIBFRAME
  else
    TRY(IO_close_ascii(status, (FILE **) f),status);
  
  RETURN(status);
}

void IO_read(bcc_status* status, int *stat, time_series *out, const void *f, const char* channel, const double epoch, const int N, const bcc_boolean from_frame)
{
  CHECKSTATUSPTR(status);

  ASSERT(stat,status,IO_ENULL,IO_MSGENULL);
  ASSERT(out,status,IO_ENULL,IO_MSGENULL);
  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(channel,status,IO_ENULL,IO_MSGENULL);
  ASSERT(N,status,IO_ENSMPL,IO_MSGENSMPL);

  if (from_frame==BCC_TRUE)
#ifdef HAVE_LIBFRAME
#ifdef HAVE_LIBFRV
    TRY(IO_read_frame(status,stat,out,(FrFile *) f,channel,epoch,N),status);
#else
      ABORT(status,IO_EFRVLIB,IO_MSGEFRVLIB);
#endif // HAVE_LIBFRV
#else
      ABORT(status,IO_EFRLIB,IO_MSGEFRLIB);
#endif // HAVE_LIBFRAME
  else
    TRY(IO_read_ascii(status,stat,out,(FILE *) f,channel,epoch,N),status);

  RETURN(status);
}

void IO_info(bcc_status* status, double *fs, const void *f, const char* channel, const double epoch, const bcc_boolean from_frame)
{
  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(channel,status,IO_ENULL,IO_MSGENULL);

  if (from_frame==BCC_TRUE)
#ifdef HAVE_LIBFRAME
    TRY(IO_info_frame(status,fs,(FrFile *) f,channel,epoch),status);
#else
    ABORT(status,IO_EFRLIB,IO_MSGEFRLIB);
#endif // HAVE_LIBFRAME
  else
    TRY(IO_info_ascii(status,fs,(FILE *) f,channel,epoch),status);

  RETURN(status);
}

void IO_create_dir(bcc_status* status, const char *dir)
{
  char infostr[BCCInfoLength];
  int stat;

  CHECKSTATUSPTR(status);

  stat = access ( dir, W_OK );
  if (stat !=0)
    {
      sprintf(infostr,"Create dir %s\n", dir);
      bcc_info(status,infostr);

      stat=mkdir(dir,0x0fff);
      if (stat<0)
        {ABORT(status,IO_EDIR, IO_MSGEDIR);}
    }

  RETURN(status);
}
