/* 
*  This file is part of BCC.
*
*  BCCsuite is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCCsuite is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "IO.h"

#ifdef HAVE_LIBFRAME

static FrameH* read_one_frame(FrFile* f, long int t)
{
  if (f==NULL)
    return(NULL);

  return FrameHReadT(f,t);
}

static FrVect* read_one_vector(FrFile* f, const double t, const char *channel)
{
  FrVect *orig, *copy;
  FrameH *frame;
  char tmp_channel [BCCStringLength];

  if (f==NULL)
    return(NULL);

  if (channel==NULL)
    return (NULL);

  sprintf(tmp_channel,"%s",channel);

  frame=FrameReadTChnl(f,t,tmp_channel);

  if (frame==NULL)
    return(NULL);

  orig=FrameFindVect(frame,tmp_channel);
  copy=FrVectCopy(orig);
  FrameFree(frame);

  if ((copy->type != FR_VECT_8R)&&(copy->type != FR_VECT_4R))
    {
      FrVectFree(copy);
      return(NULL);
    }

  return(copy);
}

static double next_gps(FrFile *f, double t)
{
  int pos;

  if (f==NULL)
    return (-1);

  pos=FrTOCFrameFindNext(f,t);
  if (pos<0)
    return(-1);
  
  if (f->toc==NULL)
    FrTOCReadFull(f);
  if (f->toc==NULL)
    return(-1);
  
  t=f->toc->GTimeS[pos] + 1.e-9 * f->toc->GTimeN[pos];

  return (t);
}

void IO_open_frame(bcc_status* status, FrFile** f, const char* name)
{
  char infostr[BCCInfoLength];
  char tmp_name[BCCStringLength];
  double start, end;

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(*f==NULL,status,IO_ENNULL,IO_MSGENNULL);  
  ASSERT(name,status,IO_ENULL,IO_MSGENULL);
  
  sprintf(tmp_name,"%s",name);
  *f=FrFileINew(tmp_name);
  if (*f==NULL)
    {ABORT(status,IO_EOPENF,IO_MSGEOPENF);}

  sprintf(infostr,"Open frame file %s",name);
  bcc_log(status,infostr);

  start=FrFileITStart(*f);
  sprintf(infostr,"Starting time is %ld",(unsigned long int) start);
  bcc_log(status,infostr);
  
  end=FrFileITEnd(*f);  
  sprintf(infostr,"Ending time is %ld",(unsigned long int) end);
  bcc_log(status,infostr);
    
  sprintf(infostr,"Duration is %ld",(unsigned long int) (end-start));
  bcc_log(status,infostr);
    
  RETURN(status);
}

void IO_close_frame(bcc_status* status, FrFile **f)
{
  char infostr[BCCInfoLength];

  CHECKSTATUSPTR(status);
  
  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(*f,status,IO_ENULL,IO_MSGENULL);
  
  sprintf(infostr,"Close frame file");
  bcc_log(status,infostr);
  
  /* Close the file. */
  FrFileIEnd (*f);
  *f=NULL;

  RETURN(status);
}

#ifdef HAVE_LIBFRV

void IO_read_frame(bcc_status* status, int *stat, time_series *out, FrFile *f, const char* channel, const double epoch, const int N)
{
  char infostr[BCCInfoLength];
  long int GPS;
  double fs,relic,frame_duration,samples_per_frame,start,end,t;
  int i,offset=0;
  FrvBuf *buffer=NULL;
  FrVect *vec=NULL;
  FrameH *frame=NULL;

  CHECKSTATUSPTR(status);

  ASSERT(out,status,IO_ENULL,IO_MSGENULL);
  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(channel,status,IO_ENULL,IO_MSGENULL);
  ASSERT(N>0,status,IO_ENSMPL,IO_MSGENSMPL);
  
  if (N>out->vec->length)
    {ABORT(status,IO_ENSMPL,IO_MSGENSMPL);}

  /* check that we seek data in the available time range */
  start=FrFileITStart(f);
  end=FrFileITEnd(f);  
  if ((epoch<start)|(epoch>end))
    {
      sprintf(infostr,"Requested epoch=%ld not in available range from %ld to %ld",(unsigned long int) epoch,(unsigned long int) start,(unsigned long int) end);
      bcc_warning(status,infostr);
      *stat=IO_FAIL;
      RETURN(status);
    }

  /* get the frame at requested epoch */
  GPS=(unsigned long int)epoch;
  frame=read_one_frame(f,GPS);
  if (frame==NULL)
    {
      sprintf(infostr,"Missing frame at GPS=%ld",GPS);
      bcc_warning(status,infostr);
      stat=IO_FAIL;
      RETURN(status);
    }

  frame_duration=frame->dt;
  GPS=frame->GTimeS;
  if (GPS==0)
    {
      sprintf(infostr,"Missing time information at GPS=%ld",GPS);
      bcc_warning(status,infostr);
      stat=IO_FAIL;
      RETURN(status);
    }

  FrameFree(frame);
  
  /* get the vector at requested epoch */
   vec=read_one_vector(f,GPS,channel);
  // vec=FrFileIGetV(f,channel,epoch,fs*N);
  if (vec==NULL)
    {
      sprintf(infostr,"Missing channel %s at GPS=%ld",channel,GPS);
      bcc_warning(status,infostr);
      stat=IO_FAIL;
      RETURN(status);
    }

  samples_per_frame=vec->nx[0];
  fs=frame_duration/((double) samples_per_frame);
    
/*   relic=epoch-GPS; */
/*   if (relic>0) */
/*     offset=(int) (relic*fs); */
  FrVectFree(vec); 

  vec=FrFileIGetV(f,channel,epoch,fs*out->vec->length);
  if (vec==NULL)
    {
      *stat=IO_FAIL;
      RETURN(status);
    }

  for (i=0; i<out->vec->length; i++)
    out->vec->data[i]=FrVectGetValueI(vec,i);

    *stat=IO_OK;
  FrVectFree(vec); 
  RETURN(status);

  /* init buffer */

/*   buffer=FrvBufNew(N,N,FR_VECT_8R,0,offset); */

/*   do */
/*     {                  */

/*       if (FrvBufFeed(buffer,vec) == 0) */
/* 	{ */
/* 	  FrvBufGetNext(buffer); */
/* 	  out->epoch=buffer->output->GTime; */
/* 	  out->deltaT=buffer->output->dx[0]; */
/* 	  for (i=0; i<out->vec->length; i++) */
/* 	    out->vec->data[i]=FrVectGetValueI(buffer->output,i); */
/* 	  FrVectFree(vec); */
/* 	  FrvBufFree(buffer); */
/* 	  *stat=IO_OK; */
/* 	  RETURN(status); */
/* 	} */
      
/*       t=next_gps(f,GPS); */

/*       if (GPS<0) */
/* 	{ */
/* 	  sprintf(infostr,"Reach EOF at GPS=%ld",(long int) t); */
/* 	  bcc_warning(status,infostr); */
/* 	  FrvBufFree(buffer); */
/* 	  *stat=IO_EOF; */
/* 	  RETURN(status); */
/* 	} */
      
/*       GPS=(unsigned long int) t; */
/*       FrVectFree(vec); */
/*       vec=read_one_vector(f,GPS,channel);       */
      
/*     }  */
/*   while(1); */
  
  /* RETURN (status);*/
}

#endif // HAVE_LIBFRV

void IO_info_frame(bcc_status* status, double *fs, FrFile *f, const char* channel, const double epoch)
{
  char infostr[BCCInfoLength];
  long int GPS;
  double frame_duration,samples_per_frame,start,end;
  FrVect *vec=NULL;
  FrameH *frame=NULL;

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(channel,status,IO_ENULL,IO_MSGENULL);
  
  /* check that we seek data in the available time range */
  start=FrFileITStart(f);
  end=FrFileITEnd(f);  
  if ((epoch<start)|(epoch>end))
    {
      sprintf(infostr,"Requested epoch=%ld not in available range from %ld to %ld",(unsigned long int) epoch,(unsigned long int) start,(unsigned long int) end);
      bcc_warning(status,infostr);
      *fs=-1;
      RETURN(status);
    }

  /* get the frame at requested epoch */
  GPS=(unsigned long int)epoch;
  frame=read_one_frame(f,GPS);
  if (frame==NULL)
    {
      sprintf(infostr,"Missing frame at GPS=%ld",GPS);
      bcc_warning(status,infostr);
      *fs=-1;
      RETURN(status);
    }

  frame_duration=frame->dt;
  GPS=frame->GTimeS;
  if (GPS==0)
    {
      sprintf(infostr,"Missing time information at GPS=%ld",GPS);
      bcc_warning(status,infostr);
      *fs=-1;
      RETURN(status);
    }

  FrameFree(frame);

  /* get the vector at requested epoch */
  vec=read_one_vector(f,GPS,channel);
  if (vec==NULL)
    {
      sprintf(infostr,"Missing channel %s at GPS=%ld",channel,GPS);
      bcc_warning(status,infostr);
      *fs=-1;
      RETURN(status);
    }

  samples_per_frame=vec->nx[0];
  *fs=frame_duration/((double) samples_per_frame);

  RETURN(status);
}


#endif // HAVE_LIBFRAME
