/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "bcc.h"

#ifdef  __cplusplus
extern "C" {
#endif
    
  static double
  path_integral (int m0, int m1, const  matrix * tfr)
  {
    int b, m, n;
    double s=0, a, dm;
    
    b=tfr->columns;
    
    if (m1==m0)
      {
	for (n=1 ; n<b ; n++)
	  {
	    s=tfr->data2[0][m1];
	    for (n=1 ; n<b ; n++)
	      s+=tfr->data2[n][m1];
	  }
      }
    else
      {
	a=(double) (m0-m1)/((double) b);
	s=tfr->data2[0][m1];
	for (n=1 ; n<b ; n++)
	  {
	    dm=m1+a*n;
	    m=(int) dm; 
	    dm-=m;
	    s+=(1-dm)*tfr->data2[n][m]+dm*tfr->data2[n][m+1];
	  }
      }
    
    return s;
  }
  
  static void 
  DP_first_interval(vector* current,const  matrix * tfr,const int_array *label, const bcc_params * p)
  {
    int m0,m1,i,cc;
    int Nr1,step,Nr2,Nf;
    
    /* maps input and params structures to simpler variables */
    Nf=tfr->rows;
    Nr1=p->n_regul1;
    step=p->step;
    Nr2=p->n_regul2;

    /* compute path integral for each chirplet in interval */
    for (m0=0 ; m0<Nf ; m0++)
      {
	cc=label->data1[m0];
	
	current->data[cc]=path_integral(m0,m0,tfr);
	
	i=1; m1=m0-step;
	while ((i<=Nr1)&(m1>=0))
	  {
	    current->data[cc-i]=path_integral(m0,m1,tfr);
	    m1-=step; i++;
	  }
	
	i=1; m1=m0+step;
	while ((i<=Nr1)&(m1<Nf))
	  {
	    current->data[cc+i]=path_integral(m0,m1,tfr);
	    m1+=step; i++;
	  }
      }
  }
  
  static int
  find_max(const double *previous, const int m0, const int m1, const int *label, const bcc_params * p)
  {
    int m2,d,d2,max,i,j,cc;
    double s=0.0;
    int Nr1,step,Nr2,Nf;
    
    /* maps input and params structures to simpler variables */
    Nf=p->n_freq_bins;
    Nr1=p->n_regul1;
    step=p->step;
    Nr2=p->n_regul2;
    
    d=m1-m0;
    m2=m1+d;  // we prolong with the chirplet in the continuity
    i=(int) (d/step);
    
    cc=label[m1];
    cc+=i;
    
    // if this chirplet is outside the TF domain (below lower boundary)
    if (m2<0) 
      {
	j=min_i(Nr2,Nr1-i);
	d2=j*step;
	if (m2+d2>=0)
	  {
	    s=previous[cc+j];
	    max=cc+j;
	    
	    j--; d2-=step;
	    while (m2+d2>=0)
	      {
		if (previous[cc+j]>s)
		  {
		    s=previous[cc+j];
		    max=cc+j;
		  }
		j--; d2-=step;
	      }
	  }
	else
	  max=-1;
	return max;
      }
    
    // if this chirplet is outside the TF domain (above upper boundary)
    if (m2>=Nf)
      {
	j=-min_i(Nr2,Nr1+i);
	d2=j*step;
	if (m2+d2<Nf)
	  {
	    s=previous[cc+j];
	    max=cc+j;
	    
	    j++; d2+=step;
	    while (m2+d2<Nf)
	      {
		if (previous[cc+j]>s)
		  {
		    s=previous[cc+j];
		    max=cc+j;
		  }
		j++; d2+=step;
	      }
	  }
	else
	  max=-1;
	return max;
      }
    
    // if this chirplet is inside the TF domain
    
    // init: prolong with chirplet in the continuity
    s=previous[cc];
    max=cc;
    
    // scan downward
    j=-1; m2=m1+d-step;
    while ((m2>=0)&(j>=-Nr2)&(i+j>=-Nr1))
      {
	if (previous[cc+j]>s)
	  {
	    s=previous[cc+j];
	    max=cc+j;
	  }
	j--; m2-=step;
      }
    
    // scan upward
    j=1; m2=m1+d+step;
    while ((m2<Nf)&(j<=Nr2)&(i+j<=Nr1))
      {
	if (previous[cc+j]>s)
	  {
	    s=previous[cc+j];
	    max=cc+j;
	  }
	j++; m2+=step;
      }
    
    return max;
  }

#ifdef BCC_NTRACE
#define SET_TRACE(trace,label) (void) (0)
#else
#define SET_TRACE(trace,label) ((void) (trace=label))
#endif // BCC_NTRACE
  
  static void
  DP_next_interval(vector *current, int* trace, const vector* previous, const  matrix * tfr, const int_array* label, const bcc_params * p)
  {
    double s;
    int i,m0,m1,max,cc;
    int Nr1,step,Nr2,Nf;
    
    /* maps input and params structures to simpler variables */
    Nf=tfr->rows;
    Nr1=p->n_regul1;
    step=p->step;
    Nr2=p->n_regul2;
    
    for (m0=0 ; m0<Nf ; m0++)
      {
	cc=label->data1[m0];
	
	s=path_integral(m0,m0,tfr);
	max=find_max(previous->data,m0,m0,label->data1,p);
	current->data[cc]=previous->data[max]+s;
	SET_TRACE(trace[cc],max);
	
	i=1; m1=m0-step;
	while ((m1>=0)&(i<=Nr1))
	  {
	    s=path_integral(m0,m1,tfr);
	    max=find_max(previous->data,m0,m1,label->data1,p);
	    if (max>=0) // is chirplet connected?
	      current->data[cc-i]=previous->data[max]+s;
	    else
	      current->data[cc-i]=-1.0;
	    SET_TRACE(trace[cc-i],max);
	    i++; m1-=step;
	  }
	
	i=1; m1=m0+step;
	while ((m1<Nf)&(i<=Nr1))
	  {
	    s=path_integral(m0,m1,tfr);
	    max=find_max(previous->data,m0,m1,label->data1,p);
	    if (max>=0) // is chirplet connected?
	      current->data[cc+i]=previous->data[max]+s;
	    else
	      current->data[cc+i]=-1.0;
	    SET_TRACE(trace[cc+i],max);
	    i++; m1+=step;
	  }
      }
  }
  
  static void 
  compute_tables(int* out, int_array *horizontal, int_array *steepest, const bcc_params * p)
  {
    int i,k,l,n=0;
    int Nr1,step,Nr2,Nf;
    
    /* maps input and params structures to simpler local variables */
    Nf=p->n_freq_bins;
    Nr1=p->n_regul1;
    step=p->step;
    Nr2=p->n_regul2;
    
    /* main loop */
    for (i=0 ; i<Nf ; i++)
      {
	l=0; k=i-step;
	while ((k>=0)&(l<Nr1))
	  {
	    k-=step;
	    l++; n++;
	  }
	horizontal->data1[i]=n; n++;
	l=0; k=i+step;
	while ((k<Nf)&(l<Nr1))
	  {
	    k+=step;
	    l++; n++;
	  }
	steepest->data1[i]=n-1;
      }    

    *out=n;
  }

#ifdef BCC_NTRACE
#define trace_chain(chain,cc,Nt,trace,label_horiz_cc,label_steep_cc) (void) (0)
#else
  void 
  trace_chain(chirplet_chain *chain, const int seed, const int step, int_array* trace,const int_array* horizontal,const int_array *steepest)
  {
    int j,k=0,n;
    
    while (steepest->data1[k]<seed)
      k++;
    chain->nodes[trace->columns+1]=k;
    
    n=seed;
    for (j=trace->columns-1 ; j>-1 ; j--)
      {
	n=trace->data2[j][n];
	k=0;
	while (steepest->data1[k]<n)
	  k++;
	chain->nodes[j+1]=k;
      }
    chain->nodes[0]=chain->nodes[1]+step*(n-horizontal->data1[k]);
  }
#endif // BCC_NTRACE
  
  void check_trace_table(const int Nt, const int Nc, int** trace)
  {
    int j,n;
    for (j=0 ; j<Nt-1 ; j++)
      for (n=0 ; n<Nc ; n++)
	if ((trace[j][n]<-1) | (trace[j][n]>=Nc))
	  printf("BCC: error: j=%d n=%d trace=%d >= Nc=%d\n",j,n,trace[j][n],Nc);
    //ABORT

    /* check connectivity */
    /* #ifndef BCC_NDEBUG */
    /*   check continuity of all chains */
    /*     check_trace_table(Nt,Nc,trace); */
    /*     for (k=0 ; k<Nc ; k++) */
    /*       if (trace[Nt-2][k]>=0) */
    /* 	trace_chain(c,k,Nt,p->step,trace,label_horiz_cc,label_steep_cc); */
    /* #endif */

  }
    
  void create_chirplet_chain(bcc_status* status, chirplet_chain **c, int length, int time_bin_length, int n_freq_bins)
  {
    CHECKSTATUSPTR(status);
    
    ASSERT(length>0, status, BCC_ECCHZ, BCC_MSGECCHZ);
    ASSERT( c != NULL, status, BCC_EVPTR, BCC_MSGEVPTR );
    ASSERT( *c == NULL, status, BCC_EUPTR, BCC_MSGEUPTR );
    
    *c=(chirplet_chain*) malloc(sizeof(chirplet_chain));
    if ( NULL == *c )
      {
	ABORT( status, BCC_EMALLOC, BCC_MSGEMALLOC );
      }
    
    (*c)->length = 0;        /* length 0 until storage allocated */
    (*c)->time_bin_length=0; /* time_bin_length 0 until storage allocated */
    (*c)->n_freq_bins=0;     /* n_freq_bins 0 until storage allocated */
    (*c)->score  = 0.0;      /* set score to zero */
    (*c)->SNR  = 0.0;        /* set SNR to zero */
    (*c)->nodes   = NULL;    /* NULL data until allocated */
    
    (*c)->nodes=(int *) malloc(length*sizeof(int));
    if ( NULL == (*c)->nodes )
      {
	free( *c );
	ABORT( status, BCC_EMALLOC, BCC_MSGEMALLOC );
      }
    
    (*c)->length=length;
    (*c)->time_bin_length=time_bin_length;
    (*c)->n_freq_bins=n_freq_bins;
    
    RETURN(status);
  }
  
  void destroy_chirplet_chain(bcc_status *status, chirplet_chain ** c)
  {
    CHECKSTATUSPTR(status);
    
    ASSERT( c != NULL, status, BCC_EVPTR, BCC_MSGEVPTR );
    ASSERT( *c == NULL, status, BCC_EUPTR, BCC_MSGEUPTR );
    ASSERT( (*c)->nodes == NULL, status, BCC_EDPTR, BCC_MSGEDPTR );
    
    free((*c)->nodes);
    free(*c);
    *c=NULL;
    
    RETURN(status);
  }
  
  void bcc (bcc_status* status, chirplet_chain* out, const vector* in, const bcc_params* p)
  {
    static int first_visit=BCC_TRUE;
    static int_array *label_horiz_cc=NULL, *label_steep_cc=NULL, *trace=NULL;
    static matrix *tfr=NULL;
    static vector *cur=NULL, *pre=NULL;
    static fftw_plan *plan=NULL;
    static int Nc=0;
    int N,b,Nt,Nf;
    int i,j,k,n,m,label_max;
    double max;

    CHECKSTATUSPTR(status);

    ASSERT(in,status,BCC_ENULL,BCC_MSGENULL);
    ASSERT(in->data,status,BCC_ENULL,BCC_MSGENULL);
    ASSERT(out,status,BCC_ENULL,BCC_MSGENULL);
    ASSERT(out->nodes,status,BCC_ENULL,BCC_MSGENULL);
    ASSERT(out->length==p->n_time_bins+1,status,BCC_ECCHZ,BCC_MSGECCHZ);
    ASSERT(p,status,BCC_ENULL,BCC_MSGENULL); 
   
    /* map input and params structures to simpler local variables */
    N=in->length;
    Nf=p->n_freq_bins;
    b=p->time_bin_length;
    Nt=p->n_time_bins;
    
    ASSERT((is_power_of_two(N)==1)&(N>0),status,BCC_EBLKZ,BCC_MSGEBLKZ);
    ASSERT((is_power_of_two(b)==1)&(b>=2)&(b<N),status,BCC_EINTZ,BCC_MSGEINTZ);
    ASSERT(Nf>=N,status,BCC_EFRQZ,BCC_MSGEFRQZ);
    
    /* memory allocation for static variables */
    if (first_visit==BCC_TRUE)
      {
	TRY(create_array(status,&label_horiz_cc,Nf,1),status);
	TRY(create_array(status,&label_steep_cc,Nf,1),status);
	TRY(compute_tables(&Nc,label_horiz_cc,label_steep_cc,p),status);
	TRY(create_vector(status,&cur,Nc),status);
	TRY(create_vector(status,&pre,Nc),status);
	
#ifdef BCC_NTRACE
	trace=NULL;
#else
	TRY(create_array(status,&trace,Nc,Nt-1),status);
#endif // BCC_NTRACE
	TRY(create_matrix(status,&tfr,Nf,b),status);
	if (p->input_is_matrix==BCC_FALSE)
	  {TRY(create_FFTW_plan(status,&plan,tfr,FFTW_ESTIMATE),status);}

	first_visit=BCC_FALSE;
      }
    
    /* reset memory */
    memset((void *)cur->data,0,cur->length*sizeof(double));
    memset((void *)pre->data,0,pre->length*sizeof(double));
    
#ifndef BCC_NTRACE
      memset((void *)trace->data1,-1,trace->columns*trace->rows*sizeof(int));
#endif // BCC_NTRACE
    
    /* main loop */
    if (p->input_is_matrix==BCC_FALSE)
      {TRY(dwv(status,tfr,in,0,plan),status);}
    else
      {
	for (n=0 ; n<b ; n++)
	  for (m=0 ; m<Nf ; m++)
	    tfr->data2[n][m]=in->data[n+N*m];      
      }

    DP_first_interval(pre,tfr,label_horiz_cc,p);

    k=b;
    for (j=0 ; j<(Nt-1) ; j++)
      {
	if (p->input_is_matrix==BCC_FALSE)
	  {TRY(dwv(status,tfr,in,k,plan),status);}
	else
	  {
	    for (n=0 ; n<b ; n++)
	      for (m=0 ; m<Nf ; m++)
		tfr->data2[n][m]=in->data[k+n+N*m];
	  }

	DP_next_interval(cur,trace->data2[j],pre,tfr,label_horiz_cc,p);
	memcpy(pre->data,cur->data,pre->length*sizeof(double));
	k+=b;
      }
    
    /* compute stat and get max */
    label_max=0;
    max=cur->data[0];
    for (i=1 ; i<Nc ; i++)
      {
	if (cur->data[i]>max)
	  {
	    max=cur->data[i];
	    label_max=i;
	  }
      }
    
    /* trace back */
    out->score=max;
    out->SNR=sqrt(max/((double) N));
    trace_chain(out,label_max,p->step,trace,label_horiz_cc,label_steep_cc);
        
    RETURN(status);
  }
  
#ifdef  __cplusplus
}
#endif
