/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "factories.h"
#include "error.h"

#ifdef  __cplusplus
extern "C" {
#endif

void create_vector(bcc_status* status, vector ** v,const int length)
{
  CHECKSTATUSPTR(status);

  ASSERT(length>0, status, BCC_FACTORIES_EOZ, BCC_FACTORIES_MSGEOZ);
  ASSERT( v != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *v == NULL, status, BCC_FACTORIES_ENNPTR, BCC_FACTORIES_MSGENNPTR );
  
  *v=(vector*) malloc(sizeof(vector));
  if ( NULL == *v )
  {
    ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
  }

  (*v)->length = 0;        /* length 0 until storage allocated */
  (*v)->data   = NULL;     /* NULL data until allocated */

  (*v)->data=(double *) malloc(length*sizeof(double));
 if ( NULL == (*v)->data )
  {
    free( *v );
    ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
  }

  (*v)->length=length;

  RETURN(status);
}

void destroy_vector(bcc_status *status, vector ** v)
{
  CHECKSTATUSPTR(status);

  ASSERT( v != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *v != NULL, status, BCC_FACTORIES_ENPTR, BCC_FACTORIES_MSGENPTR );
  ASSERT( (*v)->data != NULL, status, BCC_FACTORIES_EDPTR, BCC_FACTORIES_MSGEDPTR );
  
  free((*v)->data);
  free(*v);
  *v=NULL;

  RETURN(status);
}

void create_matrix(bcc_status *status, matrix ** m,const int rows,const int columns)
{
    int c;
    long int ptr;

    CHECKSTATUSPTR(status);

    ASSERT(rows>0, status, BCC_FACTORIES_EOZ, BCC_FACTORIES_MSGEOZ);
    ASSERT(columns>0, status, BCC_FACTORIES_EOZ, BCC_FACTORIES_MSGEOZ);
    ASSERT( m != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
    ASSERT( *m == NULL, status, BCC_FACTORIES_ENNPTR, BCC_FACTORIES_MSGENNPTR );

    *m=(matrix*) malloc(sizeof(matrix));
    if ( NULL == *m )
      {
	ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
      }

    (*m)->rows = 0;           /* rows 0 until storage allocated */
    (*m)->columns = 0;        /* columns 0 until storage allocated */
    (*m)->data1   = NULL;     /* NULL data until allocated */
    (*m)->data2   = NULL;     /* NULL data until allocated */

    (*m)->data1=(double *) malloc(columns*rows*sizeof(double));
    if ( NULL == (*m)->data1 )
      {
	free( *m);
	ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
      }

    (*m)->data2=(double **) malloc(columns*sizeof(double*));
    if ( NULL == (*m)->data2)
      {
	free((*m)->data1);
	free( *m);
	ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
      }

    ptr=0;
    for (c=0 ; c<columns ; c++)
      {
	(*m)->data2[c]=(*m)->data1+ptr;
	ptr+=rows;
      }

    (*m)->columns=columns;
    (*m)->rows=rows;

    RETURN(status);
}

void destroy_matrix(bcc_status *status, matrix ** m)
{
  ASSERT( m != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *m != NULL, status, BCC_FACTORIES_ENPTR, BCC_FACTORIES_MSGENPTR );

  ASSERT( (*m)->data1 != NULL, status, BCC_FACTORIES_EDPTR, BCC_FACTORIES_MSGEDPTR );
  free((*m)->data1);
  (*m)->data1=NULL;

  ASSERT( (*m)->data2 != NULL, status, BCC_FACTORIES_EDPTR, BCC_FACTORIES_MSGEDPTR );
  free((*m)->data2);
  (*m)->data2=NULL;

  free(*m);
  *m=NULL;
  
  RETURN(status);
}

void create_array(bcc_status *status, int_array ** a,const int rows,const int columns)
{
    int c;
    long int ptr;

    CHECKSTATUSPTR(status);

    ASSERT(rows>0, status, BCC_FACTORIES_EOZ, BCC_FACTORIES_MSGEOZ);
    ASSERT(columns>0, status, BCC_FACTORIES_EOZ, BCC_FACTORIES_MSGEOZ);
    ASSERT( a != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
    ASSERT( *a == NULL, status, BCC_FACTORIES_ENNPTR, BCC_FACTORIES_MSGENNPTR );

    *a=(int_array*) malloc(sizeof(int_array));
    if ( NULL == *a )
      {
	ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
      }

    (*a)->rows = 0;           /* rows 0 until storage allocated */
    (*a)->columns = 0;        /* columns 0 until storage allocated */
    (*a)->data1   = NULL;      /* NULL data until allocated */
    (*a)->data2   = NULL;      /* NULL data until allocated */

    (*a)->data1=(int *) malloc(columns*rows*sizeof(int));
    if ( NULL == (*a)->data1 )
      {
	free( *a);
	ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
      }
    
    (*a)->data2=(int **) malloc(columns*sizeof(int*));
    if ( NULL == (*a)->data2)
      {
	free((*a)->data1);
	free( *a);
	ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
      }
    
    ptr=0;
    for (c=0 ; c<columns ; c++)
      {
	(*a)->data2[c]=(*a)->data1+ptr;
	ptr+=rows;
      }

    (*a)->columns=columns;
    (*a)->rows=rows;

    RETURN(status);
}

void destroy_array(bcc_status *status, int_array ** a)
{
   
  ASSERT( a != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *a != NULL, status, BCC_FACTORIES_ENPTR, BCC_FACTORIES_MSGENPTR );
  
  ASSERT( (*a)->data1 != NULL, status, BCC_FACTORIES_EDPTR, BCC_FACTORIES_MSGEDPTR );
  free((*a)->data1);
  (*a)->data1=NULL;

  ASSERT( (*a)->data2 != NULL, status, BCC_FACTORIES_EDPTR, BCC_FACTORIES_MSGEDPTR );
  free((*a)->data2);
  (*a)->data2=NULL;

  free(*a);
  *a=NULL;
  
  RETURN(status);
}

void create_string(bcc_status* status, char ** s,const int length)
{
  CHECKSTATUSPTR(status);

  ASSERT(length>0, status, BCC_FACTORIES_EOZ, BCC_FACTORIES_MSGEOZ);
  ASSERT( s != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *s == NULL, status, BCC_FACTORIES_ENNPTR, BCC_FACTORIES_MSGENNPTR );
  
  *s=(char*) malloc(length*sizeof(char));
  if ( NULL == *s )
    {
      ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );
    }
  
  RETURN(status);
}

void destroy_string(bcc_status *status, char ** s)
{
  CHECKSTATUSPTR(status);

  ASSERT( s != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *s != NULL, status, BCC_FACTORIES_ENPTR, BCC_FACTORIES_MSGENPTR );
  
  free(*s);
  *s=NULL;

  RETURN(status);
}

void create_time_series(bcc_status* status, time_series ** s,const int length, const double epoch, const double deltaT)
{    
  CHECKSTATUSPTR(status);

  ASSERT( s != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *s == NULL, status, BCC_FACTORIES_ENNPTR, BCC_FACTORIES_MSGENNPTR );
  ASSERT( deltaT>=0 , status, BCC_FACTORIES_EDELTAT, BCC_FACTORIES_MSGEDELTAT );

  *s=(time_series*) malloc(sizeof(time_series));
  if ( NULL == *s )
    {ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );}

  (*s)->epoch=0.0;
  (*s)->deltaT=0.0;
  (*s)->vec=NULL;

  TRY(create_vector(status,&(*s)->vec,length),status);

  (*s)->epoch=epoch;
  (*s)->deltaT=deltaT;

  RETURN(status);
}

void destroy_time_series(bcc_status* status, time_series ** s)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( s != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *s != NULL, status, BCC_FACTORIES_ENPTR, BCC_FACTORIES_MSGENPTR );
  
  TRY(destroy_vector(status,&(*s)->vec),status);
  
  free(*s);
  *s=NULL;

  RETURN(status);
}

void create_buffer(bcc_status* status, buffer ** b,const int length, const double epoch, const double deltaT)
{    
  CHECKSTATUSPTR(status);

  ASSERT( b != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *b == NULL, status, BCC_FACTORIES_ENNPTR, BCC_FACTORIES_MSGENNPTR );
  ASSERT( deltaT>=0 , status, BCC_FACTORIES_EDELTAT, BCC_FACTORIES_MSGEDELTAT );

  *b=(buffer*) malloc(sizeof(buffer));
  if ( NULL == *b )
    {ABORT( status, BCC_FACTORIES_EMALLOC, BCC_FACTORIES_MSGEMALLOC );}

  (*b)->epoch=0.0;
  (*b)->deltaT=0.0;
  (*b)->used=0;
  (*b)->allocated=0;
  (*b)->vec=NULL;

  TRY(create_vector(status,&(*b)->vec,length),status);
  (*b)->allocated=length;

  (*b)->epoch=epoch;
  (*b)->deltaT=deltaT;

  RETURN(status);
}

void destroy_buffer(bcc_status* status, buffer ** b)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( b != NULL, status, BCC_FACTORIES_EHPTR, BCC_FACTORIES_MSGEHPTR );
  ASSERT( *b != NULL, status, BCC_FACTORIES_ENPTR, BCC_FACTORIES_MSGENPTR );
  
  TRY(destroy_vector(status,&(*b)->vec),status);
  
  free(*b);
  *b=NULL;

  RETURN(status);
}


#ifdef  __cplusplus
}
#endif
