/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "parse.h"

#ifdef  __cplusplus
extern "C" {
#endif

static char const *BCCParseFormat[]={
  " %s",      /* conversion specifier for type BCCString */
  " %d",      /* conversion specifier for type BCCBoolean */
  " %lf",     /* conversion specifier for type BCCReal */
  " %d"       /* conversion specifier for type BCCInteger */
};

void copy_file_to_buf(bcc_status* status, char** buffer, const char* file_name)
{
  FILE         *file;
  register int size, nbytes, nitem, err;

  CHECKSTATUSPTR(status);

  ASSERT(buffer, status, PRS_ENULL, PRS_MSGENULL);
  ASSERT(file_name, status, PRS_ENULL, PRS_MSGENULL);

  /* open file */
  file = fopen(file_name, "r");
  if (file==NULL)
    {
      ABORT(status, PRS_EOPEN, PRS_MSGEOPEN);
    }

  /* get byte size of file */
  fseek(file, 0, SEEK_END);
  nbytes = ftell(file);
  if (nbytes==0)
    {
      fclose(file);
      ABORT(status, PRS_EFILE, PRS_MSGEFILE);
    }
  size   = nbytes + 1;
  rewind(file);

  /* init buffer */
  TRY(create_string(status,buffer,size),status);
  bzero(*buffer, size);

  /* read file */
  nitem=fread(*buffer, 1, nbytes, file);
  if (nitem==0)
    {
      fclose(file);
      ABORT(status, PRS_EREAD, PRS_MSGEREAD);
    }

  /* close file */
  err=fclose(file);
  if (err!=0)
    {
      ABORT(status, PRS_ECLOS, PRS_MSGECLOS);
    }
  
  RETURN(status);
}

static char* get_next_line(const char* string)
{
  static char *this=(char *) NULL;
  static char  line [BCCPrsLineLength];
  char *tail=(char *) NULL;
  register int length;
  
  // if empty string reset
  if (string[0] == '\0')
    { 
      this=(char *) NULL;
      return((char *)NULL); 
    }

  if (this == NULL)
    this = string;
  
  tail = strchr(this, '\n');
  if (tail == (char *)NULL)
    { return((char *)NULL); }

  while (this==tail)
  {
    this=tail+1;
    tail = strchr(this, '\n');
    if (tail == (char *)NULL)
      { return((char *)NULL); }
  }

  length  = (int) ((tail - this)/sizeof(char));
  strncpy(line, (char *)this, length);
  line[length] = '\0';

  this=tail+1;

  return(line);
}

static void *deblank(const char *string,char *line)
{
  register int head, tail, length;
  //static char line[BCCPrsLineLength];

  /* remove trailing space, tab, vertical formfeed, etc... */

  length= strlen(string);
  
  for(head=0; (isspace((int)string[head]) != 0) && (head < length); head++);
  for(tail=length; (isspace((int)string[tail]) != 0) && (tail >= 0); tail--);

  length  = (int) ((tail - head)/sizeof(char));
  strncpy(line, (char *)string+head, length);

  line[length] = '\0';

  return(line);
}

static void scan_data(bcc_status* status, int* n, void *var, const BCCParseType t, const char* string)
{

  CHECKSTATUSPTR(status);

  ASSERT(var,status,PRS_ENULL,PRS_MSGENULL);
  ASSERT(string,status,PRS_ENULL,PRS_MSGENULL);

  *n=0;
  switch (t)
    {
    case BCCString:
      *n=sscanf(string,BCCParseFormat[t],(char **)var);
      break;
    case BCCReal:
      *n=sscanf(string,BCCParseFormat[t],(double *)var);
      break;
    case BCCInteger:
      *n=sscanf(string,BCCParseFormat[t],(int *)var);
      break;
    default:
      *n=0;
      ABORT(status,PRS_ETYPE,PRS_MSGETYPE);
    }

  RETURN(status);
}

static void set_default_string(char **var)
{
  var[0]='\0';
}

static int set_boolean(bcc_boolean *var, bcc_boolean value)
{
  *var=value;
  return(1);
}

static void set_default_real(double *var)
{
  *var=0.0;
}

static void set_default_integer(int *var)
{
  *var=0;
}

static void set_default(bcc_status* status, void *var, const BCCParseType t)
{
  CHECKSTATUSPTR(status);
  
  ASSERT(var,status,PRS_ENULL,PRS_MSGENULL);

  switch (t)
    {
    case BCCString:
      set_default_string((char**) var);
      break;
    case BCCReal:
      set_default_real((double*) var);
      break;
    case BCCBoolean:
      set_boolean((bcc_boolean*) var,BCC_FALSE);
      break;
    case BCCInteger:
      set_default_integer((int *) var);
      break;
    default:
      ABORT(status,PRS_ETYPE,PRS_MSGETYPE);
    }

  RETURN(status);
}

static void  dump_cfg_to_log(bcc_status* status, const void *var, const BCCParseType t,const char* label)
{
  char infostr[BCCInfoLength];
  const char* mask="Cfg: ";

  CHECKSTATUSPTR(status);
  
  ASSERT(var,status,PRS_ENULL,PRS_MSGENULL);
  ASSERT(label,status,PRS_ENULL,PRS_MSGENULL);

  switch (t)
    {
    case BCCString:
      sprintf(infostr,"%s%s=%s",mask,label,(char*) var);
      bcc_log(status,infostr);
      break;
    case BCCReal:
      sprintf(infostr,"%s%s=%g",mask,label,*((double*) var));
      bcc_log(status,infostr);
      break;
    case BCCBoolean:
      sprintf(infostr,"%s%s=%s",mask,label, (*((bcc_boolean*) var)==BCC_TRUE ? "TRUE":"FALSE"));
      bcc_log(status,infostr);
      break;
    case BCCInteger:
      sprintf(infostr,"%s%s=%d",mask,label,*((int*) var));
      bcc_log(status,infostr);
      break;
    default:
      ABORT(status,PRS_ETYPE,PRS_MSGETYPE);
    }

  RETURN(status);
}

void parse(bcc_status* status, bcc_boolean* success, void *var, const char *string, const char* label, const BCCParseType t)
{
  char *buffer, *line, *line_in, *sep, *first, *second;
  int n;
  
  CHECKSTATUSPTR(status);

  ASSERT(var,status,PRS_ENULL,PRS_MSGENULL);
  ASSERT(string,status,PRS_ENULL,PRS_MSGENULL);
  ASSERT(label,status,PRS_ENULL,PRS_MSGENULL);

  TRY(set_default(status,var,t),status);

  get_next_line("\0");
  buffer=(char *)malloc(strlen(string)*2*sizeof(char));
  deblank(string,buffer);
  line_in=get_next_line(buffer);
  line=(char *)malloc(BCCPrsLineLength*sizeof(char));
  while(line_in!=NULL)
    {
      deblank(line_in,line);
      n=0;
      if ((strlen(line)>0)&(line[0]!='#'))
	{
	    if (t==BCCBoolean)
	      {
		if (strcasecmp(line,label)==0)
		  n=set_boolean((bcc_boolean *)var,BCC_TRUE);
	      }
	    else
	      {
		sep=index(line,'=');
		if (sep!=NULL)
		  {
		    sep[0]='\0';
		    first=strdup(line);
		    second=strdup(sep+1);
		    if (strcasecmp(line,label)==0)
		      {TRY(scan_data(status,&n,var,t,second),status);}
		  }
	      }
	}
      if (n>0)
	break;
      line_in=get_next_line(buffer);
    }

  TRY(dump_cfg_to_log(status,var,t,label),status);

  *success=(n>0?BCC_TRUE:BCC_FALSE);

  free(line);
  free(buffer);

  RETURN(status);
}

#ifdef  __cplusplus
}
#endif
