/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "vectorops.h"

#ifdef  __cplusplus
extern "C" {
#endif

void copy_vector(bcc_status* status, vector *copy, const vector *orig)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( copy , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( copy->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( orig , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( orig->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );

  if (orig->length>copy->length)
    {ABORT(status,BCC_VECTOROPS_EVECZ,BCC_VECTOROPS_MSGEVECZ);}

  memcpy(copy->data,orig->data,orig->length*sizeof(double));

  RETURN(status);
}

void reset_vector(bcc_status* status, vector *in)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  
  memset(in->data,0,in->length*sizeof(double));
  
  RETURN(status);
}

void reset_matrix(bcc_status* status, matrix *in)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->data1 , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->rows , status, BCC_VECTOROPS_EVECZ, BCC_VECTOROPS_MSGEVECZ );
  ASSERT( in->columns , status, BCC_VECTOROPS_EVECZ, BCC_VECTOROPS_MSGEVECZ );
  
  memset(in->data1,0,in->rows*in->columns*sizeof(double));
  
  RETURN(status);
}

void copy_time_series(bcc_status* status, time_series *copy, const time_series *orig)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( copy , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( orig , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  
  TRY(copy_vector(status,copy->vec,orig->vec),status);

  copy->epoch=orig->epoch;
  copy->deltaT=orig->deltaT;

  RETURN(status);
}

void reset_time_series(bcc_status* status, time_series *in)
{
  CHECKSTATUSPTR(status);
  
  ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->vec , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  
  TRY(reset_vector(status,in->vec),status);
  in->epoch=0.0;
  in->deltaT=0.0;

  RETURN(status);
}

/* void extract_vector(bcc_status* status, vector *out, const vector *in, const int start, const int end) */
/* { */
/*   int n; */

/*   CHECKSTATUSPTR(status); */
  
/*   ASSERT( out , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( out->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( in->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */

/*   if ((start<0)|(end<=start)|(end>=in->length)) */
/*     {ABORT(status, BCC_VECTOROPS_EIRNG, BCC_VECTOROPS_MSGEIRNG );} */

/*   n=end-start+1; */

/*   if (n>out->length) */
/*     {ABORT(status, BCC_VECTOROPS_VECZ, BCC_VECTOROPS_MSGEVECZ );} */

/*   memcpy(out->data,in->data+start,n*sizeof(double)); */

/*   RETURN(status); */
/* } */

/* void concat_vector(bcc_status* status, vector *out, const vector *in1, const vector *in2) */
/* { */
/*   int n; */

/*   CHECKSTATUSPTR(status); */
  
/*   ASSERT( out , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( out->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( in1 , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( in1->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( in2 , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */
/*   ASSERT( in2->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL ); */

/*   if (in1->length+in2->length>out->length) */
/*     {ABORT(status, BCC_VECTOROPS_VECZ, BCC_VECTOROPS_MSGEVECZ );} */

/*   memcpy(out->data,in1->data,in1->length*sizeof(double)); */
/*   memcpy(out->data+in1->length,in2->data,in2->length*sizeof(double)); */

/*   RETURN(status); */
/* } */

void feed_buffer(bcc_status* status, buffer *out, const time_series *in, const int start, const int size, const bcc_boolean overwrite)
{
  int n;

  CHECKSTATUSPTR(status);
  
  ASSERT( out , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( out->vec , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( out->vec->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->vec , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->vec->data , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );

  if (size<=0)
    {RETURN(status);}

  if ((start<0)|(start+size>in->vec->length))
    {ABORT(status, BCC_VECTOROPS_EIRNG, BCC_VECTOROPS_MSGEIRNG);}

  n=(overwrite==BCC_TRUE ? 0 : out->used);
  if (n+size>out->vec->length)
    {ABORT(status,BCC_VECTOROPS_EVECZ,BCC_VECTOROPS_MSGEVECZ);}

  if (n>0)
    {
      if (in->deltaT!=out->deltaT)
	{ABORT(status,BCC_VECTOROPS_EICSR,BCC_VECTOROPS_MSGEICSR);}
      
      if (out->epoch+start*out->deltaT != in->epoch)
	{ABORT(status,BCC_VECTOROPS_ETDSC,BCC_VECTOROPS_MSGETDSC);}  
    }
  else
    {
      out->epoch=in->epoch;
      out->deltaT=in->deltaT;
    }

  out->used=n+size;

  memcpy(out->vec->data+n,in->vec->data+start,size*sizeof(double));
  
  RETURN(status);
}

void transfer_buffer(bcc_status* status, buffer *out, buffer *in)
{

  CHECKSTATUSPTR(status);
  
  ASSERT( out , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( out->vec , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->vec , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );

  if (in->used<=0)
    {RETURN(status);}

  if (in->used>out->vec->length)
    {ABORT(status,BCC_VECTOROPS_EVECZ,BCC_VECTOROPS_MSGEVECZ);}

  memcpy(out->vec->data,in->vec->data,in->used*sizeof(double));

  out->used=in->used;
  out->epoch=in->epoch;
  out->deltaT=in->deltaT;

  in->used=0;
  in->epoch=0.0;
  in->deltaT=0.0;
  
  RETURN(status);

}

void clear_buffer(bcc_status* status, buffer *in)
{

  CHECKSTATUSPTR(status);
  
  ASSERT( in , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );
  ASSERT( in->vec , status, BCC_VECTOROPS_ENULL, BCC_VECTOROPS_MSGENULL );

  in->used=0;
  in->epoch=0.0;
  in->deltaT=0.0;
  
  RETURN(status);
}

void reshape_time_series(bcc_status* status, buffer *out, buffer *buf, const time_series *in, const int segment_length, const int overlap)
{
  int size,leftovers;

  CHECKSTATUSPTR(status);

  ASSERT(out, status,BCC_VECTOROPS_ENULL,BCC_VECTOROPS_MSGENULL);
  ASSERT(buf,status,BCC_VECTOROPS_ENULL,BCC_VECTOROPS_MSGENULL);
  ASSERT(in,status,BCC_VECTOROPS_ENULL,BCC_VECTOROPS_MSGENULL);
  ASSERT(in->vec->length>=segment_length,status,BCC_VECTOROPS_EVECZ,BCC_VECTOROPS_MSGEVECZ);

  leftovers=(buf->used+in->vec->length-overlap)%(segment_length-overlap);

  /* check that data fits in output vector */
  size=buf->used+in->vec->length-leftovers;
  if (size>out->vec->length)
    {ABORT(status,BCC_VECTOROPS_EVECZ,BCC_VECTOROPS_MSGEVECZ);}

  /* copy buf if available and set epoch */
  TRY(transfer_buffer(status,out,buf),status);
  TRY(feed_buffer(status,out,in,0,in->vec->length-leftovers,BCC_FALSE),status);
  
  /* backup left over */
  size=leftovers+overlap;
  TRY(feed_buffer(status,buf,in,in->vec->length-size,size,BCC_TRUE),status);

  RETURN(status);

}

#ifdef  __cplusplus
}
#endif

