/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include <getopt.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "bcc.h"

int bccDebugLevel = 1;

extern char    *optarg;
extern int      optind;

static double fs=0.0;
static double fe=0.5;
static int N =16;
static int b =2;
static int Nf=17;
static int Nr1=2;
static int step=1;
static int Nr2=1;

static void 
Usage (const char *program,int exitcode)
{
  fprintf (stderr, "Usage: %s [options]\n", program);
  fprintf (stderr, "Options:\n");
  fprintf (stderr, "    -s starting frequency           [0.0]\n");
  fprintf (stderr, "    -e ending frequency             [0.5]\n");
  fprintf (stderr, "    -n number of samples            [128]\n");
  fprintf (stderr, "    -b number of points in interval [ 2 ]\n");
  fprintf (stderr, "    -f number of freq. bins         [128]\n");
  fprintf (stderr, "    -1 regularity param             [ 2 ]\n");
  fprintf (stderr, "    -2 regularity param             [ 1 ]\n");
  fprintf (stderr, "    -t step size                    [ 1 ]\n");
  exit (exitcode);
}

static void
ParseOptions (int argc, char *argv[])
{
  while (1)
    {
      int c = -1;
      
      c = getopt (argc, argv, "s:""e:""n:""b:""f:""1:""t:""2:");
      if (c == -1)
	break;
      
      switch (c)
	{
	case 'n': /* set number of signal samples */
	  N = atoi (optarg);
	  break;
	case 'b': /* set number of points in an interval */
	  b = atoi (optarg);
	  break;
	case 'f': /* set number of frequency bins */
	  Nf = atoi (optarg);
	  break;
	case '1': /* set number of frequency bins */
	  Nr1 = atoi (optarg);
	  break;
	case 't': /* set the step size */
	  step = atoi (optarg);
	  break;
	case '2': /* set number of frequency bins */
	  Nr2 = atoi (optarg);
	  break;
	case 's': /* set the starting frequency */
	  fs = atof (optarg);
	  break;
	case 'e': /* set the ending frequency */
	  fe = atof (optarg);
	  break;
	default:
	  Usage (rindex(argv[0],'/')+1,1);
	}
    }
}

void get_chirp(vector *x, double fs, double fe)
{
  double p,a,b;
  int n;
  
  a=M_PI*(fe-fs)/x->length;
  b=2*M_PI*fs;
  for (n=0; n<x->length; n++){
    p=a*n*n+b*n;
    x->data[n]=cos(p);
  }
}

int
main (int argc, char *argv[])
{
  int k,n,j,Nt,error;
  double ae,ac,fc,f,f0,f1,d;
  bcc_params *params=NULL;
  vector *vec=NULL;
  chirplet_chain *chain=NULL;
  static bcc_status status;

  INITSTATUSPTR(&status);

  ParseOptions (argc, argv);

  fprintf( stdout, "BCC: track linear chirp ");
  
  Nt=N/b;
  params=(bcc_params *) malloc(sizeof(bcc_params));
  params->time_bin_length=b;
  params->n_time_bins=Nt;
  params->n_freq_bins=Nf;
  params->n_regul1=Nr1;
  params->step=step;
  params->n_regul2=Nr2;
  params->input_is_matrix=BCC_FALSE;
  
  fprintf( stdout, "DWV is of size N=%d Nf=%d\n",N,Nf);
  fprintf( stdout, "args are b=%d Nr1=%d step=%d Nr2=%d\n",b,Nr1,step,Nr2);
  fprintf( stdout, "chirp signal from fs=%2.2f to fe=%2.2f\n",fs,fe);
  
  vec=(vector *) malloc(sizeof(vector));
  vec->length=N;
  vec->data=(double *) malloc (N*sizeof(double));
  get_chirp(vec,fs,fe);

  /* apply DP */

  SUB(create_chirplet_chain(&status,&chain,Nt+1,b,Nf),&status);

  SUB(bcc(&status,chain,vec,params),&status);
  
  fprintf( stdout, "chain found is    ");
  for (n=0; n<chain->length; n++)
    fprintf(stdout,"%d ",chain->nodes[n]);
  fprintf(stdout," score=%2.2f\n",chain->score);

  /* check result */
  ae=(fe-fs)/N;

  error=0; k=0;
  fprintf( stdout, "n  f   f_c   diff\n");
  for (j=0; j<N/b; j++) {
    f1=chain->nodes[j]/(2.0*Nf);
    f0=chain->nodes[j+1]/(2.0*Nf);
    ac=(f0-f1)/((double) b);
    
    for (n=0; n<b; n++)
      {
	fc=ac*n+f1;
	f=ae*k+fs;
	d=f-fc;
	
	fprintf( stdout, "%d %2.2f %2.2f %2.2f\n",k,f,fc,d);

	if (fabs(d)>1e-1)
	  {
	    fprintf( stdout, "BCC: linear chirp tracking failed!\n");
	    error=1;
	    break;
	  }
	k++;
      }
    if (error==1)
      break;
  }
  
  if (error==0)
    fprintf( stdout, "BCC: linear chirp tracking passed\n");
  
  return(EXIT_SUCCESS);
}
