/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include <getopt.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#include "resample.h"

int bccDebugLevel = BCCMSGLVL3;

#define MEAS_RMS_ERROR 0
#define MEAS_REJECT 1

/* resampling, Virgo type configuration */
/* static int N=20000; */
/* static int D=625; */
/* static int I=64; */
//static double lower_f=256./2048.;
/* static double lower_f=0.0;; */
/* static double upper_f=(1024.-128.)/2048.0; */
/* static double roll_off=256./2048.; */
/* static double reject=-3; */
/* static int n_measurement=1; */

/* resampling, LIGO type configuration */
static int N=16384;
static int D=8;
static int I=1;
static double lower_f=256./2048.;
static double upper_f=(1024.-128.)/2048.0;
static double roll_off=256./2048.;
static double reject=-3;
static int n_measurement=100;

static inline int
getmin ( int x, int y)
{
  return x < y ? x : y;
}

static void
Usage (
    const char *program,
    int         exitcode
      )
{
  fprintf (stderr, "Usage: %s [options]\n", program);
  fprintf (stderr, "Options (defaults shown in square brackets):\n\n");
  fprintf (stderr, "    -N number of samples in a block      [20000]\n");
  fprintf (stderr, "    -D decimation factor                   [625]\n");
  fprintf (stderr, "    -I interpolation factor                 [64]\n");
  fprintf (stderr, "    -l lower cut-off frequency              [??]\n");
  fprintf (stderr, "    -u upper cut-off frequency              [??]\n");
  fprintf (stderr, "    -w width of the filter roll-off         [??]\n");
  fprintf (stderr, "    -r log10 rejection outside the passband [-3]\n");
  fprintf (stderr, "    -m number of measurements              [100]\n");
  exit (exitcode);
}

static void
ParseOptions (
    int         argc,
    char       *argv[]
             )
{
  while (1)
  {
    int c = -1;

    c = getopt (argc, argv, "N:""D:""I:""r:""l:""w:");
    if (c == -1)
    {
      break;
    }
    
    switch (c)
      {
      case 'N': /* set number of signal samples */
	N = atoi (optarg);
        break;
      case 'D': /* set the decimation factor */
        D = atoi (optarg);
        break;
      case 'I': /* set the interpolation factor */
        I = atoi (optarg);
        break;
      case 'r': /* set the rejection */
	reject = atof (optarg);
	break;
      case 'l': /* set the low frequency cut-off */
	lower_f = atof (optarg);
	break;
      case 'u': /* set the upper frequency cut-off */
	upper_f = atof (optarg);
	break;
      case 'w': /* set the width of the filter roll-off */
	roll_off = atof (optarg);
	break;
      case 'm': /* set the number of measurements */
	n_measurement = atoi (optarg);
	break;
      default:
        Usage (argv[0], 1);
    }
  }

  if (optind < argc)
  {
    Usage (argv[0], 1);
  }
}

void renew_seed()
{
  int seed;
  seed=time(NULL);
  srand48(seed);
}

void get_discrep_resampled_vs_exact(bcc_status* status, double *out, const double f, const int flag, const resampling_params *p)
{
  time_series *x=NULL, *y=NULL;
  vector *z=NULL;
  resampling_buffers *bufs=NULL;
  double phase, phase0, rms, d, norm, y_exact;
  int k, M;

  CHECKSTATUSPTR(status);

  M=(int) ((I*N)/(double) D);
  
  TRY(create_vector(status,&z,2*N),status);
  TRY(create_time_series(status,&y,M,0.0,0.0),status);

  bufs=malloc(sizeof(resampling_params));
  TRY(resample_init_buffer(status,bufs,N,p),status);
  TRY(resample_clear_buffer(status,bufs),status);

  /* get random initial phase */
  renew_seed();
  phase0=0.0; //2*M_PI*drand48();

  /* generate input stream */
  for (k=0; k<2*N; k++)
    {
      phase=2.0*M_PI*f*k+phase0;
      z->data[k]=cos(phase);
    }

  x=(time_series*) malloc(sizeof(time_series));
  x->vec=(vector*) malloc(sizeof(vector));
  x->vec->length=N;
  x->deltaT=1.0;

  /* resample first segment and dump result (avoid border effect) */
  x->vec->data=z->data;
  x->epoch=0.0;
  TRY(resample(status,y,bufs,x,p),status);
  
  /* resample second segment */
  x->vec->data=z->data+N;
  x->epoch=N;
  TRY(resample(status,y,bufs,x,p),status);
  
  rms=0.0;
  norm=0.0;
  for (k=0; k<M; k++)
    {
      phase=2.0*M_PI*f*(k+y->epoch)*y->deltaT+phase0;
      y_exact=cos(phase);
      norm+=y->vec->data[k]*y->vec->data[k];
      d=(y->vec->data[k]-y_exact);
      printf("%g %g %g\n",y_exact,y->vec->data[k],d);
      rms+=d*d;
    }

  TRY(resample_purge_buffer(status,bufs),status);
  free(bufs); bufs=NULL;
  TRY(destroy_vector(status,&z),status);
  TRY(destroy_time_series(status,&y),status);

  if (flag==MEAS_RMS_ERROR)
    *out=sqrt(rms/(double) M);
  else
    *out=sqrt(norm/(double) M);

  RETURN(status);
}

int
main (int argc, char *argv[])
{
  resampling_params *p=NULL;
  int n, error, n_start, n_end, n_bin, step;
  double ratio, rms, resolution, threshold;
  static bcc_status status;
  char infostr[BCCInfoLength];

  INITSTATUSPTR(&status);

  ParseOptions (argc, argv);

  ratio=(double) I/(double) D;

  p= (resampling_params*) malloc(sizeof(resampling_params));
  p->interpolate=I;
  p->decimate=D;
  p->log10_reject=reject;
  p->roll_off_width=roll_off;
  p->lower_cutoff_f=lower_f;
  p->upper_cutoff_f=upper_f;

  sprintf(infostr,"BCC: test of the resampling procedure");
  bcc_log(&status,infostr);
  sprintf(infostr,"with args, N=%d D=%d I=%d r=%g l=%g w=%g meas=%d",N,D,I,reject,lower_f,roll_off,n_measurement);
  bcc_log(&status,infostr);

  /* check if the block size is large enough */
  SUB(smallest_block_size(&status,&n,p),&status);
  if (N<n)
    {
      sprintf(infostr,"BCC: block size must be larger than %d",n);
      bcc_error(&status,infostr);
      return (-1);
    }

  /* scan frequencies */  
  resolution=N;

  /* check lower-band attenuation if high-pass filtering */
/*   sprintf(infostr,"BCC: test of the resampling, check high-pass (low-freq attenuation)"); */
/*   bcc_log(&status,infostr);  */
  threshold=pow(10.0,reject);
/*   n_start=0.0; */
/*   n_end=(int) (ratio*(lower_f-roll_off/2.0)*resolution); */
/*   n_bin=n_end-n_start; */
/*   if (n_measurement > n_bin) */
/*     step=1; */
/*   else */
/*     step=(int) (n_bin/(double) n_measurement); */

/*   error=0; */
/*   sprintf(infostr,"idx     f      attenuation"); */
/*   bcc_log(&status,infostr); */
/*   for (n=n_start; n<n_end; n+=step) */
/*     { */
/*       SUB(get_discrep_resampled_vs_exact(&status,&rms,n/resolution,MEAS_REJECT,p),&status); */
/*       sprintf(infostr,"%d    %1.1e  %g",n,n/resolution,rms); */
/*       bcc_log(&status,infostr); */
/*       if (rms>threshold) */
/* 	{ */
/* 	  sprintf(infostr,"BCC: f=%g (n=%d) low cut-off fl=%g  error rms=%g",n/resolution,n,ratio*lower_f,rms); */
/* 	  bcc_log(&status,infostr); */
/* 	  sprintf(infostr,"BCC: resampling near-DC signal, test failed!"); */
/* 	  bcc_log(&status,infostr); */
/* 	  error=1; */
/* 	} */
/*     } */
/*   if (error==0) */
/*     { */
/*       sprintf(infostr,"BCC: resampling near-DC signal, test passed"); */
/*       bcc_log(&status,infostr); */
/*     } */

  /* check in-band reconstruction */
  bcc_log(&status,"BCC: test of the resampling, check reconstruction of in-band signals\n"); 
  n_start=(int) (ratio*(lower_f+roll_off/2.0)*resolution);
  n_end=(int) (ratio*(upper_f-roll_off/2.0)*resolution);
  n_bin=n_end-n_start;
  if (n_measurement > n_bin)
    step=1;
  else
    step=(int) (n_bin/(double) n_measurement);

  error=0;
  sprintf(infostr,"idx     f       RMS error");
  bcc_log(&status,infostr);
  n=16;
  //  for (n=n_start; n<n_end; n+=step)
    {
      SUB(get_discrep_resampled_vs_exact(&status,&rms,n/resolution,MEAS_RMS_ERROR,p),&status);
      sprintf(infostr,"%d    %1.1e  %g",n,n/resolution,rms);
      bcc_log(&status,infostr);

      if (rms>threshold)
	{
	  sprintf(infostr,"BCC: f=%g (n=%d) low cut-off fl=%g high cut-off fu=%g error rms=%g",n/resolution,n,ratio*lower_f,ratio*upper_f,rms);
	  bcc_log(&status,infostr);
	  sprintf(infostr,"BCC: resampling in-band signal, test failed!");
	  bcc_log(&status,infostr);
	  error=1;
	}
    }
  if (error==0)
    {
      sprintf(infostr,"BCC: resampling in-band signal, test passed");
      bcc_log(&status,infostr);
    }

  /* check out-band rejection */ 
/*   bcc_log(&status,"BCC: test of the resampling, check low-pass (rejection of high freq)\n");  */
/*   n_start=(int) (ratio*(upper_f+roll_off/2.0)*resolution); */
/*   n_end  =(int) (.5*resolution); */
/*   n_bin=n_end-n_start; */
/*   if (n_measurement > n_bin) */
/*     step=1; */
/*   else */
/*     step=(int) (n_bin/(double) n_measurement); */

/*   error=0; */
/*   bcc_log(&status,"idx     f      attenuation\n"); */
/*   for (n=n_start; n<n_end; n+=step) */
/*     { */
/*       SUB(get_discrep_resampled_vs_exact(&status,&rms,n/resolution,MEAS_REJECT,p),&status); */
/*       sprintf(infostr,"%d    %1.1e  %g",n,n/resolution,rms); */
/*       bcc_log(&status,infostr); */
/*       if (rms>threshold) */
/* 	{ */
/* 	  sprintf(infostr,"BCC: frequency f=%g (n=%d) RMS rejection rms=%g",n/resolution,n,rms); */
/* 	  bcc_log(&status,infostr); */
/* 	  sprintf(infostr,"BCC: resampling out-of-band signal, test failed!"); */
/* 	  bcc_log(&status,infostr); */
/* 	  error=1; */
/* 	} */
/*     } */
/*   if (error==0) */
/*     { */
/*       sprintf(infostr,"BCC: resampling out-of-band signal, test passed"); */
/*       bcc_log(&status,infostr); */
/*     } */

  return 0;
}
