function data = bccresamplehighpass(data, sampleFrequencies, sampleFrequency, highPassCutoff)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        process command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% check for sufficient command line arguments
error(nargchk(3, 4, nargin));

% force cell array
if ~iscell(data),
  data = mat2cell(data, size(data, 1), size(data, 2));
end

% force one dimensional cell arrays and vectors
data = data(:);
sampleFrequencies = sampleFrequencies(:);

% determine number of channels
numberOfChannels = length(data);

% default high pass cutoff frequency
if (nargin < 4)
  highPassCutoff=0.0;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                       validate command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% validate number of sample frequencies
if length(sampleFrequencies) ~= numberOfChannels,
  error("bccresamplehighpass.m: number of sample frequencies and channels are inconsistent");
end

% force row vectors and determine data lengths
dataLengths = zeros(numberOfChannels, 1);
for channelNumber = 1 : numberOfChannels,
  data{channelNumber} = data{channelNumber}(:).';
  dataLengths(channelNumber) = length(data{channelNumber});
end

% determine data durations
dataDurations = dataLengths ./ sampleFrequencies;

% validate equal data durations
if ~all(dataDurations == dataDurations(1)),
  error("bccresamplehighpass.m: data durations are not equal");
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%            resample and high pass filter data                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% begin loop over channels
for channelNumber = 1 : numberOfChannels,

  % closest rational approximation of ratio of sample frequencies
  [upSampleFactor, downSampleFactor] = rat(sampleFrequency / sampleFrequencies(channelNumber));

  % reduced cutoff frequency
  reducedCutoff = highPassCutoff/sampleFrequencies(channelNumber);

  % process data
  data{channelNumber} = resamplehighpass(data{channelNumber}, upSampleFactor, downSampleFactor, reducedCutoff);

% end loop over channels
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                    return                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% return to calling function
return;
