function bccscan(eventTime, configurationFile, frameCacheFile, outputDirectory)
%% BCCSCAN Create spectrograms of significant channels for candidate events
%%
%% BCCSCAN Examines gravitational-wave channels, auxiliary interferometer channels,
%% and environmental channels around times of interest.  Spectrograms are
%% generated for those channels that exhibit statistically significant behavior.
%%
%% usage: bccscan(eventTime, configurationFile, frameCacheFile, outputDirectory);
%%
%%   eventTime             GPS time of candidate event
%%   configurationFile     path name of channel configuration file
%%   frameCacheFile        path name of frame file cache file
%%   outputDirectory       path name of directory to write results
%%
%% To allow use as a stand-alone executable, the specified eventTime should be
%% a string, not a number.
%%
%% The configuration file is an ASCII text file describing the parameters for
%% each channel to be analyzed.  The entries should have the following syntax.
%%
%% {
%%   channelName:                 'V1:Pr_B1_ACp'
%%   frameType:                   'raw'
%%   sampleFrequency:             4096
%%   searchTimeRange:             16
%%   searchFrequencyRange:        [64 1024]
%%   searchQRange:                [4 64]
%%   searchMaximumEnergyLoss:     0.2
%%   whiteNoiseFalseRate:         1e-2
%%   searchWindowDuration:        0.1
%%   plotTimeRanges:              [0.1 1.0 10.0]
%%   plotFrequencyRange:          [64 1024]
%%   plotNormalizedEnergyRange:   [0 25.5]		  
%%   injectionName:               'NONE'
%%   injectionType:               ''
%%   injectionFactor:             0.0
%%   alwaysPlotFlag:              0
%% }
%%
%% Groups of related channels may optionally be separated by a section title
%% specifier with the following form.
%%
%% [index_entry:section_title]
%%
%% This will be used to generate a index entry and section title in the resulting
%% web page.
%%
%% The QCONFIGURE.SH script can be used to automatically generate a reasonable
%% configuration file for sample frame files.
%%
%% If no configuration file is specified, BCCSCAN looks for the file
%% configuration.txt in the current working directory.  Similarly, if not frame
%% cache file is specified, BCCSCAN looks for the file framecache.txt in the
%% current working directory.
%%
%% For information on the frameCacheFile, see READFRAMEDATA.
%%
%% The resulting spectrograms for statistically significant channels are written
%% to an event sub-directory that is created in the specified output directory
%% and is labelled by the GPS time of the event under analysis.  If no output
%% directory is specified, a default output directory called bccscans is created in
%% the current working directory.  A web page named index.html is also created in
%% each event sub-directory and presents the resulting spectrograms in table
%% form.
%%
%% See also QREADDATA, QRESAMPLE, QTILE, QCONDITION, QTRANSFORM, QTHRESHOLD,
%% QSELECT, QSPECTROGRAM, QEVENTGRAM, QTIMESERIES, QCONFIGURE.SH and BCCSCAN.SH.

%% Shourov K. Chatterji
%% shourov@ligo.mit.edu

%% $Id: bccscan.m,v 1.2 2008/10/13 15:59:25 ecm Exp $

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                debug settings                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% specify debug level for log output
  debugLevel = 1;
  
  %% enable or disable warnings
  if debugLevel >= 2,
    warning on BCCSCAN:incompleteConfiguration
  else
    warning off BCCSCAN:incompleteConfiguration
  endif

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                   defaults                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% default parameter file names
  defaultConfigurationFile = "configuration.txt";
  defaultFrameCacheFile = "framecache.txt";
  defaultOutputDirectory = "bccscans";
  
  %% default configuration parameters
  defaultSampleFrequency = 4096;
  defaultSearchTimeRange = 64;
  defaultSearchFrequencyRange = [0 Inf];
  defaultWhiteNoiseFalseRate = 1e-2;
  defaultSearchWindowDuration = [0.1];
  defaultWhiteningDuration = 8;
  defaultChirpletDuration = 4.8828e-04;
  defaultChirpingRateLimit1 = 3.6864e+04;
  defaultChirpingRateLimit2 = 2.5166e+07;
  defaultSnrThreshold = 20;
  defaultPlotTimeRanges = [1 4 16];
  defaultPlotFrequencyRange = [];
  defaultPlotMaximumEnergyLoss = 0.2;
  defaultAlwaysPlotFlag = 0;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            hard coded parameters                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  plotHorizontalResolution = 512;
  plotVerticalResolution = 512;
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        process command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% verify correct number of input arguments
  error(nargchk(1, 4, nargin));
  
  %% apply default arguments
  if (nargin < 2) || isempty(configurationFile),
    configurationFile = defaultConfigurationFile;
  endif

  if (nargin < 3) || isempty(frameCacheFile),
    frameCacheFile = defaultFrameCacheFile;
  endif
  
  if (nargin < 4) || isempty(outputDirectory),
    outputDirectory = defaultOutputDirectory;
  endif
  
  %% convert string event time to a number
  if ischar(eventTime),
    eventTimeString = eventTime;
    eventTime = str2num(eventTime);
  else
    eventTimeString = sprintf("%#019.9f", eventTime);
  endif
			      
  %% string name of event type
  eventType = "Event";
  
  %% event directory name
  eventDirectory = [outputDirectory "/" eventTimeString];
  
  %% name of html file
  htmlFile = "index.html";

  %% name of text summary file
  textSummaryFile = "summary.txt";
  
  %% name of xml summary file
  xmlSummaryFile = "summary.xml";
  
  %% name of context file
  contextFile = "context.html";
  
  %% set random number generator seeds based on event time
  rand("state", eventTime);
  randn("state", eventTime);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                      write log file header information                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% report status
  if debugLevel >= 0,
    fprintf(1, "BCCscan of %s at %s\n", eventType, eventTimeString);
    fprintf(1, "created by %s on %s at %s\n", ...
            getenv("USER"), datestr(clock, 29), datestr(clock, 13));
  endif

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           read configuration file                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% report status
  if debugLevel >= 0,
    fprintf(1, "reading configuration file %s...\n", configurationFile);
  endif
  
  %% initialize configuration structure
  configuration = [];
  channelNumber = 0;
  sectionIndex = [];
  sectionName = [];
  sectionNumber = 0;
  sectionStart = [];
  
  %% open configuration file for reading
  configurationFID = fopen(configurationFile, "r");
  
  %% begin loop over configuration file
  while ~feof(configurationFID),
    
    %% read one line from configuration file
    configurationLine = fgetl(configurationFID);
    
    %% remove any comments
    commentIndices = findstr(configurationLine, "#");
    if ~isempty(commentIndices),
      configurationLine = configurationLine(1 : (commentIndices(1) - 1));
    endif
    
    %% remove leading and trailing blanks
    configurationLine = fliplr(deblank(fliplr(deblank(configurationLine))));

    %% if empty line, skip to the next line
    if isempty(configurationLine),
      continue;
    endif
    
    %% check for new section
    if configurationLine(1) == "[",
      
      %% locate field separator
      commaIndex = strfind(configurationLine, ",");
      
      %% if field separator not located, report syntax error
      if isempty(commaIndex),
	error("syntax error processing configuration file \"%s\":\n%s\n", configurationFile, configurationLine);
      endif

      %% select first field separator
      commaIndex = commaIndex(1);
      
      %% increment section number
      sectionNumber = sectionNumber + 1;
      
      %% exract section index
      sectionIndex{sectionNumber} = configurationLine(2 : commaIndex - 1);
      
      %% extract section name
      sectionName{sectionNumber} = configurationLine((commaIndex + 1) : end - 1);
      
      %% remove leading and trailing blanks
      sectionIndex{sectionNumber} = ...
          fliplr(deblank(fliplr(deblank(sectionIndex{sectionNumber}))));
      sectionName{sectionNumber} = ...
          fliplr(deblank(fliplr(deblank(sectionName{sectionNumber}))));
      
      %% standardize section names
      sectionIndex{sectionNumber} = strrep(sectionIndex{sectionNumber}, ";", ":");
      sectionName{sectionNumber} = strrep(sectionName{sectionNumber}, ";", ":");
      
      %% determine initial visibility
      switch sectionIndex{sectionNumber},
	case "Context",
          sectionChecked{sectionNumber} = "checked";
          sectionDisplay{sectionNumber} = "block";
	case "Gravitational",
          sectionChecked{sectionNumber} = "checked";
          sectionDisplay{sectionNumber} = "block";
	otherwise
          sectionChecked{sectionNumber} = "checked";
          sectionDisplay{sectionNumber} = "block";
          %% sectionChecked{sectionNumber} = "";
          %% sectionDisplay{sectionNumber} = "none";
      endswitch
      
      %% record first channel in section
      sectionStart(sectionNumber) = channelNumber + 1;
      
      %% continue to next line
      continue;
      
    endif
    
    %% check for beginning of new channel configuration
    if configurationLine == "{",
      
      %% increment channel number
      channelNumber = channelNumber + 1;
      
      %% initialize configuration parameters
      configuration{channelNumber}.channelName = [];
      configuration{channelNumber}.frameType = [];
      configuration{channelNumber}.sampleFrequency = [];
      configuration{channelNumber}.searchTimeRange = [];
      configuration{channelNumber}.searchFrequencyRange = [];
      configuration{channelNumber}.searchWindowDuration = [];
      configuration{channelNumber}.searchWindowOverlap = [];
      configuration{channelNumber}.whiteningDuration = [];
      configuration{channelNumber}.chirpletDuration = [];
      configuration{channelNumber}.chirpingRateLimit1 = [];
      configuration{channelNumber}.chirpingRateLimit2 = [];
      configuration{channelNumber}.snrThreshold = [];
      configuration{channelNumber}.plotTimeRanges = [];
      configuration{channelNumber}.plotFrequencyRange = [];
      configuration{channelNumber}.plotNormalizedEnergyRange = [];
      configuration{channelNumber}.injectionNames = [];
      configuration{channelNumber}.injectionTypes = [];
      configuration{channelNumber}.injectionFactors = [];
      configuration{channelNumber}.alwaysPlotFlag = [];

      %% continue to next line
      continue;
      
    endif
    
    %% check for end of existing channel configuration
    if configurationLine == "}",

      %% validate channel configuration
      if isempty(configuration{channelNumber}.channelName),
	error("channel name not specified for channel number %d", channelNumber);
      endif
	       
      if isempty(configuration{channelNumber}.frameType),
	error("frame type not specified for channel number %d", channelNumber);
      endif

      fieldNames=fieldnames(configuration{channelNumber});
      for fieldNumber=1:length(fieldNames)
	fieldLabel=fieldNames{fieldNumber};
	if strcmp(fieldLabel,"channelName") & strcmp(fieldLabel,"channelNumber")
	  if isempty(configuration{channelNumber}.(field))
            warning("BCCSCAN: incompleteConfiguration",
		    "%s not specified for channel number %d", fieldLabel, channelNumber);
	    capitalizedFieldLabel=sprintf("%c%s",upper(fieldLabel(1)),fieldLabel(2:end));
	    eval("configuration{channelNumber}.(field) = default%s",capitalizedFieldLabel);
	  endif
	endif
      endfor

    %% continue to next line
    continue;

  endif
  
  %% locate field separator
  colonIndex = strfind(configurationLine, ":");
  
  %% if field separator not located, report syntax error
  if isempty(colonIndex),
    error("syntax error processing configuration file \"%s\":\n%s\n",configurationFile, configurationLine);
  endif
  
  %% parse configuration line
  colonIndex = colonIndex(1);
  parameterName = configurationLine(1 : colonIndex - 1);
  parameterValue = configurationLine((colonIndex + 1) : end);
  parameterName = fliplr(deblank(fliplr(deblank(parameterName))));
  parameterValue = fliplr(deblank(fliplr(deblank(parameterValue))));

  %% assign parameters based on name
  if isfield(configuration{channelNumber},parameterName)
    configuration{channelNumber}.(parameterName) = eval(parameterValue);
  else
    error("unknown configuration parameter %s",parameterName);
  endif
  
%% end loop over channel configuration file
endwhile

%% close configuration file
fclose(configurationFID);

%% number of configured channels
numberOfChannels = length(configuration);

%% number of sections
numberOfSections = length(sectionName);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            load frame cache file                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% report status
if debugLevel >= 0,
  fprintf(1, "reading framecache file %s...\n", frameCacheFile);
endif

%% load frame file cache
frameCache = loadframecache(frameCacheFile);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           create output directory                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% report status
if debugLevel >= 0,
  fprintf(1, "creating event directory...\n");
endif

%% create spectrogram directory
unix(["mkdir -p " eventDirectory]);

%% copy configuration file
unix(["cp " configurationFile " " eventDirectory "/configuration.txt"]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        initialize text summary report                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% report status
if debugLevel >= 0,
  fprintf(1, "opening text summary...\n");
endif

%% open text summary file
textSummaryFID = fopen([eventDirectory "/" textSummaryFile], "w");

%% write column definitions
fprintf(textSummaryFID, "# channelName\n");
fprintf(textSummaryFID, "# peakTime\n");
fprintf(textSummaryFID, "# peakMeanFrequency\n");
fprintf(textSummaryFID, "# peakBandwidth\n");
fprintf(textSummaryFID, "# peakSNR\n");

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        initialize xml summary report                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% report status
if debugLevel >= 0,
  fprintf(1, "opening xml summary...\n");
endif

%% open xml summary file
xmlSummaryFID = fopen([eventDirectory "/" xmlSummaryFile], "w");

%% write channel definitions
fprintf(xmlSummaryFID, "<?xml version='1.0' encoding='utf-8' ?>\n");
fprintf(xmlSummaryFID, "<LIGO_LW>\n");
fprintf(xmlSummaryFID, "<Table Name=\"bccscan:summary:table\">\n");
fprintf(xmlSummaryFID, "<Column Name=\"bccscan:summary:channelName\" Type=\"lstring\"/>\n");
fprintf(xmlSummaryFID, "<Column Name=\"bccscan:summary:peakTime\" Type=\"real_8\"/>\n");
fprintf(xmlSummaryFID, "<Column Name=\"bccscan:summary:peakMeanFrequency\" Type=\"real_8\"/>\n");
fprintf(xmlSummaryFID, "<Column Name=\"bccscan:summary:peakBandwidth\" Type=\"real_8\"/>\n");
fprintf(xmlSummaryFID, "<Column Name=\"bccscan:summary:peakSNR\" Type=\"real_8\"/>\n");
fprintf(xmlSummaryFID, "<Stream Name=\"bccscan:summary:table\" Type=\"Local\" Delimiter=\",\">\n");

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            initialize html report                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% report status
if debugLevel >= 0,
  fprintf(1, 'opening html report...\n');
endif

%% open web page
htmlFID = fopen([eventDirectory "/" htmlFile], "w");

%% begin html
fprintf(htmlFID, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"\n");
fprintf(htmlFID, "\"http://www.w3.org/TR/html4/strict.dtd\">\n");
fprintf(htmlFID, "<html>\n");
fprintf(htmlFID, "<head>\n");
fprintf(htmlFID, "<title>%s at %s</title>\n", eventType, eventTimeString);
fprintf(htmlFID, "<link rel=\"stylesheet\" href=\"bccstyle.css\" type=\"text/css\">\n");
fprintf(htmlFID, "<script type=\"text/javascript\">\n");
fprintf(htmlFID, "function showImage(gpsTime, channelName, timeRanges, imageType) {\n");
fprintf(htmlFID, "  for (var timeRangeIndex in timeRanges) {\n");
fprintf(htmlFID, "    var imageBaseName =\n");
fprintf(htmlFID, "      gpsTime + \"_\" + channelName + \"_\" + timeRanges[timeRangeIndex];\n");
fprintf(htmlFID, "    document.getElementById(\"a_\" + imageBaseName).href =\n");
fprintf(htmlFID, "      imageBaseName + \"_\" + imageType + \".png\";\n");
fprintf(htmlFID, "    document.getElementById(\"img_\" + imageBaseName).src =\n");
fprintf(htmlFID, "      imageBaseName + \"_\" + imageType + \"_thumbnail.png\";\n");
fprintf(htmlFID, "  }\n");
fprintf(htmlFID, "}\n");
fprintf(htmlFID, "function toggleVisible(division) {\n");
fprintf(htmlFID, "  if (document.getElementById(\"div_\" + division).style.display == \"none\") {\n");
fprintf(htmlFID, "    document.getElementById(\"div_\" + division).style.display = \"block\";\n");
fprintf(htmlFID, "    document.getElementById(\"input_\" + division).checked = true;\n");
fprintf(htmlFID, "  } else {\n");
fprintf(htmlFID, "    document.getElementById(\"div_\" + division).style.display = \"none\";\n");
fprintf(htmlFID, "    document.getElementById(\"input_\" + division).checked = false;\n");
fprintf(htmlFID, "  } \n");
fprintf(htmlFID, "}\n");
fprintf(htmlFID, "function gotoSection(section) {\n");
fprintf(htmlFID, "  document.getElementById(\"div_\" + section).style.display = \"block\";\n");
fprintf(htmlFID, "  document.getElementById(\"input_\" + section).checked = true;\n");
fprintf(htmlFID, "  window.location.hash = section;\n");
fprintf(htmlFID, "}\n");
fprintf(htmlFID, "</script>\n");
fprintf(htmlFID, "</head>\n");
fprintf(htmlFID, "<body>\n");
fprintf(htmlFID, "\n");
fprintf(htmlFID, "<div class=\"title\">\n");
fprintf(htmlFID, "<h1>%s at %s</h1>\n", eventType, eventTimeString);
fprintf(htmlFID, "</div>\n");
fprintf(htmlFID, "\n");

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           add index to html report                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf(htmlFID, "<div class=\"index\">\n");
fprintf(htmlFID, "<h2>Index</h2>\n");
fprintf(htmlFID, "<p>\n");
for sectionNumber = 1 : numberOfSections,
  fprintf(htmlFID, "<a href=\"javascript:gotoSection('%s')\">\n",  ...
          sectionIndex{sectionNumber});
  fprintf(htmlFID, "%s</a><br />\n", ...
          sectionIndex{sectionNumber});
endfor
fprintf(htmlFID, "<a href=\"log.txt\">\n");
fprintf(htmlFID, "Scan Log</a><br />\n");
fprintf(htmlFID, "<a href=\"%s\">\n", textSummaryFile);
fprintf(htmlFID, "Text Summary</a><br />\n");
fprintf(htmlFID, "<a href=\"%s\">\n", xmlSummaryFile);
fprintf(htmlFID, "XML Summary</a><br />\n");
fprintf(htmlFID, "<a href=\"configuration.txt\">\n");
fprintf(htmlFID, "Configuration</a><br />\n");
fprintf(htmlFID, "<a href=\"http://www.ligo.caltech.edu/~shourov/q/bccscan/\">\n");
fprintf(htmlFID, "Documentation</a><br />\n");
fprintf(htmlFID, "</p>\n");
fprintf(htmlFID, "</div>\n");
fprintf(htmlFID, "\n");
fprintf(htmlFID, "<div class=\"content\">\n");
fprintf(htmlFID, "\n");

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                       loop over configuration channels                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% initialize section status
inSectionFlag = 0;

%% begin loop over channel configurations
for channelNumber = 1 : numberOfChannels,
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                        add section to html report                          %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% check for start of new section
  sectionNumbers = find(sectionStart == channelNumber);
  
  %% if start of new section
  if ~isempty(sectionNumbers),
    
    %% begin loop over new sections
    for sectionNumber = sectionNumbers,

      %% end any previously existing section
      if inSectionFlag == 1,
        fprintf(htmlFID, "</div>\n");
        fprintf(htmlFID, "\n");
      endif

      %% label section
      fprintf(htmlFID, "<a name=\"%s\"></a>\n", sectionIndex{sectionNumber});
      fprintf(htmlFID, "<h2 class=\"%s\">\n", sectionIndex{sectionNumber});

      fprintf(htmlFID, "<input id=\"input_%s\" type=\"checkbox\" %s\n", ...
              sectionIndex{sectionNumber}, sectionChecked{sectionNumber});
      fprintf(htmlFID, "       onclick=\"toggleVisible('%s');\" />\n", ...
              sectionIndex{sectionNumber});
      fprintf(htmlFID, "%s\n", sectionName{sectionNumber});
      fprintf(htmlFID, "</h2>\n");
      fprintf(htmlFID, "\n");
      fprintf(htmlFID, "<div id=\"div_%s\" style=\"display: %s;\">\n", ...
              sectionIndex{sectionNumber}, sectionDisplay{sectionNumber});
      fprintf(htmlFID, "\n");
      
      %% record start of section
      inSectionFlag = 1;
      
      %% transcribe context information
      if any(strcmp(sectionIndex{sectionNumber}, {"Context", "Timing"})),
        fprintf(htmlFID, "<!-- BEGIN CONTEXT -->\n");
        fprintf(htmlFID, "\n");
	contextFullPath=[eventDirectory "/" contextFile];
	
        if exist(contextFullPath,"file")
          context = textread([eventDirectory "/" contextFile], "%s");
        else
          context = [];
        endif
        for lineNumber = 1 : length(context),
          fprintf(htmlFID, "%s\n", context{lineNumber});
        endfor
        fprintf(htmlFID, "\n");
        fprintf(htmlFID, "<!-- END CONTEXT -->\n");
        fprintf(htmlFID, "\n");
      elseif (strcmp(sectionIndex{sectionNumber}, "Parameters")),
        fprintf(htmlFID, "<!-- BEGIN PARAMETERS -->\n");
        fprintf(htmlFID, "\n");
        fprintf(htmlFID, "<!-- END PARAMETERS -->\n");
        fprintf(htmlFID, "\n");
      elseif (strcmp(sectionIndex{sectionNumber}, "Notes")),
        fprintf(htmlFID, "<!-- BEGIN NOTES -->\n");
        fprintf(htmlFID, "\n");
        fprintf(htmlFID, "<!-- END NOTES -->\n");
        fprintf(htmlFID, "\n");
      endif
      
      %% end loop over new sections
    endfor
    
    %% otherwise, continue
  endif
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %               identify statistically significant channels                  %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
  %% extract search configuration for channel
  fieldNames=fieldnames(configuration{channelNumber});
  for fieldNumber=1:length(fieldNames)
    eval(sprintf("%s = configuration{channelNumber}.(fieldNames{fieldNumber});",fieldNames{fieldNumber}));
  endfor

  %% whitening parameters
  whitening.duration = whiteningDuration;
  
  %% chirplet chain parameters
  chirplet.tileDuration = searchWindowDuration;
  chirplet.tileOverlap = searchWindowOverlap;
  chirplet.chirpletDuration = chirpletDuration;
  chirplet.chirpingRateLimit1 = chirpingRateLimit1;
  chirplet.chirpingRateLimit2 = chirpingRateLimit2;

  %% string name of channel
  if iscell(channelName),
    channelNameString = strrep(deblank(sprintf("%s ", channelName{:})), " ", "-");
  else
    channelNameString = channelName;
  end

  %% standardize channel names
  channelNameString = strrep(channelNameString, ";", ":");

  %% display status
  if debugLevel >= 0,
    fprintf(1, "processing channel %s...\n", channelNameString);
  end

  %% censor channels used for "blind" injections
  if any(strcmp(channelNameString, {"H1:LSC-ETMY_EXC_DAQ", ...
                                    "H2:LSC-ETMY_EXC_DAQ", ...
                                    "L1:LSC-ETMY_EXC_DAQ"})),
    fprintf(1, "  WARNING: %s censored for \"blind\" injections.\n", ...
            channelNameString);
    continue
  end

  %% override censor for channels used in "blind" injections
  if any(strcmp(channelNameString, {"H1:LSC-ETMY_EXC_DAQ!", ...
                                    "H2:LSC-ETMY_EXC_DAQ!", ...
                                    "L1:LSC-ETMY_EXC_DAQ!"})),
    channelName = strrep(channelName, "!", "");
    channelNameString = strrep(channelName, "!", "");
    fprintf(1, "  WARNING: overriding censor for %s.\n", ...
            channelNameString);
  end
  
  %% find closest sample time to event time
  centerTime = floor(eventTime) + round((eventTime - floor(eventTime)) * ...
                     sampleFrequency) / sampleFrequency;

  %% determine segment start and stop times
  startTime = centerTime - searchTimeRange / 2;
  stopTime = centerTime + searchTimeRange / 2;

  %% read data from frame file
  if debugLevel >= 2,
    fprintf(1, "  reading data...\n");
  end
  timeShifts = [];
  [rawData, rawSampleFrequency] = qreaddata(frameCache, channelName, frameType, ...
                startTime, stopTime, timeShifts, debugLevel);


  %% if any injections are requested
  injectionChannel=find(~strcmp(upper(injectionName), "NONE") & ...
                         ~strcmp(upper(injectionType), "NONE") & ...
                         (injectionFactor ~= 0));
  if injectionChannel,

    %% report status
    fprintf(1, "  reading/producing injection data\n");
    
    
    %% read injection data
    [injectionData, injectionSampleFrequency] = ...
        qreaddata(frameCache, injectionName, injectionType, startTime, ...
                  stopTime, 0.0, debugLevel);
    
    %% test for missing injection data
    if any(injectionSampleFrequency == 0),
      fprintf(1, "bccscan.m: error reading injection data\n");
      errorFlag = 1;
      continue;
    endif
    
    %% test for inconsistent injection data
    if any(injectionSampleFrequency(injectionChannel) ~= ...
           sampleFrequency(injectionChannel)),
      error("bccscan.m: inconsistent detector and injection data");
    endif
    
    %% add injection data to detector data
    data{channelNumber} = data{channelNumber} + injectionFactor * injectionData;
    
    %% free injection memory
    clear injectionData;
    clear injectionSampleFrequency;
    
  endif
  
  %% report status
  if debugLevel >= 2,
      fprintf(1, "  resampling and highpass filtering data\n");
  endif

  %% resample data
  highPassedData = bccresamplehighpass(rawData, rawSampleFrequency, sampleFrequency);

  %% report status
  if debugLevel >= 2,
      fprintf(1, "  whitening data\n");
  endif

  %% whitening data
  [whitenedData, whitening] = bccwhitening(highPassedData,whitening,chirplet.tileDuration,sampleFrequency);

  %% report status
  if debugLevel >= 2,
      fprintf(1, "  trigger generation\n");
  endif

  %% generate triggers
  withTrace=true;
  triggers = bccsearch(whitenedData,chirplet,whitening,sampleFrequency,withTrace);

  if isempty(triggers{channelNumber})
    continue
  endif

  %% report status
  if debugLevel >= 2,
      fprintf(1, "  downselect triggers\n");
  endif

  %% downselect triggers
  if alwaysPlotFlag
    triggers=bccselect(triggers,channelName,startTime,-Inf);
  else
    triggers=bccselect(triggers,channelName,startTime,snrThreshold);
  endif

### compute threshold for a given FA rate here! ###

  %% if channel not significant
  if isempty(triggers{channelNumber}.snr)

    %% report status and skip to the next channel
    if debugLevel >= 1,
      fprintf(1, ["  channel not significant at threshold %7.1f\n"], snrThreshold);
    endif
    fprintf(textSummaryFID, "%-30s %#019.9f %#09.3e %#09.3e %#09.3e\n", ...
            channelNameString, 0, 0, 0, 0, 0);
    fprintf(xmlSummaryFID,  "\"%s\",%#019.9f,%#09.3e,%#09.3f,%#09.3e,\n", ...
            channelNameString, 0, 0, 0, 0, 0);
    continue
    
    %% end test of channel significance
  endif
  
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                      add channel to summary report                         %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
  %% identify most significant tile
  if debugLevel >= 2,
    fprintf(1, "  identify most significant tile\n");
  endif
  
  %% most significant tile properties
  mostSignificant{channelNumber}.time = triggers{channelNumber}.time(1);
  mostSignificant{channelNumber}.frequency = triggers{channelNumber}.frequency{1};
  mostSignificant{channelNumber}.meanFrequency = triggers{channelNumber}.meanFrequency(1);
  mostSignificant{channelNumber}.bandwidth = triggers{channelNumber}.bandwidth(1);
  mostSignificant{channelNumber}.snr = triggers{channelNumber}.snr(1);
  
  %% write most significant tile properties to text summary file
  if debugLevel >= 2,
    fprintf(1, "  writing text summary...\n");
  endif
  fprintf(textSummaryFID, ...
          "%-30s %#019.9f %#09.3e %#09.3e %#09.3e \n", ...
          channelNameString, mostSignificant{channelNumber}.time, mostSignificant{channelNumber}.meanFrequency, ...
          mostSignificant{channelNumber}.bandwidth, mostSignificant{channelNumber}.snr);
  
  %% write most significant tile properties to xml summary file
  if debugLevel >= 2,
    fprintf(1, "  writing xml summary...\n");
    end
  fprintf(xmlSummaryFID, ...
          "\"%s\",%#019.9f,%#09.3e,%#09.3e,%#09.3e,\n", ...
          channelNameString, mostSignificant{channelNumber}.time, ...
	  mostSignificant{channelNumber}.meanFrequency, ...
          mostSignificant{channelNumber}.bandwidth, mostSignificant{channelNumber}.snr);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                        add channel to html report                          %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
  % html for most significant tile properties
  mostSignificantTimeHtml = sprintf("%.3f", mostSignificant{channelNumber}.time);
  mostSignificantMeanFrequencyHtml = ...
      sprintf("%.1f&times;10<sup>%d</sup>", ...
              mostSignificant{channelNumber}.meanFrequency / ...
              10^floor(log10(mostSignificant{channelNumber}.meanFrequency)), ...
              floor(log10(mostSignificant{channelNumber}.meanFrequency)));
  mostSignificantBandwidthHtml = ...
      sprintf("%.1f&times;10<sup>%d</sup>", ...
              mostSignificant{channelNumber}.bandwidth / ...
              10^floor(log10(mostSignificant{channelNumber}.bandwidth)), ...
              floor(log10(mostSignificant{channelNumber}.bandwidth)));
  mostSignificantSNRHtml = ...
      sprintf("%.1f&times;10<sup>%d</sup>", ...
              mostSignificant{channelNumber}.snr / ...
              10^floor(log10(mostSignificant{channelNumber}.snr)), ...
              floor(log10(mostSignificant{channelNumber}.snr)));

  % begin html channel entry
  if debugLevel >= 2,
    fprintf(1, "  writing html report...\n");
  end
  fprintf(htmlFID, "<a name=\"%s\"></a>\n", channelNameString);
  fprintf(htmlFID, "<h3>\n");
  fprintf(htmlFID, "<input id=\"input_%s\" type=\"checkbox\" checked\n", ...
          channelNameString);
  fprintf(htmlFID, "       onclick=\"toggleVisible('%s');\" />\n", ...
          channelNameString);
  fprintf(htmlFID, ["<a href=\"http://ldas-jobs.ligo.caltech.edu/cgi-bin/chanwiki?%s\">\n"], channelNameString);
  fprintf(htmlFID, "%s\n", channelNameString);
  fprintf(htmlFID, "</a>\n");
  fprintf(htmlFID, "</h3>\n");
  fprintf(htmlFID, ["(t = %s s,\n" ...
                    " f = %s Hz,\n" ...
                    " B = %s Hz,\n" ...
		    " SNR = %s,\n" ...
                    "<br />\n"], ...
          mostSignificantTimeHtml, mostSignificantMeanFrequencyHtml, ...
          mostSignificantBandwidthHtml, mostSignificantSNRHtml);
  fprintf(htmlFID, "<div id=\"div_%s\" style=\"display: block;\">\n", ...
          channelNameString);

  timeRangeString = sprintf("'%.2f', ", plotTimeRanges);
  timeRangeString = timeRangeString(1 : end - 2);

  fprintf(htmlFID, "  time series:\n");
  fprintf(htmlFID, "<a href=\"javascript:showImage('%s', '%s',\n", eventTimeString, channelNameString);
  fprintf(htmlFID, "[%s], 'timeseries_raw');\">raw</a>,\n", timeRangeString);
  fprintf(htmlFID, "<a href=\"javascript:showImage('%s', '%s',\n", eventTimeString, channelNameString);
  fprintf(htmlFID, "[%s], 'timeseries_highpassed');\">high passed</a>,\n", timeRangeString);
  fprintf(htmlFID, "<a href=\"javascript:showImage('%s', '%s',\n", eventTimeString, channelNameString);
  fprintf(htmlFID, "[%s], 'timeseries_whitened');\">whitened</a>\n", timeRangeString);
  fprintf(htmlFID, "| spectrogram:\n");
  fprintf(htmlFID, "<a href=\"javascript:showImage('%s', '%s',\n", eventTimeString, channelNameString);
  fprintf(htmlFID, "[%s], 'spectrogram_raw');\">raw</a>,\n", ...
          timeRangeString);
  fprintf(htmlFID, "<a href=\"javascript:showImage('%s', '%s',\n", ...
          eventTimeString, channelNameString);
  fprintf(htmlFID, "[%s], 'spectrogram_whitened');\">whitened</a>,\n", ...
          timeRangeString);
  fprintf(htmlFID, "<a href=\"javascript:showImage('%s', '%s',\n", ...
          eventTimeString, channelNameString);
  fprintf(htmlFID, "[%s], 'spectrogram_autoscaled');\">autoscaled</a>\n", ...
          timeRangeString);
  fprintf(htmlFID, "<br />\n");

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                       begin loop over time ranges                          %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % begin loop over time ranges
  for plotTimeRange = plotTimeRanges,

    % disable X11 output
    handle = figure (1);
    set (handle, "visible", "off");

    % report status
    if debugLevel >= 2,
      fprintf(1, "  processing %.2f second time range...\n", plotTimeRange);
    end

    % determine start and stop times of figures
    plotStartTime = searchTimeRange / 2 - plotTimeRange / 2;
    plotStopTime = searchTimeRange / 2 + plotTimeRange / 2;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                         plot raw time series                             %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot raw time series
    if debugLevel >= 2,
      fprintf(1, "    plotting raw time series...\n");
    end
    clf;
    bcctimeseries(rawData, sampleFrequency, startTime, centerTime, ...
                plotTimeRange * [-1 +1] / 2, channelNameString);

    % print raw time series to file
    if debugLevel >= 2,
      fprintf(1, "    printing raw time series...\n");
    end
    imageFile = sprintf("%s_%s_%.2f_timeseries_raw.png", ...
                        eventTimeString, channelNameString, plotTimeRange);
    print("-dpng", [eventDirectory "/" imageFile]);

    % create thumbnail image
    thumbnailFile = sprintf("%s_%s_%.2f_timeseries_raw_thumbnail.png", ...
                            eventTimeString, channelNameString, plotTimeRange);
    unix(["convert -resize 300x " eventDirectory "/" imageFile ...
          " -strip -depth 8 -colors 256 " eventDirectory "/" thumbnailFile]);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                 plot high pass filtered time series                      %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot high pass filtered time series
    if debugLevel >= 2,
      fprintf(1, "    plotting high pass filtered time series...\n");
    end
    bcctimeseries(highPassedData, sampleFrequency, startTime, centerTime, ...
                plotTimeRange * [-1 +1] / 2, channelNameString);

    % print high pass filtered time series to file
    if debugLevel >= 2,
      fprintf(1, "    printing high passed time series...\n");
    end
    imageFile = sprintf("%s_%s_%.2f_timeseries_highpassed.png", ...
                        eventTimeString, channelNameString, plotTimeRange);
    print("-dpng", [eventDirectory "/" imageFile]);

    % create thumbnail image
    thumbnailFile = sprintf("%s_%s_%.2f_timeseries_highpassed_thumbnail.png", ...
                            eventTimeString, channelNameString, plotTimeRange);
    unix(["convert -resize 300x " eventDirectory "/" imageFile ...
          " -strip -depth 8 -colors 256 " eventDirectory "/" thumbnailFile]);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                      plot whitened time series                           %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot whitened time series
    if debugLevel >= 2,
      fprintf(1, "    plotting whitened time series...\n");
    end
    bcctimeseries(whitenedData, sampleFrequency, startTime, centerTime, ...
                plotTimeRange * [-1 +1] / 2, channelNameString);

    % print whitened time series to file
    if debugLevel >= 2,
      fprintf(1, "    printing whitened time series...\n");
    end
    imageFile = sprintf("%s_%s_%.2f_timeseries_whitened.png", ...
                        eventTimeString, channelNameString, plotTimeRange);
    print("-dpng", [eventDirectory "/" imageFile]);

    % create thumbnail image
    thumbnailFile = sprintf("%s_%s_%.2f_timeseries_whitened_thumbnail.png", ...
                            eventTimeString, channelNameString, plotTimeRange);
    unix(["convert -resize 300x " eventDirectory "/" imageFile ...
          " -strip -depth 8 -colors 256 " eventDirectory "/" thumbnailFile]);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                         plot raw spectrogram                             %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot raw spectrogram
    if debugLevel >= 2,
      fprintf(1, "    plotting raw spectrogram...\n");
    end

    clf;
    bccspectrogram(rawData, mostSignificant, sampleFrequency, startTime, centerTime, ...
		   plotTimeRange * [-1 +1] / 2, plotFrequencyRange,plotNormalizedEnergyRange,...
		   channelNameString, plotHorizontalResolution, plotVerticalResolution);    

    % print autoscaled spectrogram to file
    if debugLevel >= 2,
      fprintf(1, "    printing raw spectrogram...\n");
    end
    imageFile = sprintf("%s_%s_%.2f_spectrogram_raw.png", ...
                        eventTimeString, channelNameString, plotTimeRange);
    print("-dpng", [eventDirectory "/" imageFile]);

    % create thumbnail image
    thumbnailFile = sprintf("%s_%s_%.2f_spectrogram_raw_thumbnail.png", ...
                            eventTimeString, channelNameString, plotTimeRange);
    unix(["convert -resize 300x " eventDirectory "/" imageFile ...
          " -strip -depth 8 -colors 256 " eventDirectory "/" thumbnailFile]);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                      plot whitened spectrogram                           %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot whitened spectrogram
    if debugLevel >= 2,
      fprintf(1, "    plotting whitened spectrogram...\n");
    end

    clf;
    bccspectrogram(whitenedData, mostSignificant, sampleFrequency, startTime, centerTime, ...
		   plotTimeRange * [-1 +1] / 2, plotFrequencyRange,plotNormalizedEnergyRange,...
		   channelNameString, plotHorizontalResolution,plotVerticalResolution);

    % print whitened spectrogram to file
    if debugLevel >= 2,
      fprintf(1, "    printing whitened spectrogram...\n");
    end
    imageFile = sprintf("%s_%s_%.2f_spectrogram_whitened.png", ...
                        eventTimeString, channelNameString, plotTimeRange);
    print("-dpng", [eventDirectory "/" imageFile]);

    % create thumbnail image
    thumbnailFile = sprintf("%s_%s_%.2f_spectrogram_whitened_thumbnail.png", ...
                            eventTimeString, channelNameString, plotTimeRange);
    unix(["convert -resize 300x " eventDirectory "/" imageFile ...
          " -strip -depth 8 -colors 256 " eventDirectory "/" thumbnailFile]);

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                      plot autoscaled spectrogram                           %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % plot whitened spectrogram
    if debugLevel >= 2,
      fprintf(1, "    plotting autoscaled spectrogram...\n");
    end

    clf;
    autoNormalizedEnergyRange = [];
    bccspectrogram(whitenedData, mostSignificant, sampleFrequency, startTime, centerTime, ...
		   plotTimeRange * [-1 +1] / 2, plotFrequencyRange,autoNormalizedEnergyRange, ...
		   channelNameString, plotHorizontalResolution,plotVerticalResolution);

    % print whitened spectrogram to file
    if debugLevel >= 2,
      fprintf(1, "    printing autoscaled spectrogram...\n");
    end
    imageFile = sprintf("%s_%s_%.2f_spectrogram_autoscaled.png", ...
                        eventTimeString, channelNameString, plotTimeRange);
    print("-dpng", [eventDirectory "/" imageFile]);

    % create thumbnail image
    thumbnailFile = sprintf("%s_%s_%.2f_spectrogram_autoscaled_thumbnail.png", ...
                            eventTimeString, channelNameString, plotTimeRange);
    unix(["convert -resize 300x " eventDirectory "/" imageFile ...
          " -strip -depth 8 -colors 256 " eventDirectory "/" thumbnailFile]);


    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                      add images to html report                           %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % create html link
    if debugLevel >= 2,
      fprintf(1, "    linking images...\n");
    end
    fprintf(htmlFID, "<a id=\"a_%s_%s_%.2f\"\n", ...
            eventTimeString, channelNameString, plotTimeRange);
    fprintf(htmlFID, "   href=\"%s_%s_%.2f_spectrogram_whitened.png\">\n", ...
            eventTimeString, channelNameString, plotTimeRange);
    fprintf(htmlFID, "<img id=\"img_%s_%s_%.2f\"\n", ...
            eventTimeString, channelNameString, plotTimeRange);
    fprintf(htmlFID, "     src=\"%s_%s_%.2f_spectrogram_whitened_thumbnail.png\"\n", ...
            eventTimeString, channelNameString, plotTimeRange);
    fprintf(htmlFID, "     alt=\"%s_%s_%.2f\" /></a>\n", ...
            eventTimeString, channelNameString, plotTimeRange);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                        end loop over time ranges                           %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % end loop time ranges
  end

  % end html table row
  fprintf(htmlFID, "<br />\n");
  fprintf(htmlFID, "</div>\n");
  fprintf(htmlFID, "\n");

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     end loop over configuration channels                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% end loop over configuration channels
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                              close html report                               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
if debugLevel >= 0,
  fprintf(1, "closing html report...\n");
end

% end any previously existing section
if inSectionFlag == 1,
  fprintf(htmlFID, "</div>\n");
  fprintf(htmlFID, "\n");
end

% end html
fprintf(htmlFID, "</div>\n");
fprintf(htmlFID, "\n");
fprintf(htmlFID, "<div class=\"footer\">\n");
fprintf(htmlFID, "Created by user %s on %s at %s<br />\n", ...
        getenv("USER"), datestr(clock, 29), datestr(clock, 13));
fprintf(htmlFID, "</div>\n");
fprintf(htmlFID, "\n");
fprintf(htmlFID, "</body>\n");
fprintf(htmlFID, "</html>\n");

% close html file
fclose(htmlFID);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          close text summary report                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
if debugLevel >= 0,
  fprintf(1, "closing text summary...\n");
end

% close text summary file
fclose(textSummaryFID);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           close xml summary report                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
if debugLevel >= 0,
  fprintf(1, "closing xml summary...\n");
end

% end summary table
fprintf(xmlSummaryFID, "</Stream>\n");
fprintf(xmlSummaryFID, "</Table>\n");
fprintf(xmlSummaryFID, "</LIGO_LW>\n");

% close xml summary file
fclose(xmlSummaryFID);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                     exit                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report completion
if debugLevel >= 0,
  fprintf(1, "finished on %s at %s\n", ...
          datestr(clock, 29), datestr(clock, 13));
end

% close all figures and quit
% exit
