#!/bin/sh
#
# qscan.sh
#
# Shell script to run the Q Scan utility after setting the appropriate
# envioronment variables.
#
# usage: qscan.sh triggerTime [configurationFile frameCacheFile outputDirectory]
#
#   triggerTime         gps center time of trigger
#   configurationFile   qscan configuration file
#   frameCacheFile      readframedata frame cache file
#   outputDirectory     directory to put results
#
# If no configuration and frame cache file are specified, QScan selects
# between one of two default configurations depending on the requested
# time.  If the requested time is less than 5 days old, QScan selects the
# @online configuration if it exists.  Otherwise, QScan selects the
# @default configuration.
#
# QScan places the resulting image files and an index.html file for web
# based viewing of the results in a subdirectory of the specified output
# directory.  The subdirectory is named by the specified event time.  If no
# output directory is specified, QScan uses ${HOME}/public_html/qscans/ by
# default.
#
# If the specified frameCacheFile argument points to a directory instead of
# a file, qscan.sh will automatically generate a temporary frame cache file
# for the specified directory.
#
# This script sources the qsetup.sh shell script in order to setup the
# necessary environment for Q transform based tools.

# Shourov K. Chatterji
# shourov@ligo.caltech.edu

# $Id: bccscan.sh,v 1.1 2008/09/03 17:25:17 ecm Exp $

# full path to current directory
currentDirectory=`pwd`

# full path to qscan installation
cd `dirname $0`/..
qscanDirectory=`pwd`
cd ${currentDirectory}

# source q environment setup script
. ${qscanDirectory}/bin/qsetup.sh

# parse command line arguments
triggerTime=$1
if [ $# -eq 1 ]; then
  configurationFile=
  frameCacheFile=
  outputDirectory=${HOME}/public_html/qscans/
elif [ $# -eq 4 ]; then
  configurationFile=$2
  frameCacheFile=$3
  outputDirectory=$4
else
  echo "usage: `basename $0` triggerTime [configurationFile frameCacheFile outputDirectory]" 1>&2
  exit 1
fi

# output log information to standard out
quiet="true"
# quiet=

# determine platform name
platform=`uname`

# find last X11 tcp socket
if [ "${platform}" = "Linux" ]; then
  lastTcpSocket=`netstat -lnt | grep tcp | \
                 awk ' { print $4 } ' | grep ':60[0-9][0-9]$' | \
                 sed -e 's|.*:60||' -e 's|^0||' | \
                 sort -n | uniq | tail -1`
elif [ "${platform}" = "SunOS" ]; then
  lastTcpSocket=`netstat -aP tcp -f inet6 | grep '*.60' | \
                 awk ' { print $1 } ' | grep '\.60[0-9][0-9]$' | \
                 sed -e 's|.*\.60||' -e 's|^0||' | \
                 sort -n | uniq | tail -1`
fi

# find last X11 unix socket
lastUnixSocket=`/bin/ls /tmp/.X11-unix/X* 2>/dev/null | \
                sed -e 's|.*X||' | sort -n | tail -1`

# find last X11 display number
if [ "0${lastTcpSocket}" -gt "0${lastUnixSocket}" ]; then
  lastDisplay=${lastTcpSocket}
else
  lastDisplay=${lastUnixSocket}
fi

# find unused X11 display number
minimumDisplayNumber=50
if [ -z "${lastDisplay}" ]; then
  displayNumber=${minimumDisplayNumber};
elif [ ${lastDisplay} -lt ${minimumDisplayNumber} ]; then
  displayNumber=${minimumDisplayNumber};
else
  displayNumber=`expr ${lastDisplay} + 1`
fi

# create virtual X display
xvfbTrials=5
while [ ${xvfbTrials} -gt 0 ]; do
  if [ "${platform}" = "Linux" ]; then
    Xvfb :${displayNumber} -screen 0 1280x1024x24 -audit 0 -auth /dev/null -nolisten tcp 2>/dev/null &
  elif [ "${platform}" = "SunOS" ]; then
    if [ `uname -r` = "5.8" ]; then
      Xvfb :${displayNumber} -audit 0 -auth /dev/null 2>/dev/null &
    else
      Xsun :${displayNumber} -audit 0 -auth /dev/null +nkeyboard +nmouse \
                             -nolisten tcp -defdepth 24 -dev vfb 2>/dev/null &
    fi
  fi
  xvfbProcessID=$!
  sleep 5
  xvfbSuccess=`/bin/ps -p ${xvfbProcessID} -o pid | sed -e 1d`
  if [ -n "${xvfbSuccess}" ]; then
    break;
  fi
  xvfbTrials=`expr ${xvfbTrials} - 1`
  displayNumber=`expr ${displayNumber} + 1`
done
if [ -z "${xvfbSuccess}" ]; then
  echo "error: unable to open display" 1>&2
  exit 1
fi

# destroy vitual X display on exit
trap "kill ${xvfbProcessID} >/dev/null 2>&1;" EXIT

# point to virtual X display
DISPLAY=":${displayNumber}.0"
export DISPLAY

# prevent core dumps
ulimit -c 0

# create output directory
mkdir -p ${outputDirectory}

# event subdirectory name
eventDirectory=${outputDirectory}/${triggerTime}

# test for previously existing scan
if [ -d ${eventDirectory} ]; then
  echo "error: ${eventDirectory} already exists" 1>&2
  kill ${xvfbProcessID}
  exit 1
fi

# create event directory
mkdir -p ${eventDirectory}

# predefined configuration file directory
configurationDirectory=${qscanDirectory}/configurations

# predefined frame cache file directory
frameCacheDirectory=${qscanDirectory}/framecaches

# round trigger time down to nearest integer GPS second
integerTriggerTime=`echo ${triggerTime} | sed -e s'|\..*$||'`

# start time of online data availability
if [ "${LIGOTOOLS:-unset}" != "unset" ]; then
  onlineStartTime=`tconvert now - 5days`
else
  onlineStartTime=${integerTriggerTime}
fi

# default configuration file
if [ -z "${configurationFile}" ]; then
  if [ ${integerTriggerTime} -gt ${onlineStartTime} ]; then
    configurationFile="@online"
  else
    configurationFile="@default"
  fi
fi

# default frame cache file
if [ -z "${frameCacheFile}" ]; then
  if [ ${integerTriggerTime} -gt ${onlineStartTime} ]; then
    frameCacheFile="@online"
  else
    frameCacheFile="@default"
  fi
fi

# configuration file for online analysis
if [ "${configurationFile}" = "@online" ]; then
  if [ -f "${configurationDirectory}/online.txt" -o \
       -h "${configurationDirectory}/online.txt" ]; then
    configurationFile=${configurationDirectory}/online.txt
  else
    configurationFile="@default"
  fi
fi

# frame cache file for online analysis
if [ ${frameCacheFile} = "@online" ]; then
  if [ -d "/frames/full" ]; then
    frameCacheFile="/frames/full"
  elif [ -f "/virgoData/ffl/raw.ffl" ]; then
    frameCacheFile="/virgoData/ffl/raw.ffl"
  else
    frameCacheFile="@default"
  fi
fi

# frame cache file for default analysis
if [ ${frameCacheFile} = "@default" ]; then
  if [ ${integerTriggerTime} -lt 728955008 ]; then
    frameCacheFile="@S1"
  elif [ ${integerTriggerTime} -lt 751505200 ]; then
    frameCacheFile="@S2"
  elif [ ${integerTriggerTime} -lt 757711008 ]; then
    frameCacheFile="@S3"
  elif [ ${integerTriggerTime} -lt 792866944 ]; then
    frameCacheFile="@A3"
  elif [ ${integerTriggerTime} -lt 796144256 ]; then
    frameCacheFile="@S4"
  elif [ ${integerTriggerTime} -lt 815153408 ]; then
    frameCacheFile="@A4"
  else
    frameCacheFile="@S5"
  fi
fi

# handle predefined configuration files
configurationFile=`echo ${configurationFile} | \
                   sed -e "s|^@\(.*\)$|${configurationDirectory}/\1.txt|"`

# handle predefined frame cache files
frameCacheFile=`echo ${frameCacheFile} | \
                sed -e "s|^@\(.*\)$|${frameCacheDirectory}/\1.txt|"`

# generate frame cache file if requested
if [ -d "${frameCacheFile}" ]; then
  frameFileDirectory=${frameCacheFile}
  dateString=`date +%Y%m%d%H%M%S`
  frameCacheFile=/tmp/framecache_${dateString}_${triggerTime}_$$.txt
  rm -f ${frameCacheFile}
  createframecache.pl ${frameCacheFile} ${frameFileDirectory} >/dev/null
  temporaryFrameCacheFile=true
elif [ -n "`echo ${frameCacheFile} | grep '\.ffl$'`" ]; then
  fflFile=${frameCacheFile}
  dateString=`date +%Y%m%d%H%M%S`
  frameCacheFile=/tmp/framecache_${dateString}_${triggerTime}_$$.txt
  rm -f ${frameCacheFile}
  convertfflcache.pl ${fflFile} ${frameCacheFile}
  temporaryFrameCacheFile=true
fi

# open context file
contextFile=${eventDirectory}/context.html
rm -f ${contextFile}
touch ${contextFile}

# add timing, detector state, data quality, and detector log information
if [ -n "`grep Timing ${configurationFile}`" ]; then
  if [ "${LIGOTOOLS:-unset}" != "unset" ]; then
    qcontext.sh ${triggerTime} ${contextFile} timing logs
  else
    echo '<h3>Context information is not available<h3><br />' >>${contextFile}
  fi
elif [ -n "`grep Context ${configurationFile}`" ]; then
  if [ "${LIGOTOOLS:-unset}" != "unset" ]; then
    qcontext.sh ${triggerTime} ${contextFile} timing segments logs
  else
    echo '<h3>Context information is not available<h3><br />' >>${contextFile}
  fi
fi

# copy html style sheet to event directory
cp ${qscanDirectory}/bin/qstyle.css ${eventDirectory}

# log file name
logFile=${eventDirectory}/log.txt

# call the qscan executable
if [ -n "${quiet}" ]; then
  qscan ${triggerTime} ${configurationFile} ${frameCacheFile} ${outputDirectory} \
        >${logFile} 2>&1
else
  qscan ${triggerTime} ${configurationFile} ${frameCacheFile} ${outputDirectory} \
        2>&1 | tee ${logFile}
fi

# remove any temporary frame cache file
if [ "${temporaryFrameCacheFile:-false}" = "true" ]; then
  rm -f ${frameCacheFile}
fi

# destroy vitual X display
kill ${xvfbProcessID}

# exit
exit 0
