function events = bccselect(triggers, channelNames, blockStartTime, threshold, maximumEvents, debugLevel)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        process command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% verify correct number of input arguments
error(nargchk(4, 6, nargin));

if (nargin < 5) || isempty(maximumEvents),
  maximumEvents = 1;
end
if (nargin < 6) || isempty(debugLevel),
  debugLevel = 1;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                       validate command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

numberOfChannels=length(triggers);

if (length(channelNames) ~= numberOfChannels)
  error("bccselect.m: the number of channel names is inconsistent with the number of analyses");
endif;

% begin loop over channels

events=cell(numberOfChannels);

for channelNumber = 1 : numberOfChannels,

  if isempty(triggers{channelNumber})
    continue
  endif

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                        compute GPS time of the triggers                    %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  if isfield(triggers{channelNumber},"timeInBlock")
    triggers{channelNumber}.time = blockStartTime + triggers{channelNumber}.timeInBlock;
  endif

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                   sort by decreasing SNR                                   %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % sort tile indices by normalized energy
  [sortedSnr, sortedIndices] = sort(triggers{channelNumber}.snr,"descend");

  % select SNR above threshold
  aboveThreshold = max(find(sortedSnr > threshold));

  % get list of field names available for this channel
  fieldNames = fieldnames(triggers{channelNumber});

  % number of field names available for this channel
  numberOfFields = length(fieldNames);

  % reorder significant tile properties by decreasing normalized energy
  for fieldNumber = 1 : numberOfFields,
    events{channelNumber}.(fieldNames{fieldNumber}) = ...
      triggers{channelNumber}.(fieldNames{fieldNumber})(sortedIndices(1:aboveThreshold));
  endfor;

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                    check for excessive number of events                    %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % determine number of significant tiles in channel
  numberOfEvents = length(events{channelNumber}.time);

  % if maximum allowable number of significant tiles is exceeded
  if numberOfEvents > maximumEvents,

    % issue warning
    if debugLevel > 0,
      fprintf(1, "WARNING: %s: maximum number of events exceeded.\n", channelNames{channelNumber});
    end

    % set overflow flag
    events{channelNumber}.overflowFlag = 1;

    % indices of most significant tiles
    maximumIndices = 1 : maximumEvents;

    % truncate lists of significant event properties
    for fieldNumber = 1 : numberOfFields,
      events{channelNumber}.(fieldNames{fieldNumber}) = ...
        events{channelNumber}.(fieldNames{fieldNumber})(maximumIndices);
    end

  % otherwise continue
  end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            end loop over channels                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% end loop over channels
end
