function handles = bccspectrogram(data, event, sampleFrequency, startTime, referenceTime, ...
                                timeRange, frequencyRange, normalizedEnergyRange, ...
				  channelNames, timeResolution, frequencyResolution)
% $Id: bccspectrogram.m,v 1.1 2008/09/03 17:25:18 ecm Exp $

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            hard coded parameters                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% maximal number of horizontal and vertical pixels in image
defaultTimeResolution = 1024;
defaultFrequencyResolution = 1024;

% spectrogram boundary
spectrogramLeft = 0.14;
spectrogramWidth = 0.80;
spectrogramBottom = 0.28;
spectrogramHeight = 0.62;
spectrogramPosition = [spectrogramLeft spectrogramBottom ...
                       spectrogramWidth spectrogramHeight];

% colorbar position
colorbarLeft = spectrogramLeft;
colorbarWidth = spectrogramWidth;
colorbarBottom = 0.12;
colorbarHeight = 0.02;
colorbarPosition = [colorbarLeft colorbarBottom ...
                    colorbarWidth colorbarHeight];

% time scales for labelling
millisecondThreshold = 0.5;
secondThreshold = 3 * 60;
minuteThreshold = 3 * 60 * 60;
hourThreshold = 3 * 24 * 60 * 60;
dayThreshold = 365.25 * 24 * 60 * 60;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        process command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% verify number of input arguments
error(nargchk(3, 11, nargin));

% apply default arguments
if (nargin < 4) || isempty(startTime),
  startTime = 0;
end
if (nargin < 5) || isempty(referenceTime),
  referenceTime = 0;
end
if (nargin < 6) || isempty(timeRange),
  timeRange = [-Inf Inf];
end
if (nargin < 7) || isempty(frequencyRange),
  frequencyRange = [0 sampleFrequency/2.0];
end
if (nargin < 8) || isempty(normalizedEnergyRange),
  normalizedEnergyRange = [];
end
if (nargin < 9) || isempty(channelNames),
  channelNames = [];
end
if (nargin < 10) || isempty(timeResolution),
  timeResolution = defaultTimeResolution;
end
if (nargin < 11) || isempty(frequencyResolution),
  frequencyResolution = defaultFrequencyResolution;
end

% force cell arrays
if ~iscell(data),
  transforms = mat2cell(data, size(data, 1), size(data, 2));
end
if ~isempty(channelNames) & ~iscell(channelNames),
  channelNames = mat2cell(channelNames, size(channelNames, 1), ...
                          size(channelNames, 2));
end

% force one dimensional cell arrays
data = data(:);
channelNames = channelNames(:);

% determine number of channels
numberOfChannels = length(data);

% provide default channel names
if isempty(channelNames),
  channelNames = cell(numberOfChannels, 1);
  for channelNumber = 1 : numberOfChannels,
    channelNames{channelNumber} = ['Channel ' int2str(channelNumber)];
  end
end

% force ranges to be monotonically increasing column vectors
timeRange = unique(timeRange(:));
frequencyRange = unique(frequencyRange(:));
if ~isempty(normalizedEnergyRange),
  normalizedEnergyRange = unique(normalizedEnergyRange(:));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                       validate command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% validate channel names
if ~isempty(channelNames) & (length(channelNames) ~= numberOfChannels),
  error(['channel names is inconsistent with number of transform channels']);
end

% check for valid sampling frequency
if sampleFrequency <= 0
  error('incorrect sampling rate');
end

% Check for two component range vectors
if length(timeRange) ~= 2,
  error('Time range must be two component vector [tmin tmax].');
end
if length(frequencyRange) ~= 2,
  error('Frequency range must be two component vector [fmin fmax].');
end
if ~isempty(normalizedEnergyRange) & length(normalizedEnergyRange) ~= 2,
  error('Normalized energy range must be two component vector [Zmin Zmax].');
end

% initialize handle vector
handles = zeros(numberOfChannels, 1);

% check consistency of the data lengths
dataLength = length(data{1});
if numberOfChannels > 1
  for channelNumber = 2 : numberOfChannels,
    if length(data{channelNumber}) ~= dataLength,
      error('Channels have inconsistent data lengths');
    endif
  endfor
endif

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          identify times to display                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

duration=dataLength/sampleFrequency;

% default start time for display is start time of available data
if timeRange(1) == -Inf,
  timeRange(1) = startTime - referenceTime;
end

% default stop time for display is stop time of available data
if timeRange(2) == +Inf,
  timeRange(2) = startTime - referenceTime + duration;
end

% validate requested time range
if (timeRange(1) < startTime - referenceTime) | ...
   (timeRange(2) > startTime - referenceTime + duration),
  error('requested time range exceeds available data');
end

% vector of times to display
times = linspace(min(timeRange), max(timeRange), timeResolution);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           begin loop over channels                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% loop over channels
for channelNumber = 1 : numberOfChannels,

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                           compute spectrogram                            %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    

    % determine length of analysis window
    windowLength = floor(sampleFrequency*(timeRange(2)-timeRange(1))/16);
    windowLength += !rem(windowLength,2); % next odd if even
    windowCenterIndex = (windowLength+1)/2;

    % generate analysis window
    window = bccwindow(windowLength);
    window=window(:)';

    % requested number of frequency bins
    frequencyBins = max(windowLength,frequencyResolution*sampleFrequency/diff(frequencyRange));
    frequencyBins = 2^nextpow2(frequencyBins);

    % window time slices
    windowedTimeSlices = zeros(frequencyBins,timeResolution);

    for column=1:timeResolution,
      timeIndex = round(sampleFrequency*(referenceTime+times(column)-startTime)); 
      availableIndices= -min([frequencyBins/2-1,windowCenterIndex-1,timeIndex-1]):...
	  min([frequencyBins/2-1,windowCenterIndex-1,dataLength-timeIndex]);
      indices = rem(frequencyBins+availableIndices,frequencyBins)+1;

      % apply window at each analysis times
      windowedTimeSlices(indices,column) = data{channelNumber}(timeIndex+availableIndices).*...
	  window(windowCenterIndex+availableIndices)/norm(window(windowCenterIndex+availableIndices));
    endfor;
    
    % Fourier transform
    normalizedEnergies=abs(fft(windowedTimeSlices)).^2; 
    clear windowedTimeSlices;
    

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                   identify frequency rows to display                     %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    frequencies=sampleFrequency*(0:frequencyBins/2)/frequencyBins;

    % find rows within requested frequency range
    rowIndices = find((frequencies >= min(frequencyRange)) & ...
                      (frequencies <= max(frequencyRange)));

    numberOfIndices=length(rowIndices);

    % select the required number of frequency bins
    selectedRowIndices=1:round(numberOfIndices/frequencyResolution):numberOfIndices;

    % vector of frequencies to display
    frequencies = frequencies(selectedRowIndices);

    % restrict to selected frequency range
    normalizedEnergies=normalizedEnergies(selectedRowIndices,:);

    % mean energies at each frequency bins
    meanEnergies=mean(normalizedEnergies.').';

    % normalize by mean energy
    normalizedEnergies=repmat(1./meanEnergies,1,size(normalizedEnergies,2)).*normalizedEnergies;

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                         set colormap scaling                             %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % if normalized energy range is not specified
    if isempty(normalizedEnergyRange),

      % normalized energy range to code on colormap
      colormapScale = [0 max(max(normalizedEnergies))];

    % or if autoscaling of upper limit is requested,
    elseif normalizedEnergyRange(2) == Inf,

      % normalized energy range to code on colormap
      colormapScale = [normalizedEnergyRange(1) max(max(normalizedEnergies))];

    % otherwise,
    else

      % use specified range
      colormapScale = normalizedEnergyRange(:).';

    % continue
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                            plot spectrogram                              %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % if plotting more than one figure
    if numberOfChannels > 1,

      % select figure to plot in
      figure(channelNumber);

    % continue
    end

    % reset figure
    clf;
    set(gca, 'FontSize', 16);

    % plot spectrogram
    if abs(diff(timeRange)) < millisecondThreshold,
      surf(times * 1e3, frequencies, normalizedEnergies, normalizedEnergies);
    elseif abs(diff(timeRange)) < secondThreshold,
      surf(times * 1, frequencies, normalizedEnergies, normalizedEnergies);
    elseif abs(diff(timeRange)) < minuteThreshold,
      surf(times / 60, frequencies, normalizedEnergies, normalizedEnergies);
    elseif abs(diff(timeRange)) < hourThreshold,
      surf(times / 3600, frequencies, normalizedEnergies, normalizedEnergies);
    elseif abs(diff(timeRange)) < dayThreshold,
      surf(times / 86400, frequencies, normalizedEnergies, normalizedEnergies);
    else
      surf(times / 31557600, frequencies, normalizedEnergies, normalizedEnergies);
    end

    % apply colormap scaling
    colormap('default');
    caxis(colormapScale);

    % set axis position
    set(gca, 'Position', spectrogramPosition);

    % set axis range
    if abs(diff(timeRange)) < millisecondThreshold,
      axis([min(timeRange) * 1e3 max(timeRange) * 1e3 ...
            min(frequencyRange) max(frequencyRange)]);
    elseif abs(diff(timeRange)) < secondThreshold,
      axis([min(timeRange) * 1 max(timeRange) * 1 ...
            min(frequencyRange) max(frequencyRange)]);
    elseif abs(diff(timeRange)) < minuteThreshold,
      axis([min(timeRange) / 60 max(timeRange) / 60 ...
            min(frequencyRange) max(frequencyRange)]);
    elseif abs(diff(timeRange)) < hourThreshold,
      axis([min(timeRange) / 3600 max(timeRange) / 3600 ...
            min(frequencyRange) max(frequencyRange)]);
    elseif abs(diff(timeRange)) < dayThreshold,
      axis([min(timeRange) / 86400 max(timeRange) / 86400 ...
            min(frequencyRange) max(frequencyRange)]);
    else
      axis([min(timeRange) / 31557600 max(timeRange) / 31557600 ...
            min(frequencyRange) max(frequencyRange)]);
    end

    % set view angle
    view(0,90);

    % disable coordinate grid
    grid off;

    % enable interpolated shading
    shading interp;

    % set y axis properties
    ylabel('Frequency [Hz]');
    set(gca, 'YScale', 'log');
    set(gca, 'TickDir', 'out');
    set(gca, 'TickLength', [0.01 0.025]);
    if min(frequencyRange) >= 0.0625,
      set(gca, 'YMinorTick', 'off');
      set(gca, 'YTick', 2.^(ceil(log2(min(frequencyRange))) : 1 : ...
                            floor(log2(max(frequencyRange)))));
    end

    % set vertical height based on frequency range

    % set x axis properties
    if abs(diff(timeRange)) < millisecondThreshold,
      xlabel('Time [milliseconds]');
    elseif abs(diff(timeRange)) < secondThreshold,
      xlabel('Time [seconds]');
    elseif abs(diff(timeRange)) < minuteThreshold,
      xlabel('Time [minutes]');
    elseif abs(diff(timeRange)) < hourThreshold,
      xlabel('Time [hours]');
    elseif abs(diff(timeRange)) < dayThreshold,
      xlabel('Time [days]');
    else
      xlabel('Time [years]');
    end

    % set title properties
    titleString = sprintf("%s at %.3f", channelNames{channelNumber}, referenceTime);
    titleString = strrep(titleString, '_', '\_');
    title(titleString);

    % set figure background color
    set(gca, 'Color', [1 1 1]);
    set(gcf, 'Color', [1 1 1]);
    set(gcf, 'InvertHardCopy', 'off');

    % append current axis handle to list of handles
    handles(channelNumber) = gca;

    % Display colorbar
    colorbar("SouthOutside");

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %                      plot event frequency path                           %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    if ~isempty(event{channelNumber})
      hold on
      eventTimeAxis = event{channelNumber}.time - referenceTime + ...
	  (0:length(event{channelNumber}.frequency)-1)/sampleFrequency;
      if abs(diff(timeRange)) < millisecondThreshold,
	handle = plot(eventTimeAxis * 1e3,event{channelNumber}.frequency, "y-");
      elseif abs(diff(timeRange)) < secondThreshold,
	handle = plot(eventTimeAxis * 1, event{channelNumber}.frequency, "y-");
      elseif abs(diff(timeRange)) < minuteThreshold,
	handle = plot(eventTimeAxis * 60, event{channelNumber}.frequency, "y-");
      elseif abs(diff(timeRange)) < hourThreshold,
	handle = plot(eventTimeAxis / 3600, event{channelNumber}.frequency, "y-");
      elseif abs(diff(timeRange)) < dayThreshold,
	handle = plot(eventTimeAxis / 86400, event{channelNumber}.frequency, "y-");
      else
	handle = plot(eventTimeAxis / 31557600, event{channelNumber}.frequency, "y-");
      endif
      set(handle,"linewidth",2);
      hold off
    endif
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            end loop over channels                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% end loop over channels
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          return to calling function                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% return to calling function
return;
