
function [data, whitening] = bccwhitening(rawData,whitening,param1,param2)

## -*- texinfo -*-
## @deftypefn {Function File}  {[@var{out} @var{filter}]=} bccwhitening(@var{data},@var{params},@var{duration},@var{freq})
## @deftypefnx {Function File}  {[@var{out}]=} bccwhitening(@var{data},@var{params},@var{freq})
## Applies a whitening filter. Compute the spectrum of @var{data} from
## a Welch estimate and take the inverse square root to obtain a
## whitening filter. The data are filtered in the frequency domain
## using the Overlap-and-add method. @var{params} is a structure with
## members @var{duration}, @var{overlap}, @var{windowType} and
## @var{double} which sets the characteristics of the whitening filter. 
## @var{params.duration} fixes the duration of the time
## intervals used for the spectrum estimate (hence the spectral
## resolution of the whitening filter) and @var{params.overlap}, how
## they overlap. @var{params.windowType} sets the type of the window
## (Hanning by default).  @var{params.double} is a flag to switch from
## "single" whitening (default) to "double" (use the inverse of the
## spectrum, no square-root). @var{freq} is the sampling frequency of
## the input data. The transfer function @var{filter} of the whitening
## filter can be optionally requested. Its spectral resolution is specified
## by @var{duration}.
## @end deftypefn

## Copyright (C) 2008 Eric Chassande-Mottin, CNRS (France)

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, see .

## $Id: bccwhitening.m,v 1.2 2008/09/15 14:24:08 ecm Exp $

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        process command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% verify correct number of input arguments
if nargout==1
  error(nargchk(3, 3, nargin));
  sampleFrequency=param1;
endif

if nargout==2
  error(nargchk(4, 4, nargin));
  filterDuration=param1;
  sampleFrequency=param2;
endif


% force cell arrays
if ~iscell(rawData),
  rawData = mat2cell(rawData, size(rawData, 1), size(rawData, 2));
end

% force one dimensional cell arrays
rawData = rawData(:);

% set default parameters
if ~isfield(whitening,"overlap") || isempty(whitening.overlap),
  whitening.overlap = 0.0;
end
if ~isfield(whitening,"double") || isempty(whitening.double),
  whitening.double = false;
end
if ~isfield(whitening,"windowType"),
  whitening.windowType = [];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                       validate command line arguments                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% determine number of channels
numberOfChannels = length(rawData);

% return if nothing to do
if numberOfChannels==0
  data=[];
  return
endif

% determine required data lengths
dataLength = length(rawData{1});

% validate data length
for channelNumber=1:numberOfChannels,
  if length(rawData{channelNumber}) ~= dataLength,
    error("bccwhitening.m: data lengths are not consistent");
  endif
endfor

% check whitening duration
if ~isfield(whitening,"duration"),
    error("bccwhitening.m: whitening duration is missing");
end

% determine window length
windowLength=fix(whitening.duration*sampleFrequency);
windowLength+=!rem(windowLength,2); % next odd if even

% determine frequency resolution
numberOfFreqBins=2^(nextpow2(windowLength)+1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            filter  data                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% generate overlap-and-add window
[window olaStride]=bccwindow(windowLength,whitening.windowType);

normalizationFactor = zeros(numberOfChannels,1);
for channelNumber = 1:numberOfChannels,

  % estimate spectrum
  spectrum = welch(rawData{channelNumber},window,whitening.overlap,numberOfFreqBins);

  % indices of non singular bins
  indices = find(spectrum > 0);

  % initialize whitening filter
  whiteningFilter = zeros(numberOfFreqBins,1);

  if whitening.double

    %% double whitening filter
    whiteningFilter(indices) = 1./spectrum(indices);

    %% normalize
    normalizationFactor(channelNumber) = sqrt(2.0 * sum(whiteningFilter) / numberOfFreqBins);
    whiteningFilter /= normalizationFactor(channelNumber);

  else

    %% single whitening filter
    whiteningFilter(indices) = 1./sqrt(spectrum(indices));

  endif

  data{channelNumber} = olafilter(whiteningFilter,windowLength,...
				  window,olaStride,rawData{channelNumber});

endfor


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     whitening filter at requested sampling                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargout == 2 

  %% compute window length
  nFilterSamples = fix(filterDuration * sampleFrequency);
  windowLength = nFilterSamples - !rem(windowLength,2);
  
  %% generate window
  window = bccwindow(windowLength,whitening.windowType);
  
  whitening.filter=cell(numberOfChannels,1);
  
  for channelNumber = 1:numberOfChannels,
    
    %% estimate spectrum
    spectrum = welch(rawData{channelNumber},window,whitening.overlap,nFilterSamples);
    
    %% non singular frequency bins
    indices = find(spectrum > 0);
    
    %% initialize filter
    whitening.filter{channelNumber} = zeros(nFilterSamples,1);
    
    if whitening.double
      whitening.filter{channelNumber}(indices) = ...
	  (normalizationFactor(channelNumber)^2*spectrum(indices)).^(-1);
    else
      whitening.filter{channelNumber}(indices) = (spectrum(indices)).^(-1/2);
    endif
    
  endfor

endif

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                    return                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% return to calling function
return;
