## INJECT_GWCHIRP  Gravitational wave chirp injection
##                 X=INJECT_GWCHIRP(NET,S,SNR,PHI,THETA,EPSILON,PSI,NORM,PAD,FS) generates network
##                 data in which a gravitational wave chirp from a specified source as been
##                 injected.
##
##                 NET     :   network descriptor
##                 S       :   (complex) GW chirp with the convention: s=plus + i cross
##                 SNR     :   signal to noise ratio (assuming white Gaussian noise with unit variance)
##                 PHI     :   source direction grid, azimuth angle from x-axis
##                 THETA   :   source direction grid, polar angle from zenith
##                 EPSILON :   inclination angle of the source orbital plane
##                 PSI     :   wave polarization angle
##                 NORM    :   normalization coefficients
##                 PAD     :   size of zero pad before and after the injected signal
##                 FS      :   sampling frequency
##                             (if FS=1.0, we work with normalized units. c=1.0 ms^-1)

##
## Copyright (C) 2007 Eric Chassande-Mottin, Olivier Rabaste CNRS (France)
##
## This program may be freely copied under the terms of the 
## GNU General Public License (GPL), version 2, or at your 
## option any later version.
##

## $Id: inject_gwchirp.m,v 1.4 2008/09/03 14:00:52 ecm Exp $

function x=inject_gwchirp(net,s,snr,phi,theta,epsilon,psi,norm,pad,fs)

  if (nargin<10)
    error("Usage: x=inject_gwchirp(net,s,snr,phi,theta,epsilon,psi,norm,pad,fs)");
  end;

  if common_size(norm,phi,theta),
    error("inject_gwchirp: size(norm) should be size(phi or theta) times length(det)");
  end;

  ## -- init network data
  d=length(net);
  N=length(s);
  x=zeros(2*pad+N,d);

  if (snr==0.0),
    return;
  end;

  ## -- antenna pattern and delays at source position
  ap=antenna(net,phi,theta);
  tplus=gelfand(2,+2,psi,epsilon,0.0);
  tminus=gelfand(2,-2,psi,epsilon,0.0);
  F=tplus*ap+tminus*conj(ap);
  global E;
  n=round(tof([E net],phi,theta,fs==1.0)*fs);

  if any(n>pad)
    error("inject_gwchirp: insufficient zero pad");
  end;

  ## -- inject waveform
  t=0:N-1;
  if (snr>0.0)

    ## -- calibrate to requested SNR
    calD=squeeze(sum(abs(ap).^2 .* norm,3));
    Delta=squeeze(sum(ap.^2 .* norm,3));

    delta=Delta/abs(Delta);
    factor1=(calD+abs(Delta))*(abs(conj(delta)*tminus+tplus)^2+abs(delta*tplus+tminus)^2);
    factor2=(calD-abs(Delta))*(abs(conj(delta)*tminus-tplus)^2+abs(delta*tplus-tminus)^2);
    factor=factor1+factor2;
    a=2*sqrt(2)*snr/sqrt(N*factor);

    for j=1:d,
      x(pad+n(:,:,j)+t,j)+=a*real(conj(F(:,:,j))*s);
    end;

  else

    ## -- no calibration
    for j=1:d,
      x(pad+n(:,:,j)+t,j)+=real(conj(F(:,:,j))*s);
    end;

  end;

endfunction;
