
// Copyright (C) 2008 Eric Chassande-Mottin, CNRS (France)

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see .

#include <octave/config.h>
#include <octave/defun-dld.h>
#include <octave/error.h>
#include <octave/gripes.h>
#include <octave/oct-obj.h>
#include <octave/pager.h>
#include <octave/quit.h>
#include <octave/variables.h>

static inline int
min ( int x, int y)
{
  return x < y ? x : y;
}

inline static int min (const int32NDArray& v)
{
  int l = v.length ();

  if (l == 0)
    return octave_NaN;
  
  int retval=v(0);
  for (int i=1; i<l; i++)
    retval=( (int) v(i) < retval ? (int) v(i) : retval);
  
  return retval;

}

inline static int max (const int32NDArray& v)
{
  int l = v.length ();

  if (l == 0)
    return octave_NaN;
  
  int retval=v(0);
  for (int i=1; i<l; i++)
    retval=( (int) v(i) > retval ? (int) v(i) : retval);
  
  return retval;

}

DEFUN_DLD (lcf, args, ,
  "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{out} =} lcf (@var{in},@var{index})\n\
Local auto or crosscorrelation.\n\
If @var{in} is a column vector, its local autocorrelation matrix is computed at indices @var{index}.\n\
If @var{in} has two columns, the local crosscorrelation matrix between the first and second column\n\
is computed at indices @var{index}.\n\
@end deftypefn\n\n")
{
  octave_value_list retval;
  
  int nargin = args.length ();
  
  if (nargin < 2)
    {
      print_usage();
      return retval;
    }

  /* get the data */

  ComplexMatrix in(args(0).complex_matrix_value());

  if (error_state) 
    { 
      gripe_wrong_type_arg("lcf",args(0));
      return retval; 
    }

  int N=in.rows();

  /* get the indices */

  int32NDArray idx(args(1).int32_array_value());

  if (error_state) 
    { 
      gripe_wrong_type_arg("lcf",args(1));
      return retval; 
    }

  if ((min(idx)<0)|(max(idx)>=N))
    {
      error("lcf: invalid index range");
      return retval;
    }

  /* define if we compute cross or autocorrelation */

  int first=0;
  int second=0;
  if (in.columns()==2)
    {
      first=0;
      second=1;
    }
  else if (in.columns() > 2)
    {
      error("lcf: IN should have two columns at most");
      return retval;
    }
  
  /* create output data */

  int c=idx.length();
  ComplexMatrix out(N,c,0);

  /* compute the cross/autocorrelation matrix */

  int taumax;
  for (int n=0 ; n<c ; n++)
    {    

      int ti= (int) idx(n)-1; 
      taumax=min(ti,N-1-ti);
      for (int tau=-taumax; tau<=taumax ; tau++)
	out((N+tau)%N,n)=in(ti+tau,first)*conj(in(ti-tau,second));
      
    }
  
  retval(0)=out;
  
  return retval;
}





