
// Copyright (C) 2008 Eric Chassande-Mottin, CNRS (France)

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see .

#include <octave/config.h>
#include <octave/defun-dld.h>
#include <octave/error.h>
#include <octave/gripes.h>
#include <octave/oct-obj.h>
#include <octave/utils.h>

#define RANDMAX 4096

int debug_flag=0;

static int is_power_of_two(int n)
{
  if (n>0)
    {
      while(!(n & 1))
	n = n>>1;
    }
  return(n==1);
}

inline static int min ( int x, int y)
{
  return x < y ? x : y;
}

inline static double getrand(void)
{ 
  static int first_visit=0;

  if (first_visit==0)
    srand(time(NULL));
  first_visit=1;

  return (rand() % RANDMAX)/((double) RANDMAX);
}

DEFUN_DLD (randchirp, args, ,
  "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {[@var{phase} @var{ifl} @var{index}]=} randchirp (@var{N},@var{b},@var{Nr1},@var{step},@var{Nr2})\n\
@deftypefnx {Loadable Function} {[@var{phase} @var{ifl} @var{index}]=} randchirp (@var{N},@var{b},@var{Nr1},@var{step},@var{Nr2},@var{f})\n\
Generate a random chain of chirplets. Chirplets are\n\
frequency modulated signals whose frequency joins linearly\n\
two points of the discrete frequency axis @var{f}.\n\
The regularity of the chirplet chain is specified by \n\
the parameters @var{b},@var{Nr1},@var{step} and @var{Nr2}.\n\
By default, the frequency samples regularly the range from 0\n\
to the Nyquist frequency 0.5 with @var{N}+1 samples.\n\
@end deftypefn\n\n")
{
  octave_value_list retval;
  
  int nargin=args.length ();
  
  if ((nargin < 5))
    {
      print_usage();
      return retval;
    }

  /* get the data */
  int N=args(0).int_value();

  if (error_state) 
    { 
      gripe_wrong_type_arg("randchirp",args(0));
      return retval; 
    }

  if (!is_power_of_two(N))
    {
      error("randchirp: size of chain should be a power of 2");
      return retval;
    }

  /* get the size of the interval */
  int b=args(1).int_value();

  if (error_state) 
    { 
      gripe_wrong_type_arg("randchirp",args(1));
      return retval; 
    }

  if (!is_power_of_two(b))
    {
      error("randchirp: chirplet size should be a power of 2");
      return retval;
    }

  if ((b<2)|(b>=N))
    {
      error("randchirp: chirplet size b should be 0 < b < N");
      return retval;
    }

  int Nt=(int) N/b;

  /* get the number of steps, constraint 1: 1st derivative */
  int Nr1=args(2).int_value();

  if (error_state) 
    { 
      gripe_wrong_type_arg("randchirp",args(2));
      return retval; 
    }

  /* get the step size*/
  int step =args(3).int_value();

  if (error_state) 
    { 
      gripe_wrong_type_arg("randchirp",args(3));
      return retval; 
    }

  /* get the number of steps, constraint 2: 2nd derivative */
  int Nr2 =args(4).int_value();

  if (error_state) 
    { 
      gripe_wrong_type_arg("randchirp",args(4));
      return retval; 
    }

  if (Nr2>=(2*Nr1))
      warning("randchirp: warning! no effective constraint on 2nd derivative");

  /* get the frequency axis */
  ColumnVector f;

  if (nargin > 5)
    {
      f = ColumnVector (args(5).vector_value ());
      
      if (error_state) 
	{ 
	  gripe_wrong_type_arg("randchirp",args(5));
	  return retval; 
	}

    }
  
  if (f.length() == 0)
    {
      /* set default f axis */
      f.resize(dim_vector(N+1));
      for (int k=0; k<=N; k++)
	f(k)=(double) k/(2.0*(double) N);
    }

  int Nf=f.length();

  /* pick first chirplet */
  ColumnVector m(Nt+1);
  m(0)=round((Nf-1)*getrand());

  int n=0,dn=0,mn=0;
  int dl=-Nr1, du=Nr1;

  // make steps ahead and decrease upper bound of the chirplet slope if necessary
  do
    {
      n=1; 
      dn=du;
      mn=(int) m(0)+dn;
      while ((n<=Nt)&(mn<Nf)&(dn>0))
	{
	  dn-=Nr2;
	  mn+=dn;
	  n++;
	}

      if (mn>=Nf)
	du-=step; 
      else
	break;
    }
  while (1);

  // make steps ahead and increase lower bound of the chirplet slope if necessary
  do
    {
      n=1; 
      dn=dl;
      mn=(int) m(0)+dn;
      while ((n<=Nt)&(mn>=0)&(dn<0))
	{
	  dn+=Nr2;
	  mn+=dn;
	  n++;
	}
      
      if (mn<0)
	dl+=step;
      else
	break;
    }
  while (1);

  int d=(int) round(dl+(du-dl)*getrand());
  m(1)=m(0)+d;

  /* main loop */
  for (int j=2; j<Nt; j++)
    {
  
      // define bounds for chirplet variation
      dl=-min((int) m(j-1),min(Nr1,Nr2-d));
      du=min(Nf-1-(int) m(j-1),min(Nr1,Nr2+d));
      
      // make steps ahead and decrease upper bound of the chirplet slope if necessary
      if (du>0)
	{
	  do
	    {
	      n=j+1; 
	      dn=du;
	      mn=(int) m(j-1)+dn;
	      while ((n<=Nt)&(mn<Nf)&(dn>0))
		{
		  dn-=Nr2;
		  mn+=dn;
		  n++;
		}

	      if (mn>=Nf)
		du-=step; 
	      else
		break;
	    }
	  while (1);
	}

      // make steps ahead and increase lower bound of the chirplet slope if necessary
      if (dl<0)
	{
	  do
	    {
	      n=j+1; 
	      dn=dl;
	      mn=(int) m(j-1)+dn;
	      while ((n<=Nt)&(mn>=0)&(dn<0))
		{
		  dn+=Nr2;
		  mn+=dn;
		  n++;
		}

	      if (mn<0)
		dl+=step;
	      else
		break;
	    }
	  while (1);
	}

      d=(int) round(dl+(du-dl)*getrand());
      m(j)=m(j-1)+d;

    }
  
  dl=-min((int) m(Nt-1),min(Nr1,Nr2-d));
  du= min(Nf-1-(int) m(Nt-1),min(Nr1,Nr2+d));
  d=(int) round(dl+(du-dl)*getrand());
  m(Nt)=m(Nt-1)+d;

  /* create output signal */
  ColumnVector out(N);
  out.fill(0.0);

  /* create chirplet chain frequency */
  ColumnVector ifl(N);
  ifl.fill(0.0);

  /* convert chain of indices into chirp signal and frequency */
  double a2,a1,b2,b1;
  double theta=0.0;

  int i=0;
  for (int j=0 ; j<Nt ; j++)
    {
      b2=(f((int) m(j+1))-f((int) m(j)))/((double) b);
      b1=f((int) m(j));
      a2=M_PI*b2;
      a1=2.0*M_PI*f((int) m(j));
      out(i)=theta;
      ifl(i)=b1; i++;
      n=1;
      while (n<b)
        {
          out(i)=a2*n*n+a1*n+theta;
          ifl(i)=b2*n+b1;
          n++;i++;
        }
      theta+=M_PI*((double) b)*(f((int) m(j+1))+f((int) m(j)));
    }

  retval(0)=out;
  retval(1)=ifl;
  retval(2)=m;

  return retval;
}
