
// Copyright (C) 2008 Eric Chassande-Mottin, CNRS (France)

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see .


#include <octave/config.h>
#include <octave/defun-dld.h>
#include <octave/error.h>
#include <octave/gripes.h>
#include <octave/oct-obj.h>
#include <octave/utils.h>

static inline int
min ( int x, int y)
{
  return x < y ? x : y;
}

DEFUN_DLD (weyl_remap, args, ,
  "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{out} =} weyl_remap (@var{in})\n\
Matrix remapping used to compute Weyl symbols.\n\
@end deftypefn\n\n")
{
  octave_value_list retval;
  
  int nargin = args.length ();
  
  if (nargin < 1)
    {
      print_usage();
      return retval;
    }

  octave_value arg = args(0);
  
  octave_idx_type r = arg.rows ();
  octave_idx_type c = arg.columns ();

  int arg_is_empty = empty_arg ("weyl_remap", r, c);

  if (arg_is_empty > 0)
    return octave_value (Matrix ());
  
  if (c!=r)
    {
      gripe_square_matrix_required ("weyl_remap: ");
      return retval;
    }
  
  int N=2*c;
  
  /* remap matrix */

  if (arg.is_complex_type())
    {
      // Do everything in complex arithmetic
      
      ComplexMatrix in = arg.complex_matrix_value ();

      if (error_state)
	{
	  gripe_wrong_type_arg("weyl_remap",args);
	  return retval;
	}

      ComplexMatrix out(N,c,0.0);

      for (int n=0 ; n<c ; n++)
	{    
	  int kmax=min(2*n,N-1-2*n);
	  for (int k=-kmax; k<=kmax ; k++)
	    out((N+k)%N,n)=in(floor(n+.5*(double) k),floor(n-.5*(double)k));
	}

      retval(0)=out;

    }
  else
    {
      // Do everything in real arithmetic
      
      Matrix in = arg.matrix_value ();

      if (error_state)
	{
	  gripe_wrong_type_arg("weyl_remap",args);
	  return retval;
	}

      Matrix out(N,c,0.0);

      for (int n=0 ; n<c ; n++)
	{    
	  int kmax=min(2*n,N-1-2*n);
	  for (int k=-kmax; k<=kmax ; k++)
	    out((N+k)%N,n)=in(floor(n+.5*(double) k),floor(n-.5*(double)k));
	}

      retval(0)=out;

    }

  return retval;
}
