/*
 * parcours.c: Statemachine zum Adventure One aus dem Mayan-Adventure-Buch
 *
 */

/*  BSD-License:

Copyright (c) 2010 by Matthias Bunte, Germany

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
  * Neither the names of the authors the name nicai-systems nor
    the names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// Use RC5Rcv - or not
#undef  USE_RC5RCV
//#define USE_RC5RCV

#include <stdlib.h>         //
#include <systemtimer.h>           // LED-Steuerung
//#include <leds.h>           // LED-Steuerung
#include <motorpid.h>         // RC5 reveiver (IR remote control)

#ifdef USE_RC5RCV
 #include <rc5rcv.h>        // RC5 reveiver (IR remote control)
#else
 #include <feelers.h>       // Feelers
#endif

#define PARCOURS_INIT_TIME  (10)

#define PARCOURS_MOTORSTOPP     (0)   // Stopp-Geschwindigkeit (in % von Max)
#define PARCOURS_MOTORFORWARD   (80)  // Fahr-Geschwindigkeit (in % von Max)
#define PARCOURS_RECHTSKURVE    (22)  // 
#define PARCOURS_LINKSKURVE     (-PARCOURS_RECHTSKURVE)  // 
#define PARCOURS_KURVENWEG      (43)  // Strecke der Aussenkurven
#define PARCOURS_TURN_COMPLETE      (-100)
#define PARCOURS_TURN_COMPLETE_WAY  (19)

/*
#define PARCOURS_TUNNELFIRST    (230)
#define PARCOURS_TUNNELSECOND   (90)
#define PARCOURS_TUNNELTHIRD    (30)
*/
#define PARCOURS_TUNNELFIRST    (30)
#define PARCOURS_TUNNELSECOND   (20)
#define PARCOURS_TUNNELTHIRD    (10)

// Hardware initialisieren
void Parcours_Init(void)
{
  // Library initialization
  SystemTimer_CheckAndInit();
//  LEDS_INIT();
  // Motorregelung initialisieren
//  MotorPID_Init(Parcours_MotorPID_EventFkt);
  MotorPID_Init(NULL);
  // Initialization of RC5 receiver
#ifdef USE_RC5RCV
  RC5Rcv_CheckAndInit();
#else
  Feelers_Init(NULL);
#endif
}

void Parcours_StateMachine(void)
{
  static enum
  {
    Parcours_State_Init = 1,
    Parcours_State_Stopped,
#ifdef USE_RC5RCV
    Parcours_State_StoppedWaitIR,
#endif
    Parcours_State_TunnelFirst,
    Parcours_State_TurnLeftFirst,
    Parcours_State_TunnelSecond,
    Parcours_State_TurnLeftSecond,
    Parcours_State_TunnelThird,
    Parcours_State_TurnCompleteFirst,
    Parcours_State_TunnelThirdBack,
    Parcours_State_TurnRightFirst,
    Parcours_State_TunnelSecondBack,
    Parcours_State_TurnRightSecond,
    Parcours_State_TunnelFirstBack,
    Parcours_State_TurnCompleteSecond,
  }
  Parcours_States;
  int16_t Parcours_InitTime;
  
  // Statemachine
  switch (Parcours_States)
  {
  case Parcours_State_Init:
    // Wait a startup time befor initializing
    Parcours_InitTime = SystemTimer_GetOffset(PARCOURS_INIT_TIME);
    while (!SystemTimer_IsExpired(Parcours_InitTime));
    // Initialisierung und Uebergang zum State Angehalten (stopped)
    Parcours_Init();
    Parcours_States = Parcours_State_Stopped;
    break;
  case Parcours_State_Stopped:
#ifdef USE_RC5RCV
    // RC5 receiving
    RC5Rcv_Start();
    Parcours_States = Parcours_State_StoppedWaitIR;
    break;
  case Parcours_State_StoppedWaitIR:
    // Tasten abfragen
    // gegebenenfalls Status wechseln
    if (RC5Rcv_CodeSampled())
    {
      // RC5-Code auswerten
      RC5Rcv_ReadCode();
#else
    if (    (Feelers_Read_KeysChanged())
         && GET_BIT(Feelers_Read_KeysState(), IO_FEELERS_BIT_L_BACK)
         && GET_BIT(Feelers_Read_KeysState(), IO_FEELERS_BIT_R_BACK) )
    {
      // Tastenbedienung zuruecksetzen
      Feelers_Write_KeysChanged(0);
#endif
      // Motoren starten
      MotorPID_StartStrecke(PARCOURS_MOTORFORWARD, PARCOURS_TUNNELFIRST);
      Parcours_States = Parcours_State_TunnelFirst;
    }
    break;
  case Parcours_State_TunnelFirst:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartKurvenStrecke(PARCOURS_MOTORFORWARD, PARCOURS_LINKSKURVE, PARCOURS_KURVENWEG);
      Parcours_States = Parcours_State_TurnLeftFirst;
    }
    break;
  case Parcours_State_TurnLeftFirst:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartStrecke(PARCOURS_MOTORFORWARD, PARCOURS_TUNNELSECOND);
      Parcours_States = Parcours_State_TunnelSecond;
//      Parcours_States = Parcours_State_Stopped;
    }
    break;
  case Parcours_State_TunnelSecond:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartKurvenStrecke(PARCOURS_MOTORFORWARD, PARCOURS_LINKSKURVE, PARCOURS_KURVENWEG);
      Parcours_States = Parcours_State_TurnLeftSecond;
//      Parcours_States = Parcours_State_Stopped;
    }
    break;
  case Parcours_State_TurnLeftSecond:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartStrecke(PARCOURS_MOTORFORWARD, PARCOURS_TUNNELTHIRD);
      Parcours_States = Parcours_State_TunnelThird;
    }
    break;
  case Parcours_State_TunnelThird:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartKurvenStrecke((PARCOURS_MOTORFORWARD / 2), PARCOURS_TURN_COMPLETE, PARCOURS_TURN_COMPLETE_WAY);
      Parcours_States = Parcours_State_TurnCompleteFirst;
//      Parcours_States = Parcours_State_Stopped;
    }
    break;
  case Parcours_State_TurnCompleteFirst:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartStrecke(PARCOURS_MOTORFORWARD, PARCOURS_TUNNELTHIRD);
      Parcours_States = Parcours_State_TunnelThirdBack;
    }
    break;
  case Parcours_State_TunnelThirdBack:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartKurvenStrecke(PARCOURS_MOTORFORWARD, PARCOURS_RECHTSKURVE, PARCOURS_KURVENWEG);
      Parcours_States = Parcours_State_TurnRightFirst;
//      Parcours_States = Parcours_State_Stopped;
    }
    break;
  case Parcours_State_TurnRightFirst:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartStrecke(PARCOURS_MOTORFORWARD, PARCOURS_TUNNELSECOND);
      Parcours_States = Parcours_State_TunnelSecondBack;
    }
    break;
  case Parcours_State_TunnelSecondBack:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartKurvenStrecke(PARCOURS_MOTORFORWARD, PARCOURS_RECHTSKURVE, PARCOURS_KURVENWEG);
      Parcours_States = Parcours_State_TurnRightSecond;
//      Parcours_States = Parcours_State_Stopped;
    }
    break;
  case Parcours_State_TurnRightSecond:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartStrecke(PARCOURS_MOTORFORWARD, PARCOURS_TUNNELFIRST);
      Parcours_States = Parcours_State_TunnelFirstBack;
    }
    break;
  case Parcours_State_TunnelFirstBack:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      // Motoren starten
      MotorPID_StartKurvenStrecke((PARCOURS_MOTORFORWARD / 2), PARCOURS_TURN_COMPLETE, PARCOURS_TURN_COMPLETE_WAY);
      Parcours_States = Parcours_State_TurnCompleteSecond;
    }
    break;
  case Parcours_State_TurnCompleteSecond:
    if (MotorPID_GetLastEvent() == MotorPID_EventFinished)
    {
      Parcours_States = Parcours_State_Stopped;
    }
    break;
  default:
    // Ein anderer State ist unzulaessig -> Neu initialisieren
    Parcours_States = Parcours_State_Init;
    break;
  }
}

