/*  BSD-License:

Copyright (c) 2010 by Matthias Bunte, Germany

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
  * Neither the names of the authors the name nicai-systems nor
    the names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*! @file    systemtimer.c
 *  @brief   System timer - the base of the library
 *  @author  Matthias Bunte (m_bunte@arcor.de)
 *  @date    2010-02-02
 */

#include <stdint.h>
#include "iodefs.h"
#include "systemtimer.h"
#include "hal_systemtimer.h"

#define SYSTEMTIMER_MAX_ENTRIES 25
#define SYSTEMTIMER_EMPTY       255

/*
 * @brief Macro zur Pruefung, ob ein Timer abgelaufen ist
 *
 * Prueft, ob der in TIMER_FOR_CHECK_VAR gespeicherte Zeitpunkt _vor_ dem in
 * PRESENT_TIME_VAR gespeichterten Zeitpunkt liegt. Dies ist der Fall, wenn 
 * 1. das High-Byte identisch und das Low-Byte des zu pruefenden Timers 
 *    kleiner oder gleich der aktuellen Zeit ist
 * oder
 * 2. das um eins erhoehte High-Byte des zu pruefenden Timers gleich
 *    dem High-Byte der aktuellen Zeit ist.
 *
 * Die zweite Bedingung ist noetig, um einen Ueberlauf des Low-Bytes zu
 * erkennen. Anderenfalls werden abgelaufene Timer, deren Low-Byte 255
 * nicht erkannt, falls die Abfrage nicht innerhalb der Tick-Zeit erfolgt.
 * So duerfen zwischen dem Ablauf des Timers und Erkennen des Timers max. 256
 * Ticks vergehen.
 * Dies wird erkauft, indem sich der laengste Timer-Intervall von 65280 um 256
 * auf 65024 Ticks verringert.
 */
#define SYSTEMTIMER_IS_EXPIRED(TIMER_FOR_CHECK_VAR,PRESENT_TIME_VAR) ( \
   ( (HIBYTE(TIMER_FOR_CHECK_VAR) == HIBYTE(PRESENT_TIME_VAR)) && \
     (LOBYTE(TIMER_FOR_CHECK_VAR) <= LOBYTE(PRESENT_TIME_VAR)) ) \
   || \
     ((HIBYTE(TIMER_FOR_CHECK_VAR) + 1) == HIBYTE(PRESENT_TIME_VAR)) )

/*
 * @brief Macro zur Pruefung, welcher von zwei Timern frueher zur Ausfuehrung kommen soll
 *
 * Prueft, ob der in TIMER_FOR_CHECK_VAR gespeicherte Zeitpunkt _frueher_ als
 * der in TIMER_NEXT_TO_RUN ("fruehester" Timer) gespeicherte Zeitpunkt bezogen
 * auf den aktuellen Zeitpunkt PRESENT_TIME_VAR ist. Dies ist der Fall, wenn
 * 1. das High-Byte des zu pruefenden Timers kleiner als das High-Byte
 *    des bisherigen "fruehesten" Timers ist
 * oder
 * 2. die High-Bytes der beiden Timer gleich sind und das Low-Byte des zu
 *    pruefenden Timers kleiner als das Low-Byte des bisherigen "fruehesten"
 *    Timers ist
 */
#define SYSTEMTIMER_IS_NEXT_TO_RUN(TIMER_FOR_CHECK_VAR,TIMER_NEXT_TO_RUN,PRESENT_TIME_VAR) ( \
  ( (unsigned)(HIBYTE(TIMER_FOR_CHECK_VAR) - HIBYTE(PRESENT_TIME_VAR)) < \
    (unsigned)(HIBYTE(TIMER_NEXT_TO_RUN)   - HIBYTE(PRESENT_TIME_VAR)) ) \
  || \
    ( (HIBYTE(TIMER_FOR_CHECK_VAR) == HIBYTE(TIMER_NEXT_TO_RUN)) && \
      (LOBYTE(TIMER_FOR_CHECK_VAR) <  LOBYTE(TIMER_NEXT_TO_RUN)) ) )

/*
 * @brief Loescht einen Timer-Eintrag
 *
 * Der Timer-Eintrag an der Stelle ELEMENT_VAL wird ersetzt durch den letzten
 * Eintrag im Array und die Anzahl der Timereintraege um einen heruntergezaehlt.
 */
#define SYSTEMTIMER_DELETE_TIMER(ELEMENT_VAL) do { \
  pSystemTimer_SchedulerFunktionen[ELEMENT_VAL] = pSystemTimer_SchedulerFunktionen[SystemTimer_LastElement]; \
  SystemTimer_SchedulerWerte[ELEMENT_VAL]       = SystemTimer_SchedulerWerte[SystemTimer_LastElement]; \
  SystemTimer_SchedulerZeiten[ELEMENT_VAL]      = SystemTimer_SchedulerZeiten[SystemTimer_LastElement]; \
  SystemTimer_LastElement--; } while (0)

// Initialisierungszustand
static uint8_t  SystemTimer_InitState = 0;

// Internal Timer Value in ms
static volatile uint16_t SystemTimer_InternalMSTimerVal = 0;

// Internes Array fuer Pointer auf die aufzurufenden Funktionen
static pSystemTimer_Fkt_t pSystemTimer_SchedulerFunktionen[SYSTEMTIMER_MAX_ENTRIES];
// Uebergabevariablen fuer diese Funktionen
static uint16_t SystemTimer_SchedulerWerte[SYSTEMTIMER_MAX_ENTRIES];
// Internes Array fuer zugehoerige Zeiten
static uint16_t SystemTimer_SchedulerZeiten[SYSTEMTIMER_MAX_ENTRIES];
// Nummer des letzten gueltigen Array-Eintrags; "SYSTEMTIMER_EMPTY", wenn leer
static uint8_t SystemTimer_LastElement = SYSTEMTIMER_EMPTY;

static inline uint16_t SystemTimer_ReadTimer(void)
{
  // Zwischenspeicher fuer SREG
  uint8_t   sreg_copy;
  uint16_t  Time;

  // Interuptsicher aktuellen Zeitstempel auslesen
  sreg_copy = SREG;
  cli();
  Time = SystemTimer_InternalMSTimerVal;
  SREG = sreg_copy;
  // Und als Ergebnis die Zeit liefern  
  return Time;
}

/*
 * @brief Loescht einen Timer-Eintrag
 *
 * Der Timer-Eintrag an der Stelle ELEMENT_VAL wird ersetzt durch den letzten
 * Eintrag im Array und die Anzahl der Timereintraege um einen heruntergezaehlt.
 */
static inline void SystemTimer_DeleteTimer(uint8_t ElementVal)
{
  pSystemTimer_SchedulerFunktionen[ElementVal] = pSystemTimer_SchedulerFunktionen[SystemTimer_LastElement];
  SystemTimer_SchedulerWerte[ElementVal]       = SystemTimer_SchedulerWerte[SystemTimer_LastElement];
  SystemTimer_SchedulerZeiten[ElementVal]      = SystemTimer_SchedulerZeiten[SystemTimer_LastElement];
  SystemTimer_LastElement--;
}

/*
 * Initialisierung 
 */
void SystemTimer_Init(void)
{
  // Timer HAL initialisieren
  SYSTEMTIMER_INIT();
  // Initialisierung ist nun abgeschlossen
  SystemTimer_InitState = 1;
}

// Initialisierungszustand auslesen
uint8_t SystemTimer_Initialized(void)
{
  return SystemTimer_InitState;
}

/*
 * Liefert die aktuelle Systemzeit zurueck
 * Rueckgabewert in Millisekunden
 */
uint16_t SystemTimer_GetTime(void)
{
  return SystemTimer_ReadTimer();
}

/*
 * Liefert den Wert, den der Timer nach der Zeit Offset hat
 * Einheit von Offset und Rueckgabewert ist Milisekunden
 */
uint16_t SystemTimer_GetOffset(uint16_t Offset)
{
  return (SystemTimer_ReadTimer() + Offset);
}

/*
 * Prueft, ob der Timer abgelaufen ist
 */
uint8_t SystemTimer_IsExpired(uint16_t TimerVal)
{
  uint16_t  Time;
  
  // Zeit auslesen
  Time = SystemTimer_ReadTimer();  

  // Pruefen, ob der Timer abgelaufen ist
  return ((uint8_t) SYSTEMTIMER_IS_EXPIRED(TimerVal, Time));
}

/*
 * SystemTimer_SchedulerAdd() - Funktion zur Timerliste zufuegen
 */
void SystemTimer_SchedulerAdd(uint16_t TimerVal, pSystemTimer_Fkt_t pFunktion, uint16_t Wert)
{
  // Das letzte Element um eins erhoehen
  SystemTimer_LastElement++;
  // Ist es noch kleiner als die maximale Anzahl der Elemente?
  if (SystemTimer_LastElement < SYSTEMTIMER_MAX_ENTRIES)
  {
    // Ja -> Zeiger auf die Funktion und Zeitstempel abspeichern
    uint16_t Time;

    Time = SystemTimer_ReadTimer();
    pSystemTimer_SchedulerFunktionen[SystemTimer_LastElement] = pFunktion;
    SystemTimer_SchedulerWerte[SystemTimer_LastElement] = Wert;
    SystemTimer_SchedulerZeiten[SystemTimer_LastElement] = (TimerVal + Time);
  } else {
    // ... sonst LastElement auf das Maximum setzen
    SystemTimer_LastElement = (SYSTEMTIMER_MAX_ENTRIES - 1); 
  }
  return;
}

/*
 * SystemTimer_SchedulerDel() - Funktion von Timerliste entfernen
 */
//void SystemTimer_SchedulerDel(pSystemTimer_Fkt_t pFunktion, uint16_t Wert)
void SystemTimer_SchedulerDel(pSystemTimer_Fkt_t pFunktion)
{
  uint8_t   Element;
  uint16_t  Time;
  uint8_t   NextEntryToRun;
 
  // Nicht bei leerer Liste ausfuehren
  if (SystemTimer_LastElement != SYSTEMTIMER_EMPTY)
  {
    // Hole die Systemzeit
    Time = SystemTimer_ReadTimer();
    // Initialisiere NextElement
    NextEntryToRun = SYSTEMTIMER_EMPTY;
    // Suche passende Funktion
    for (Element = 0; Element <= SystemTimer_LastElement; Element++)
    {
      // Passenden Eintrag gefunden
      if ( pSystemTimer_SchedulerFunktionen[Element] == pFunktion )
      {
        // Pruefen, ob der aktuelle Timereintrag frueher ausgefuehrt werden soll
        // als der Timereintrag, der als "Fruehester" gespeichert ist. Wenn dies
        // der Fall ist, den aktuellen Eintrag als "Fruehesten" uebernehmen.
        // Falls als "Fruehester" noch kein Eintrag gespeichert wurde, soll
        // ebenfalls der aktuelle Eintrag uebernommen werden.
        if (SYSTEMTIMER_IS_NEXT_TO_RUN(SystemTimer_SchedulerZeiten[Element],
                                       SystemTimer_SchedulerZeiten[NextEntryToRun],
                                       Time)
          ||
            (NextEntryToRun == SYSTEMTIMER_EMPTY) )
        {
          NextEntryToRun = Element;
        }
      }
    }
    // Es wurde eine Funktion gefunden -> loeschen
    if (NextEntryToRun != SYSTEMTIMER_EMPTY)
    {
      // Diesen Eintrag komplett loeschen
//      SYSTEMTIMER_DELETE_TIMER(NextEntryToRun);
      SystemTimer_DeleteTimer(NextEntryToRun);
    }
  }
}

/*
 * SystemTimer_SchedulerRun() - Liste von gespeicherten Timereintraegen
 * durchsuchen und ausfuehren
 */
void SystemTimer_SchedulerRun(void)
{
  // Letzter Ausfuehrungszeitpunkt
  static uint16_t LastTime = 0;
  // Aktuelle Zeit
  uint16_t Time;
  // Schleifenindex fuer Vergleich
  uint8_t Element;

  // Nicht bei leerer Liste ausfuehren
  if (SystemTimer_LastElement != SYSTEMTIMER_EMPTY)
  {
    Time = SystemTimer_ReadTimer();
    // Nur ausfuehren, wenn der Zeitstempel aktualisiert wurde
    if (LastTime != Time)
    {
      // Zeitstempel merken
      LastTime = Time;
      // Fuehre erste Funktion von Beginn der Liste aus, deren Zeitstempel abgelaufen ist
      for (Element = 0; Element <= SystemTimer_LastElement; Element++)
      {
        // Pruefen, ob der Timereintrag abgelaufen ist
        if ( SYSTEMTIMER_IS_EXPIRED(SystemTimer_SchedulerZeiten[Element], Time) )
        {
          pSystemTimer_Fkt_t  pFunktion_copy;
          uint16_t            Wert_copy;
          
          // Funktion zwischenspeichern
          pFunktion_copy = pSystemTimer_SchedulerFunktionen[Element];
          Wert_copy = SystemTimer_SchedulerWerte[Element];
          // Eintrag aus der Liste loeschen
//          SYSTEMTIMER_DELETE_TIMER(Element);
          SystemTimer_DeleteTimer(Element);
          // Funktion aufrufen
          pFunktion_copy(Wert_copy);
          // Schleife abbrechen
          break;
        }
      }
    }
  }  
}

/*
 * Timer2 Overflow IRQ
 *
 * Scales Timer2 IRQs to 1 ms ticks in systemtimer internal variable
 */
ISR(TIMER2_COMPA_vect)
{
  // Count one up
  SystemTimer_InternalMSTimerVal++;
  // Fractional counter for presize counting may be implemented here
  // Change bytes of counter to geht a more random pattern
  // Check if the counter value is bigger than calculated
  // Yes - New TOP is the bigger value
  // No -  New TOP is the lower value
}

#ifdef __cplusplus
} // extern "C"
#endif

