/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: FeaEngineInteractive.java,v 1.7 2003/11/17 11:00:35 evertonm Exp $

package bera.fea;

import java.io.InputStream;
import java.io.IOException;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.Vector;
import java.util.Hashtable;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;

import bera.util.Tokenizer;
import bera.BeraAssert;

class FeaEngineInteractive extends Thread implements FeaEngineAdaptor {

    static private Logger logger = Logger.getLogger(FeaEngineInteractive.class);

    private Vector      interactiveIfaceList = new Vector();
    private InputStream interactiveInput;

    FeaEngineInteractive(InputStream is) {
	interactiveInput = is;
    }

    private static void inputError(String msg) {
	logger.error("input error: " + msg);
    }

    public Vector addRoutes(Vector routeList) {
	logger.debug("FIXME: addRoutes()");
	return null;
    }

    public Vector withdrawRoutes(Vector routeList) {
	logger.debug("FIXME: withdrawRoutes()");
	return null;
    }

    ////////////////////////////////////////////
    // These methods expose concurrent access to
    // interactiveIfaceList

    // Used by FeaRpcHandler thread
    public Vector getInterfaces() {
	synchronized (interactiveIfaceList) {
	    return (Vector) interactiveIfaceList.clone();
	}
    }

    private static Hashtable findIface(Vector ifaceList, String ifaceName) {
	int size = ifaceList.size();
	for (int i = 0; i < size; ++i) {
	    Hashtable iface = (Hashtable) ifaceList.get(i);
	    String name = (String) iface.get("interfaceName");
	    if (ifaceName.equals(name))
		return iface;
	}
	return null;
    }

    private static Hashtable findAddr(Vector addrList, String family, String value, Integer prefixLength) {
	int size = addrList.size();
	for (int i = 0; i < size; ++i) {
	    Hashtable addr = (Hashtable) addrList.get(i);

	    String addrFamily = (String) addr.get("addressFamily");
	    String addrValue = (String) addr.get("addressValue");
	    Integer addrPrefixLength = (Integer) addr.get("addressPrefixLength");

	    if (family.equals(addrFamily) && value.equals(addrValue) && prefixLength.equals(addrPrefixLength))
		return addr;
	}

	return null;
    }

    // Used by interactive input thread (FeaEngineInteractive)
    private void doAdd(StringTokenizer toker, int tokenCount) {

	//
	// Parsing
	//

	if (tokenCount != 7) {
	    inputError("doAdd: bad token count: " + tokenCount);
	    return;
	}

	String tokenIfaceName    = toker.nextToken();
	String tokenMetric       = toker.nextToken();
	String tokenAddrFamily   = toker.nextToken();
	String tokenAddr         = toker.nextToken();
	String tokenPrefixLength = toker.nextToken();
	String tokenIfaceOptions = toker.nextToken();

	Integer prefixLength = Integer.decode(tokenPrefixLength);
	Integer metric = Integer.decode(tokenMetric);

	// Options

	boolean multiaccessSupport = false;
	boolean broadcastSupport = false;

	String[] optionList = Tokenizer.split(tokenIfaceOptions, ",");
	for (int i = 0; i < optionList.length; ++i) {
	    if (optionList[i].equals("multiaccess")) {
		multiaccessSupport = true;
		continue;
	    }
	    if (optionList[i].equals("broadcast")) {
		broadcastSupport = true;
		continue;
	    }
	}

	//
	// Adding interface
	//

	synchronized (interactiveIfaceList) {
	    Hashtable iface = findIface(interactiveIfaceList, tokenIfaceName);
	    if (iface == null) {
		iface = new Hashtable();
		interactiveIfaceList.add(iface);

		iface.put("interfaceInetAddresses", new Vector());

		logger.info("new interface added: " + tokenIfaceName);
	    }

	    iface.put("interfaceName", tokenIfaceName);
	    iface.put("interfaceMetric", metric);
	    iface.put("interfaceCanBroadcast", new Boolean(broadcastSupport));
	    iface.put("interfaceCanMultiaccess", new Boolean(multiaccessSupport));

	    Vector addrList = (Vector) iface.get("interfaceInetAddresses");

	    Hashtable addr = findAddr(addrList, tokenAddrFamily, tokenAddr, prefixLength);
	    if (addr == null) {
		addr = new Hashtable();
		addrList.add(addr);

		logger.info("new address added: " + tokenIfaceName + " " + tokenAddrFamily + "/" + tokenAddr + "/" + prefixLength);
	    }

	    addr.put("addressFamily", tokenAddrFamily);
	    addr.put("addressValue", tokenAddr);
	    addr.put("addressPrefixLength", prefixLength);

	    logger.info("interface updated: " + tokenIfaceName);
	}
    }

    // Used by interactive input thread (FeaEngineInteractive)
    private void doDel(StringTokenizer toker, int tokenCount) {
	//
	// Parsing
	//

	if (tokenCount != 5) {
	    inputError("doDel: bad token count: " + tokenCount);
	    return;
	}

	String tokenIfaceName    = toker.nextToken();
	String tokenAddrFamily   = toker.nextToken();
	String tokenAddr         = toker.nextToken();
	String tokenPrefixLength = toker.nextToken();

	Integer prefixLength = Integer.decode(tokenPrefixLength);

	//
	// Removing interface
	//

	synchronized (interactiveIfaceList) {
	    Hashtable iface = findIface(interactiveIfaceList, tokenIfaceName);
	    if (iface == null) {
		logger.info("interface not found: " + tokenIfaceName);
		return;
	    }

	    Vector addrList = (Vector) iface.get("interfaceInetAddresses");

	    Hashtable addr = findAddr(addrList, tokenAddrFamily, tokenAddr, prefixLength);
	    if (addr == null) {
		logger.info("address not found: " + tokenIfaceName + " " + tokenAddrFamily + "/" + tokenAddr + "/" + prefixLength);
		return;
	    }

	    BeraAssert.require(addrList.size() > 0);
	    
	    boolean addrRemoved = addrList.remove(addr);
	    BeraAssert.require(addrRemoved);
	    logger.info("address removed: " + tokenIfaceName + " " + tokenAddrFamily + "/" + tokenAddr + "/" + prefixLength);

	    if (addrList.size() == 0) {
		boolean ifaceRemoved = interactiveIfaceList.remove(iface);
		BeraAssert.require(ifaceRemoved);
		logger.info("interface removed: " + tokenIfaceName);
	    }
	}
    }

    // These methods expose concurrent access to
    // interactiveIfaceList
    ////////////////////////////////////////////

    public void run() {

	BufferedReader in = new BufferedReader(new InputStreamReader(interactiveInput));

	logger.info("reading commands from interactive input");

	for (;;) {
	    String line;
	    try {
		line = in.readLine();
	    }
	    catch (IOException e) {
		inputError("could not read line: " + e);
		return;
	    }
	    if (line == null)
		break;

	    StringTokenizer toker = new StringTokenizer(line);
	    int tokenCount = toker.countTokens();
	    if (tokenCount < 1) {
		inputError("bad token count: " + tokenCount);
		continue;
	    }
	    String tokenCmd = toker.nextToken();

	    if (tokenCmd.equals("add")) {
		doAdd(toker, tokenCount);
		continue;
	    }

	    if (tokenCmd.equals("del")) {
		doDel(toker, tokenCount);
		continue;
	    }

	    inputError("unknown command: " + line);
	}

	inputError("interactive input has been closed (!)");
    }
}
