/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques
  Copyright (C) 2003  Fabricio Matheus Goncalves

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: FeaEngineLinuxIPRoute.java,v 1.9 2003/12/02 00:55:37 evertonm Exp $

package bera.fea;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.util.Vector;
import java.util.Hashtable;

//
// gcj 3.3.2 doesn't support java.util.regex
//
//import java.util.regex.Pattern;
//import java.util.regex.Matcher;
//
// Wrapper for Jakarta Regexp
//
import bera.util.regex.Pattern;
import bera.util.regex.Matcher;

import org.apache.log4j.Logger;

import bera.BeraAssert;

class FeaEngineLinuxIPRoute implements FeaEngineAdaptor {

    static private Logger logger = Logger.getLogger(FeaEngineLinuxIPRoute.class);

    static private final Vector emptyVector = new Vector();

    // From: /usr/include/linux/rtnetlink.h

    private static final int RTPROT_UNSPEC   = 0;
    private static final int RTPROT_REDIRECT = 1;  /* Route installed by ICMP redirects */
    private static final int RTPROT_KERNEL   = 2;  /* Route installed by kernel */
    private static final int RTPROT_BOOT     = 3;  /* Route installed during boot */
    private static final int RTPROT_STATIC   = 4;  /* Route installed by administrator */
    private static final int RTPROT_GATED    = 8;  /* Apparently, GateD */
    private static final int RTPROT_RA       = 9;  /* RDISC/ND router advertisements */
    private static final int RTPROT_MRT      = 10; /* Merit MRT */
    private static final int RTPROT_ZEBRA    = 11; /* Zebra/Quagga */
    private static final int RTPROT_BIRD     = 12; /* BIRD */
    private static final int RTPROT_DNROUTED = 13; /* DECnet routing daemon */
    private static final int RTPROT_BERA     = 14; /* BERA */

    private void routeFlush() {
	logger.debug("flushing existing route table");

	String cmd = "ip route flush proto " + RTPROT_BERA;

	Runtime rt = Runtime.getRuntime();

	Process proc;
	try {
	    proc = rt.exec(cmd);
	}
	catch (IOException e) {
	    logger.error("failure calling '" + cmd + "': " + e);
	    return;
	}

	try { proc.waitFor(); } catch (InterruptedException e) { }
	int exitStatus = proc.exitValue();
	if (exitStatus != 0)
	    logger.error("command '" + cmd + "' returned bad exit status: " + exitStatus);

	logger.debug("flushing existing route table - done");
    }

    FeaEngineLinuxIPRoute() {
	routeFlush();
    }

    public Vector getInterfaces() {
	Vector ifaceList = new Vector();

	Runtime rt = Runtime.getRuntime();

	String cmd = "ip addr";

	Process proc;
	try {
	    proc = rt.exec(cmd);
	}
	catch (IOException e) {
	    logger.error("failure calling '" + cmd + "': " + e);
	    return ifaceList;
	}

	BufferedReader reader = new BufferedReader(new InputStreamReader(proc.getInputStream()));

	Pattern ifaceLine = Pattern.compile("^\\d+:\\s+([^:]+):.*");
	Pattern broad = Pattern.compile(".*BROADCAST.*");
	Pattern mcast = Pattern.compile(".*MULTICAST.*");
	Pattern addrLine = Pattern.compile("^\\s*inet\\s+(\\d+\\.\\d+\\.\\d+\\.\\d+).*?/(\\d+).*");
	Matcher m;

	int status = 0; // find interface

	Hashtable ifaceData;
	Vector addrList = null;

	for (;;) {
	    String line;

	    try {
		line = reader.readLine();
	    }
	    catch (IOException e) {
		logger.error("error reading output from '" + cmd + "': " + e);
		break;
	    }

	    if (line == null)
		break;

	    //logger.debug("LINE: " + line);

	    m = ifaceLine.matcher(line);
	    if (m.matches()) {
		String ifaceName = m.group(1);

		ifaceData = new Hashtable();
		ifaceList.add(ifaceData);
		ifaceData.put("interfaceName", ifaceName);

		logger.debug("FIXME: forcing bogus metric: 1");
		ifaceData.put("interfaceMetric", new Integer(1));

		m = mcast.matcher(line);
		if (m.matches())
		    ifaceData.put("interfaceCanMultiaccess", Boolean.valueOf(true));
		else
		    ifaceData.put("interfaceCanMultiaccess", Boolean.valueOf(false));

		m = broad.matcher(line);
		if (m.matches())
		    ifaceData.put("interfaceCanBroadcast", Boolean.valueOf(true));
		else
		    ifaceData.put("interfaceCanBroadcast", Boolean.valueOf(false));

		addrList = new Vector();
		ifaceData.put("interfaceInetAddresses", addrList);

		continue;
	    }

	    m = addrLine.matcher(line);
	    if (m.matches()) {
		String ifaceAddr = m.group(1);
		String ifacePrefixLen = m.group(2);
		Integer prefixLen = new Integer(ifacePrefixLen);

		BeraAssert.require(addrList != null);

		Hashtable addrData = new Hashtable();
		addrList.add(addrData);

		addrData.put("addressFamily",       "IPv4");
		addrData.put("addressValue",        ifaceAddr);
		addrData.put("addressPrefixLength", prefixLen);

		continue;
	    }
	}

	try { proc.waitFor(); } catch (InterruptedException e) { }
	int exitStatus = proc.exitValue();
	if (exitStatus != 0)
	    logger.error("command '" + cmd + "' returned bad exit status: " + exitStatus);

	return ifaceList;
    }

    private void exec(String cmd) {
	logger.debug("exec: " + cmd);
    }

    private void routeAdd(String protocol, String instance, String addrFamily, String network, int prefixLen, String nextHop) {
	logger.debug("add: " + protocol + "/" + instance + ": " + addrFamily + "/" + network + "/" + prefixLen + "->" + nextHop);
	exec("route add " + network + "/" + prefixLen + " gw " + nextHop + " proto " + RTPROT_BERA);
    }

    private void routeDel(String protocol, String instance, String addrFamily, String network, int prefixLen, String nextHop) {
	logger.debug("del: " + protocol + "/" + instance + ": " + addrFamily + "/" + network + "/" + prefixLen + "->" + nextHop);
	exec("route del " + network + "/" + prefixLen + " gw " + nextHop + " proto " + RTPROT_BERA);
    }

    public Vector addRoutes(Vector routeList) {
	logger.debug("addRoutes called");

	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    Hashtable route = (Hashtable) routeList.get(i);

	    String routeProtocol = (String) route.get("routeProtocol");
	    String routeProtocolInstance = (String) route.get("routeProtocolInstance");
	    String routeAddressFamily = (String) route.get("routeAddressFamily");
	    String routeNetwork  = (String) route.get("routeNetwork");
	    int routePrefixLength = ((Integer) route.get("routePrefixLength")).intValue();
	    String routeNextHop = (String) route.get("routeNextHop");

	    routeAdd(routeProtocol, routeProtocolInstance,
		     routeAddressFamily, routeNetwork,
		     routePrefixLength, routeNextHop);
	}

	return emptyVector;
    }

    public Vector withdrawRoutes(Vector routeList) {
	logger.debug("withdrawRoutes called");

	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    Hashtable route = (Hashtable) routeList.get(i);

	    String routeProtocol = (String) route.get("routeProtocol");
	    String routeProtocolInstance = (String) route.get("routeProtocolInstance");
	    String routeAddressFamily = (String) route.get("routeAddressFamily");
	    String routeNetwork  = (String) route.get("routeNetwork");
	    int routePrefixLength = ((Integer) route.get("routePrefixLength")).intValue();
	    String routeNextHop = (String) route.get("routeNextHop");

	    routeDel(routeProtocol, routeProtocolInstance,
		     routeAddressFamily, routeNetwork,
		     routePrefixLength, routeNextHop);
	}

	return emptyVector;
    }

}
