/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: BeraManager.java,v 1.16 2003/11/15 00:06:05 evertonm Exp $

package bera.manager;

import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.Hashtable;

import org.apache.log4j.Logger;

import bera.util.Tokenizer;
import bera.manager.config.ConfigNode;
import bera.manager.config.SingleNode;
import bera.manager.config.MultiNode;
import bera.manager.config.ConfigValue;
import bera.manager.config.SingleValue;
import bera.manager.config.IntegerLeafNode;
import bera.manager.config.BooleanLeafNode;
import bera.manager.config.Inet4AddressLeafNode;

public class BeraManager {

    static private Logger logger = Logger.getLogger(BeraManager.class);

    private ConfigNode configTree;
    private ConfigValue valueTree;
    private String configDir;
    private String startupConfigFile = "manager.cfg";

    static private ConfigNode buildConfigTree() {
	SingleNode rootNode = new SingleNode("root");

	//
	// Add "router" section to config tree
	//

	SingleNode routerNode = new SingleNode("router");
	rootNode.addChild(routerNode);

	//
	// Add "router rip" section to config tree
	//

	MultiNode ripNode = new MultiNode("rip");
	routerNode.addChild(ripNode);

	ripNode.addChild(new IntegerLeafNode("pulse-interval", 1, 60, 5));
	ripNode.addChild(new IntegerLeafNode("trigger-holdtime", 1, 10, 5));
	ripNode.addChild(new IntegerLeafNode("update-interval", 10, 120, 30));
	ripNode.addChild(new IntegerLeafNode("timeout", 10, 720, 180));
	ripNode.addChild(new IntegerLeafNode("garbage", 10, 480, 120));

	MultiNode iface = new MultiNode("interface");
	ripNode.addChild(iface);
	iface.addChild(new BooleanLeafNode("poisoned-reverse", true));

	MultiNode neigh = new MultiNode("neighbor");
	ripNode.addChild(neigh);
	neigh.addChild(new Inet4AddressLeafNode("address"));

	return rootNode;
    }

    private static ConfigValue buildValueTree(SingleNode rootNode) {
	ConfigValue valueRoot = new SingleValue(rootNode);
	return valueRoot;
    }

    synchronized private void loadConfig() {
	String fullpath = configDir + "/" + startupConfigFile;

	File inputFile = new File(fullpath);

	BufferedReader reader;
	try {
	    reader = new BufferedReader(new FileReader(inputFile));
	}
	catch (IOException e) {
	    logger.error("can't load config file: " + e);
	    return;
	}

	LogErrorSender errorSender = new LogErrorSender(logger);

	for (;;) {
	    String line;

	    try {
		line = reader.readLine();
	    }
	    catch (IOException e) {
		logger.error("failure reading config file: " + e);
		break;
	    }

	    if (line == null)
		break;

	    if (line.length() == 0)
		continue;

	    String args[] = Tokenizer.split(line);

	    if (args.length < 2) {
		logger.error("loading config command: missing argument");
		continue;
	    }

	    try {
		valueTree.setValue(errorSender, args, 1);
	    }
	    catch (BadCommandException e) {
		logger.error("loading config command: " + e.getMessage());
	    }

	}

	try {
	    reader.close();
	}
	catch (IOException e) {
	    logger.error("failure closing config file: " + e);
	}
    }

    BeraManager(String dir) {
	configDir = dir;
	configTree = buildConfigTree();
	valueTree = buildValueTree((SingleNode) configTree);

	loadConfig();
    }

    public ConfigContext getConfigContext() {
	return new ConfigContext(configTree, valueTree);
    }

    //
    // Manager Services
    //

    // Services for bera.manager.command.CommandInterpreter

    synchronized public void performSetValue(LineSender sender, ConfigContext config, String args[], int firstArg) throws BadCommandException {
	config.getConfigValue().setValue(sender, args, firstArg);
    }

    synchronized public void performUnsetValue(LineSender sender, ConfigContext config, String args[], int firstArg) throws BadCommandException {
	config.getConfigValue().unsetValue(sender, args, firstArg);
    }

    synchronized public void performListValueTree(LineSender sender, ConfigContext config, int depth, int inc) {
	config.getConfigValue().listTree(sender, depth, inc);
    }

    synchronized public void performListConfigTree(LineSender sender, ConfigContext config, int depth, int inc) {
	config.getConfigNode().listTree(sender, depth, inc);
    }

    synchronized public void performSaveConfigTree(LineSender sender, String filename, String prefix) {

	if (filename == null)
	    filename = startupConfigFile;

	String fullpath = configDir + "/" + filename;

	FileSaver saver;
	try {
	    saver = new FileSaver(fullpath);
	}
	catch (IOException e) {
	    logger.error("could not write to file: " + fullpath + ": " + e);
	    sender.sendLine("write to file '" + filename + "' failed");
	    return;
	}

	((SingleValue) valueTree).saveChildren(saver, prefix);

	saver.close();

	if (saver.error())
	    sender.sendLine("error saving file");
    }

    // Services for bera.manager.ManagerRpcHandler

    synchronized Hashtable rpcDumpConfig() {
	Hashtable configDump = new Hashtable();
	valueTree.rpcDump(configDump);
	return configDump;
    }
}

