/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: CommandInterpreter.java,v 1.19 2003/12/01 10:50:59 evertonm Exp $

package bera.manager.command;

import org.apache.log4j.Logger;

import bera.BeraAssert;
import bera.BeraConstants;
import bera.manager.BeraManager;
import bera.manager.LineSender;
import bera.manager.ConfigContext;
import bera.manager.BadCommandException;
import bera.telnet.TelnetClient;
import bera.util.Tokenizer;
import bera.manager.command.show.DoShow;

public class CommandInterpreter implements LineSender {

    static private Logger logger = Logger.getLogger(CommandInterpreter.class);

    static private final int TERM_HEIGHT = 20;

    static private final String HELLO_BANNER = 
	"\n" +
	"BERA is free software, covered by the GNU General Public License, and\n" +
	"you are welcome to change it and/or distribute copies of it under\n" +
	"certain conditions. Type 'license' to see the conditions.\n" +
	"There is absolutely no warranty for BERA. Type 'warranty' for details.\n" +
	"\n";

    private static final int STATUS_LOGIN    = 0;
    private static final int STATUS_PASSWORD = 1;
    private static final int STATUS_COMMAND  = 2;
    private static final int STATUS_PAGER    = 3;

    static final Command commandTable[] = {
	new Exit("exit", "close session"),
	new DoGet("get", "get current configuration"),
	new Help("help", "issue list of available commands"),
        new DoList("list", "list valid syntax tree"),
        new DoLicense("license", "show licensing terms for the software"),
	new DoSave("save", "save current configuration to persistent storage"),
	new DoSet("set", "define configuration item"),
	new DoShow("show", "show information"),
	new DoUnset("unset", "delete configuration item"),
	new DoVersion("version", "show software version"),
        new DoNoWarranty("warranty", "show warranty disclaiment for the software")
	    };
    
    private static Command findCommand(String cmd) {
	for (int i = 0; i < commandTable.length; ++i) 
	    if (commandTable[i].matchPrefix(cmd))
		return commandTable[i];

	return null;
    }

    private TelnetClient  telnetClient;
    private int           clientStatus;
    private ConfigContext configContext;
    private String        pagerList[];
    private int           pagerLine;
    private BeraManager   beraManager;

    public CommandInterpreter(BeraManager manager, TelnetClient client) {
	telnetClient = client;
	clientStatus = STATUS_LOGIN;
	configContext = manager.getConfigContext();
	beraManager = manager;
    }

    public void send(String line) {
	telnetClient.output(line);
    }

    public void sendLine(String line) {
	telnetClient.outputln(line);
    }

    public void sendLine() {
	telnetClient.outputln(BeraConstants.EMPTY_STRING);
    }

    private void pagerStop() {
	BeraAssert.require(clientStatus == STATUS_PAGER);
	pagerList = null; // help garbage collection (could weak references act here?)
	clientStatus = STATUS_COMMAND;
    }

    private void pagerSendPage() {
	BeraAssert.require(clientStatus == STATUS_PAGER);

	for (int i = 0; i < TERM_HEIGHT; ++i) {
	    if (pagerLine >= pagerList.length) {
		pagerStop();
		break;
	    }

	    sendLine(pagerList[pagerLine]);
	    ++pagerLine;
	}
    }

    public void pagerStart(String lines[]) {
	BeraAssert.require(clientStatus == STATUS_COMMAND);

	pagerList = lines;
	pagerLine = 0;
	clientStatus = STATUS_PAGER;

	pagerSendPage();
    }

    private void executeCommand(String line) {
	String[] args = Tokenizer.split(line);

	// Empty command?
	if (args.length < 1)
	    return;
	
	Command command = findCommand(args[0]);
	if (command == null) {
	    sendLine("command not found: " + line);
	    return;
	}

	if (command.execute(this, line, args)) {
	    sendLine("bye");
	    telnetClient.flush();
	    telnetClient.close();
	}
    }

    void managerSetValue(String args[], int firstArg) throws BadCommandException {
	beraManager.performSetValue(this, configContext, args, firstArg);
    }

    void managerUnsetValue(String args[], int firstArg) throws BadCommandException {
	beraManager.performUnsetValue(this, configContext, args, firstArg);
    }

    void managerListValueTree(int depth, int inc) {
	beraManager.performListValueTree(this, configContext, depth, inc);
    }

    void managerListConfigTree(int depth, int inc) {
	beraManager.performListConfigTree(this, configContext, depth, inc);
    }

    void managerSaveConfigTree(LineSender sender, String filename, String prefix) {
	beraManager.performSaveConfigTree(sender, filename, prefix);
    }

    public void execute(String buf) {
	switch (clientStatus) {
	case STATUS_LOGIN:
	    clientStatus = STATUS_PASSWORD;
	    break;
	case STATUS_PASSWORD:
	    logger.debug("FIXME: perform user/password authentication");
	    send(HELLO_BANNER);
	    executeCommand("version");
	    sendLine("");
	    sendLine("Type 'help' for available commands.");
	    sendLine("");
    	    clientStatus = STATUS_COMMAND;
	    break;
	case STATUS_COMMAND:
	    executeCommand(buf);
	    break;
	case STATUS_PAGER:
	    if (buf.length() == 0)
		pagerSendPage();
	    else if ("quit".startsWith(buf))
		pagerStop();
	    else
		pagerSendPage();
	    break;
	default:
	    logger.error("execute(): unknown status: " + clientStatus);
	}

    }

    public String getPrompt() {
	switch (clientStatus) {
	case STATUS_LOGIN:
	    return "username: ";
	case STATUS_PASSWORD:
	    return "password: ";
	case STATUS_COMMAND:
	    return "command> ";
	case STATUS_PAGER:
	    return "ENTER=next q=quit> ";
	default:
	    logger.error("getPrompt(): unknown status: " + clientStatus);
	    return "?> ";
	}
    }
}

