/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: InternalValue.java,v 1.6 2003/11/11 14:05:13 evertonm Exp $

package bera.manager.config;

import java.util.List;
import java.util.ArrayList;
import java.util.Hashtable;

import bera.BeraAssert;
import bera.manager.LineSender;
import bera.manager.BadCommandException;

abstract class InternalValue extends ConfigValue {

    protected List childList = new ArrayList();

    InternalValue(InternalNode node) {
	super(node);
	node.enableAutomaticChildrenValues(childList);
    }

    protected void listChildren(LineSender sender, int depth, int inc) {
	for (int i = 0; i < childList.size(); ++i) {
	    ConfigValue value = (ConfigValue) childList.get(i);
	    value.listTree(sender, depth, inc);
	}
    }

    public void saveChildren(LineSender sender, String prefix) {
	for (int i = 0; i < childList.size(); ++i) {
	    ConfigValue value = (ConfigValue) childList.get(i);
	    value.saveTree(sender, prefix);
	}
    }

    private ConfigValue findChildByLabel(String label) {
	for (int i = 0; i < childList.size(); ++i) {
	    ConfigValue configValue = (ConfigValue) childList.get(i);
	    if (configValue.getConfigNode().getLabel().equals(label))
		return configValue;
	}
	return null;
    }

    private MultiValue findChildByLabel(String label, String name) {
	for (int i = 0; i < childList.size(); ++i) {
	    ConfigValue configValue = (ConfigValue) childList.get(i);
	    ConfigNode configNode = configValue.getConfigNode();
	    if (configNode.getLabel().equals(label)) {
		if (configValue instanceof MultiValue) {
		    BeraAssert.require(configNode instanceof MultiNode);
		    MultiValue multiValue = (MultiValue) configValue;
		    if (multiValue.getName().equals(name))
			return multiValue;
		}
	    }
	}
	return null;
    }

    public void setValue(LineSender sender, String args[], int arg) throws BadCommandException {
	BeraAssert.require(args.length > 0);
	BeraAssert.require(arg >= 0);
	BeraAssert.require(arg < args.length);

	int lastArg = args.length - 1;
	String childLabel = args[arg];

	//
	// Is the child label valid?
	//
	ConfigNode childNode = ((InternalNode) getConfigNode()).findChild(childLabel);
	if (childNode == null)
	    throw new BadCommandException("invalid config label: " + childLabel);

	//
	// Is child multi node?
	//
	if (childNode instanceof MultiNode) {

	    if (arg == lastArg)
		throw new BadCommandException("missing name for config label: " + childLabel);

	    int childNameArg = arg + 1;

	    String childName = args[childNameArg];
	    MultiValue childMultiValue = findChildByLabel(childLabel, childName);
	    if (childMultiValue != null) {

		if (childNameArg == lastArg) {
		    sender.sendLine("FIXME DEBUG: nothing to do");
		    return;
		}

		childMultiValue.setValue(sender, args, childNameArg + 1);
		return;
	    }

	    //
	    // Multi child value does not exist
	    //

	    if (childNameArg != lastArg)
		throw new BadCommandException("config value not found: '" + childLabel + " " + childName + "'");

	    childList.add(new MultiValue((MultiNode) childNode, childName));

	    sender.sendLine("FIXME DEBUG: added MultiNode value");

	    return;
	}

	//
	// Find non-multi-node child
	//
	ConfigValue childValue = findChildByLabel(childLabel);
	if (childValue != null) {

	    BeraAssert.require(childValue.getConfigNode() == childNode);

	    if (arg == lastArg) {

		// LeafNode requires a value argument
		if (childNode instanceof LeafNode)

		    // Except BooleanLeafNode which does not use an explicit value
		    if (((LeafNode) childNode).useValue())
			throw new BadCommandException("missing value for label: " + childLabel);

		sender.sendLine("FIXME DEBUG: nothing to do");
		return;
	    }

	    childValue.setValue(sender, args, arg + 1);

	    return;
	}

	//
	// Add the new child
	//

	// Must be SingleNode ?
	if (childNode instanceof SingleNode) {

	    if (arg != lastArg)
		throw new BadCommandException("extra argument after label '" + childLabel + "': " + args[arg + 1]);

	    childList.add(new SingleValue((SingleNode) childNode));

	    sender.sendLine("FIXME DEBUG: added SingleNode value");

	    return;
	}

	//
	// Add leaf node child
	//

	// Must be BooleanLeafNode ?
	if (childNode instanceof BooleanLeafNode) {

	    if (arg != lastArg)
		throw new BadCommandException("extra argument after label '" + childLabel + "': " + args[arg + 1]);

	    BooleanLeafValue newBoolValue;
	    try {
		newBoolValue = (BooleanLeafValue) ((LeafNode) childNode).createConfigValue(null);
	    }
	    catch (BadValueException e) {
		e.printStackTrace();
		BeraAssert.require(false); // NOT REACHED
		return; // keep java happy about non-initialized newBoolValue
	    }

	    childList.add(newBoolValue);

	    return;
	}

	// 
	// Add non-boolean leaf node child
	//

	if (arg == lastArg)
	    throw new BadCommandException("missing value for label: " + childLabel);

	int valueArg = arg + 1;
	String strValue = args[valueArg];

	if (valueArg != lastArg)
	    throw new BadCommandException("extra argument after value '" + strValue + "': " + args[valueArg + 1]);

	ConfigValue newValue;
	try {
	    newValue = ((LeafNode) childNode).createConfigValue(strValue);
	}
	catch (BadValueException e) {
	    throw new BadCommandException("bad value '" + strValue + "': " + e.getMessage());
	}

	childList.add(newValue);
    }

    private void prune(ConfigValue child) {
	boolean success = childList.remove(child);
	BeraAssert.require(success);
    }

    public void unsetValue(LineSender sender, String args[], int arg) throws BadCommandException {
	BeraAssert.require(args.length > 0);
	BeraAssert.require(arg >= 0);
	BeraAssert.require(arg < args.length);

	int lastArg = args.length - 1;
	String childLabel = args[arg];

	//
	// Is the child label valid?
	//
	ConfigNode childNode = ((InternalNode) getConfigNode()).findChild(childLabel);
	if (childNode == null)
	    throw new BadCommandException("invalid config label: " + childLabel);

	//
	// Is child multi node?
	//
	if (childNode instanceof MultiNode) {

	    if (arg == lastArg)
		throw new BadCommandException("missing name for config label: " + childLabel);

	    int childNameArg = arg + 1;

	    String childName = args[childNameArg];
	    MultiValue childMultiValue = findChildByLabel(childLabel, childName);
	    if (childMultiValue == null)
		throw new BadCommandException("config value not found: '" + childLabel + " " + childName + "'");

	    // Found

	    // Last argument reached?
	    if (childNameArg == lastArg) {
		prune(childMultiValue);
		sender.sendLine("FIXME DEBUG: config value pruned");
		return;
	    }
	    
	    // Keep searching downstream
	    
	    childMultiValue.unsetValue(sender, args, childNameArg + 1);
	    return;
	}

	//
	// Find non-multi-node child
	//
	ConfigValue childValue = findChildByLabel(childLabel);
	if (childValue == null)
	    throw new BadCommandException("config value not found: " + childLabel);

	BeraAssert.require(childValue.getConfigNode() == childNode);

	// LeafNode child?
	if (childNode instanceof LeafNode)
	    if (arg != lastArg)
		if (!((LeafNode) childNode).useValue())
		    throw new BadCommandException("extra argument after label '" + childLabel + "': " + args[arg + 1]);

	//
	// SingleNode child
	//

	if (arg == lastArg) {
	    prune(childValue);
	    sender.sendLine("FIXME DEBUG: config value pruned");
	    return;
	}

	// Keep searching downstream
	childValue.unsetValue(sender, args, arg + 1);
    }

    public void rpcDump(Hashtable configMap) {
	for (int i = 0; i < childList.size(); ++i) {
	    ConfigValue childValue = (ConfigValue) childList.get(i);
	    ConfigNode childNode = childValue.getConfigNode();
	    String childLabel = childNode.getLabel();

	    if (childNode instanceof MultiNode) {
		Hashtable childMap = (Hashtable) configMap.get(childLabel);
		if (childMap == null) {
		    childMap = new Hashtable();
		    configMap.put(childLabel, childMap);
		}

		MultiValue multiChildValue = (MultiValue) childValue;
		String multiChildName = multiChildValue.getName();
		BeraAssert.require(childMap.get(multiChildName) == null);
		Hashtable multiMap = new Hashtable();
		childMap.put(multiChildName, multiMap);

		childValue.rpcDump(multiMap);
		return;
	    }

	    if (childNode instanceof SingleNode) {
		Hashtable map = new Hashtable();
		configMap.put(childLabel, map);
		childValue.rpcDump(map);
		return;
	    }

	    childValue.rpcDump(configMap);
	}
    }
}


