/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: RipPacket.java,v 1.14 2003/11/15 00:06:06 evertonm Exp $

package bera.rip;

import java.net.InetAddress;
import java.util.List;
import java.util.ArrayList;

import org.apache.log4j.Logger;

import bera.BeraAssert;
import bera.net.BeraNetworkInterface;
import bera.net.BeraInterfaceAddress;

class RipPacket {

    static private final Logger logger = Logger.getLogger(RipPacket.class);

    static final int PACKET_HEADER_SIZE          = 4;
    static final int PACKET_ENTRY_SIZE           = 20;
    static final int PACKET_MIN_ENTRIES          = 1;
    static final int PACKET_MAX_ENTRIES          = 25;
    static final int PACKET_MIN_SIZE             = PACKET_HEADER_SIZE + PACKET_ENTRY_SIZE * PACKET_MIN_ENTRIES;
    static final int PACKET_MAX_SIZE             = PACKET_HEADER_SIZE + PACKET_ENTRY_SIZE * PACKET_MAX_ENTRIES;
    static final int PACKET_ENTRY_METRIC_OFFSET  = 16;
    static final int PACKET_ENTRY_NEXTHOP_OFFSET = 12;

    static final byte CMD_REQUEST     = 1;
    static final byte CMD_RESPONSE    = 2;
    static final byte VERSION_TWO     = 2;
    static final byte METRIC_INFINITY = 16;

    private int  packetCommand;
    private int  packetVersion;
    private List packetEntryList;

    static int getPacketSize(int entries) {
	return PACKET_HEADER_SIZE + PACKET_ENTRY_SIZE * entries;
    }

    static boolean badLength(int length) {
	if (length < PACKET_MIN_SIZE) {
	    logger.error("packet too short: size=" + length);
	    return true;
	}

	if (length > PACKET_MAX_SIZE) {
	    logger.error("packet too long: size=" + length);
	    return true;
	}
	
	if ((length % PACKET_ENTRY_SIZE) != PACKET_HEADER_SIZE) {
	    logger.error("bad packet size: size=" + length);
	    return true;
	}

	return false;
    }

    static RipPacket parse(byte[] buffer, int length) {

	if (length > buffer.length) {
	    logger.error("bad buffer length: capacity=" + buffer.length + " size=" + length);
	    return null;
	}
	
	if (badLength(length))
	    return null;
	    
	int command = buffer[0];
	int version = buffer[1];

	if (version != VERSION_TWO) {
	    logger.error("unknown protocol version: " + version);
	    return null;
	}

	logger.debug("FIXME: detect authentication entry");

	int entries = (length - PACKET_HEADER_SIZE) / PACKET_ENTRY_SIZE;

	List entryList = new ArrayList(entries);

	for (int i = 0; i < entries; ++i) {

	    boolean acceptNullAddressFamily = (entries == 1) && (command == CMD_REQUEST);

	    RipPacketEntry entry = RipPacketEntry.parse(buffer, 
							i * PACKET_ENTRY_SIZE + PACKET_HEADER_SIZE,
							acceptNullAddressFamily);
	    if (entry == null) {
		logger.error("bad packet");
		return null;
	    }
	    entryList.add(entry);
	}

	return new RipPacket(command, version, entryList);
    }

    static int buildResponse(byte[] buffer, List routeList, int firstRoute, int lastRoute) {

	int totalRoutes = routeList.size();

	BeraAssert.require(firstRoute <= lastRoute);
	BeraAssert.require(firstRoute >= 0);
	BeraAssert.require(lastRoute < totalRoutes);
	BeraAssert.require(!badLength(buffer.length));
	BeraAssert.require(totalRoutes >= PACKET_MIN_ENTRIES);
	BeraAssert.require(totalRoutes <= PACKET_MAX_ENTRIES);

	buffer[0] = CMD_RESPONSE;
	buffer[1] = VERSION_TWO;
	buffer[2] = 0;
	buffer[3] = 0;

	int offset = PACKET_HEADER_SIZE;
	for (int i = firstRoute; i <= lastRoute; ++i) {
	    RipRoute ripRoute = (RipRoute) routeList.get(i);
	    ripRoute.build(buffer, offset, false /* poisoned reverse */);
	    offset += PACKET_ENTRY_SIZE;
	}

	return offset;
    }

    static int buildResponse(byte[] buffer, List routeList, int firstRoute, int lastRoute,
			     BeraNetworkInterface outputIface, 
			     BeraInterfaceAddress outputIfaceAddr,
			     RipConfig ripConfig) {

	int totalRoutes = routeList.size();

	BeraAssert.require(firstRoute <= lastRoute);
	BeraAssert.require(firstRoute >= 0);
	BeraAssert.require(lastRoute < totalRoutes);
	BeraAssert.require(!badLength(buffer.length));
	BeraAssert.require(totalRoutes >= PACKET_MIN_ENTRIES);
	BeraAssert.require(totalRoutes <= PACKET_MAX_ENTRIES);

	buffer[0] = CMD_RESPONSE;
	buffer[1] = VERSION_TWO;
	buffer[2] = 0;
	buffer[3] = 0;

	int offset = PACKET_HEADER_SIZE;
	for (int i = firstRoute; i <= lastRoute; ++i) {
	    RipRoute ripRoute = (RipRoute) routeList.get(i);

	    InetAddress nextHop = ripRoute.getNextHop();
	    if (nextHop != null) {

		// Perform split horizon on this interface?
		if (outputIfaceAddr.sameNetwork(nextHop)) {
		    logger.debug("performing split horizon");

		    // Perform poisoned reverse on this interface?
		    if (!ripConfig.usePoisonedReverse(outputIfaceAddr.getAddress())) {

			// Multiaccess interfaces can't use
			// silent (non-poisoned) split horizon,
			// otherwise we risk leaving some neighbors
			// without update information
			//
			// Thus, for multiaccess interfaces, we
			// force poisoned reverse split horizon
			if (!outputIface.multiaccess())
			    continue; // skip route if split horizon without poisoned reverse
		    }

		    logger.debug("performing poisoned reverse");
		    
		    ripRoute.build(buffer, offset, true /* poisonedReverse */);
		    offset += PACKET_ENTRY_SIZE;
		    
		    continue;
		}
	    }

	    ripRoute.build(buffer, offset, false /* poisonedReverse */);
	    offset += PACKET_ENTRY_SIZE;
	}

	return offset;
    }

    RipPacket(int command, int version, List entryList) {
	packetCommand   = command;
	packetVersion   = version;
	packetEntryList = entryList;
    }

    void addEntry(RipPacketEntry packetEntry) {
	packetEntryList.add(packetEntry);
    }

    int getCommand() {
	return packetCommand;
    }

    List getEntryList() {
	return packetEntryList;
    }
}
