/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: RipPacketEntry.java,v 1.10 2003/11/15 00:06:06 evertonm Exp $

package bera.rip;

import java.net.InetAddress;

import org.apache.log4j.Logger;

import bera.BeraAssert;
import bera.net.AddressFamily;
import bera.util.Conversion;

class RipPacketEntry {

    static private final Logger logger = Logger.getLogger(RipPacketEntry.class);

    int    entryAddressFamily;
    int    entryRouteTag;
    byte[] entryAddress;
    byte[] entrySubnetMask;
    byte[] entryNextHop;
    int    entryMetric;

    String dumpString() {
	return "RipPacketEntry: AF=" + entryAddressFamily + " tag=" + entryRouteTag + " net=" + debugAddr(entryAddress) + " mask=" + debugAddr(entrySubnetMask) + " NH=" + debugAddr(entryNextHop) + " m=" + entryMetric;
    }

    RipPacketEntry(int addressFamily, int routeTag, byte[] netAddr, byte[] subnetMask, byte[] nextHop, int metric) {
	entryAddressFamily = addressFamily;
	entryRouteTag      = routeTag;
	entryAddress       = netAddr;
	entrySubnetMask    = subnetMask;
	entryNextHop       = nextHop;
	entryMetric        = metric;
    }

    static RipPacketEntry parse(byte[] buffer, int offset, boolean acceptNullAddressFamily) {

	if (offset < RipPacket.PACKET_HEADER_SIZE) {
	    logger.error("bad buffer offset: " + offset);
	    return null;
	}

	if ((offset + RipPacket.PACKET_ENTRY_SIZE) > buffer.length) {
	    logger.error("bad buffer offset: length=" + buffer.length + " offset=" + offset);
	    return null;
	}

	int addressFamily = Conversion.byteToShort(buffer, offset);

	if (addressFamily != AddressFamily.AF_INET) {
	    if (!acceptNullAddressFamily || (addressFamily != 0)) {
		logger.error("bad address family: " + addressFamily);
		return null;
	    }
	}

	int routeTag = Conversion.byteToShort(buffer, offset + 2);

	byte[] netAddr = new byte[4];
	for (int i = 0; i < 4; ++i)
	    netAddr[i] = buffer[offset + i + 4];

	byte[] subnetMask = new byte[4];
	for (int i = 0; i < 4; ++i)
	    subnetMask[i] = buffer[offset + i + 8];

	byte[] nextHop = new byte[4];
	for (int i = 0; i < 4; ++i)
	    nextHop[i] = buffer[offset + i + RipPacket.PACKET_ENTRY_NEXTHOP_OFFSET];
	
	int metric = Conversion.byteToInt(buffer, offset + RipPacket.PACKET_ENTRY_METRIC_OFFSET);
	if ((metric < 1) || (metric > RipPacket.METRIC_INFINITY)) {
	    logger.error("bad metric: " + metric);
	    return null;
	}

	RipPacketEntry entry = new RipPacketEntry(addressFamily, routeTag, netAddr, subnetMask, nextHop, metric);

	//logger.debug(entry.dumpString());

	return entry;
    }
    
    private static String debugAddr(byte[] buffer) {
	String str = "";

	for (int i = 0; i < buffer.length; ++i)
	    str += (int) (0xFF & buffer[i]) + ".";

	return str;
    }

    static void build(byte[] buffer, int offset, 
		      int addressFamily,
		      int routeTag,
		      InetAddress networkAddress,
		      int prefixLen,
		      InetAddress nextHop,
		      int metric) {

	BeraAssert.require(offset >= RipPacket.PACKET_HEADER_SIZE);
	BeraAssert.require(offset < buffer.length);
	BeraAssert.require(offset + RipPacket.PACKET_ENTRY_SIZE <= buffer.length);
	BeraAssert.require(!RipPacket.badLength(buffer.length));

	// Address family
	Conversion.shortToByte(buffer, offset, addressFamily);

	// Route tag
	Conversion.shortToByte(buffer, offset + 2, routeTag);

	// Network address
	int addrOff = offset + 4;
	byte[] netAddr = networkAddress.getAddress();
	for (int i = 0; i < netAddr.length; ++i)
	    buffer[addrOff + i] = netAddr[i];

	// Subnet mask
	Conversion.lengthToMask(buffer, 4, offset + 8, prefixLen);

	// Next hop
	byte[] nextHopAddr = nextHop.getAddress();
	int nhOff = offset + RipPacket.PACKET_ENTRY_NEXTHOP_OFFSET;
	for (int i = 0; i < nextHopAddr.length; ++i)
	    buffer[nhOff + i] = nextHopAddr[i];

	// Metric
	Conversion.intToByte(buffer, offset + RipPacket.PACKET_ENTRY_METRIC_OFFSET, metric);
    }

}


