/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: RipQuery.java,v 1.5 2003/11/15 00:06:06 evertonm Exp $

package bera.rip;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.net.InetAddress;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.UnknownHostException;
import java.util.List;
import java.util.ArrayList;
import java.util.StringTokenizer;

import org.apache.log4j.BasicConfigurator;

import bera.BeraAssert;

public class RipQuery {

    private static void go(String hostname) {

	InetAddress hostAddr;
	try {
	    hostAddr = InetAddress.getByName(hostname);
	}
	catch (UnknownHostException e) {
	    System.err.println("could not solve hostname: " + hostname);
	    return;
	}

	System.err.println("enter 'addressFamily networkAddress prefixLength' tuples, one per line");
	System.err.println("example: 2 192.168.1.0 24");

	//
	// Load input
	//

	List entryList = new ArrayList();
	BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
	for (;;) {
	    String line;
	    try {
		line = in.readLine();
	    }
	    catch (IOException e) {
		System.err.println("could not read line: " + e);
		return;
	    }
	    if (line == null)
		break;

	    StringTokenizer toker = new StringTokenizer(line);
	    int tokenCount = toker.countTokens();
	    if (tokenCount != 3) {
		System.err.println("bad token count: " + tokenCount);
		continue;
	    }
	    String tokenAddressFamily = toker.nextToken();
	    String tokenNetAddr = toker.nextToken();
	    String tokenPrefixLength = toker.nextToken();
	    
	    short       addrFamily;
	    InetAddress netAddr;
	    short       prefixLen;

	    addrFamily = Short.parseShort(tokenAddressFamily);
	    prefixLen  = Short.parseShort(tokenPrefixLength);

	    try {
		netAddr = InetAddress.getByName(tokenNetAddr);
	    }
	    catch (UnknownHostException e) {
		System.err.println("could not solve network address: " + tokenNetAddr + ": " + e);
		continue;
	    }

	    entryList.add(new Entry(addrFamily, netAddr, prefixLen));
	}

	System.err.println("building request packet");

	//
	// Build request packet
	//

	int entries = entryList.size();
	int bufferSize = RipPacket.getPacketSize(entries);
	byte[] packetBuffer = new byte[bufferSize];

	packetBuffer[0] = RipPacket.CMD_REQUEST;
	packetBuffer[1] = RipPacket.VERSION_TWO;
	packetBuffer[2] = 0;
	packetBuffer[3] = 0;

	int offset = RipPacket.PACKET_HEADER_SIZE;
	for (int i = 0; i < entries; ++i) {
	    Entry e = (Entry) entryList.get(i);
	    RipPacketEntry.build(packetBuffer, offset, 
				 e.entryAddrFamily,
				 0,
				 e.entryNetAddr,
				 e.entryPrefixLen,
				 RipServer.getWildcardAddress(),
				 RipPacket.METRIC_INFINITY);
	    offset += RipPacket.PACKET_ENTRY_SIZE;
	}

	BeraAssert.require(offset == bufferSize);

	//
	// Send request packet
	//

	System.err.println("sending request packet");

	DatagramPacket packet = new DatagramPacket(packetBuffer, packetBuffer.length);

	packet.setAddress(hostAddr);
	packet.setPort(RipServer.RIP_UDP_PORT);

	DatagramSocket socket;
	try {
            socket = new DatagramSocket();
	}
	catch (Exception e) {
            System.err.println("could not create socket: " + e);
            return;
	}

	try {
	    socket.send(packet);
	}
	catch (IOException e) {
	    System.err.println("could not send packet: " + e);
	}

	//
	// Receive response packets
	//

	System.err.println("waiting response packets");

	byte[] recvBuffer = new byte[RipPacket.PACKET_MAX_SIZE];

	DatagramPacket recvPacket = new DatagramPacket(recvBuffer, recvBuffer.length);

	for (;;) {
	    try {
		socket.receive(recvPacket);
	    }
	    catch (IOException e) {
		System.err.println("could not receive packet: " + e);
		continue;
	    }

	    System.err.println("packet received");

	    byte[] buf = recvPacket.getData();

	    RipPacket ripPacket = RipPacket.parse(buf, recvPacket.getLength());
	    if (ripPacket == null) {
		System.err.println("ignoring bad packet");
		continue;
	    }

	    List eList = ripPacket.getEntryList();
	    int sz = eList.size();
	    for (int i = 0; i < sz; ++i) {
		RipPacketEntry entry = (RipPacketEntry) eList.get(i);

		System.err.println(entry.dumpString());
	    }

	}
    }

    public static void main(String args[]) {
	if (args.length != 1) {
	    System.err.println("usage: bera.rip.RipQuery hostname");
	    System.exit(1);
	}

	BasicConfigurator.configure();

	go(args[0]);
    }
}

class Entry {
    short       entryAddrFamily;
    InetAddress entryNetAddr;
    short       entryPrefixLen;

    Entry(short addrFamily, InetAddress netAddr, short prefixLen) {
	entryAddrFamily = addrFamily;
	entryNetAddr    = netAddr;
	entryPrefixLen  = prefixLen;
    }
}
