/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: RipRoute.java,v 1.17 2003/10/19 05:36:02 evertonm Exp $

package bera.rip;

import java.net.InetAddress;

import org.apache.log4j.Logger;

import bera.BeraAssert;
import bera.util.Conversion;
import bera.net.AddressFamily;

class RipRoute {

    static private final Logger logger = Logger.getLogger(RipRouteTable.class);

    static private final String AF_IPV4 = "IPv4";

    static private RipConfig ripConfig = null;
    static private RipRouteTable ripRouteTable = null;

    // Used by RipRouteTable
    static void setConfig(RipConfig config) {
	BeraAssert.require(config != null);
	ripConfig = config;
    }

    // Used by RipRouteTable
    static void setRouteTable(RipRouteTable routeTable) {
	BeraAssert.require(routeTable != null);
	ripRouteTable = routeTable;
    }

    private InetAddress routeNetwork;
    private int         routePrefixLength;
    private int         routeMetric;
    private InetAddress routeNextHop;
    private boolean     routeChangedFlag;
    private int         routeTimerTimeout;           // seconds
    private int         routeTimerGarbageCollection; // seconds
    private int         routeTag;
    private boolean     routeRibPending;

    private void assertRequire() {
	BeraAssert.require(
			   ((routeTimerTimeout > 0) && (routeMetric < RipPacket.METRIC_INFINITY))
			   ||
			   ((routeTimerTimeout == 0) && (routeMetric == RipPacket.METRIC_INFINITY))
			   );
	BeraAssert.require(ripConfig != null);
	BeraAssert.require(ripRouteTable != null);
    }

    private void routeChanged() {
	routeChangedFlag = true;
	ripRouteTable.triggerUpdate();
    }

    RipRoute(InetAddress netAddr, int length, InetAddress nextHop, int metric) {
	BeraAssert.require(netAddr != null);
	BeraAssert.require(length >= 0);
	BeraAssert.require(length <= 32);
	BeraAssert.require(metric >= 1);
	BeraAssert.require(metric < RipPacket.METRIC_INFINITY);

	routeNetwork                = netAddr;
	routePrefixLength           = length;
	routeMetric                 = metric;
	routeNextHop                = nextHop;
	routeTag                    = 0;
	routeRibPending             = true; // send to RIB
	routeTimerTimeout           = ripConfig.getRipTimerTimeout(); // 180
	routeTimerGarbageCollection = ripConfig.getRipTimerGarbage(); // 120

	routeChanged();

	assertRequire();
    }

    String getAddressFamily() {
	return AF_IPV4;
    }

    boolean ribIsPending() {
	return routeRibPending;
    }

    boolean updateIsPending() {
	return routeChangedFlag;
    }

    void sentToRib() {
	BeraAssert.require(routeRibPending);
	routeRibPending = false;
    }

    void refresh() {
	
	// (routeTimerTimeout == 0) <=> (metric == RipPacket.METRIC_INFINITY)
	//
	// Thus: to reset timeout to 180 implies on non-infinity metric

	// It's not allowed to refresh a route with invalid metric
	BeraAssert.require(routeMetric >= 1);
	BeraAssert.require(routeMetric < RipPacket.METRIC_INFINITY);

	// If we're refreshing an expired route, we may need to insert into RIB
	if (expired())
	    routeRibPending = true;

	routeTimerTimeout           = ripConfig.getRipTimerTimeout();
	routeTimerGarbageCollection = ripConfig.getRipTimerGarbage();

	assertRequire();
    }

    void setNextHop(InetAddress nextHop) {

	// Nothing changed
	if (nextHop == null) {
	    if (nextHop == routeNextHop)
		return;
	}
	else if (nextHop.equals(routeNextHop))
	    return;

	// If the route has changed, re-insert into RIB
	routeRibPending = true;

	routeNextHop     = nextHop;

	routeChanged();
    }

    void setMetric(int metric) {

	// It's not allowed to set an invalid metric
	BeraAssert.require(metric >= 1);
	BeraAssert.require(metric <= RipPacket.METRIC_INFINITY);

	// Nothing changed
	if (metric == routeMetric)
	    return;

	// If we're recycling an expired route, we may need to insert into RIB
	if (expired())
	    if (metric != RipPacket.METRIC_INFINITY)
		routeRibPending = true;

	routeMetric      = metric;

	// Ensure metric/timer consistency
	if (metric == RipPacket.METRIC_INFINITY)
	    routeTimerTimeout = 0;
	else
	    refresh();

	routeChanged();

	assertRequire();
    }

    void updateSent() {
	routeChangedFlag = false;
    }

    void age(int elapsed) {

	BeraAssert.require(elapsed > 0);
	BeraAssert.require(routeTimerTimeout >= 0);
	BeraAssert.require(routeTimerTimeout <= ripConfig.getRipTimerTimeout());
	BeraAssert.require(routeTimerGarbageCollection >= 0);
	BeraAssert.require(routeTimerGarbageCollection <= ripConfig.getRipTimerGarbage());

	if (routeTimerTimeout > 0) {
	    routeTimerTimeout -= elapsed;
	    if (routeTimerTimeout < 0)
		routeTimerTimeout = 0;
	    if (routeTimerTimeout == 0) {                      // deletion
		routeMetric       = RipPacket.METRIC_INFINITY; // process
		routeChanged();
	    }

	    logger.debug("route aged: " + dumpString() + " elapsed=" + elapsed + " timeout=" + routeTimerTimeout + " garbage=" + routeTimerGarbageCollection);

	    assertRequire();
	    return;
	}

	if (routeTimerGarbageCollection > 0) {
	    routeTimerGarbageCollection -= elapsed;
	    if (routeTimerGarbageCollection < 0)
		routeTimerGarbageCollection = 0;
	}

	logger.debug("route aged: " + dumpString() + " elapsed=" + elapsed + " timeout=" + routeTimerTimeout + " garbage=" + routeTimerGarbageCollection);

	BeraAssert.require(routeTimerTimeout >= 0);
	BeraAssert.require(routeTimerTimeout <= ripConfig.getRipTimerTimeout());
	BeraAssert.require(routeTimerGarbageCollection >= 0);
	BeraAssert.require(routeTimerGarbageCollection <= ripConfig.getRipTimerGarbage());

	assertRequire();
    }

    // Used to expire a removed connected address
    void expire() {
	if (!expired())
	    routeChanged();

	routeTimerTimeout = 0;
	routeMetric       = RipPacket.METRIC_INFINITY;

	BeraAssert.require(expired());

	assertRequire();
    }

    // Should the route be used to forward traffic?
    boolean expired() {
	return routeTimerTimeout == 0;
    }

    // Should the route be deleted right now?
    boolean dead() {
	BeraAssert.require(routeTimerGarbageCollection >= 0);
	BeraAssert.require(routeTimerGarbageCollection <= ripConfig.getRipTimerGarbage());

	return routeTimerGarbageCollection == 0;
    }

    boolean match(InetAddress netAddr, int prefixLength) {
	return (routePrefixLength == prefixLength) && (routeNetwork.equals(netAddr));
    }

    InetAddress getNetwork() {
	return routeNetwork;
    }

    int getPrefixLength() {
	return routePrefixLength;
    }

    InetAddress getNextHop() {
	return routeNextHop;
    }

    int getMetric() {
	return routeMetric;
    }

    int getTimeout() {
	return routeTimerTimeout;
    }

    void build(byte[] buffer, int offset, boolean poisonedReverse) {
	InetAddress nextHop;
	if (routeNextHop != null)
	    nextHop = routeNextHop;
	else
	    nextHop = RipServer.getWildcardAddress();

	int metric = poisonedReverse ? RipPacket.METRIC_INFINITY : routeMetric;

	RipPacketEntry.build(buffer, offset,
			     AddressFamily.AF_INET,
			     routeTag,
			     routeNetwork,
			     routePrefixLength,
			     nextHop,
			     metric);
    }

    private String dumpString() {
	return "RipRoute:" + routeNetwork.getHostAddress() + "/" + routePrefixLength + "@" + routeMetric + "->" + routeNextHop;
    }
}
