/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: RipRouteTable.java,v 1.20 2003/12/03 20:27:01 evertonm Exp $

package bera.rip;

import java.util.Vector;
import java.util.Hashtable;
import java.util.List;
import java.util.ArrayList;
import java.net.InetAddress;
import java.net.UnknownHostException;

import org.apache.log4j.Logger;
import org.apache.xmlrpc.XmlRpcClientLite;
import org.apache.xmlrpc.XmlRpcException;

import bera.BeraAssert;
import bera.BeraServices;
import bera.util.Conversion;

class RipRouteTable {

    static private final Logger logger = Logger.getLogger(RipRouteTable.class);

    private RipConfig ripConfig;
    private List      routeList = new ArrayList();
    private boolean   triggeredUpdatePending = false;
    private String    instanceLabel;

    RipRouteTable(RipConfig config, String ripInstanceLabel) {
	ripConfig = config;
	instanceLabel = ripInstanceLabel;
	RipRoute.setConfig(ripConfig);
	RipRoute.setRouteTable(this);
    }

    // Used by RipRoute
    void triggerUpdate() {
	triggeredUpdatePending = true;
    }

    // Used by RipServer
    void triggeredUpdateSent() {
	triggeredUpdatePending = false;
    }

    boolean triggeredUpdateIsPending() {
	return triggeredUpdatePending;
    }

    RipRoute findRoute(InetAddress netAddr, int prefixLen) {
	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    RipRoute route = (RipRoute) routeList.get(i);
	    if (route.match(netAddr, prefixLen))
		return route;
	}
	return null;
    }

    void expireDeadConnectedAddresses(RipServer ripServer) {
	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    RipRoute route = (RipRoute) routeList.get(i);

	    // Is a connected route?
	    if (route.getNextHop() == null) {

		// Has expired?
		if (!route.expired()) {

		    // Expire route, it does not exist locally
		    if (!ripServer.findConnectedNetwork(route.getNetwork(), route.getPrefixLength())) {

			route.expire();

			BeraAssert.require(route.expired());
		    }
		}
	    }
	}
    }

    void addConnectedRoute(InetAddress netAddr, int prefixLength, int metric) {

	BeraAssert.require(metric >= 1);
	BeraAssert.require(metric < RipPacket.METRIC_INFINITY);

	RipRoute existingRoute = findRoute(netAddr, prefixLength);
	if (existingRoute == null) {

	    logger.debug("adding new connected route: " + netAddr.getHostAddress() + "/" + prefixLength);

	    RipRoute connectedRoute = new RipRoute(netAddr, prefixLength, null, metric);
	    routeList.add(connectedRoute);
	    return;
	}

	//logger.debug("updating existing connected route: " + netAddr + "/" + prefixLength);

	existingRoute.setNextHop(null);  // mark as directly connected
	existingRoute.setMetric(metric); // update metric
	
	// metric is non-infinity, so we can reset the counters
	existingRoute.refresh();         
    }

    void addRoute(byte[] address, byte[] mask, int metric, InetAddress nextHop) {

	BeraAssert.require(metric >= 1);
	BeraAssert.require(metric <= RipPacket.METRIC_INFINITY);

	InetAddress netAddr;
	try {
	    netAddr = InetAddress.getByAddress(address);
	}
	catch (UnknownHostException e) {
	    logger.error("bad network address: " + e);
	    return;
	}

	int prefixLength = Conversion.maskToLength(mask);

	RipRoute existingRoute = findRoute(netAddr, prefixLength);
	if (existingRoute == null) {

	    if (metric == RipPacket.METRIC_INFINITY) {
		logger.debug("refusing to learn new route with infinity metric: " + netAddr + "/" + prefixLength);
		return;
	    }

	    logger.debug("learning new route: " + netAddr + "/" + prefixLength);

	    RipRoute newRoute = new RipRoute(netAddr, prefixLength, nextHop, metric);
	    routeList.add(newRoute);
	    return;
	}

	//
	// From RFC 2453, page 27, last paragraph and forth
	//

	if (nextHop.equals(existingRoute.getNextHop())) {

	    // can't refresh route with metric infinity
	    if (existingRoute.getMetric() != RipPacket.METRIC_INFINITY)
		existingRoute.refresh();

	    if (metric != existingRoute.getMetric()) {
		// setNextHop -> redundant
		existingRoute.setMetric(metric); // may start deletion process
		return;
	    }
	}

	if (metric < existingRoute.getMetric()) {
	    existingRoute.setNextHop(nextHop);
	    existingRoute.setMetric(metric);   // non-infinity, refresh
	    return;
	}

	// Heuristic to replace a expiring by the new one
	if (metric == existingRoute.getMetric())
	    if (metric != RipPacket.METRIC_INFINITY)
		if (existingRoute.getTimeout() < (ripConfig.getRipTimerTimeout() >> 1)) {
		    existingRoute.setNextHop(nextHop);
		    existingRoute.setMetric(metric);   // non-infinity, refresh
		    return;
		}

	// If this point is reached, the route is simply ignored
	//logger.debug("existing route is better than the received one");
    }

    List getRouteList() {
	return routeList;
    }

    void ageRoutes(int elapsed) {
	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    RipRoute route = (RipRoute) routeList.get(i);
	    route.age(elapsed);
	}
    }

    void sendRibRoutes() {
	Vector pendingRouteList = new Vector();

	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    RipRoute route = (RipRoute) routeList.get(i);
	    if (route.ribIsPending()) {
		BeraAssert.require(!route.expired());

		Hashtable pendingRoute = new Hashtable();
		pendingRouteList.add(pendingRoute);

		pendingRoute.put("routeProtocol", "rip");
		pendingRoute.put("routeProtocolInstance", instanceLabel);
		pendingRoute.put("routeAddressFamily", route.getAddressFamily());
		pendingRoute.put("routeNetwork", route.getNetwork().getHostAddress());
		pendingRoute.put("routePrefixLength", new Integer(route.getPrefixLength()));

		InetAddress nextHop = route.getNextHop();
		InetAddress nh = (nextHop == null) ? RipServer.getWildcardAddress() : nextHop;
		
		pendingRoute.put("routeNextHop", nh.getHostAddress());

		route.sentToRib();
	    }
	}

	if (pendingRouteList.size() == 0)
	    return;

	String method = "rib.addRoutes";
	logger.debug("calling XMLRPC method " + method + " on URL " + BeraServices.RIB_URL);

	XmlRpcClientLite ribClient;
	try {
	    ribClient = new XmlRpcClientLite(BeraServices.RIB_URL);
	}
	catch (Exception e) {
            logger.error("could not create XML-RPC client: " + e);
	    return;
        }

	Vector params = new Vector();
	params.add(pendingRouteList);

	try {
	    Object result = ribClient.execute(method, params);
	}
	catch (Exception e) {
            logger.error("could not call XML-RPC method: " + e);
            return;
        }
    }

    void withdrawExpiredRoutes() {
	Vector expiredRouteList = new Vector();

	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    RipRoute route = (RipRoute) routeList.get(i);
	    if (route.expired()) {
		BeraAssert.require(!route.ribIsPending());

		Hashtable expiredRoute = new Hashtable();
		expiredRouteList.add(expiredRoute);

		expiredRoute.put("routeProtocol", "rip");
		expiredRoute.put("routeProtocolInstance", instanceLabel);
		expiredRoute.put("routeAddressFamily", route.getAddressFamily());
		expiredRoute.put("routeNetwork", route.getNetwork().getHostAddress());
		expiredRoute.put("routePrefixLength", new Integer(route.getPrefixLength()));

		InetAddress nextHop = route.getNextHop();
		InetAddress nh = (nextHop == null) ? RipServer.getWildcardAddress() : nextHop;
		
		expiredRoute.put("routeNextHop", nh.getHostAddress());
	    }
	}

	if (expiredRouteList.size() == 0)
	    return;

	String method = "rib.withdrawRoutes";
	logger.debug("calling XMLRPC method " + method + " on URL " + BeraServices.RIB_URL);

	XmlRpcClientLite ribClient;
	try {
	    ribClient = new XmlRpcClientLite(BeraServices.RIB_URL);
	}
	catch (Exception e) {
            logger.error("could not create XML-RPC client: " + e);
	    return;
        }

	Vector params = new Vector();
	params.add(expiredRouteList);

	try {
	    Object result = ribClient.execute(method, params);
	}
	catch (Exception e) {
            logger.error("could not call XML-RPC method: " + e);
            return;
        }
    }

    void deleteDeadRoutes() {
	int size = routeList.size();
	for (int i = 0; i < size; ++i) {
	    RipRoute route = (RipRoute) routeList.get(i);
	    if (route.dead()) {
		routeList.remove(i);
		--i;
		size = routeList.size();
	    }
	}
    }
}
