/*-GNU-GPL-BEGIN-*
  bera - Buriti Experimental Routing Architecture
  Copyright (C) 2003  Everton da Silva Marques

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*-GNU-GPL-END-*/

// $Id: RipServer.java,v 1.40 2003/12/01 23:09:26 evertonm Exp $

package bera.rip;

import java.util.Enumeration;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Timer;
import java.util.TimerTask;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.net.MulticastSocket;
import java.net.SocketException;
import java.io.IOException;

import org.apache.log4j.Logger;
import org.apache.xmlrpc.XmlRpcClientLite;
import org.apache.xmlrpc.XmlRpcException;

import bera.BeraListen;
import bera.BeraServices;
import bera.BeraConstants;
import bera.BeraAssert;
import bera.net.BeraNetworkInterface;
import bera.net.BeraInterfaceAddress;
import bera.net.BeraInterfaceAddressIterator;
import bera.util.Conversion;

class RipServer {

    static private final Logger logger = Logger.getLogger(RipServer.class);

    static final String RIP_WILDCARD_ADDRESS  = BeraConstants.IPV4_WILDCARD_STRING;
    static final String RIP_MULTICAST_ADDRESS = BeraConstants.RIP_MULTICAST_ADDRESS;
    static final int    RIP_UDP_PORT          = BeraConstants.RIP_UDP_PORT;

    private Bool                 periodicTaskIsPending  = new Bool(false);
    private Timer                ripTimer               = new Timer(true);
    private EventLoopBlocker     eventLoopBlocker       = new EventLoopBlocker();
    private ThreadBlocker        multicastBlocker       = new ThreadBlocker();
    private RipConfig            ripConfig              = new RipConfig();
    private RipRouteTable        routeTable;
    private byte[]               sendBuffer             = new byte[RipPacket.PACKET_MAX_SIZE];
    private DatagramPacket       sendPacket             = new DatagramPacket(sendBuffer, sendBuffer.length);
    private RipMulticastListener multicastListener;
    private MulticastSocket      multicastSenderSocket;
    private List                 currentIfaceList       = null;
    private int                  periodicUpdateTimer;
    private int                  triggeredUpdateTimer;
    private String               ripInstanceLabel;

    static InetAddress getWildcardAddress() {
	InetAddress inetAddr = null;
	try {
	    inetAddr = InetAddress.getByName(RIP_WILDCARD_ADDRESS);
	}
	catch (UnknownHostException e) {
	    logger.error("could not solve wildcard address");
	    BeraAssert.require(false);
	}
	return inetAddr;
    }

    static InetAddress getMulticastAddress() {
	InetAddress inetAddr = null;
	try {
	    inetAddr = InetAddress.getByName(RIP_MULTICAST_ADDRESS);
	}
	catch (UnknownHostException e) {
	    logger.error("could not solve multicast address");
	    BeraAssert.require(false);
	}
	return inetAddr;
    }

    RipServer(String instanceLabel) {
	ripInstanceLabel = instanceLabel;
	routeTable = new RipRouteTable(ripConfig, ripInstanceLabel);
    }

    // Callback for periodic task thread.
    // A one-time scheduled RipTask thread calls this method.
    // There is no RipTask scheduled once this method is called.
    void periodicTaskCall() {

	// Both main loop thread and RipTask thread can access periodicTaskIsPending
	synchronized (periodicTaskIsPending) {
	    BeraAssert.require(!periodicTaskIsPending.value);
	    
	    // mark periodic task as pending for main loop thread
	    periodicTaskIsPending.value = true; 
	}

	logger.debug("periodic task call");

	// wake event loop thread
	eventLoopBlocker.eventAdded();
    }

    private void findNewRoutes() {

	if (currentIfaceList == null) {
	    logger.error("findNewRoutes(): empty interface list");
	    return;
	}

	for (Iterator ifaceIt = currentIfaceList.iterator();
	     ifaceIt.hasNext();
	     ) {
	    BeraNetworkInterface iface = (BeraNetworkInterface) ifaceIt.next();

	    int metric = iface.getMetric();
	    if ((metric < 1) || (metric >= RipPacket.METRIC_INFINITY)) {
		logger.error("ignoring connected route with bad metric: " + metric);
		continue;
	    }

	    for (Iterator addrIt = iface.getAddresses().iterator();
		 addrIt.hasNext();
		 ) {
		BeraInterfaceAddress ifaceAddr = (BeraInterfaceAddress) addrIt.next();
	
		InetAddress ipAddr = ifaceAddr.getAddress();
	        int addrPrefixLength = ifaceAddr.getPrefixLength();

		byte[] addr = (byte[]) ipAddr.getAddress().clone();
		BeraInterfaceAddress.forceNetworkPrefix(addr, addrPrefixLength);

		InetAddress netAddr;
		try {
		    netAddr = InetAddress.getByAddress(addr);
		}
		catch (UnknownHostException e) {
		    logger.error("could not make network address: " + ipAddr + ": " + e);
		    continue;
		}

		routeTable.addConnectedRoute(netAddr, addrPrefixLength, metric);
	    }
	}
    }

    private boolean sendResponsePacket(byte[] buf, int offset, int size, InetAddress outIface, InetAddress dstAddr, int dstPort) {
	BeraAssert.require(outIface != null);
	BeraAssert.require(dstAddr != null);

	sendPacket.setData(buf, offset, size);
	sendPacket.setAddress(dstAddr);
	sendPacket.setPort(dstPort);

	try {
	    multicastSenderSocket.setInterface(outIface);
	}
	catch (SocketException e) {
	    logger.error("failure defining output interface as: " + outIface.getHostAddress() + ": " + e);
	    return true;
	}

	logger.debug("sending " + size + " bytes to " + dstAddr.getHostAddress() + ":" + dstPort + " on interface " + outIface.getHostAddress());

	try {
	    multicastSenderSocket.send(sendPacket);
	}
	catch (IOException e) {
	    logger.error("failure writing response packet: " + e);
	    return true;
	}

	return false;
    }

    private void multicastBatch(List routeList, int firstRoute, int lastRoute) {

	if (currentIfaceList == null)
	    return;

	//
	// Scan local interfaces
	//

	int ifaceListSize = currentIfaceList.size();
	for (int j = 0; j < ifaceListSize; ++j) {
	    BeraNetworkInterface outIface = (BeraNetworkInterface) currentIfaceList.get(j);
	    List outIfaceAddrList = outIface.getAddresses();

	    int addrListSize = outIfaceAddrList.size();
	    for (int i = 0; i < addrListSize; ++i) {
		BeraInterfaceAddress outIfaceAddr = (BeraInterfaceAddress) outIfaceAddrList.get(i);
	    
		//
		// Build packet data,
		// checking split-horizon/poisoned-reverse
		// on a per-interface basis
		//
		int length = RipPacket.buildResponse(sendBuffer, routeList, firstRoute, lastRoute,
						     outIface, outIfaceAddr, ripConfig);
		BeraAssert.require(!RipPacket.badLength(length));
		
		//
		// Send packet on specific interface
		// addressed to multicast-address/RIP-port 
		//
		if (sendResponsePacket(sendBuffer, 0, length, outIfaceAddr.getAddress(), getMulticastAddress(), RIP_UDP_PORT)) {
		    logger.error("failure writing multicast batch");
		    continue;
		}
	    }
	}
    }

    private void multicastFullTable() {

	List routeList = routeTable.getRouteList();
	int totalRoutes = routeList.size();

	logger.debug("multicast full table: " + totalRoutes + " route(s)");

	//
	// Scan batches of routes
	//

	int lastRoute;
	for (int firstRoute = 0; firstRoute < totalRoutes; firstRoute = lastRoute + 1) {

	    lastRoute = firstRoute + RipPacket.PACKET_MAX_ENTRIES - 1;
	    if (lastRoute >= totalRoutes)
		lastRoute = totalRoutes - 1;

	    BeraAssert.require(firstRoute <= lastRoute);

	    logger.debug("sending batch " + firstRoute + "-" + lastRoute + " of " + totalRoutes + " total route(s)");

	    //
	    // Send batch for every local interface
	    //
	    multicastBatch(routeList, firstRoute, lastRoute);
	}
    }

    private void fullyUpdateExplicitNeighbors() {

	//
	// Some neighbors can't be reached through multicast on
	// local interfaces
	//
	// For example, neighbors on NBMA networks won't receive
	// multicasts/broadcasts
	//
	// These neighbors must be explicitely listed in the
	// RIP configuration
	//
	// This method is intended to scan that explicit neighbor
	// list and send the full RIP table for every neighbor
	//

	logger.debug("FIXME: RipServer.fullyUpdateExplicitNeighbors()");
    }

    private void sendFullTable(InetAddress dstAddr, int dstPort) {

	List routeList = routeTable.getRouteList();
	int totalRoutes = routeList.size();

	//
	// Send batches of routes
	//

	int lastRoute;
	for (int firstRoute = 0; firstRoute < totalRoutes; firstRoute = lastRoute + 1) {

	    lastRoute = firstRoute + RipPacket.PACKET_MAX_ENTRIES - 1;
	    if (lastRoute >= totalRoutes)
		lastRoute = totalRoutes - 1;

	    BeraAssert.require(firstRoute <= lastRoute);

	    logger.debug("sending batch " + firstRoute + "-" + lastRoute + " of " + totalRoutes + " to " + dstAddr.getHostAddress() + ":" + dstPort);

	    //
	    // Build packet data
	    //
	    int length = RipPacket.buildResponse(sendBuffer, routeList, firstRoute, lastRoute);
	    BeraAssert.require(!RipPacket.badLength(length));

	    //
	    // Send packet on all interfaces to given addr/port
	    //
	    if (sendResponsePacket(sendBuffer, 0, length, getWildcardAddress(), dstAddr, dstPort)) {
		logger.error("failure writing response packet");
		continue;
	    }
	}
    }

    private void sendPeriodicUpdate() {
	logger.debug("sending periodic update");
	multicastFullTable();
	fullyUpdateExplicitNeighbors();
	clearTriggerUpdate();
    }

    private void sendTriggeredUpdate() {
	logger.debug("FIXME: sendTriggeredUpdate()");

	List routeList = routeTable.getRouteList();
	int totalRoutes = routeList.size();

	//
	// Disable route changed flag
	//
	for (int i = 0; i < totalRoutes; ++i) {
	    RipRoute r = (RipRoute) routeList.get(i);
	    r.updateSent();
	}
    }

    private void clearTriggerUpdate() {
	routeTable.triggeredUpdateSent();
	triggeredUpdateTimer = ripConfig.getRipTriggerHoldtime();
    }

    private void checkTriggeredUpdate(int elapsed) {
	if (!routeTable.triggeredUpdateIsPending())
	    return;

	triggeredUpdateTimer -= elapsed;

	if (triggeredUpdateTimer < 0)
	    triggeredUpdateTimer = 0;

	if (triggeredUpdateTimer > 0)
	    return;

	sendTriggeredUpdate();

	BeraAssert.require(routeTable.triggeredUpdateIsPending());
	clearTriggerUpdate();
	BeraAssert.require(!routeTable.triggeredUpdateIsPending());
    }

    private void checkPeriodicUpdate(int elapsed) {

	periodicUpdateTimer -= elapsed;

	if (periodicUpdateTimer < 0)
	    periodicUpdateTimer = 0;

	if (periodicUpdateTimer > 0)
	    return;

	sendPeriodicUpdate();

	periodicUpdateTimer = ripConfig.getRipUpdateInterval();
    }

    private void fetchConfig() {

	logger.debug("fetching config from MANAGER");

	String method = "manager.dumpConfig";
	logger.debug("calling XMLRPC method " + method + " on URL " + BeraServices.MANAGER_URL);

	XmlRpcClientLite client;
	try {
	    client = new XmlRpcClientLite(BeraServices.MANAGER_URL);
	}
	catch (Exception e) {
            logger.error("could not create XML-RPC client: " + e);
	    return;
        }

	Object result;
	Vector params = new Vector();
	try {
	    result = client.execute(method, params);
	}
	catch (Exception e) {
            logger.error("could not call XML-RPC method: " + e);
            return;
        }

	Hashtable configMap = (Hashtable) result;
	if (configMap == null) {
	    logger.error("bad config dump from manager");
	    ripConfig.update(null);
	    return;
	}

	Hashtable routerMap = (Hashtable) configMap.get("router");
	if (routerMap == null) {
	    logger.error("missing router section in config dump");
	    ripConfig.update(null);
	    return;
	}

	Hashtable ripMap = (Hashtable) routerMap.get("rip");
	if (ripMap == null) {
	    logger.error("missing rip section in config dump");
	    ripConfig.update(null);
	    return;
	}

	Hashtable instanceMap = (Hashtable) ripMap.get(ripInstanceLabel);
	if (instanceMap == null) {
	    logger.error("missing rip instance in config dump");
	    ripConfig.update(null);
	    return;
	}

	logger.debug("rip config instance loaded");

	ripConfig.update(instanceMap);
    }

    private void scanInterfaces() {
	// Scan interfaces
	//
	// Methods must properly handle a null
	// interface list; otherwise, RIP will
	// crash if FEA is unavailable.
	//
	Vector currentInterfaceVector = BeraNetworkInterface.feaFetchInterfaces();
	currentIfaceList = BeraNetworkInterface.makeInterfaceList(currentInterfaceVector);
    }

    private void handlePeriodicTask(int elapsed) {
	//
	// BEGIN: There is no RipTask thread running
	//
	// Thus, the main loop thread can access periodicTaskIsPending
	// without synchronizing
	//
	logger.debug("periodic task - beginnig execution");

	fetchConfig();

	scanInterfaces();

	// Update route timers
	routeTable.ageRoutes(elapsed);

	// Expire routes for deleted addresses
	routeTable.expireDeadConnectedAddresses(this);

	// Discover connected routes
	findNewRoutes();

	// Add new routes to RIB
	routeTable.sendRibRoutes();

	// Remove expired routes from RIB
	routeTable.withdrawExpiredRoutes();

	// Delete dead routes from RIP route table
	routeTable.deleteDeadRoutes();

	checkPeriodicUpdate(elapsed);
	
	checkTriggeredUpdate(elapsed);

	logger.debug("periodic task - finished");
	//
	// END: There is no RipTask thread running
	//
    }

    private void checkPeriodicTask() {

	// Both main loop thread and RipTask thread can access periodicTaskIsPending
	synchronized (periodicTaskIsPending) {
	    if (!periodicTaskIsPending.value)
		return;
	}

	int pulseInterval = ripConfig.getRipPulseInterval();

	handlePeriodicTask(pulseInterval);

	// Periodic task performed
	periodicTaskIsPending.value = false;
	
	new RipTask(this, ripTimer, pulseInterval); // the constructor schedules the one-time task
	eventLoopBlocker.eventHandled();
    }

    // Callback for multicast listener thread.
    void multicastPacketCall(DatagramPacket multicastPacket) {

	//logger.debug("multicast packet call");

	// wake event loop thread
	eventLoopBlocker.eventAdded();

	// block multicast listener thread
	multicastBlocker.put(multicastPacket);
    }

    private boolean selfAddress(InetAddress addr) {

	if (currentIfaceList == null)
	    return false;

	for (Iterator addrIt = new BeraInterfaceAddressIterator(currentIfaceList);
	     addrIt.hasNext(); ) {
	    BeraInterfaceAddress ifaceAddr = (BeraInterfaceAddress) addrIt.next();
	    if (ifaceAddr.getAddress().equals(addr))
		return true;
	}

	return false;
    }

    boolean findConnectedNetwork(InetAddress netAddr, int prefixLength) {

	if (currentIfaceList == null)
	    return false;

	for (Iterator addrIt = new BeraInterfaceAddressIterator(currentIfaceList);
	     addrIt.hasNext(); ) {
	    BeraInterfaceAddress ifaceAddr = (BeraInterfaceAddress) addrIt.next();
	    if (ifaceAddr.sameNetwork(netAddr, prefixLength))
		return true;
	}

	return false;
    }

    // Check if 'addr' belongs to a directly connected network
    private boolean neighborAddress(InetAddress addr) {

	if (currentIfaceList == null)
	    return false;

	for (Iterator addrIt = new BeraInterfaceAddressIterator(currentIfaceList);
	     addrIt.hasNext(); ) {
	    BeraInterfaceAddress ifaceAddr = (BeraInterfaceAddress) addrIt.next();
	    if (ifaceAddr.sameNetwork(addr))
		return true;
	}

	return false;
    }

    private int findInterfaceCost(InetAddress addr) {
	logger.debug("FIXME: find local interface cost for source address: " + addr);
	return 1;
    }

    private void handleMulticastPacket(DatagramPacket multicastPacket) {

	int srcPort = multicastPacket.getPort();
	InetAddress srcAddr = multicastPacket.getAddress();

	logger.debug("packet received from: " + srcAddr.getHostAddress() + ":" + srcPort);

	byte[] packetBuffer = multicastPacket.getData();
	int packetBufferLength = multicastPacket.getLength();

	RipPacket ripPacket = RipPacket.parse(packetBuffer, packetBufferLength);
	if (ripPacket == null) {
	    logger.error("ignoring bad packet");
	    return;
	}

	int command = ripPacket.getCommand();

	if (command == RipPacket.CMD_REQUEST) {
	    //logger.debug("handling received rip multicast REQUEST packet");

	    // RFC 2453, section 3.9.1 prohibits requests from RIP port
	    if (srcPort == RIP_UDP_PORT) {
		logger.debug("ignoring request packet from rip port");
		return;
	    }

	    //
	    // Is this a request for full routing table?
	    //

	    List entryList = ripPacket.getEntryList();
	    int size = entryList.size();
	    if (size == 1) {
		RipPacketEntry entry = (RipPacketEntry) entryList.get(0);
		if ((entry.entryAddressFamily == 0) && (entry.entryMetric == RipPacket.METRIC_INFINITY)) {
		    logger.debug("sending FULL RESPONSE back");
		    sendFullTable(srcAddr, srcPort);
		    return;
		}
	    }

	    //
	    // Scan packet entries, modifying the received buffer
	    //

	    for (int i = 0; i < size; ++i) {
		RipPacketEntry entry = (RipPacketEntry) entryList.get(i);

		InetAddress netAddr;
		try {
		    netAddr = InetAddress.getByAddress(entry.entryAddress);
		}
		catch (UnknownHostException e) {
		    logger.error("could not parse address for packet entry");
		    continue;
		}

		int prefixLength = Conversion.maskToLength(entry.entrySubnetMask);

		int         responseMetric;
		InetAddress responseNextHop;

		RipRoute route = routeTable.findRoute(netAddr, prefixLength);
		if (route == null) {
		    responseMetric = RipPacket.METRIC_INFINITY;
		    responseNextHop = null;
		}
		else {
		    responseMetric = route.getMetric();
		    responseNextHop = route.getNextHop(); // may be null
		}

		if (responseNextHop == null)
		    responseNextHop = RipServer.getWildcardAddress();

		byte[] nextHopBuf = responseNextHop.getAddress();

		// The change of next hop may be a violation of RFC 2453
		// This is done because we want to be able to use rip
		// queries for debug purposes
		for (int j = 0; j < nextHopBuf.length; ++j) {
		    int off = RipPacket.PACKET_HEADER_SIZE + 
			(i * RipPacket.PACKET_ENTRY_SIZE) +
			RipPacket.PACKET_ENTRY_NEXTHOP_OFFSET;
		    packetBuffer[off] = nextHopBuf[j];
		}

		Conversion.intToByte(packetBuffer,
				     RipPacket.PACKET_HEADER_SIZE +
				     (i * RipPacket.PACKET_ENTRY_SIZE) +
				     RipPacket.PACKET_ENTRY_METRIC_OFFSET, 
				     responseMetric);

		logger.debug("response metric set to: " + responseMetric);
	    }

	    packetBuffer[0] = RipPacket.CMD_RESPONSE;

	    if (sendResponsePacket(packetBuffer, 0, packetBufferLength, getWildcardAddress(), srcAddr, srcPort)) {
		logger.error("failure writing query response packet");
		return;
	    }

	    return;
	}

	if (command == RipPacket.CMD_RESPONSE) {
	    //logger.debug("handling received rip multicast RESPONSE packet");

	    if (selfAddress(srcAddr)) {
		logger.debug("ignoring packet from myself");
		return;
	    }

	    if (!neighborAddress(srcAddr)) {
		logger.debug("ignoring packet from non-neighbor");
		return;
	    }

	    if (srcPort != RIP_UDP_PORT) {
		logger.debug("ignoring response packet from non-rip port");
		return;
	    }

	    int ifaceCost = findInterfaceCost(srcAddr);

	    //
	    // Scan packet entries
	    //

	    List entryList = ripPacket.getEntryList();
	    int size = entryList.size();
	    for (int i = 0; i < size; ++i) {
		RipPacketEntry entry = (RipPacketEntry) entryList.get(i);

		int metric = entry.entryMetric + ifaceCost;
		if (metric > RipPacket.METRIC_INFINITY)
		    metric = RipPacket.METRIC_INFINITY;

		InetAddress nextHop;
		try {
		    nextHop = InetAddress.getByAddress(entry.entryNextHop);

		    if (nextHop.equals(getWildcardAddress()))
			nextHop = srcAddr;
		    else if (!neighborAddress(nextHop))
			nextHop = srcAddr;
		}
		catch (UnknownHostException e) {
		    logger.error("could not interpret packet entry nexthop");
		    nextHop = srcAddr;
		}

		routeTable.addRoute(entry.entryAddress, entry.entrySubnetMask, metric, nextHop);
	    }

	    return;
	}

	logger.error("ignoring packet with unknown command: " + command);
    }

    private void checkMulticastPacket() {

	// Is there a multicast packet pending?
	DatagramPacket multicastPacket = (DatagramPacket) multicastBlocker.get();
	if (multicastPacket == null)
	    return;

	handleMulticastPacket(multicastPacket);

	multicastBlocker.release();
	eventLoopBlocker.eventHandled();
    }

    private static MulticastSocket makeMulticastServerSocket() {

	InetAddress multicastAddress;
        try {
	    multicastAddress = InetAddress.getByName(RipServer.RIP_MULTICAST_ADDRESS);
        }
        catch (UnknownHostException e) {
            logger.error("could not solve multicast address: " + e);
	    return null;
        }

	MulticastSocket multicastSocket;
	try {
            multicastSocket = new MulticastSocket(RipServer.RIP_UDP_PORT);
	}
	catch (SocketException e) {
            logger.error("could not create server multicast socket: " + e);
            return null;
	}
	catch (IOException e) {
            logger.error("could not create server multicast socket: " + e);
            return null;
	}

	try {
	    multicastSocket.setLoopbackMode(true); // disable
	}
	catch (SocketException e) {
	    logger.warn("could not disable multicast socket loopback mode: " + e);
	}

	logger.debug("multicast socket bound to address: " + RipServer.RIP_MULTICAST_ADDRESS + ":" + RipServer.RIP_UDP_PORT);

	return multicastSocket;
    }

    public void run() {

	logger.info("starting");

	//
	// Create multicast sender
	//
	multicastSenderSocket = makeMulticastServerSocket();
	if (multicastSenderSocket == null) {
	    logger.error("could not create multicast sender socket");
	    return;
	}

	//
	// Fetch currentIfaceList from FEA
	//
	scanInterfaces();

	Iterator addrIt = new BeraInterfaceAddressIterator(currentIfaceList);

	//
	// Start multicast listener
	//
	try {
	    multicastListener = new RipMulticastListener(this, addrIt);
	}
	catch (BeraListen e) {
	    logger.error("could not listen on multicast socket: " + e);
	    return;
	}
	multicastListener.start();

	periodicUpdateTimer = ripConfig.getRipUpdateInterval();
	triggeredUpdateTimer = ripConfig.getRipTriggerHoldtime();

	//
	// Start periodic task
	//
	// the RipTask constructor schedules the one-time task
	//
	new RipTask(this, ripTimer, ripConfig.getRipPulseInterval()); 

	logger.debug("entering main event loop");
	eventLoopBlocker.blockIfEmpty();

	//
	// Rip Main Event Loop
	//
	for (;;) {
	    checkPeriodicTask();
	    checkMulticastPacket();
	}

    }
}

class EventLoopBlocker {

    static private Logger logger = Logger.getLogger(EventLoopBlocker.class);

    private int pendingEvents = 0;

    synchronized void blockIfEmpty() {
	BeraAssert.require(pendingEvents >= 0);

	if (pendingEvents != 0)
	    return;

	logger.debug("no pending event, going to sleep");
	
	// Blocks event loop thread, until some other thread 
	// [periodic task or packet receiving] calls eventAdded()
	try {
	    wait();
	}
	catch (InterruptedException e) {
	}

	logger.debug("waking up due to new event");

	BeraAssert.require(pendingEvents > 0);
    }

    // Periodic task thread and packet receiving threads use this 
    // to signal the event loop thread of a new pending event.
    synchronized void eventAdded() {
	BeraAssert.require(pendingEvents >= 0);
	++pendingEvents;
        notify();
    }

    // Event loop thread uses this to signal handling of an event,
    // and possibly block waiting for events.
    synchronized void eventHandled() {
	BeraAssert.require(pendingEvents > 0);
	--pendingEvents;
	blockIfEmpty();
    }
}

class ThreadBlocker {

    private Object slot = null;

    // Auxiliary thread puts data and blocks
    // until event loop calls release()
    synchronized void put(Object obj) {
	BeraAssert.require(obj != null);
	BeraAssert.require(slot == null);
	slot = obj;
	try { wait(); } catch (InterruptedException e) { }
    }

    // Main event loop thread consumes data
    synchronized Object get() {
	return slot;
    }

    // Main event loop unblocks auxiliary thread
    synchronized void release() {
	BeraAssert.require(slot != null);
	slot = null;
	notify();
    }
}

class Bool {
    boolean value;

    Bool(boolean v) {
	value = v;
    }
}
